package com.bizunited.platform.user.service.feign;

import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.common.interceptor.FeignRequestInterceptor;
import com.bizunited.platform.user.common.vo.PositionVo;
import com.bizunited.platform.user.service.feign.fallback.PositionFeignClientFallback;
import org.springframework.cloud.openfeign.FeignClient;
import org.springframework.data.repository.query.Param;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;

import java.security.Principal;
import java.util.List;

/**
 * 岗位的feign远程调用
 * @Author: zengxingwang
 * @Date: 2020/3/22 20:23
 */
@FeignClient(qualifier = "PositionFeignClient", name = "${user.application.name}", path = "/v1/nebula/position", fallback = PositionFeignClientFallback.class, configuration = FeignRequestInterceptor.class)
public interface PositionFeignClient {

  /**
   * 按照条件搜索岗位
   * @param nameEqual
   * @param tstatus
   * @param flag
   * @param ids
   * @param codes
   * @param name
   * @param userName
   * @param positionLevel
   * @param orgName
   * @param page
   * @param size
   * @return
   */
  @GetMapping("/findByConditions")
  ResponseModel findByConditions(@RequestParam("nameEqual") String nameEqual,
                                 @RequestParam("tstatus") Integer tstatus,
                                 @RequestParam("flag") Integer flag,
                                 @RequestParam("ids") List<String> ids,
                                 @RequestParam("codes") List<String> codes,
                                 @RequestParam("name") String name,
                                 @RequestParam("userName") String userName,
                                 @RequestParam("positionLevel") String positionLevel,
                                 @RequestParam("orgName") String orgName,
                                 @RequestParam("page") int page,
                                 @RequestParam("size") int size);

  /**
   * 根据ID查询岗位的详细信息，包括相关组织、角色、用户等
   * @param positionId
   * @return
   */
  @GetMapping("/find/{positionId}")
  ResponseModel findDetailsById(@PathVariable("positionId") String positionId);

  /**
   * 添加岗位
   * @param position
   * @return
   */
  @PostMapping("")
  ResponseModel create(@RequestBody PositionVo position);

  /**
   * 修改岗位
   * @param position
   * @return
   */
  @PatchMapping("update")
  ResponseModel update(@RequestBody PositionVo position);

  /**
   * 修改岗位状态（启用禁用）
   * @param positionId
   * @return
   */
  @PatchMapping("updateStatus/{positionId}")
  ResponseModel updateStatus(@PathVariable("positionId") String positionId);

  /**
   * 绑定指定的岗位和指定的组织机构
   * @param orgId
   * @param positionId
   */
  @PatchMapping("/bindOrg")
  ResponseModel bindOrg(@RequestParam(name = "positionId") String positionId, @RequestParam(name = "orgId") String orgId);

  /**
   * 解绑指定的岗位和任何组织机构的绑定关系
   * @param positionId
   */
  @PatchMapping("/unbindOrg")
  ResponseModel unbindOrg(@RequestParam(name = "positionId") String positionId);

  /**
   * 在忽略指定岗位是否已经绑定组织机构的前提下，重新为这个指定的岗位绑定新的组织机构
   * @param orgId
   * @param positionId
   */
  @PatchMapping("/reBindOrg")
  ResponseModel reBindOrg(@RequestParam(name = "orgId") String orgId, @RequestParam(name = "positionId") String positionId);
  
  /**
   * 将指定一个岗位和指定的多个用户形成绑定关系
   * @param userIds
   * @param positionId
   */
  @PatchMapping("/bindUsersAndPosition")
  ResponseModel bindUsers(@RequestParam(name = "positionId") String positionId,
                          @RequestParam(name = "userIds") String[] userIds);

  /**
   * 解绑多个用户与某一指定的岗位
   * @param userIds
   * @param positionId
   */
  @PatchMapping("/unbindUsersAndPosition")
  ResponseModel unbindUsersAndPosition(@RequestParam(name = "positionId") String positionId,
                                       @RequestParam(name = "userIds") String[] userIds);

  /**
   * 预制控件查询，查询用户岗位相关信息
   * @param userId
   * @return
   */
  @GetMapping("/findByUserId")
  ResponseModel findByUserId(@RequestParam("userId") String userId);

  /**
   * 预制控件查询，查询用户岗位相关信息
   * @param userAccount
   * @return
   */
  @GetMapping("/findByUserAccount")
  ResponseModel findByUserAccount(@RequestParam("userAccount") String userAccount);

  /**
   * 查询多个用户岗位相关信息
   * @param ids
   * @return
   */
  @PostMapping("/findByIds")
  ResponseModel findByIds(@RequestBody List<String> ids);

  /**
   * 根据岗位编码查询
   *
   * @param code
   * @return
   */
  @GetMapping("findByCode")
  ResponseModel findByCode(@RequestParam("code") String code);

  /**
   * 切换岗位
   *
   * @param positionId
   * @param userId
   * @return
   */
  @PostMapping("changePosition")
  ResponseModel changePosition(@RequestParam(name = "positionId") String positionId,
                               @RequestParam(name = "userId") String userId);

  /**
   * 查询Redis中主岗位，如果redis中不存在，则默认主岗位为数据库查询的第一条数据，并且将该数据存入redis中
   *
   * @param userId
   * @return
   */
  @GetMapping("/findMainPositionByUserId")
  ResponseModel findMainPositionByUserId(@RequestParam(name = "userId") String userId);

  /**
   * 重新解绑某一角色与某一岗位关系
   * @param positionId
   * @param userIds
   */
  @PatchMapping("/rebindUsers")
  ResponseModel rebindUsers(@RequestParam(name = "positionId") String positionId, @RequestParam(name = "userIds") String[] userIds);

  /**
   * 查询所有岗位信息（根据创建时间排序）
   * @return
   */
  @GetMapping("/findAll")
  ResponseModel findAll();

  /**
   * 根据条件查询所有岗位
   * @param status
   * @return
   */
  @GetMapping("findByStatus")
  ResponseModel findByStatus(@RequestParam(name = "status") Integer status);

  /**
   * 根据登陆人关联的岗位查询
   * @param type
   * @return
   */
  @GetMapping("findByPrincipal")
  ResponseModel findByPrincipal(@RequestParam("type") Integer type, @RequestParam("principal") Principal principal);

  /**
   * 根据登陆人关联的岗位和输入层级查询控件
   * @param type
   * @param principal
   * @param level
   * @return
   */
  @GetMapping("findByTypeAndLevel")
  ResponseModel findByTypeAndLevel(@RequestParam("type") Integer type, @RequestParam("principal") Principal principal, @RequestParam("level") Integer level);

  /**
   * 根据职级创建相关职位
   * @param levelCode
   * @return
   */
  @GetMapping("createByPositionLevelCode")
  ResponseModel createByPositionLevelCode(@RequestParam("levelCode") String levelCode);

  /**
   * 根据组织编码查询职位（包括关联信息）
   * @param orgCode
   * @return
   */
  @GetMapping("findDetailsByOrgCode")
  ResponseModel findDetailsByOrgCode(@RequestParam("orgCode") String orgCode);

  /**
   * 根据编码查询职位的所有下级信息
   * @param code
   * @return
   */
  @GetMapping("findDetailsByCode")
  ResponseModel findDetailsByCode(@RequestParam("code") String code);

  /**
   * 根据编码查询职位详情，包含组织信息、用户信息
   * @param code
   * @return
   */
  @GetMapping("findPositionDetailByCode")
  ResponseModel findPositionDetailByCode(@RequestParam("code") String code);

  /**
   * 根据职级批量查询关联的职位
   * @param positionLevelIds 职级主键列表
   * @return
   */
  @GetMapping("findByPositionLevelIds")
  ResponseModel findByPositionLevelIds(@Param("positionLevelIds") String[] positionLevelIds);

  /**
   * 根据职位Id批量禁用职位
   * @param ids 职位Id列表
   */
  @PostMapping("disableByIds")
  ResponseModel disableByIds(@Param("ids")String[] ids);

  /**
   * 根据职位Id批量启用职位
   * @param ids 职位Id列表
   */
  @PostMapping("enableByIds")
  ResponseModel enableByIds(@Param("ids")String[] ids);

  /**
   * 多个职位重新绑定组织
   * @param codes
   * @return
   */
  @PatchMapping("reBindOrgCode")
  ResponseModel reBindOrgCode(@RequestParam("codes") String[] codes, @RequestParam("orgCode") String orgCode);

  /**
   * 根据职级分页查询职位，带出用户拓展以及组织
   * @param name
   * @param orgName
   * @param userName
   * @param positionLevel
   * @param page
   * @param size
   * @return
   */
  @GetMapping("findByPositionLevelAndConditions")
  ResponseModel findByPositionLevelAndConditions(@RequestParam("name") String name,
                                                 @RequestParam("orgName") String orgName,
                                                 @RequestParam("userName") String userName,
                                                 @RequestParam("positionLevel") String positionLevel,
                                                 @RequestParam("page") int page,
                                                 @RequestParam("size") int size);

  /**
   * 根据用户id查询所有下属职位
   * @param userId
   * @return
   */
  @GetMapping("findByUserIdStrDescendant")
  ResponseModel findByUserIdStruDescendant(@RequestParam("userId") String userId);

  /**
   * 绑定上级职位
   * @param id
   * @param parentId
   * @return
   */
  @PatchMapping("bindParent")
  ResponseModel bindParent(@RequestParam("id") String id, @RequestParam("parentId") String parentId);
}
