package com.bizunited.platform.user.service.feign;

import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.common.interceptor.FeignRequestInterceptor;
import com.bizunited.platform.user.common.vo.UserVo;
import com.bizunited.platform.user.service.feign.fallback.UserFeignClientFallback;
import org.springframework.cloud.openfeign.FeignClient;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;

import java.util.Date;
import java.util.List;

/**
 * 用户的feign远程调用
 * @Author: zengxingwang
 * @Date: 2020/3/22 20:19
 */
@FeignClient(qualifier = "UserFeignClient", name = "${user.application.name}", path = "/v1/nebula/users", fallback = UserFeignClientFallback.class, configuration = FeignRequestInterceptor.class)
public interface UserFeignClient {

  /**
   * 添加一个新的用户信息（后台管理人员，业务功能操作者）
   * @param user
   * @return
   */
  @PostMapping("")
  ResponseModel create(@RequestBody UserVo user);

  /**
   * 修改一个已经存在的用户信息
   * @param user
   * @return
   */
  @PatchMapping("")
  ResponseModel update(@RequestBody UserVo user);

  /**
   * 重置指定用户的密码信息
   * @param userId
   * @param newPassword 没有经过加密的新的密码
   * @param oldPassword 没有经过加密的原始密码
   * @return
   */
  @PatchMapping("/updatePassword")
  ResponseModel updatePassword(@RequestParam("userId") String userId,
                               @RequestParam("oldPassword") String oldPassword,
                               @RequestParam("newPassword") String newPassword);

  /**
   * 修改指定用户的密码信息
   * @param userId
   * @param newPassword 没有经过加密的新的密码
   * @return
   */
  @PatchMapping("updatePasswordIgnoreOld")
  ResponseModel updatePasswordIgnoreOld(@RequestParam("userId") String userId,
                                        @RequestParam("newPassword") String newPassword);

  /**
   * 根据账号修改状态
   * @param account
   * @param flag
   * @return
   */
  @PatchMapping("/updateStatus")
  ResponseModel updateStatus(@RequestParam("account") String account,
                             @RequestParam("flag") boolean flag);

  /**
   * 查询当前用户信息，且包括当前用户已经绑定（包括间接绑定）的角色信息、用户组、组织机构和岗位
   * @param userId 指定的用户id
   * @return
   */
  @GetMapping("/{userId}")
  ResponseModel findDetailsById(@PathVariable("userId") String userId);

  /**
   * 查询当前用户信息(按照用户账号查询)，不包括当前用户已经绑定（包括间接绑定）的角色信息、用户组、组织机构和岗位
   * @param account 账户名
   * @return
   */
  @GetMapping("/findByAccount")
  ResponseModel findByAccount(@RequestParam("account") String account);

  /**
   * 查询登录用户信息
   * @return
   */
  @GetMapping("/findByPrincipal")
  ResponseModel findByPrincipal();

  /**
   * 预制控件查询，暂时查询当前操作用户基础信息
   * @param userId
   * @return
   */
  @GetMapping("/findByUserId")
  ResponseModel findByUserId(@RequestParam("userId") String userId);

  /**
   * 按照条件搜索用户，支持模糊查询
   * @param userVo
   * @param page
   * @param size
   * @return
   */
  @GetMapping("/findByConditions")
  ResponseModel findByConditions(
      @RequestParam("userName") String userName,
      @RequestParam("account") String account,
      @RequestParam("idcard") String idcard,
      @RequestParam("phone") String phone,
      @RequestParam("createTime") Date createTime,
      @RequestParam("entryTime") Date entryTime,
      @RequestParam("lastloginTime") Date lastloginTime,
      @RequestParam("useStatus") Integer useStatus,
      @RequestParam("userHead") String userHead,
      @RequestParam("gender") Integer gender,
      @RequestParam("extend1") String extend1,
      @RequestParam("extend2") String extend2,
      @RequestParam("extend3") String extend3,
      @RequestParam("extend4") String extend4,
      @RequestParam("extend5") String extend5,
      @RequestParam("extend6") String extend6,
      @RequestParam("extend7") String extend7,
      @RequestParam("extend8") String extend8,
      @RequestParam("extend9") String extend9,
      @RequestParam("extend10") String extend10,
      @RequestParam("page") Integer page,
      @RequestParam("size") Integer size);

  /**
   * 查询多个用户信息
   * @param ids
   * @return
   */
  @PostMapping("/findByIds")
  ResponseModel findByIds(@RequestBody List<String> ids);

  /**
   * 根据用户账号与名称模糊查询
   * @param param
   * @return
   */
  @GetMapping("/findByAccountLikeOrNameLike")
  ResponseModel findByAccountLikeOrNameLike(@RequestParam("param") String param);

  /**
   * 按照指定的手机号，查询账户基本信息，该方法的查询不包括任何关联信息，例如角色信息。
   *
   * @param phone 指定的用户手机号
   * @return
   */
  @GetMapping("/findByPhone")
  ResponseModel findByPhone(@RequestParam(name = "phone") String phone);

  /**
   * 用户启用
   *
   * @param account
   * @return
   */
  @PatchMapping("/enabled")
  ResponseModel enabled(@RequestParam(name = "account") String account);

  /**
   * 用户禁用
   *
   * @param account
   * @return
   */
  @PatchMapping("/disable")
  ResponseModel disable(@RequestParam(name = "account") String account);

  /**
   * 成功登陆后，更新用户最后一次登录时间
   *
   * @param account 按照用户账号更新
   */
  @PatchMapping("/updateLastloginTime")
  ResponseModel updateLastloginTime(@RequestParam(name = "account") String account);

  /**
   * 根据组织机构ID查询绑定的用户
   * @param orgId
   * @return
   */
  @GetMapping("findByOrgId")
  ResponseModel findByOrgId(@RequestParam("orgId") String orgId);

  /**
   * 根据岗位获取绑定的用户
   * @param positionId
   * @return
   */
  @GetMapping("findByPositionId")
  ResponseModel findByPositionId(@RequestParam("positionId") String positionId);

  /**
   * 根据用户账号和获取下属的类型，查询指定用户所有的下属（子孙级）
   * @param account 用户账号
   * @param type 查询方式，1:以职位的方式获取下属
   * @return
   */
  @GetMapping("findByAccountAndTypeStruDescendant")
  ResponseModel findByAccountAndTypeStruDescendant(@RequestParam("account") String account, @RequestParam("type") Integer type);

  /**
   * 判断用户密码是否默认密码
   * @param account 用户账号
   * @return
   */
  @GetMapping("/isDefaultPassword")
  ResponseModel isDefaultPassword(@RequestParam("account") String account);
}
