package com.bizunited.platform.user.service.feign;

import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.common.interceptor.FeignRequestInterceptor;
import com.bizunited.platform.user.common.vo.UserGroupVo;
import com.bizunited.platform.user.service.feign.fallback.UserGroupFeignClientFallback;
import org.springframework.cloud.openfeign.FeignClient;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;

import java.util.List;

/**
 * 用户组的feign远程调用
 * @Author: zengxingwang
 * @Date: 2020/3/22 20:21
 */
@FeignClient(qualifier = "UserGroupFeignClient", name = "${user.application.name}", path = "/v1/nebula/userGroups", fallback = UserGroupFeignClientFallback.class, configuration = FeignRequestInterceptor.class)
public interface UserGroupFeignClient {

  /**
   * 根据指定用户组名称、状态查询
   * @param groupName
   * @param status
   * @param page
   * @param size
   * @return
   */
  @GetMapping("/findByCondition")
  ResponseModel findByCondition(@RequestParam(required = false, name = "groupName") String groupName,
                                @RequestParam(required = false, name = "groupCode") String groupCode,
                                @RequestParam(required = false, name = "status") Integer status,
                                @RequestParam("page") Integer page,
                                @RequestParam("size") Integer size);

  /**
   * 新增用户组
   * @param userGroup 用户组信息
   * @return
   */
  @PostMapping("")
  ResponseModel create(@RequestBody UserGroupVo userGroup);

  /**
   * 修改用户组
   * @param userGroupVo 用户组相关信息
   * @return
   */
  @PatchMapping("/update")
  ResponseModel update(@RequestBody UserGroupVo userGroupVo);

  /**
   * 修改用户组状态
   * @param groupId 用户组id
   * @return
   */
  @PatchMapping("/updateStatus/{groupId}")
  ResponseModel updateStatus(@PathVariable("groupId") String groupId);

  /**
   * 用户与用户组绑定
   * @param groupId 用户组id
   * @param userIds 用户id
   */
  @PatchMapping("/bindUser")
  ResponseModel bindUser(@RequestParam("groupId") String groupId,
                         @RequestParam("userIds") String[] userIds);

  /**
   * 用户与用户组解除绑定
   * @param groupId 用户组id
   * @param userIds  用户id
   */
  @PatchMapping("/unBindUser")
  ResponseModel unBindUser(@RequestParam("groupId") String groupId,
                           @RequestParam("userIds") String[] userIds);

  /**
   * 用户组与角色绑定
   * @param groupId 用户组id
   * @param roleIds  角色id
   */
  @PatchMapping("/bindRole")
  ResponseModel bindRole(@RequestParam("groupId") String groupId,
                         @RequestParam("roleIds") String[] roleIds);

  /**
   * 用户组与角色解除绑定
   * @param groupId 用户组id
   * @param roleIds  角色id
   */
  @PatchMapping("/unBindRole")
  ResponseModel unBindRole(@RequestParam("groupId") String groupId,
                           @RequestParam("roleIds") String[] roleIds);

  /**
   * 查询指定用户组的基本详情
   * @param id 用户组id
   * @return
   */
  @GetMapping("/findDetailsById")
  ResponseModel findDetailsById(@RequestParam("id") String id);

  /**
   * 查询多个用户组基本信息
   * @param ids
   * @return
   */
  @GetMapping("/findByIds")
  ResponseModel findByIds(@RequestBody List<String> ids);

  /**
   * 根据分组名称查询
   *
   * @param groupName
   * @return
   */
  @GetMapping("findByGroupName")
  ResponseModel findByGroupName(@RequestParam(name = "groupName") String groupName);

  /**
   * 查询所有用户组信息（根据创建时间排序）
   *
   * @return
   */
  @GetMapping("findAll")
  ResponseModel findAll();

  /**
   * 按照指定的人员数据库编号，查询人员已经绑定的一个或者多个用户组信息。<p>
   * 注意：用户组信息中包括了用户组已经绑定的角色信息。
   *
   * @param userId
   * @return
   */
  @GetMapping("findByUserId")
  ResponseModel findByUserId(@RequestParam(name = "userId") String userId);

  /**
   * 根据用户组编号查询对应的用户组
   * @param groupCode
   * @return
   */
  @GetMapping("findByCode")
  ResponseModel findByCode(@RequestParam(name = "groupCode") String groupCode);

  /**
   * 用户与用户组的重新绑定
   * @param groupId
   * @param userIds
   * @return
   */
  @PatchMapping("/reBindUser")
  ResponseModel reBindUser(@RequestParam(name = "groupId") String groupId, @RequestParam(name = "userIds") String[] userIds);

  /**
   * 根据用户组编码查询多个用户组信息
   * @param groupCodes
   * @return
   */
  @GetMapping("/findByGroupCodes")
  ResponseModel findByGroupCodes(@RequestParam(name = "groupCodes") List<String> groupCodes);
}
