package com.bizunited.platform.user.service.feign.service.internal;

import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.common.util.JsonUtils;
import com.bizunited.platform.core.annotations.NebulaServiceMethod;
import com.bizunited.platform.core.annotations.ServiceMethodParam;
import com.bizunited.platform.core.service.invoke.InvokeParams;
import com.bizunited.platform.user.common.enums.RegionLevelEnum;
import com.bizunited.platform.user.common.service.region.AdministrativeRegionService;
import com.bizunited.platform.user.common.vo.AdministrativeRegionVo;
import com.bizunited.platform.user.service.feign.AdministrativeRegionFeignClient;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.Sets;
import org.apache.commons.compress.utils.Lists;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * 行政区域，通过feign远程调用的实现类
 * @Author: chenrong
 * @Date: 2020/10/14 13:57
 */
@Service("AdministrativeRegionServiceImpl")
public class AdministrativeRegionServiceImpl implements AdministrativeRegionService {

  @Autowired
  private AdministrativeRegionFeignClient administrativeRegionFeignClient;

  @Override
  public AdministrativeRegionVo create(AdministrativeRegionVo region) {
    Validate.notNull(region, "行政区域不能为空！");
    ResponseModel responseModel = administrativeRegionFeignClient.create(region);
    Validate.isTrue(responseModel.getSuccess() != null && responseModel.getSuccess(), responseModel.getErrorMsg());
    return JsonUtils.convert(responseModel.getData(), AdministrativeRegionVo.class);
  }

  @Override
  @NebulaServiceMethod(name = "AdministrativeRegionService.findDetailsById" , desc = "根据ID查询明细", scope = NebulaServiceMethod.ScopeType.READ, returnPropertiesFilterB = "..children" )
  public AdministrativeRegionVo findDetailsById(@ServiceMethodParam(name = "id") String id) {
    if(StringUtils.isBlank(id)){
      return null;
    }
    ResponseModel response = administrativeRegionFeignClient.findDetailsById(id);
    return JsonUtils.convert(response.getSuccessDataOrElse(null), AdministrativeRegionVo.class);
  }

  @Override
  public AdministrativeRegionVo findById(String id) {
    if(StringUtils.isBlank(id)){
      return null;
    }
    ResponseModel response = administrativeRegionFeignClient.findById(id);
    return JsonUtils.convert(response.getSuccessDataOrElse(null), AdministrativeRegionVo.class);
  }

  @Override
  public List<AdministrativeRegionVo> findByConditions(Map<String, Object> conditions) {
    String regionName = (String) conditions.get("regionName");
    String regionCode = (String) conditions.get("regionCode");
    String longitude = (String) conditions.get("longitude");
    String latitude = (String) conditions.get("latitude");
    ResponseModel response = administrativeRegionFeignClient.findByConditions(regionName, regionCode, longitude, latitude);
    return JsonUtils.convert(response.getSuccessDataOrElse(Lists.newArrayList()), new TypeReference<List<AdministrativeRegionVo>>() {});
  }

  @Override
  @NebulaServiceMethod(name="AdministrativeRegionService.findByConditions" , desc="多条件查询，返回树结构" , returnPropertiesFilterB = "..parent" , scope= NebulaServiceMethod.ScopeType.READ)
  public List<AdministrativeRegionVo> findByConditions(InvokeParams invokeParams) {
    invokeParams = ObjectUtils.defaultIfNull(invokeParams, new InvokeParams());
    return this.findByConditions(invokeParams.getInvokeParams());
  }

  @Override
  @NebulaServiceMethod(name="AdministrativeRegionService.findByParentIdOrConditions" , desc="根据上级组织ID或多条件查询" , returnPropertiesFilterB = "..parent" , scope= NebulaServiceMethod.ScopeType.READ)
  public List<AdministrativeRegionVo> findByParentIdOrConditions(InvokeParams invokeParams) {
    invokeParams = ObjectUtils.defaultIfNull(invokeParams, new InvokeParams());
    Map<String, Object> params = invokeParams.getInvokeParams();
    // 清除空的值
    params.entrySet().removeIf(entry -> entry.getValue() == null || (CharSequence.class.isAssignableFrom(entry.getValue().getClass()) && StringUtils.isBlank((CharSequence) entry.getValue())));
    String parentId = (String) params.get("parentId");
    params.remove("parentId");
    if(CollectionUtils.isEmpty(params)) {
      // 如果没有其他条件，则根据上级ID查询
      return this.findByParent(parentId);
    }
    return this.findByConditions(params);
  }

  @Override
  public Set<AdministrativeRegionVo> findByRegionLevel(Integer regionLevel) {
    if(regionLevel == null){
      return Sets.newHashSet();
    }
    ResponseModel response = administrativeRegionFeignClient.findByRegionLevel(regionLevel);
    return JsonUtils.convert(response.getSuccessDataOrElse(Sets.newHashSet()), new TypeReference<Set<AdministrativeRegionVo>>() {});
  }

  @Override
  @NebulaServiceMethod(name = "AdministrativeRegionService.findByOrgId", desc = "根据组织ID查询关联区域", returnPropertiesFilter = "", scope = NebulaServiceMethod.ScopeType.READ)
  public List<AdministrativeRegionVo> findByOrgId(String orgId) {
    if(StringUtils.isBlank(orgId)) {
      return Lists.newArrayList();
    }
    ResponseModel response = administrativeRegionFeignClient.findByOrgId(orgId);
    return JsonUtils.convert(response.getSuccessDataOrElse(Lists.newArrayList()), new TypeReference<List<AdministrativeRegionVo>>() {});
  }

  @Override
  @NebulaServiceMethod(name = "AdministrativeRegionService.findByParent", desc = "根据父ID查询子区域", returnPropertiesFilter = "", scope = NebulaServiceMethod.ScopeType.READ)
  public List<AdministrativeRegionVo> findByParent(@ServiceMethodParam(name = "parentId") String parentId) {
    ResponseModel response = administrativeRegionFeignClient.findByParent(parentId);
    return JsonUtils.convert(response.getSuccessDataOrElse(Lists.newArrayList()), new TypeReference<List<AdministrativeRegionVo>>() {});
  }

  @Override
  @NebulaServiceMethod(name = "AdministrativeRegionService.findByCode", desc = "根据编码查询行政区域", returnPropertiesFilter = "", scope = NebulaServiceMethod.ScopeType.READ)
  public AdministrativeRegionVo findByCode(@ServiceMethodParam(name = "code") String code) {
    if(StringUtils.isBlank(code)){
      return null;
    }
    ResponseModel response = administrativeRegionFeignClient.findByCode(code);
    return JsonUtils.convert(response.getSuccessDataOrElse(null), AdministrativeRegionVo.class);
  }

  @Override
  public Integer updateOrgMappingLevel(Integer level) {
    Validate.notNull(level, "关联层级不能为空");
    RegionLevelEnum regionLevelEnum = RegionLevelEnum.valueOfLevel(level);
    Validate.notNull(regionLevelEnum, "不支持的区域层级");
    ResponseModel responseModel = administrativeRegionFeignClient.updateOrgMappingLevel(level);
    Validate.isTrue(responseModel.getSuccess(), "更新关联区域层级失败");
    return (Integer) responseModel.getData();
  }

  @Override
  public Integer findOrgMappingLevel() {
    ResponseModel responseModel = administrativeRegionFeignClient.findOrgMappingLevel();
    return (Integer) responseModel.getSuccessDataOrElse(RegionLevelEnum.AREA.getLevel());
  }
}
