package com.bizunited.platform.user.service.feign.service.internal;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.common.util.JsonUtils;
import com.bizunited.platform.core.annotations.NebulaServiceMethod;
import com.bizunited.platform.core.annotations.ServiceMethodParam;
import com.bizunited.platform.core.service.invoke.InvokeParams;
import com.bizunited.platform.user.common.service.organization.OrganizationService;
import com.bizunited.platform.user.common.vo.OrganizationVo;
import com.bizunited.platform.user.service.feign.OrganizationFeignClient;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;

import java.security.Principal;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * 组织机构的feign远程调用实现
 * @Author: zengxingwang
 * @Date: 2020/3/22 20:11
 */
public class OrganizationServiceImpl implements OrganizationService {

  @Autowired
  private OrganizationFeignClient organizationFeignClient;

  /**
   * 添加组织机构
   * @param organization 组织机构相关信息
   * @return
   */
  @Override
  @NebulaServiceMethod(name="OrganizationService.create" , desc="组织机构创建服务源" , returnPropertiesFilter="" , scope= NebulaServiceMethod.ScopeType.WRITE)
  public OrganizationVo create(OrganizationVo organization) {
    Validate.notNull(organization, "组织机构信息不能为空，请检查");
    Validate.isTrue(StringUtils.isBlank(organization.getId()), "添加组织机构时，不能传入组织机构id,请重新添加!!");
    String orgName = organization.getOrgName();
    Validate.notBlank(orgName, "组织机构名称不能为空，请检查!!");
    Validate.notBlank(organization.getDescription(), "组织机构描述不能为空，请检查!!");
    Validate.notNull(organization.getType(), "组织机构类型不能为空，请检查!!");
    String code = organization.getCode();
    Validate.notNull(code, "组织机构编号不能为空");
    ResponseModel response = organizationFeignClient.create(organization);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
    return JsonUtils.convert(response.getSuccessDataOrElse(null), OrganizationVo.class);
  }

  /**
   * 修改组织机构
   * @param organization 组织机构相关信息
   * @return
   */
  @Override
  @NebulaServiceMethod(name="OrganizationService.update" , desc="组织机构更新服务源" , returnPropertiesFilter="" , scope= NebulaServiceMethod.ScopeType.WRITE)
  public OrganizationVo update(OrganizationVo organization) {
    Validate.notNull(organization, "组织机构信息不能为空，请检查");
    Validate.notBlank(organization.getId(), "修改组织机构时，id不能为空,请重新添加!!");
    String orgName = organization.getOrgName();
    Validate.notBlank(orgName, "组织机构名称不能为空，请检查!!");
    Validate.notBlank(organization.getDescription(), "组织机构描述不能为空，请检查!!");
    ResponseModel response = organizationFeignClient.update(organization);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
    return JsonUtils.convert(response.getSuccessDataOrElse(null), OrganizationVo.class);
  }

  /**
   * 修改组织机构状态---(启用、禁用)
   * @param orgId 组织机构id
   * @return
   */
  @Override
  public OrganizationVo updateStatus(String orgId) {
    Validate.notBlank(orgId, "组织机构id不能为空，请检查");
    ResponseModel response = organizationFeignClient.updateStatus(orgId);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
    return JsonUtils.convert(response.getSuccessDataOrElse(null), OrganizationVo.class);
  }

  /**
   * 批量启用
   * @param orgIds 组织机构ids
   * @return
   */
  @Override
  public Set<OrganizationVo> enables(String[] orgIds) {
    Validate.notEmpty(orgIds, "组织机构id不能为空，请检查");
    ResponseModel response = organizationFeignClient.enables(orgIds);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
    return JsonUtils.convert(response.getSuccessDataOrElse(Sets.newHashSet()), new TypeReference<Set<OrganizationVo>>() {});
  }

  /**
   * 批量禁用
   * @param orgIds 组织机构ids
   * @return
   */
  @Override
  public Set<OrganizationVo> disables(String[] orgIds) {
    Validate.notEmpty(orgIds, "组织机构id不能为空，请检查");
    ResponseModel response = organizationFeignClient.disables(orgIds);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
    return JsonUtils.convert(response.getSuccessDataOrElse(Sets.newHashSet()), new TypeReference<Set<OrganizationVo>>() {});
  }

  /**
   * 对一个指定的currentOrgId进行其父级组织机构parentOrgId的绑定操作（原有currentOrgId的父级信息将会被取消）
   * @param currentOrgId
   * @param parentOrgId
   */
  @Override
  public void bindParent(String currentOrgId, String parentOrgId) {
    Validate.notBlank(currentOrgId, "指定的组织机构必须传入!!");
    Validate.notBlank(parentOrgId, "指定的父级组织机构必须传入!!");
    Validate.isTrue(!StringUtils.equals(currentOrgId, parentOrgId), "绑定组织机构时，不能自己绑定自己!!");
    ResponseModel response = organizationFeignClient.bindParent(currentOrgId, parentOrgId);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
  }

  /**
   * 对一个指定的currentOrgId取消其当前父级关系的绑定信息
   * @param currentOrgId
   */
  @Override
  public void unbindParent(String currentOrgId) {
    Validate.notBlank(currentOrgId, "进行绑定结构消除时，必须指定当前的组织机构信息!!");
    ResponseModel response = organizationFeignClient.unbindParent(currentOrgId);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
  }

  @Override
  public void rebindParent(String currentOrgId, String parentOrgId) {
    ResponseModel response = organizationFeignClient.reBindParent(currentOrgId, parentOrgId);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
  }
  /**
   * 将指定组织机构（currentOrgId）和指定的一个/多个用户数据编号（userId）形成绑定关系——一个用户只能属于一个组织机构
   * @param currentOrgId
   * @param userIds      指定的一个/多个用户数据编号
   */
  @Override
  public void bindUsers(String currentOrgId, String[] userIds) {
    Validate.notBlank(currentOrgId, "进行组织机构-人员绑定时，必须指定当前的组织机构信息!!");
    Validate.isTrue(userIds != null && userIds.length > 0, "进行组织机构-人员绑定时,至少传入一个人员编号信息!!");
    ResponseModel response = organizationFeignClient.bindUsers(currentOrgId, userIds);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
  }

  @Override
  public void rebindUsers(String currentOrgId, String[] userIds) {
    Validate.notBlank(currentOrgId, "进行组织机构-人员绑定时，必须指定当前的组织机构信息!!");
    ResponseModel response = organizationFeignClient.rebindUsers(currentOrgId, userIds);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
  }

  /**
   * 解除指定用户和指定组织机构的绑定关系
   * @param currentOrgId 指定的组织机构数据库编号
   * @param userId       指定的人员编号
   */
  @Override
  public void unbindUsers(String currentOrgId, String[] userId) {
    Validate.notBlank(currentOrgId, "进行组织机构-人员解绑时，必须指定当前的组织机构信息!!");
    Validate.notNull(userId, "进行组织机构-人员解绑时，必须指定当前的人员信息!!");
    ResponseModel response = organizationFeignClient.unbindUsers(currentOrgId, userId);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
  }

  /**
   * 查询目前没有指定父级组织机构的所有组织机构信息，无论其状态是否正常（status无论什么值）
   * @return
   */
  @Override
  public List<OrganizationVo> findByNullParent() {
    ResponseModel response = organizationFeignClient.findByNullParent();
    return JsonUtils.convert(response.getSuccessDataOrElse(Lists.newArrayList()), new TypeReference<List<OrganizationVo>>() {});
  }

  /**
   * 查询指定父级组织机构下的所有子级组织机构信息，无论其状态是否正常（status无论什么值）
   * @param parentId
   * @return
   */
  @Override
  public List<OrganizationVo> findByParent(String parentId) {
    ResponseModel response = organizationFeignClient.findByParent(parentId);
    return JsonUtils.convert(response.getSuccessDataOrElse(Lists.newArrayList()), new TypeReference<List<OrganizationVo>>() {});
  }

  @Override
  public List<OrganizationVo> findByParentAndFilterConditions(InvokeParams invokeParams) {
    invokeParams = ObjectUtils.defaultIfNull(invokeParams, new InvokeParams());
    Map<String, Object> parameters = invokeParams.getInvokeParams();
    String parentId = parameters.get("parentId") == null ? null : parameters.get("parentId").toString();
    String id = parameters.get("id") == null ? null : parameters.get("id").toString();
    String code = parameters.get("code") == null ? null : parameters.get("code").toString();
    String orgName = parameters.get("orgName") == null ? null : parameters.get("orgName").toString();
    ResponseModel response = organizationFeignClient.findByParentAndFilterConditions(parentId,id, code, orgName);
    return JsonUtils.convert(response.getSuccessDataOrElse(Lists.newArrayList()), new TypeReference<List<OrganizationVo>>() {});
  }

  @Override
  @NebulaServiceMethod(name = "OrganizationService.findTreeByConditions", desc = "根据条件查询组织机构树", returnPropertiesFilter = "parent", scope = NebulaServiceMethod.ScopeType.READ)
  public List<OrganizationVo> findTreeByConditions(InvokeParams invokeParams) {
    invokeParams = ObjectUtils.defaultIfNull(invokeParams, new InvokeParams());
    Map<String, Object> parameters = invokeParams.getInvokeParams();
    String id = parameters.get("id") == null ? null : parameters.get("id").toString();
    String code = parameters.get("code") == null ? null : parameters.get("code").toString();
    String orgName = parameters.get("orgName") == null ? null : parameters.get("orgName").toString();
    Integer tstatus = parameters.get("tstatus") == null ? null : (Integer)parameters.get("tstatus");
    ResponseModel response = organizationFeignClient.findTreeByConditions(id, code, orgName, tstatus);
    return JsonUtils.convert(response.getSuccessDataOrElse(Lists.newArrayList()), new TypeReference<List<OrganizationVo>>() {});
  }

  /**
   * 查询目前没有指定父级组织机构的所有组织机构信息，并且依据其状态进行查询
   * @param status
   * @return
   */
  @Override
  @NebulaServiceMethod(name="OrganizationService.findByNullParentAndStatus" , desc="根据状态查询组织机构的最顶级数据" , returnPropertiesFilter="" , scope= NebulaServiceMethod.ScopeType.READ)
  public List<OrganizationVo> findByNullParentAndStatus(@ServiceMethodParam(name = "status") Integer status) {
    ResponseModel response = organizationFeignClient.findByNullParentAndStatus(status);
    return JsonUtils.convert(response.getSuccessDataOrElse(Lists.newArrayList()), new TypeReference<List<OrganizationVo>>() {});
  }

  /**
   * 查询指定父级组织机构下的所有子级组织机构信息，并且依据其状态进行查询
   * @param parentId
   * @param status
   * @return
   */
  @Override
  @NebulaServiceMethod(name="OrganizationService.findByParentAndStatus" , desc="根据父级ID和状态查询组织机构的最顶级数据" , returnPropertiesFilter="" , scope= NebulaServiceMethod.ScopeType.READ)
  public List<OrganizationVo> findByParentAndStatus(@ServiceMethodParam(name = "parentId") String parentId, @ServiceMethodParam(name = "status") Integer status) {
    ResponseModel response = organizationFeignClient.findByParentAndStatus(parentId, status);
    return JsonUtils.convert(response.getSuccessDataOrElse(Lists.newArrayList()), new TypeReference<List<OrganizationVo>>() {});
  }

  /**
   * 根据组织机构名称，查询这个组织机构以及这个组织（这些组织机构）对应的所有父级、子级结构信息
   * @param orgName 组织机构名称信息
   * @return
   */
  @Override
  public JSONArray findByOrgName(String orgName) {
    ResponseModel response = organizationFeignClient.findByOrgName(orgName);
    if (response.getSuccess() && response.getData() != null) {
      return new JSONArray();
    }
    return JSON.parseArray(JSON.toJSONString(response.getData()));
  }

  /**
   * 根据指定的用户信息，查询这个用户所绑定的组织机构信息(一个或者多个)，以及这个组织机构的所有父级组织机构信息
   * @param userId
   * @return
   */
  @Override
  public JSONArray findByUserId(String userId) {
    ResponseModel response = organizationFeignClient.findByUserId(userId);
    if(response.getData() == null) {
      return new JSONArray();
    }
    return JSON.parseArray(JSON.toJSONString(response.getData()));
  }

  /**
   * 查询指定组织下的详细信息，无论其状态是否正常（status无论什么值）
   * @param id 组织机构id
   * @return
   */
  @Override
  @NebulaServiceMethod(name="OrganizationService.findDetailsById" , desc="根据组织机构ID查询明细服务源" , returnPropertiesFilter="parent,child,roles,users,positions" , scope= NebulaServiceMethod.ScopeType.READ)
  public OrganizationVo findDetailsById(@ServiceMethodParam(name = "id") String id) {
    if(StringUtils.isBlank(id)) {
      return null;
    }
    ResponseModel response = organizationFeignClient.findDetailsById(id);
    return JsonUtils.convert(response.getSuccessDataOrElse(null), OrganizationVo.class);
  }

  /**
   * 查询指定组织下的详细信息，无论其状态是否正常（status无论什么值）
   * @param orgId
   * @return
   */
  @Override
  public Object[] findUnbindOrgById(String orgId) {
    ResponseModel response = organizationFeignClient.findUnbindOrgById(orgId);
    return JsonUtils.convert(response.getSuccessDataOrElse(new Object[0]), new TypeReference<Object[]>() {});
  }

  /**
   * 查询所有的组织机构信息，无论其状态是否正常（status无论什么值）
   * @return
   */
  @Override
  public List<OrganizationVo> findAll() {
    ResponseModel response = organizationFeignClient.findAll();
    return JsonUtils.convert(response.getSuccessDataOrElse(Lists.newArrayList()), new TypeReference<List<OrganizationVo>>() {});
  }

  @Override
  public OrganizationVo findMainOrgByUserId(String userId) {
    if(StringUtils.isBlank(userId)) {
      return null;
    }
    ResponseModel response = organizationFeignClient.findMainOrgByUserId(userId);
    return JsonUtils.convert(response.getSuccessDataOrElse(null), OrganizationVo.class);
  }

  @Override
  public void updateUserMainOrg(String userId, String orgId) {
    Validate.notBlank(userId, "用户ID不能为空");
    Validate.notBlank(orgId, "组织机构ID不能为空");
    ResponseModel response = organizationFeignClient.updateUserMainOrg(userId, orgId);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
  }

  /**
   * 预制控件查询，查询用户组织机构相关信息
   * @param userId
   * @return
   */
  @Override
  public Set<OrganizationVo> findOrgByUserId(String userId) {
    if(StringUtils.isBlank(userId)) {
      return Sets.newHashSet();
    }
    ResponseModel response = organizationFeignClient.findOrgByUserId(userId);
    return JsonUtils.convert(response.getSuccessDataOrElse(Sets.newHashSet()), new TypeReference<Set<OrganizationVo>>() {});
  }

  /**
   * 查询用户组织机构相关信息
   * @param ids
   * @return
   */
  @Override
  public Set<OrganizationVo> findByIds(List<String> ids) {
    ResponseModel response = organizationFeignClient.findByIds(ids);
    return JsonUtils.convert(response.getSuccessDataOrElse(Sets.newHashSet()), new TypeReference<Set<OrganizationVo>>() {});
  }

  /**
   * 组织机构-树结构查询-根据指定组织机构ID，查询这个组织机构以及这个组织机构对应的所有子级结构信息
   * @param id
   * @return
   */
  @Override
  public OrganizationVo findById(String id) {
    ResponseModel response = organizationFeignClient.findById(id);
    return JsonUtils.convert(response.getSuccessDataOrElse(null), OrganizationVo.class);
  }

  /**
   * 组织机构-根据组织机构名称实现模糊查询
   * @param orgName
   * @return
   */
  @Override
  public Set<OrganizationVo> findByOrgNameLike(String orgName) {
    ResponseModel response = organizationFeignClient.findByOrgNameLike(orgName);
    return JsonUtils.convert(response.getSuccessDataOrElse(Sets.newHashSet()), new TypeReference<Set<OrganizationVo>>() {});
  }

  /**
   * 根据组织编码查询
   * @param code
   * @return
   */
  @Override
  public OrganizationVo findByCode(String code) {
    ResponseModel response = organizationFeignClient.findByCode(code);
    return JsonUtils.convert(response.getSuccessDataOrElse(null), OrganizationVo.class);
  }

  /**
   * 获取组织的上级组织
   * @param childId
   * @return
   */
  @Override
  public OrganizationVo findByChild(String childId) {
    if(StringUtils.isBlank(childId)) {
      return null;
    }
    ResponseModel response = organizationFeignClient.findByChild(childId);
    return JsonUtils.convert(response.getSuccessDataOrElse(null), OrganizationVo.class);
  }

  /**
   * 根据登陆人关联的组织查询
   * @param type
   * @return
   */
  @Override
  public Set<OrganizationVo> findByPrincipal(Integer type, Principal principal) {
    Validate.notNull(type, "传入类型不能为空，请检查");
    ResponseModel response = organizationFeignClient.findByPrincipal(type,principal);
    return JsonUtils.convert(response.getSuccessDataOrElse(Sets.newHashSet()), new TypeReference<Set<OrganizationVo>>() {});
  }

  /**
   * 根据登陆人关联的组织和输入层级查询控件
   * @param type
   * @param account
   * @param level
   * @return
   */
  @Override
  public Set<OrganizationVo> findByTypeAndLevel(Integer type, String account, Integer level) {
    Validate.notNull(type, "传入类型不能为空，请检查");
    Validate.notNull(account, "传入账号名不能为空，请检查");
    Validate.notNull(level, "传入层级不能为空，请检查");
    ResponseModel response = organizationFeignClient.findByTypeAndLevel(type, account, level);
    return JsonUtils.convert(response.getSuccessDataOrElse(Sets.newHashSet()), new TypeReference<Set<OrganizationVo>>() {});
  }

  @Override
  public void mergeOrgById(String id, String currentOrgId) {
    Validate.notBlank(currentOrgId, "合并指定的组织机构必须传入!!");
    Validate.notBlank(id, "被合并的组织机构必须传入!!");
    Validate.isTrue(!StringUtils.equals(currentOrgId, id), "不能合并自己，请检查!");
    ResponseModel response = organizationFeignClient.mergeOrgById(currentOrgId, id);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
  }


  @Override
  @NebulaServiceMethod(name = "OrganizationService.bindRegions", desc = "绑定指定组织机构的行政区域", returnPropertiesFilter = "", scope = NebulaServiceMethod.ScopeType.WRITE)
  public void bindRegions(@ServiceMethodParam(name = "orgId") String orgId, @ServiceMethodParam(name = "regionIds") String[] regionIds) {
    Validate.notBlank(orgId, "组织机构ID不能为空");
    Validate.notEmpty(regionIds, "行政区域ID集合不能为空");
    ResponseModel response = organizationFeignClient.bindRegions(orgId, regionIds);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
  }

  @Override
  @NebulaServiceMethod(name = "OrganizationService.rebindRegions", desc = "重新绑定指定组织机构的行政区域", returnPropertiesFilter = "", scope = NebulaServiceMethod.ScopeType.WRITE)
  public void rebindRegions(@ServiceMethodParam(name = "orgId") String orgId, @ServiceMethodParam(name = "regionIds") String[] regionIds) {
    Validate.notBlank(orgId, "组织机构ID不能为空");
    ResponseModel response = organizationFeignClient.rebindRegions(orgId, regionIds);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
  }

  @Override
  @NebulaServiceMethod(name = "OrganizationService.unbindRegions", desc = "解绑指定组织机构的行政区域", returnPropertiesFilter = "", scope = NebulaServiceMethod.ScopeType.WRITE)
  public void unbindRegions(@ServiceMethodParam(name = "orgId") String orgId, @ServiceMethodParam(name = "regionIds") String[] regionIds) {
    Validate.notBlank(orgId, "组织机构ID不能为空");
    Validate.notEmpty(regionIds, "行政区域ID集合不能为空");
    ResponseModel response = organizationFeignClient.unbindRegions(orgId, regionIds);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
  }

  @Override
  public List<OrganizationVo> findChildrenByIdsOrCodes(String[] ids, String[] codes) {
    ResponseModel response = organizationFeignClient.findChildrenByIdsOrCodes(ids, codes);
    return JsonUtils.convert(response.getSuccessDataOrElse(Lists.newArrayList()), new TypeReference<List<OrganizationVo>>() {});
  }

  @Override
  @NebulaServiceMethod(name="OrganizationService.findByIdStruSelfAndDescendant" , desc="根据ID查询当前组织及当前组织的所有下级,以一维数组的格式返回数据" , returnPropertiesFilter="parent" , scope= NebulaServiceMethod.ScopeType.READ)
  public List<OrganizationVo> findByIdStruSelfAndDescendant(@ServiceMethodParam(name = "id") String id) {
    ResponseModel response = organizationFeignClient.findByIdStruSelfAndDescendant(id);
    return JsonUtils.convert(response.getSuccessDataOrElse(Lists.newArrayList()), new TypeReference<List<OrganizationVo>>() {});
  }

  @Override
  @NebulaServiceMethod(name="OrganizationService.findByLevel" , desc="根据组织层级查询组织机构" , returnPropertiesFilter="" , scope= NebulaServiceMethod.ScopeType.READ)
  public List<OrganizationVo> findByLevel(@ServiceMethodParam(name = "level") Integer level) {
    if(level == null) {
      return Lists.newArrayList();
    }
    ResponseModel response = organizationFeignClient.findByLevel(level);
    return JsonUtils.convert(response.getSuccessDataOrElse(Lists.newArrayList()), new TypeReference<List<OrganizationVo>>() {});
  }

  @Override
  @NebulaServiceMethod(name="OrganizationService.findParentsByLevel" , desc="根据层级查询上级组织机构" , returnPropertiesFilter="" , scope= NebulaServiceMethod.ScopeType.READ)
  public List<OrganizationVo> findParentsByLevel(@ServiceMethodParam(name = "level") Integer level) {
    if(level == null || level <= 1) {
      return Lists.newArrayList();
    }
    level--;
    return this.findByLevel(level);
  }
}
