package com.bizunited.platform.user.service.feign.service.internal;

import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.common.util.JsonUtils;
import com.bizunited.platform.common.util.PageUtils;
import com.bizunited.platform.core.annotations.NebulaServiceMethod;
import com.bizunited.platform.core.annotations.ServiceMethodParam;
import com.bizunited.platform.core.service.invoke.InvokeParams;
import com.bizunited.platform.user.common.service.position.PositionLevelEventListener;
import com.bizunited.platform.user.common.service.position.PositionLevelService;
import com.bizunited.platform.user.common.vo.PositionLevelVo;
import com.bizunited.platform.user.service.feign.PositionLevelFeignClient;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.Lists;
import java.util.List;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.util.CollectionUtils;

/**
 * @author: weikangning
 * @Date: 2020/8/24 14:50
 * @Description: 职位级别的服务实现，此服务的实现是通过feign的方式远程调用
 */
public class PositionLevelServiceImpl implements PositionLevelService {
  @Autowired
  private PositionLevelFeignClient positionLevelFeignClient;
  @Autowired(required = false)
  private List<PositionLevelEventListener> positionLevelEventListener;

  /**
   * 增加职位级别
   * @author: weikangning
   * @Param:
   * @Return:
   */
  @Override
  public PositionLevelVo create(PositionLevelVo positionLevelVo) {
    Validate.notNull(positionLevelVo, "职位级别信息不能为空,请检查");
    Validate.isTrue(StringUtils.isBlank(positionLevelVo.getId()), "添加职位级别时，不能传入岗位职位级别ID,请重新添加!!");
    Validate.notBlank(positionLevelVo.getName(), "职位级别名称不能为空，请检查");
    Validate.notBlank(positionLevelVo.getCode(), "职位级别编码不能为空，请检查");
    ResponseModel response = positionLevelFeignClient.create(positionLevelVo);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
    PositionLevelVo dbPositionLevel = JsonUtils.convert(response.getSuccessDataOrElse(null), PositionLevelVo.class);
    dbPositionLevel.setRoleCodes(positionLevelVo.getRoleCodes());
    if(CollectionUtils.isEmpty(positionLevelEventListener)) {
      for (PositionLevelEventListener eventListener : positionLevelEventListener) {
        eventListener.onPositionLevelCreated(dbPositionLevel);
      }
    }
    return dbPositionLevel;
  }

  /**
   * @Description: 更新职位级别
   * @author: weikangning
   * @Param:
   * @Return:
   */
  @Override
  public PositionLevelVo update(PositionLevelVo positionLevelVo) {
    this.updateValidation(positionLevelVo);
    ResponseModel response = positionLevelFeignClient.update(positionLevelVo);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
    PositionLevelVo dbPositionLevel = JsonUtils.convert(response.getSuccessDataOrElse(null), PositionLevelVo.class);
    dbPositionLevel.setRoleCodes(positionLevelVo.getRoleCodes());
    if(CollectionUtils.isEmpty(positionLevelEventListener)) {
      for (PositionLevelEventListener eventListener : positionLevelEventListener) {
        eventListener.onPositionLevelUpdated(dbPositionLevel);
      }
    }
    return dbPositionLevel;
  }

  /**
   * 更新校验
   * @param positionLevelVo
   */
  private void updateValidation(PositionLevelVo positionLevelVo){
    Validate.notNull(positionLevelVo, "职位级别信息不能为空，请检查");
    Validate.isTrue(!StringUtils.isBlank(positionLevelVo.getId()), "修改职位级别时，必须传入职位级别id,请重新修改!!");
    Validate.notBlank(positionLevelVo.getName(), "职位级别名称不能为空，请检查!!");
  }

  /**
   * 禁用一个或多个职位级别
   * @Param:
   * @Return:
   * @Author: weikangning
   * @Date: 2020/8/28 14:25
   */
  @Override
  public void disableByIds(String[] ids) {
    Validate.notEmpty(ids, "职级ID不能为空");
    ResponseModel response = positionLevelFeignClient.disableByIds(ids);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
  }

  /**
   * 启用一个或多个职位级别
   * @Param:
   * @Return:
   * @Author: weikangning
   * @Date: 2020/8/28 14:26
   */
  @Override
  public void enableByIds(String[] positionLevelIds) {
    Validate.notEmpty(positionLevelIds, "职级ID不能为空");
    ResponseModel response = positionLevelFeignClient.enableByIds(positionLevelIds);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
  }

  /**
   * 条件分页查询
   * @author: weikangning
   * @Param:
   * @Return:
   */
  @Override
  @NebulaServiceMethod(name = "PositionLevelService.findByConditions" , desc = "条件查询职级列表", scope = NebulaServiceMethod.ScopeType.READ, returnPropertiesFilter = "" )
  public Page<PositionLevelVo> findByConditions(InvokeParams invokeParams, @ServiceMethodParam(name = "pageable") Pageable pageable) {
    invokeParams = ObjectUtils.defaultIfNull(invokeParams, new InvokeParams());
    String name = (String) invokeParams.getInvokeParam("name");
    String code = (String) invokeParams.getInvokeParam("code");
    ResponseModel response = positionLevelFeignClient.findByConditions(name, code, pageable.getPageNumber(), pageable.getPageSize());
    if (response == null || response.getData() == null) {
      return Page.empty(pageable);
    }
    return PageUtils.convert2Page(JsonUtils.toJSONObject(response), new TypeReference<List<PositionLevelVo>>() {});
  }


  /**
   * 根据职位级别编码查询职位级别
   * @author: weikangning
   * @Param:
   * @Return:
   */
  @Override
  public PositionLevelVo findByCode(String code) {
    ResponseModel response = positionLevelFeignClient.findByCode(code);
    return JsonUtils.convert(response.getSuccessDataOrElse(null), PositionLevelVo.class);
  }


  /**
   * 根据职位级别ID查询职位级别
   * @author: weikangning
   * @Param:
   * @Return:
   */
  @Override
  public PositionLevelVo findById(String positionLevelId) {
    ResponseModel response = positionLevelFeignClient.findById(positionLevelId);
    return JsonUtils.convert(response.getSuccessDataOrElse(null), PositionLevelVo.class);
  }

  @Override
  public void deleteByIds(String[] ids) {
    Validate.notEmpty(ids, "职级ID不能为空");
    ResponseModel response = positionLevelFeignClient.deleteByIds(ids);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
    for (String id : ids) {
      PositionLevelVo positionLevel = this.findById(id);
      Validate.notNull(positionLevel, "职位级别不存在:%s", id);
      this.sendDeleteEvent(positionLevel);
    }
  }

  /**
   * 推送职级删除事件
   * @param positionLevel
   */
  private void sendDeleteEvent(PositionLevelVo positionLevel) {
    if(CollectionUtils.isEmpty(this.positionLevelEventListener) || positionLevel == null){
      return;
    }
    this.positionLevelEventListener.forEach(listener -> listener.onPositionLevelDeleted(positionLevel));
  }

  /**
   * 根据删除状态查询
   * @return
   */
  @Override
  public List<PositionLevelVo> findAll() {
    ResponseModel response = positionLevelFeignClient.findAll();
    return JsonUtils.convert(response.getSuccessDataOrElse(Lists.newArrayList()), new TypeReference<List<PositionLevelVo>>() {});
  }
}
