package com.bizunited.platform.user.service.feign.service.internal;

import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.common.util.JsonUtils;
import com.bizunited.platform.common.util.PageUtils;
import com.bizunited.platform.core.service.invoke.InvokeParams;
import com.bizunited.platform.core.annotations.NebulaServiceMethod;
import com.bizunited.platform.core.annotations.ServiceMethodParam;
import com.bizunited.platform.user.common.service.position.PositionService;
import com.bizunited.platform.user.common.vo.PositionSimpleVo;
import com.bizunited.platform.user.common.vo.PositionVo;
import com.bizunited.platform.user.common.vo.UserVo;
import com.bizunited.platform.user.service.feign.PositionFeignClient;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.util.CollectionUtils;

import java.security.Principal;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * 岗位的服务实现，此服务的实现是通过feign的方式远程调用
 * @Author: zengxingwang
 * @Date: 2020/3/22 20:12
 */
public class PositionServiceImpl implements PositionService {

  @Autowired
  private PositionFeignClient positionFeignClient;

  /**
   * 按照条件查询职位
   * @param invokeParams
   * @param pageable
   * @return
   */
  @SuppressWarnings("unchecked")
  @Override
  public Page<PositionVo> findByConditions(Map<String, Object> conditions, Pageable pageable) {
    // 如果当前没有设定分页信息，则默认第一页，每页50条数据
    if(conditions == null) {
      conditions = Maps.newHashMap();
    }
    String nameEqual = (String) conditions.get("nameEqual");
    Integer tstatus = (Integer) conditions.get("tstatus");
    Integer flag = (Integer) conditions.get("flag");
    List<String> ids = (List<String>) conditions.get("ids");
    List<String> codes = (List<String>) conditions.get("codes");
    String name = (String) conditions.get("name");
    String userName = (String) conditions.get("userName");
    String positionLevel = (String) conditions.get("positionLevel");
    String orgName = (String) conditions.get("orgName");
    ResponseModel response = positionFeignClient.findByConditions(nameEqual, tstatus, flag, ids, codes, name, userName,
            positionLevel, orgName, pageable.getPageNumber(), pageable.getPageSize());
    Object data = response.getSuccessDataOrElse(null);
    if(data == null) {
      return new PageImpl<>(Lists.newArrayList(), pageable, 0);
    }
    return PageUtils.convert2Page(JsonUtils.toJSONObject(data), new TypeReference<List<PositionVo>>() {});
  }

  @Override
  @NebulaServiceMethod(name = "PositionService.findByConditions", desc = "条件查询职位列表", scope = NebulaServiceMethod.ScopeType.READ, returnPropertiesFilter = "positionLevel,parent,organization")
  public Page<PositionVo> findByConditions(@ServiceMethodParam(name = "pageable") Pageable pageable, InvokeParams invokeParams) {
    pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    invokeParams = ObjectUtils.defaultIfNull(invokeParams, new InvokeParams());
    Map<String, Object> params = invokeParams.getInvokeParams();
    return this.findByConditions(params, pageable);
  }

  /**
   * 添加岗位
   * @param position
   * @return
   */
  @Override
  public PositionVo create(PositionVo position) {
    Validate.notNull(position, "岗位信息不能为空，请检查");
    Validate.isTrue(StringUtils.isBlank(position.getId()), "添加岗位时，不能传入岗位id,请重新添加!!");
    Validate.notBlank(position.getName(), "岗位名称不能为空，请检查!!");
    Validate.notNull(position.getCode(), "岗位编号不能为空");
    ResponseModel response = positionFeignClient.create(position);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
    return JsonUtils.convert(response.getSuccessDataOrElse(null), PositionVo.class);
  }

  /**
   * 修改岗位
   * @param position
   * @return
   */
  @Override
  public PositionVo update(PositionVo position) {
    Validate.notNull(position, "岗位信息不能为空，请检查");
    Validate.isTrue(!StringUtils.isBlank(position.getId()), "修改岗位时，必须传入岗位id,请重新修改!!");
    Validate.notBlank(position.getName(), "岗位名称不能为空，请检查!!");
    ResponseModel response = positionFeignClient.update(position);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
    return JsonUtils.convert(response.getSuccessDataOrElse(null), PositionVo.class);
  }

  /**
   * 修改岗位状态（启用禁用）
   * @param positionId
   * @return
   */
  @Override
  public PositionVo updateStatus(String positionId) {
    Validate.notBlank(positionId, "岗位id不能为空，请检查");
    ResponseModel response = positionFeignClient.updateStatus(positionId);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
    return JsonUtils.convert(response.getSuccessDataOrElse(null), PositionVo.class);
  }


  @Override
  public void bindOrg(String orgId, String positionId) {
    Validate.notBlank(orgId, "组织机构ID不能为空!");
    Validate.notBlank(positionId, "岗位ID不能为空!");
    ResponseModel response = positionFeignClient.bindOrg(orgId, positionId);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
  }


  @Override
  public void unbindOrg(String positionId) {
    Validate.notBlank(positionId, "岗位ID不能为空!");
    ResponseModel response = positionFeignClient.unbindOrg(positionId);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
  }


  @Override
  public void reBindOrg(String orgId, String positionId) {
    Validate.notBlank(orgId, "组织机构ID不能为空!");
    Validate.notBlank(positionId, "岗位ID不能为空!");
    ResponseModel response = positionFeignClient.reBindOrg(orgId, positionId);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
  }

  /**
   * 将指定一个岗位和指定的多个用户形成绑定关系
   * @param userIds
   * @param positionId
   */
  @Override
  public void bindUsersAndPosition(List<String> userIds, String positionId) {
    Validate.isTrue(!CollectionUtils.isEmpty(userIds), "用户ID集合不能为空，请检查！");
    String[] array = new String[userIds.size()];
    userIds.toArray(array);
    ResponseModel response = positionFeignClient.bindUsers(positionId, array);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
  }

  /**
   * 解绑多个用户与某一指定的岗位
   * @param userIds
   * @param positionId
   */
  @Override
  public void unbindUsersAndPosition(List<String> userIds, String positionId) {
    Validate.isTrue(!CollectionUtils.isEmpty(userIds), "用户ID集合不能为空，请检查！");
    String[] array = new String[userIds.size()];
    userIds.toArray(array);
    ResponseModel response = positionFeignClient.unbindUsersAndPosition(positionId, array);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
  }

  /**
   * 根据ID查询岗位的详细信息，包括相关组织、角色、用户等
   * @param id
   * @return
   */
  @Override
  public PositionVo findDetailsById(String id) {
    ResponseModel response = positionFeignClient.findDetailsById(id);
    return JsonUtils.convert(response.getSuccessDataOrElse(null), PositionVo.class);
  }

  /**
   * 预制控件查询，查询用户岗位相关信息
   * @param userId
   * @return
   */
  @Override
  public List<PositionVo> findByUserId(String userId) {
    Validate.notBlank(userId, "传入的用户ID不能为空！");
    ResponseModel response = positionFeignClient.findByUserId(userId);
    return JsonUtils.convert(response.getSuccessDataOrElse(Lists.newArrayList()), new TypeReference<List<PositionVo>>() {
    });
  }

  @Override
  public List<PositionVo> findByUserAccount(String userAccount) {
    Validate.notBlank(userAccount, "传入的用户Account不能为空！");
    ResponseModel response = positionFeignClient.findByUserAccount(userAccount);
    return JsonUtils.convert(response.getSuccessDataOrElse(Lists.newArrayList()), new TypeReference<List<PositionVo>>() {
    });
  }

  /**
   * 查询多个用户岗位相关信息
   * @param ids
   * @return
   */
  @Override
  public Set<PositionVo> findByIds(List<String> ids) {
    ResponseModel response = positionFeignClient.findByIds(ids);
    return JsonUtils.convert(response.getSuccessDataOrElse(Sets.newHashSet()), new TypeReference<Set<PositionVo>>() {
    });
  }

  /**
   * 根据岗位编码查询
   * @param code
   * @return
   */
  @Override
  public PositionVo findByCode(String code) {
    ResponseModel response = positionFeignClient.findByCode(code);
    return JsonUtils.convert(response.getSuccessDataOrElse(null), PositionVo.class);
  }

  /**
   * 切换岗位
   * @param positionId
   * @param userId
   * @return
   */
  @Override
  public UserVo changePosition(String positionId, String userId) {
    Validate.notBlank(positionId, "切换岗位id不能为空");
    Validate.notBlank(userId, "用户id不能为空");
    ResponseModel response = positionFeignClient.changePosition(positionId, userId);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
    return JsonUtils.convert(response.getSuccessDataOrElse(null), UserVo.class);
  }

  /**
   * 查询Redis中主岗位，如果redis中不存在，则默认主岗位为数据库查询的第一条数据，并且将该数据存入redis中
   * @param userId
   * @return
   */
  @Override
  public PositionVo findMainPositionByUserId(String userId) {
    Validate.notBlank(userId, "传入的用户id不能为空");
    ResponseModel response = positionFeignClient.findMainPositionByUserId(userId);
    return JsonUtils.convert(response.getSuccessDataOrElse(null), PositionVo.class);
  }

  @Override
  public void rebindUsers(String[] userIds, String positionId) {
    Validate.notBlank(positionId, "岗位ID不能为空!");
    ResponseModel response = positionFeignClient.rebindUsers(positionId, userIds);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
  }

  /**
   * 查询所有岗位信息（根据创建时间排序）
   * @return
   */
  @Override
  public List<PositionVo> findAll() {
    ResponseModel response = positionFeignClient.findAll();
    return JsonUtils.convert(response.getSuccessDataOrElse(Lists.newArrayList()), new TypeReference<List<PositionVo>>() {
    });
  }

  /**
   * 根据条件查询所有岗位
   * @param status
   * @return
   */
  @Override
  public List<PositionVo> findByStatus(Integer status) {
    Validate.notNull(status, "状态不能为空");
    ResponseModel response = positionFeignClient.findByStatus(status);
    return JsonUtils.convert(response.getSuccessDataOrElse(Lists.newArrayList()), new TypeReference<List<PositionVo>>() {
    });
  }

  /**
   * 根据登陆人关联的岗位查询控件
   * @param type
   * @return
   */
  @Override
  public Set<PositionVo> findByPrincipal(Integer type, Principal principal) {
    Validate.notNull(type, "传入类型不能为空，请检查");
    ResponseModel response = positionFeignClient.findByPrincipal(type,principal);
    return JsonUtils.convert(response.getSuccessDataOrElse(Sets.newHashSet()), new TypeReference<Set<PositionVo>>() {});
  }

  /**
   * 根据登陆人关联的岗位和输入层级查询控件
   * @param type
   * @param principal
   * @param level
   * @return
   */
  @Override
  public Set<PositionVo> findByTypeAndLevel(Integer type, Principal principal, Integer level) {
    Validate.notNull(type, "传入类型不能为空，请检查");
    Validate.notNull(level, "传入层级不能为空，请检查");
    ResponseModel response = positionFeignClient.findByTypeAndLevel(type, principal, level);
    return JsonUtils.convert(response.getSuccessDataOrElse(Sets.newHashSet()), new TypeReference<Set<PositionVo>>() {});
  }

  /**
   * 根据层级创建职位
   * @param levelCode
   * @return
   */
  @Override
  public PositionVo createByPositionLevelCode(String levelCode) {
    Validate.notNull(levelCode, "传入层级编码不能为空，请检查");
    ResponseModel response = positionFeignClient.createByPositionLevelCode(levelCode);
    return JsonUtils.convert(response.getSuccessDataOrElse(null), PositionVo.class);
  }

  /**
   * 根据组织编码查询职位（包括关联信息）
   * @param orgCode
   * @return
   */
  @Override
  @NebulaServiceMethod(name = "PositionService.findDetailsByOrgCode" , desc = "根据组织编码相关信息（包括组织，职级，用户，下级职位）", scope = NebulaServiceMethod.ScopeType.READ, returnPropertiesFilter = "positionLevel,users,organization,children" )
  public List<PositionVo> findDetailsByOrgCode(String orgCode) {
    if(StringUtils.isBlank(orgCode)){
      return Lists.newArrayList();
    }
    ResponseModel response = positionFeignClient.findDetailsByOrgCode(orgCode);
    return JsonUtils.convert(response.getSuccessDataOrElse(Lists.newArrayList()), new TypeReference<List<PositionVo>>() {});
  }

  /**
   * 根据编码查询职位的所有下级信息
   * @param code
   * @return
   */
  @Override
  public List<PositionVo> findDetailsByCode(String code) {
    if(StringUtils.isBlank(code)){
      return Lists.newArrayList();
    }
    ResponseModel response = positionFeignClient.findDetailsByCode(code);
    return JsonUtils.convert(response.getSuccessDataOrElse(Lists.newArrayList()), new TypeReference<List<PositionVo>>() {});
  }

  /**
   * 根据编码查询职位详情，包含组织信息、用户信息
   *
   * @param code
   * @return
   */
  @Override
  public PositionVo findPositionDetailByCode(String code) {
    ResponseModel response = positionFeignClient.findPositionDetailByCode(code);
    return JsonUtils.convert(response.getSuccessDataOrElse(null), PositionVo.class);
  }

  /**
   * 根据职级批量查询关联的职位
   * @param positionLevelIds 职级主键列表
   * @return
   */
  @Override
  public Set<PositionVo> findByPositionLevelIds(String[] positionLevelIds) {
    ResponseModel response = positionFeignClient.findByPositionLevelIds(positionLevelIds);
    return JsonUtils.convert(response.getSuccessDataOrElse(Sets.newHashSet()), new TypeReference<Set<PositionVo>>() {});
  }

  /**
   * 根据职位Id批量禁用职位
   * @param ids 职位Id列表
   */
  @Override
  public void disableByIds(String[] ids) {
    Validate.notEmpty(ids, "职位id不能为空，请检查");
    ResponseModel response = positionFeignClient.disableByIds(ids);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
  }

  /**
   * 根据职位Id批量启用职位
   *
   * @param ids 职位Id列表
   */
  @Override
  public void enableByIds(String[] ids) {
    Validate.notEmpty(ids, "职位id不能为空，请检查");
    ResponseModel response = positionFeignClient.enableByIds(ids);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
  }

  @Override
  public void reBindOrgCode(String[] codes, String orgCode) {
    Validate.notEmpty(codes, "岗位不能为空!");
    Validate.notBlank(orgCode, "组织机构不能为空!");
    ResponseModel response = positionFeignClient.reBindOrgCode(codes, orgCode);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
  }

  /**
   * 根据职级分页查询职位，带出用户拓展以及组织
   *
   * @param pageable
   * @param conditions
   * @return
   */
  @Override
  public Page<PositionSimpleVo> findByPositionLevelAndConditions(Pageable pageable, Map<String, Object> conditions) {
    if(conditions == null){
      conditions = Maps.newHashMap();
    }
    String name = (String) conditions.get("name");
    String orgName = (String) conditions.get("orgName");
    String userName = (String) conditions.get("userName");
    String positionLevel = (String) conditions.get("positionLevel");
    ResponseModel response = positionFeignClient.findByPositionLevelAndConditions(name, orgName, userName, positionLevel, pageable.getPageNumber(), pageable.getPageSize());
    return JsonUtils.convert(response.getSuccessDataOrElse(null), new TypeReference<Page<PositionSimpleVo>>(){});
  }

  @Override
  public List<PositionVo> findByUserIdStruDescendant(String userId) {
    if(StringUtils.isBlank(userId)){
      return Lists.newArrayList();
    }
    ResponseModel response = this.positionFeignClient.findByUserIdStruDescendant(userId);
    return JsonUtils.convert(response.getSuccessDataOrElse(Lists.newArrayList()), new TypeReference<List<PositionVo>>() {});
  }

  @Override
  public PositionVo bindParent(String id, String parentId) {
    Validate.notBlank(id, "当前职位不能为空");
    Validate.notBlank(parentId, "父级职位id不能为空");
    ResponseModel response = this.positionFeignClient.bindParent(id, parentId);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
    return JsonUtils.convert(response.getSuccessDataOrElse(null), PositionVo.class);
  }

  @Override
  public void unbindParent(String id) {
    throw new UnsupportedOperationException("暂不支持远端解绑父级职位");
  }
}