package com.bizunited.platform.user.service.feign.service.internal;

import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.common.util.JsonUtils;
import com.bizunited.platform.common.util.PageUtils;
import com.bizunited.platform.user.common.service.userGroup.UserGroupService;
import com.bizunited.platform.user.common.vo.UserGroupVo;
import com.bizunited.platform.user.service.feign.UserGroupFeignClient;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;

import java.util.List;
import java.util.Set;

/**
 * 用户组的服务实现，通过feign方式远程调用实现
 * TODO 所有public服务的入参均为校验
 * @Author: zengxingwang
 * @Date: 2020/3/22 20:13
 */
public class UserGroupServiceImpl implements UserGroupService {

  @Autowired
  private UserGroupFeignClient userGroupFeignClient;

  /**
   * 新增用户组
   * @param userGroup 用户组信息
   * @return
   */
  @Override
  public UserGroupVo create(UserGroupVo userGroup) {
    Validate.notNull(userGroup, "分组信息不能为空，请检查");
    Validate.isTrue(StringUtils.isBlank(userGroup.getId()), "添加分组时，不能传入分组id,请重新添加!!");
    String groupName = userGroup.getGroupName();
    Validate.notBlank(groupName, "分组名称不能为空，请检查!!");
    Validate.notBlank(userGroup.getGroupDescription(), "分组描述不能为空，请检查!!");
    Validate.notNull(userGroup.getTstatus(), "分组状态不能为空，请检查!!");
    ResponseModel response = userGroupFeignClient.create(userGroup);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
    return JsonUtils.convert(response.getSuccessDataOrElse(null), UserGroupVo.class);
  }

  /**
   * 修改用户组
   * @param userGroup 用户组相关信息
   * @return
   */
  @Override
  public UserGroupVo update(UserGroupVo userGroup) {
    Validate.notNull(userGroup, "分组信息不能为空，请检查");
    String groupId = userGroup.getId();
    Validate.notNull(groupId, "分组id不能为空，请检查");
    ResponseModel response = userGroupFeignClient.update(userGroup);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
    return JsonUtils.convert(response.getSuccessDataOrElse(null), UserGroupVo.class);
  }

  /**
   * 修改用户组状态
   * @param groupId 用户组id
   * @return
   */
  @Override
  public UserGroupVo updateStatus(String groupId) {
    Validate.notNull(groupId, "分组id不能为空，请检查");
    ResponseModel response = userGroupFeignClient.updateStatus(groupId);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
    return JsonUtils.convert(response.getSuccessDataOrElse(null), UserGroupVo.class);
  }

  /**
   * 用户与用户组绑定
   * @param groupId 用户组id
   * @param userIds 用户id
   */
  @Override
  public void bindUser(String groupId, String[] userIds) {
    Validate.notNull(groupId, "分组id不能为空，请检查");
    Validate.isTrue(userIds != null && userIds.length > 0, "用户id不能为空（至少需要传入一个），请检查");
    ResponseModel response = userGroupFeignClient.bindUser(groupId, userIds);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
  }

  /**
   * 用户与用户组解除绑定
   * @param groupId 用户组id
   * @param userIds  用户id
   */
  @Override
  public void unBindUser(String groupId, String[] userIds) {
    Validate.notNull(groupId, "分组id不能为空，请检查");
    Validate.isTrue(userIds != null && userIds.length > 0, "用户id不能为空（至少需要传入一个），请检查");
    ResponseModel response = userGroupFeignClient.unBindUser(groupId, userIds);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
  }

  /**
   * 根据编码查询用户组
   * @param groupCode
   * @return
   */
  @Override
  public UserGroupVo findByCode(String groupCode) {
    ResponseModel response = userGroupFeignClient.findByCode(groupCode);
    return JsonUtils.convert(response.getSuccessDataOrElse(null), UserGroupVo.class);
  }

  /**
   * 根据分组名称查询
   * @param userId
   * @return
   */
  @Override
  public Set<UserGroupVo> findByUserId(String userId) {
    ResponseModel response = userGroupFeignClient.findByUserId(userId);
    return JsonUtils.convert(response.getSuccessDataOrElse(Sets.newHashSet()), new TypeReference<Set<UserGroupVo>>() {});
  }

  /**
   * 根据指定用户组名称、状态查询
   * @param groupName 用户组名称
   * @param status    用户组状态
   * @param pageable
   * @return
   */
  @Override
  public Page<UserGroupVo> findByCondition(String groupName, String groupCode, Integer status, Pageable pageable) {
    ResponseModel response = userGroupFeignClient.findByCondition(groupName, groupCode, status, pageable.getPageNumber(), pageable.getPageSize());
    Object data = response.getSuccessDataOrElse(null);
    if(data == null) {
      return new PageImpl<>(Lists.newArrayList(), pageable, 0);
    }
    return PageUtils.convert2Page(JsonUtils.toJSONObject(data), new TypeReference<List<UserGroupVo>>() {});
  }

  /**
   * 查询指定用户组的基本详情
   * @param id 用户组id
   * @return
   */
  @Override
  public UserGroupVo findDetailsById(String id) {
    ResponseModel response = userGroupFeignClient.findDetailsById(id);
    return JsonUtils.convert(response.getSuccessDataOrElse(Sets.newHashSet()), new TypeReference<UserGroupVo>() {});
  }

  /**
   * 查询多个用户组基本信息
   * @param ids
   * @return
   */
  @Override
  public Set<UserGroupVo> findByIds(List<String> ids) {
    ResponseModel response = userGroupFeignClient.findByIds(ids);
    return JsonUtils.convert(response.getSuccessDataOrElse(Sets.newHashSet()), new TypeReference<Set<UserGroupVo>>() {});
  }

  /**
   * 根据分组名称查询
   * @param groupName
   * @return
   */
  @Override
  public UserGroupVo findByGroupName(String groupName) {
    ResponseModel response = userGroupFeignClient.findByGroupName(groupName);
    return JsonUtils.convert(response.getSuccessDataOrElse(null), UserGroupVo.class);
  }

  /**
   * 查询所有用户组信息（根据创建时间排序）
   * @return
   */
  @Override
  public List<UserGroupVo> findAll() {
    ResponseModel response = userGroupFeignClient.findAll();
    return JsonUtils.convert(response.getSuccessDataOrElse(Lists.newArrayList()), new TypeReference<List<UserGroupVo>>() {});
  }

  /**
   * 用户与用户组的重新绑定
   * @param groupId
   * @param userIds
   */
  @Override
  public void reBindUser(String groupId, String[] userIds) {
    Validate.notBlank(groupId, "用户与用户组的重新绑定时，用户组Id不能为空");
    ResponseModel response = userGroupFeignClient.reBindUser(groupId, userIds);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
  }

  /**
   * 根据用户组编码查询多个用户组信息
   * @param groupCodes
   * @return
   */
  @Override
  public List<UserGroupVo> findByGroupCodes(List<String> groupCodes) {
    ResponseModel response = userGroupFeignClient.findByGroupCodes(groupCodes);
    return JsonUtils.convert(response.getSuccessDataOrElse(Lists.newArrayList()), new TypeReference<List<UserGroupVo>>() {});
  }
}
