package com.bizunited.platform.user.service.feign.service.internal;

import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.common.util.JsonUtils;
import com.bizunited.platform.common.util.PageUtils;
import com.bizunited.platform.core.annotations.NebulaServiceMethod;
import com.bizunited.platform.core.annotations.ServiceMethodParam;
import com.bizunited.platform.core.service.invoke.InvokeParams;
import com.bizunited.platform.user.common.service.user.UserEventListener;
import com.bizunited.platform.user.common.service.user.UserService;
import com.bizunited.platform.user.common.vo.UserVo;
import com.bizunited.platform.user.service.feign.UserFeignClient;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;

import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * 用户的服务实现，通过feign调用方式实现
 * @Author: zengxingwang yinwenjie
 * @Date: 2020/3/22 20:13
 */
public class UserServiceImpl implements UserService {

  @Autowired
  private UserFeignClient userFeignClient;
  /**
   * 可能已注册到系统中的用户事件监听器
   */
  @Autowired(required = false)
  private List<UserEventListener> localUserEventListeners;

  /**
   * 添加一个新的用户信息（后台管理人员，业务功能操作者）
   * @param user
   * @return
   */
  @Override
  public UserVo create(UserVo user) {
    Validate.notNull(user , "必须传入要添加的用户信息!");
    Validate.notBlank(user.getUserName() , "用户真实姓名必须传入!");
    Validate.notBlank(user.getAccount() , "用户账号信息（一般是手机号）必须传入!");
    ResponseModel response = userFeignClient.create(user);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
    // 激活监听器，以便本地应用系统在完成诸如创建基础角色在内的扩展动作
    if(localUserEventListeners != null) {
      for (UserEventListener eventListener : localUserEventListeners) {
        eventListener.onUserCreated(user);
      }
    }

    return JsonUtils.convert(response.getSuccessDataOrElse(null), UserVo.class);
  }

  /**
   * 修改一个已经存在的用户信息
   * @param user
   * @return
   */
  @Override
  public UserVo update(UserVo user) {
    Validate.notNull(user , "必须传入要修改的用户信息!");
    Validate.notBlank(user.getId() , "修改时，用户数据的编号信息必须传入!");
    Validate.notBlank(user.getUserName() , "用户真实姓名必须传入!");
    Validate.notBlank(user.getAccount() , "用户账号信息（一般是手机号）必须传入!");
    ResponseModel response = userFeignClient.update(user);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
    // 触发监听器
    this.doUpdateNotify(user);
    return JsonUtils.convert(response.getSuccessDataOrElse(null), UserVo.class);
  }

  /**
   * 调用用户更新的监听器
   * @param user
   */
  private void doUpdateNotify(UserVo user) {
    if(localUserEventListeners != null) {
      for (UserEventListener eventListener : localUserEventListeners) {
        eventListener.onUserUpgraded(user);
      }
    }
  }

  /**
   * 修改指定用户的密码信息
   * @param userId
   * @param newPassword 没有经过加密的新的密码
   * @return
   */
  @Override
  public UserVo updatePassword(String userId, String newPassword) {
    Validate.notBlank(userId, "修改密码时 ,用户编号不能为空!!");
    Validate.notBlank(newPassword, "修改密码时 ,新的密码不能为空!!");
    UserVo userVo = this.findByUserId(userId);
    Validate.notNull(userVo, "未在远程系统中发现指定的用户信息，请检查!!");
    ResponseModel response = userFeignClient.updatePasswordIgnoreOld(userId, newPassword);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
    // 触发监听器
    this.doUpdateNotify(userVo);
    return JsonUtils.convert(response.getSuccessDataOrElse(null), UserVo.class);
  }

  /**
   * 重置指定用户的密码信息
   * @param userId
   * @param newPassword 没有经过加密的新的密码
   * @param oldPassword 没有经过加密的原始密码
   * @return
   */
  @Override
  public UserVo updatePassword(String userId, String newPassword, String oldPassword) {
    Validate.notBlank(userId, "修改密码时 ,用户编号不能为空!!");
    Validate.notBlank(newPassword, "修改密码时 ,新的密码不能为空!!");
    Validate.notBlank(oldPassword, "修改密码时 ,原有的密码不能为空!!");
    UserVo userVo = this.findByUserId(userId);
    Validate.notNull(userVo, "未在远程系统中发现指定的用户信息，请检查!!");
    ResponseModel response = userFeignClient.updatePassword(userId, newPassword, oldPassword);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
    // 触发监听器
    this.doUpdateNotify(userVo);
    return JsonUtils.convert(response.getSuccessDataOrElse(null), UserVo.class);
  }

  /**
   * 成功登陆后，更新用户最后一次登录时间
   * @param account 按照用户账号更新
   */
  @Override
  public void updateLastloginTime(String account) {
    Validate.notBlank(account, "修改最后登录时间时 ,用户账号不能为空!!");
    UserVo userVo = this.findByAccount(account);
    Validate.notNull(userVo, "未在远程系统中发现指定的用户信息，请检查!!");
    ResponseModel response = userFeignClient.updateLastloginTime(account);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
    // 触发监听器
    this.doUpdateNotify(userVo);
  }

  /**
   * 修改指定的用户信息的状态，包括可用状态和不可用状态
   * @param account 用户账号，一般来说是他的手机号
   * @return
   */
  @Override
  public UserVo enabled(String account) {
    Validate.notBlank(account, "用户账号必须传入!!");
    UserVo userVo = this.findByAccount(account);
    Validate.notNull(userVo, "未在远程系统中发现指定的用户信息，请检查!!");
    ResponseModel response = userFeignClient.enabled(account);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
    // 触发监听器
    this.doUpdateNotify(userVo);
    return JsonUtils.convert(response.getSuccessDataOrElse(null), UserVo.class);
  }

  /**
   * 修改指定的用户信息的状态，包括可用状态和不可用状态
   * @param account 用户账号，一般来说是他的手机号
   * @return
   */
  @Override
  public UserVo disable(String account) {
    Validate.notBlank(account, "用户账号必须传入!!");
    UserVo userVo = this.findByAccount(account);
    Validate.notNull(userVo, "未在远程系统中发现指定的用户信息，请检查!!");
    ResponseModel response = userFeignClient.disable(account);
    Validate.isTrue(response.getSuccess(), response.getErrorMsg());
    // 触发监听器
    this.doUpdateNotify(userVo);
    return JsonUtils.convert(response.getSuccessDataOrElse(null), UserVo.class);
  }

  /**
   * 查询当前用户信息，且包括当前用户已经绑定（包括间接绑定）的角色信息、用户组、组织机构和岗位
   * @param userId 指定的用户id
   * @return
   */
  @Override
  public UserVo findDetailsById(String userId) {
    ResponseModel response = userFeignClient.findDetailsById(userId);
    return JsonUtils.convert(response.getSuccessDataOrElse(null), UserVo.class);
  }

  /**
   * 查询当前用户信息(按照用户账号查询)，不包括当前用户已经绑定（包括间接绑定）的角色信息、用户组、组织机构和岗位
   * @param account 账户名
   * @return
   */
  @Override
  public UserVo findByAccount(String account) {
    ResponseModel response = userFeignClient.findByAccount(account);
    return JsonUtils.convert(response.getSuccessDataOrElse(null), UserVo.class);
  }

  /**
   * 根据用户账号与名称模糊查询
   * @param param
   * @return
   */
  @Override
  public List<UserVo> findByAccountLikeOrNameLike(String param) {
    ResponseModel response = userFeignClient.findByAccountLikeOrNameLike(param);
    return JsonUtils.convert(response.getSuccessDataOrElse(Lists.newArrayList()), new TypeReference<List<UserVo>>() {
    });
  }

  /**
   * 按照指定的手机号，查询账户基本信息，该方法的查询不包括任何关联信息，例如角色信息。
   * @param phone 指定的用户手机号
   * @return
   */
  @Override
  public UserVo findByPhone(String phone) {
    ResponseModel response = userFeignClient.findByPhone(phone);
    return JsonUtils.convert(response.getSuccessDataOrElse(null), UserVo.class);
  }

  /**
   * 按照条件搜索用户，支持模糊查询
   * @param userVo 用户信息
   * @param pageable 分页设置信息
   * @return
   */
  @Override
  public Page<UserVo> findByConditions(UserVo userVo, Pageable pageable) {
    ResponseModel response = userFeignClient.findByConditions(userVo.getUserName(), userVo.getAccount(), userVo.getIdcard(),
        userVo.getPhone(), userVo.getCreateTime(), userVo.getEntryTime(), userVo.getLastloginTime(), userVo.getUseStatus(),
        userVo.getUserHead(), userVo.getGender(), userVo.getExtend1(), userVo.getExtend2(), userVo.getExtend3(), userVo.getExtend4(),
        userVo.getExtend5(), userVo.getExtend6(), userVo.getExtend7(), userVo.getExtend8(), userVo.getExtend9(), userVo.getExtend10(),
        pageable.getPageNumber(), pageable.getPageSize());
    Object data = response.getSuccessDataOrElse(null);
    if(data == null) {
      return new PageImpl<>(Lists.newArrayList(), pageable, 0);
    }
    return PageUtils.convert2Page(JsonUtils.toJSONObject(data), new TypeReference<List<UserVo>>() {});
  }

  @Override
  public List<UserVo> findByAccountAndTypeStruDescendant(String userId, Integer type) {
    if(StringUtils.isBlank(userId)){
      return Lists.newArrayList();
    }
    ResponseModel response = userFeignClient.findByAccountAndTypeStruDescendant(userId, type);
    return JsonUtils.convert(response.getSuccessDataOrElse(Lists.newArrayList()), new TypeReference<List<UserVo>>() {});
  }

  /**
   * 预制控件查询，暂时查询当前操作用户基础信息
   * @param userId
   * @return
   */
  @Override
  public UserVo findByUserId(String userId) {
    Validate.notBlank(userId, "传入的用户ID不能为空！");
    ResponseModel response = userFeignClient.findByUserId(userId);
    return JsonUtils.convert(response.getSuccessDataOrElse(null), UserVo.class);
  }
  /**
   * 查询多个用户信息
   * @param ids
   * @return
   */
  @Override
  public Set<UserVo> findByIds(List<String> ids) {
    ResponseModel response = userFeignClient.findByIds(ids);
    return JsonUtils.convert(response.getSuccessDataOrElse(Sets.newHashSet()), new TypeReference<Set<UserVo>>() {
    });
  }

  @Override
  public Set<UserVo> findByOrgId(String orgId) {
    if(StringUtils.isBlank(orgId)) {
      return Sets.newHashSet();
    }
    ResponseModel response = userFeignClient.findByOrgId(orgId);
    return JsonUtils.convert(response.getSuccessDataOrElse(Sets.newHashSet()), new TypeReference<Set<UserVo>>() {
    });
  }

  @Override
  public Set<UserVo> findByPositionId(String positionId) {
    if(StringUtils.isBlank(positionId)) {
      return Sets.newHashSet();
    }
    ResponseModel response = userFeignClient.findByPositionId(positionId);
    return JsonUtils.convert(response.getSuccessDataOrElse(Sets.newHashSet()), new TypeReference<Set<UserVo>>() {
    });
  }

  @Override
  @NebulaServiceMethod(name = "UserService.findByConditions", desc = "条件查询用户列表", returnPropertiesFilter = "positions,groups,orgs,roles", scope = NebulaServiceMethod.ScopeType.READ)
  public Page<UserVo> findByConditions(@ServiceMethodParam(name = "pageable") Pageable pageable, InvokeParams invokeParams) {
    pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    invokeParams = ObjectUtils.defaultIfNull(invokeParams, new InvokeParams());
    Map<String, Object> params = invokeParams.getInvokeParams();
    UserVo user = JsonUtils.convert(params, UserVo.class);
    return this.findByConditions(user, pageable);
  }

  @Override
  public Boolean isDefaultPassword(String account) {
    ResponseModel response = userFeignClient.isDefaultPassword(account);
    return JsonUtils.convert(response.getSuccessDataOrElse(Sets.newHashSet()), new TypeReference<Boolean>() {
    });
  }
}
