package com.bizunited.platform.user.service.local.service.internal;

import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.common.constant.Constants;
import com.bizunited.platform.common.enums.RbacRelationEnum;
import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.user.common.service.UserMappingService;
import com.bizunited.platform.user.common.service.position.PositionService;
import com.bizunited.platform.user.common.service.user.UserService;
import com.bizunited.platform.user.common.vo.PositionVo;
import com.bizunited.platform.user.common.vo.UserMappingVo;
import com.bizunited.platform.user.common.vo.UserVo;
import com.bizunited.platform.user.service.local.entity.OrganizationEntity;
import com.bizunited.platform.user.service.local.entity.PositionEntity;
import com.bizunited.platform.user.service.local.entity.UserEntity;
import com.bizunited.platform.user.service.local.repository.OrganizationRepository;
import com.bizunited.platform.user.service.local.repository.PositionRepository;
import com.bizunited.platform.user.service.local.repository.UserRepository;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.redisson.api.RedissonClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.security.Principal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * @description:
 * @author: yanwe
 * @date: 17/Jan/2019 15:01
 */
public class PositionServiceImpl implements PositionService {

  @Autowired 
  private PositionRepository positionRepository;
  @Autowired 
  private OrganizationRepository organizationRepository;
  @Autowired
  private UserService userService;
  @Autowired
  private UserMappingService userMappingService;
  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private RedissonClient redissonClient;
  @Autowired
  private UserRepository userRepository;

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.PositionService#create(com.bizunited.platform.rbac.server.vo.PositionVo)
   */
  @Override
  @Transactional
  public PositionVo create(PositionVo positionVo) {
    Validate.notNull(positionVo, "岗位信息不能为空，请检查");
    Validate.isTrue(StringUtils.isBlank(positionVo.getId()), "添加岗位时，不能传入岗位id,请重新添加!!");
    Validate.notBlank(positionVo.getName(), "岗位名称不能为空，请检查!!");
    Validate.notNull(positionVo.getCode(), "岗位编号不能为空");
    positionVo.setCreateTime(new Date());
    PositionEntity currentPosition = positionRepository.findByCode(positionVo.getCode());
    Validate.isTrue(null == currentPosition,"存在重复的code,请重新输入岗位编码!!");
    if(positionVo.getParent() != null){
      Validate.notBlank(positionVo.getParent().getId(), "未找到该岗位的上级岗位的id，请检查！");
      PositionEntity parent = positionRepository.findById(positionVo.getParent().getId()).orElse(null);
      Validate.notNull(parent, "未找到该岗位的上级岗位，请检查！");
    }
    // 转换后进行保存
    PositionEntity positionEntity = this.nebulaToolkitService.copyObjectByWhiteList(positionVo, PositionEntity.class, HashSet.class, ArrayList.class, "parent");
    positionRepository.save(positionEntity);
    positionVo.setId(positionEntity.getId());
    return positionVo;
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.PositionService#update(com.bizunited.platform.rbac.server.vo.PositionVo)
   */
  @Override
  @Transactional
  public PositionVo update(PositionVo positionVo) {
    // 验证
    Validate.notNull(positionVo, "岗位信息不能为空，请检查");
    Validate.isTrue(!StringUtils.isBlank(positionVo.getId()), "修改岗位时，必须传入岗位id,请重新修改!!");
    Validate.notBlank(positionVo.getName(), "岗位名称不能为空，请检查!!");
    Optional<PositionEntity> op = positionRepository.findById(positionVo.getId());
    PositionEntity position = op.orElse(null);
    Validate.notNull(position, "未在数据层找到对应的岗位信息");
    if(positionVo.getParent() != null){
      String parentId = positionVo.getParent().getId();
      Validate.notBlank(parentId, "未找到该岗位的上级岗位的id，请检查！");
      Validate.isTrue(!positionVo.getId().equals(parentId), "禁止将该岗位本身设置为上级岗位");
      PositionEntity parent = positionRepository.findById(parentId).orElse(null);
      Validate.notNull(parent, "未找到该岗位的上级岗位，请检查！");
      position.setParent(parent);
      Set<String> positionStack = new HashSet<>();
      positionStack.add(parentId);
      this.validateCircular(position, positionStack);
    }else {
      position.setParent(null);
    }
    // 赋值并保存
    position.setName(positionVo.getName());
    positionRepository.save(position);
    // 转换后返回
    return this.nebulaToolkitService.copyObjectByWhiteList(position, PositionVo.class, HashSet.class, ArrayList.class);
  }

  /**
   *判断是否形成循环依赖
   * @param parent
   * @param positionStack
   */
  private void validateCircular(PositionEntity parent, Set<String> positionStack) {
    if(CollectionUtils.isEmpty(parent.getChildren())) {
      return;
    }
    for (PositionEntity positionEntity : parent.getChildren()) {
      Validate.isTrue(!positionStack.contains(positionEntity.getId()), "形成循环依赖，更新失败，请检查！");
      positionStack.add(positionEntity.getId());
      this.validateCircular(positionEntity, positionStack);
    }
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.PositionService#updateStatus(java.lang.String)
   */
  @Override
  @Transactional
  public PositionVo updateStatus(String positionId) {
    Validate.notBlank(positionId, "岗位id不能为空，请检查");
    Optional<PositionEntity> op = positionRepository.findById(positionId);
    PositionEntity position = op.orElse(null);
    Validate.notNull(position, "没有该岗位，请检查!!");
    Integer status = position.getTstatus();
    
    // 反转状态，0:禁用,1:启用
    status = (status == 1 ? 0 : 1);
    position.setTstatus(status);
    positionRepository.save(position);
    
    // 转换后返回
    return this.nebulaToolkitService.copyObjectByWhiteList(position, PositionVo.class, HashSet.class, ArrayList.class);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.PositionService#findById(java.lang.String)
   */
  @Override
  public PositionVo findDetailsById(String positionId) {
    if (StringUtils.isBlank(positionId)) {
      return null;
    }
    PositionEntity positionEntity = positionRepository.findDetailsById(positionId);
    Validate.notNull(positionEntity , "未找到指定的岗位信息，请检查传参!!");
    
    // 转换后返回
    return this.nebulaToolkitService.copyObjectByWhiteList(positionEntity, PositionVo.class, LinkedHashSet.class, ArrayList.class, "organization" , "roles" , "users");
  }

  @Override
  @Transactional
  public void bindOrg(String orgId, String positionId) {
    Validate.notBlank(orgId, "组织机构ID不能为空!");
    Validate.notBlank(positionId, "岗位ID不能为空!");
    Optional<OrganizationEntity> optional = organizationRepository.findById(orgId);
    Validate.isTrue(optional.isPresent(), "未找到指定的组织机构!");
    OrganizationEntity organization = optional.get();
    Optional<PositionEntity> op = positionRepository.findById(positionId);
    PositionEntity position = op.orElse(null);
    Validate.notNull(position, "未找到指定的岗位!");
    Validate.isTrue(null == position.getOrganization(), "该岗位已绑定有组织机构，请检查！");
    position.setOrganization(organization);
    positionRepository.saveAndFlush(position);
  }
  
  @Override
  @Transactional
  public void unbindOrg(String positionId) {
    Validate.notBlank(positionId, "岗位ID不能为空!");
    Optional<PositionEntity> op = positionRepository.findById(positionId);
    PositionEntity position = op.orElse(null);
    Validate.notNull(position, "未找到指定的岗位!");
    
    // 解除绑定关系
    position.setOrganization(null);
    positionRepository.saveAndFlush(position);
  }

  @Override
  @Transactional
  public void reBindOrg(String orgId, String positionId) {
    Validate.notBlank(orgId, "组织机构ID不能为空!");
    Validate.notBlank(positionId, "岗位ID不能为空!");
    Optional<OrganizationEntity> optional = organizationRepository.findById(orgId);
    Validate.isTrue(optional.isPresent(), "未找到指定的组织机构!");
    OrganizationEntity organization = optional.get();
    Optional<PositionEntity> op = positionRepository.findById(positionId);
    PositionEntity position = op.orElse(null);
    Validate.notNull(position, "未找到指定的岗位!");
    
    // 重新设定指定岗位的组织机构信息
    position.setOrganization(organization);
    positionRepository.saveAndFlush(position);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.PositionService#bindUsersAndPosition(java.util.List, java.lang.String)
   */
  @Override
  @Transactional
  public void bindUsersAndPosition(List<String> userIds, String positionId) {
    Validate.isTrue(!CollectionUtils.isEmpty(userIds), "用户ID集合不能为空，请检查！");
    UserMappingVo mappingVo = userMappingService.find();
    userIds.stream().forEach(o -> bindUserAndPosition(mappingVo, o, positionId));
  }
  /**
   * 验证绑定用户和岗位
   * @param userId
   * @param positionId
   * @param mappingVo
   */
  private void validateBindUser(UserMappingVo mappingVo,String positionId, String userId) {
    if (mappingVo == null) {
      long count = positionRepository.countByUserAndPosition(userId, positionId);
      Validate.isTrue(count == 0, "在批量绑定时,指定用户[%s]已经绑定了，不能重复绑定，请检查!!", userId);
    } else {
      int bindRelation = mappingVo.getUserPositionRelation();
      //如果用户和岗位绑定关系为多对多,多对一，一对多，一对一分别控制判断
      if (bindRelation == RbacRelationEnum.MANY_TO_MANY.getRelation()) {
        long count = positionRepository.countByUserAndPosition(userId, positionId);
        Validate.isTrue(count == 0, "在批量绑定时,用户和岗位多对多，指定用户[%s]已经绑定了，不能重复绑定，请检查!!", userId);
      }
      if (bindRelation == RbacRelationEnum.MANY_TO_ONE.getRelation()) {
        long count = positionRepository.countPositionByUserId(userId);

        Validate.isTrue(count == 0, "在批量绑定时,用户和岗位多对一，指定用户[%s]已经绑定了岗位，不能继续绑定，请检查!!", userId);
      }
      if (bindRelation == RbacRelationEnum.ONE_TO_MANY.getRelation()) {
        long count = positionRepository.countUserByPositionId(positionId);
        Validate.isTrue(count == 0, "在批量绑定时,用户和岗位一对多，指定岗位[%s]已经绑定了用户，不能继续绑定，请检查!!", positionId);
      }
      if (bindRelation == RbacRelationEnum.ONE_TO_ONE.getRelation()) {
        long countUser = positionRepository.countUserByPositionId(positionId);
        Validate.isTrue(countUser == 0, "在批量绑定时,用户和岗位一对一，指定岗位[%s]已经绑定了用户，不能继续绑定，请检查!!", positionId);
        long countPosition = positionRepository.countPositionByUserId(userId);
        Validate.isTrue(countPosition == 0, "在批量绑定时,用户和岗位一对一，指定用户[%s]已经绑定了岗位，不能继续绑定，请检查!!", userId);
      }

    }
  }
  /**
   * 绑定用户和岗位
   * @param userId
   * @param positionId
   */
  private void bindUserAndPosition(UserMappingVo mappingVo,String userId, String positionId) {
    Validate.notBlank(userId, "用户ID不能为空!");
    Validate.notBlank(positionId, "岗位ID不能为空!");
    UserVo user = userService.findByUserId(userId);
    Validate.notNull(user, "未找到指定的用户!");

    Optional<PositionEntity> optional = positionRepository.findById(positionId);
    Validate.isTrue(optional.isPresent(), "未找到指定的岗位!");
    this.validateBindUser(mappingVo, positionId, userId);
    positionRepository.bindUser(userId, positionId);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.PositionService#unbindUsersAndPosition(java.util.List, java.lang.String)
   */
  @Override
  @Transactional
  public void unbindUsersAndPosition(List<String> userIds, String positionId) {
    Validate.notEmpty(userIds, "用户ID集合不能为空，请检查！");
    userIds.stream().forEach(o -> unbindUserAndPosition(o, positionId));
  }

  private void unbindUserAndPosition(String userId, String positionId) {
    Validate.notBlank(userId, "用户ID不能为空!");
    Validate.notBlank(positionId, "岗位ID不能为空!");
    UserVo user = userService.findByUserId(userId);
    Validate.notNull(user, "未找到指定的用户!");
    PositionEntity position = positionRepository.findById(positionId).orElse(null);
    Validate.notNull(position, "未找到指定的岗位!");
    Validate.isTrue(0L != positionRepository.countByUserAndPosition(userId, positionId), "该用户与该岗位未有绑定关系，请检查！");
    //主岗位不能解绑
    PositionVo positionVo = null;
    String key = Constants.REDIS_KEY_RBAC_USER + userId;
    Map<String, String> mainPosition = redissonClient.getMap(key);
    if(mainPosition.containsKey(Constants.MAIN_POSITION)){
      positionVo = JSONObject.parseObject(mainPosition.get(Constants.MAIN_POSITION), PositionVo.class);
    }
    if(positionVo != null) {
      Validate.isTrue(!positionId.equals(positionVo.getId()), "[%s]是用户的主岗位不能解绑", position.getName());
    }
    positionRepository.unbindUser(userId, positionId);
  }
  
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.PositionService#findByUserId(java.lang.String)
   */
  @Override
  public List<PositionVo> findByUserId(String userId) {
    if(StringUtils.isBlank(userId)) {
      return Lists.newArrayList();
    }
    Set<PositionEntity> positions = positionRepository.findByUserId(userId);
    if(CollectionUtils.isEmpty(positions)) {
      return Lists.newArrayList();
    }
    Collection<PositionVo> positionVos = this.nebulaToolkitService.copyCollectionByWhiteList(positions, PositionEntity.class, PositionVo.class, HashSet.class, ArrayList.class);
    return Lists.newArrayList(positionVos);
  }

  @Override
  public Set<PositionVo> findByIds(List<String> ids) {
    if(CollectionUtils.isEmpty(ids)){
      return Sets.newHashSet();
    }
    Set<PositionEntity> pos = positionRepository.findByIds(ids);
    if(CollectionUtils.isEmpty(pos)){
      return Sets.newHashSet();
    }
    Collection<PositionVo> pvo = this.nebulaToolkitService.copyCollectionByWhiteList(pos,PositionEntity.class,PositionVo.class,HashSet.class,ArrayList.class);
    return Sets.newHashSet(pvo);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.PositionService#findByConditions(java.lang.String, java.lang.String, java.lang.Integer, org.springframework.data.domain.Pageable)
   */
  @Override
  public Page<PositionVo> findByConditions(String code, String name, Integer tstatus, Pageable pageable) {
    Map<String, Object> conditions = new HashMap<>();
    if (StringUtils.isNotBlank(code)) {
      conditions.put("code", code);
    }
    if (StringUtils.isNotBlank(name)) {
      conditions.put("name", name);
    }
    if (tstatus != null) {
      conditions.put("tstatus", tstatus);
    }
    // 如果当前没有设定分页信息，则默认第一页，每页50条数据
    if (pageable == null) {
      pageable = PageRequest.of(0, 50);
    }
  
    Page<PositionEntity> positionEntityPage = positionRepository.queryPage(pageable, conditions);
    // 分页信息需要自行转换
    List<PositionEntity> positionEntitys = positionEntityPage.getContent();
    Page<PositionVo> positionVoPage = null;
    if(!positionEntitys.isEmpty()) {
      Collection<PositionVo> positionVos = this.nebulaToolkitService.copyCollectionByWhiteList(positionEntitys, PositionEntity.class, PositionVo.class, LinkedHashSet.class, ArrayList.class, "users", "parent", "children");
      positionVoPage = new PageImpl<>(new ArrayList<>(positionVos), pageable, positionEntityPage.getTotalElements());
    } else {
      positionVoPage = new PageImpl<>(Lists.newArrayList(), pageable, 0l);
    }
    return positionVoPage;
  }

  @Override
  public PositionVo findByCode(String code) {
    if(StringUtils.isBlank(code)) {
      return null;
    }
    PositionEntity position = positionRepository.findByCode(code);
    if(position == null) {
      return null;
    }
    return nebulaToolkitService.copyObjectByWhiteList(position, PositionVo.class, HashSet.class, ArrayList.class);
  }

  /**
   * 切换岗位
   * @param positionId
   */
  @Override
  public UserVo changePosition(String positionId, String userId){
    Validate.notBlank(positionId, "切换岗位id不能为空");
    Validate.notBlank(userId, "用户id不能为空");

    UserVo user = userService.findByUserId(userId);
    Validate.notNull(user, "未获取到用户信息，请检查");

    PositionEntity positionEntity = positionRepository.findById(positionId).orElse(null);
    Validate.notNull(positionEntity, "未获取到岗位信息，请检查");
    PositionVo positionVo = nebulaToolkitService.copyObjectByWhiteList(positionEntity, PositionVo.class, HashSet.class, ArrayList.class);

    List<PositionVo> positionVos = this.findByUserId(userId);
    user.setPositions(Sets.newLinkedHashSet(positionVos));
    user.setMainPosition(positionId);
    Map<String, String> mainPosition = redissonClient.getMap(Constants.REDIS_KEY_RBAC_USER + userId);
    mainPosition.put(Constants.MAIN_POSITION, JSONObject.toJSON(positionVo).toString());
    return user;
  }

  /**
   * 查询Redis中主岗位，如果redis中不存在，则默认主岗位为数据库查询的第一条数据，并且将该数据存入redis中
   * @param userId
   * @return
   */
  @Override
  public PositionVo findMainPositionByUserId(String userId){
    Validate.notBlank(userId, "传入的用户id不能为空");
    PositionVo positionVo = null;
    String key = Constants.REDIS_KEY_RBAC_USER + userId;
    Map<String, String> mainPosition = redissonClient.getMap(key);
    if(mainPosition.containsKey(Constants.MAIN_POSITION)){
      positionVo = JSONObject.parseObject(mainPosition.get(Constants.MAIN_POSITION), PositionVo.class);
    }
    if(positionVo == null){
      List<PositionVo> positionVoList = this.findByUserId(userId);
      if(CollectionUtils.isEmpty(positionVoList)){
        return null;
      }else {
        positionVo = positionVoList.get(0);
      }
      String json = JSONObject.toJSON(positionVo).toString();
      mainPosition.put(Constants.MAIN_POSITION, json);
    }
    return positionVo;
  }

  /**
   * 重新给岗位绑定用户关系
   * @param userIds
   * @param positionId
   */
  @Override
  @Transactional
  public void rebindUsers(String[] userIds, String positionId) {
    /***
     * 1.用户已绑定不再重复绑定
     * 2.用户未绑定进行绑定
     * 3.该岗位以前的用户进行解绑
     */
    Validate.notBlank(positionId, "岗位ID不能为空!");
    //已绑定过的用户id
    Set<UserVo> users = userService.findByPositionId(positionId);
    if(users == null) {
      users = Sets.newHashSet();
    }
    Set<String> bindedUserIds = users.stream().map(UserVo::getId).collect(Collectors.toSet());
    //传入的用户id
    Set<String> currentUserIds = new HashSet<>();
    if (userIds != null){
      for (String userId : userIds) {
        currentUserIds.add(userId);
      }
    }
    //解绑用户
    Set<String> needUnbindsUserIds = Sets.difference(bindedUserIds, currentUserIds);
    if (!CollectionUtils.isEmpty(needUnbindsUserIds)){
      for (String userId : needUnbindsUserIds) {
        if (StringUtils.isBlank(userId)){
          continue;
        }
        this.unbindUserAndPosition(userId, positionId);
      }
    }
    //绑定用户
    Set<String> needbindsUserIds = Sets.difference(currentUserIds, bindedUserIds);
    UserMappingVo mappingVo = userMappingService.find();
    if (!CollectionUtils.isEmpty(needbindsUserIds)){
      for (String userId : needbindsUserIds) {
        if (StringUtils.isBlank(userId)){
          continue;
        }
        this.bindUserAndPosition(mappingVo, userId, positionId);
      }
    }
  }

  /**
   * 查询所有岗位信息（根据创建时间排序）
   * @return
   */
  @Override
  public List<PositionVo> findAll(){
    Sort sort = Sort.by(Sort.Order.desc("createTime"));
    List<PositionEntity> positions = positionRepository.findAll(sort);
    if (CollectionUtils.isEmpty(positions)) {
      return Lists.newArrayList();
    }
    Collection<PositionVo> positionVos = nebulaToolkitService.copyCollectionByWhiteList(positions, PositionEntity.class, PositionVo.class, HashSet.class, ArrayList.class, "parent");
    return Lists.newArrayList(positionVos);
  }

  /**
   * 根据条件查询所有岗位
   * @param status
   * @return
   */
  @Override
  public List<PositionVo> findByStatus(Integer status) {
    List<PositionEntity> positions = positionRepository.findByTstatus(status);
    if(CollectionUtils.isEmpty(positions)) {
      return Lists.newArrayList();
    }
    Collection<PositionVo> collection = nebulaToolkitService.copyCollectionByWhiteList(positions, PositionEntity.class, PositionVo.class, HashSet.class, ArrayList.class);
    return Lists.newArrayList(collection);
  }

  /**
   * 根据登陆人关联的岗位查询
   * @param type
   * @return
   */
  @Override
  public Set<PositionVo> findByPrincipal(Integer type, Principal principal) {
    Validate.notNull(type, "传入类型不能为空，请检查");
    String account = principal.getName();
    UserEntity user = userRepository.findByAccount(account);
    Validate.notNull(user, "未获取登陆人信息，请检查");
    Set<PositionVo> positions = null;
    switch (type) {
      case 1:
        positions = this.findParent(user);
        break;
      case 2:
        positions = this.findChildren(user);
        break;
      case 3:
        positions = this.findParentAndChildren(user);
        break;
      case 4:
        positions = this.findPosition(user);
        break;
      default:
        break;
    }
    if(CollectionUtils.isEmpty(positions)){
      return Sets.newHashSet();
    }
    return positions;
  }

  /**
   * 根据登陆人关联的岗位查询上级
   * @param user
   * @return
   */
  private Set<PositionVo> findParent(UserEntity user) {
    Set<PositionVo> result = new HashSet<>();
    Set<PositionVo> removes = new HashSet<>();
    Map<String, PositionVo> parentMap = new HashMap<>();
    Set<PositionVo> rootPositions = this.findPosition(user);
    for(PositionVo rootPosition : rootPositions) {
      rootPosition.setParent(this.findAllParent(rootPosition, parentMap));
      result.add(rootPosition);
    }
    //过滤
    for(PositionVo item : result){
      if(parentMap.containsKey(item.getId())){
        removes.add(item);
      }
    }
    for(PositionVo remove : removes){
      result.remove(remove);
    }
    return result;
  }

  /**
   * 根据登陆人关联的岗位查询
   * @param user
   * @return
   */
  private Set<PositionVo> findPosition(UserEntity user) {
    Set<PositionEntity> positions = positionRepository.findByUserId(user.getId());
    Collection<PositionVo> result = nebulaToolkitService.copyCollectionByWhiteList(positions, PositionEntity.class, PositionVo.class, LinkedHashSet.class, ArrayList.class);
    return Sets.newHashSet(result);
  }

  /**
   * 查询父级并将下级信息存入map
   * @param rootPosition
   * @param parentMap
   * @return
   */
  private PositionVo findAllParent(PositionVo rootPosition, Map<String, PositionVo> parentMap) {
    // 查询父级结构
    PositionEntity currentPosition = positionRepository.findById(rootPosition.getId()).orElse(null);
    PositionEntity parent = currentPosition.getParent();
    if(parent == null){
      return null;
    }
    PositionVo positionVo = this.nebulaToolkitService.copyObjectByWhiteList(parent, PositionVo.class, HashSet.class, ArrayList.class);
    parentMap.put(positionVo.getId(), positionVo);
    positionVo.setParent(this.findAllParent(positionVo, parentMap));
    return positionVo;
  }

  /**
   * 根据登陆人关联的岗位查询下级
   * @param user
   * @return
   */
  private Set<PositionVo> findChildren(UserEntity user) {
    Set<PositionVo> result = new HashSet<>();
    Set<PositionVo> removes = new HashSet<>();
    Map<String, PositionVo> childMap = new HashMap<>();
    Set<PositionVo> rootPositions = this.findPosition(user);
    for(PositionVo rootPosition : rootPositions) {
      rootPosition.setChildren(this.findAllChildren(rootPosition, childMap));
      result.add(rootPosition);
    }
    //过滤
    for(PositionVo item : result){
      if(childMap.containsKey(item.getId())){
        removes.add(item);
      }
    }
    for(PositionVo remove : removes){
      result.remove(remove);
    }
    return result;
  }

  /**
   * 查询下级并将下级信息存入map
   * @param position
   * @return
   */
  private Set<PositionVo> findAllChildren(PositionVo position, Map<String, PositionVo> childMap) {
    Set<PositionVo> result = new HashSet<>();
    // 查询子级结构
    PositionEntity currentPosition = positionRepository.findById(position.getId()).orElse(null);
    Set<PositionEntity> children = currentPosition.getChildren();
    if(CollectionUtils.isEmpty(children)){
      return Sets.newHashSet();
    }
    for (PositionEntity child : children) {
      PositionVo positionVo = this.nebulaToolkitService.copyObjectByWhiteList(child, PositionVo.class, HashSet.class, ArrayList.class);
      childMap.put(positionVo.getId(), positionVo);
      positionVo.setChildren(this.findAllChildren(positionVo, childMap));
      result.add(positionVo);
    }
    return result;
  }

  /**
   * 根据登陆人关联的岗位查询上下级
   * @param user
   * @return
   */
  private Set<PositionVo> findParentAndChildren(UserEntity user) {
    Set<PositionVo> result = new HashSet<>();
    Set<PositionVo> removes = new HashSet<>();
    Map<String, PositionVo> childMap = new HashMap<>();
    Map<String, PositionVo> parentMap = new HashMap<>();
    Set<PositionVo> rootPositions = this.findPosition(user);
    for(PositionVo rootPosition : rootPositions) {
      rootPosition.setChildren(this.findAllChildren(rootPosition, childMap));
      rootPosition.setParent(this.findAllParent(rootPosition, parentMap));
      result.add(rootPosition);
    }
    //过滤
    for(PositionVo item : result){
      if(childMap.containsKey(item.getId())){
        removes.add(item);
      }
    }
    for(PositionVo remove : removes){
      result.remove(remove);
    }
    return result;
  }

  /**
   * 根据登陆人关联的岗位和输入层级查询控件
   * @param type
   * @param principal
   * @param level
   * @return
   */
  @Override
  public Set<PositionVo> findByTypeAndLevel(Integer type, Principal principal, Integer level) {
    Validate.notNull(type, "传入类型不能为空，请检查");
    Validate.notNull(level, "传入层级不能为空，请检查");
    String account = principal.getName();
    UserEntity user = userRepository.findByAccount(account);
    Validate.notNull(user, "未获取登陆人信息，请检查");
    Set<PositionVo> positions = null;
    String[] propertiesFilter = new String[level];
    Set<PositionVo> newPositions = new HashSet<>();
    switch (type) {
      case 1:
        positions = this.findParent(user);
        this.bindPropertiesFilter(level, propertiesFilter, "parent");
        break;
      case 2:
        positions = this.findChildren(user);
        this.bindPropertiesFilter(level, propertiesFilter, "children");
        break;
      default:
        break;
    }
    if(CollectionUtils.isEmpty(positions)){
      return Sets.newHashSet();
    }
    for (PositionVo position : positions) {
      PositionVo positionVo = this.nebulaToolkitService.copyObjectByWhiteList(position, PositionVo.class, HashSet.class, ArrayList.class, propertiesFilter);
      newPositions.add(positionVo);
    }
    return newPositions;
  }

  /**
   * 处理参数
   * @param level
   * @param propertiesFilter
   * @return
   */
  private void bindPropertiesFilter(Integer level, String[] propertiesFilter, String type) {
    String temp = "";
    for(int i = 0 ; i < level; i++){
      temp = temp + type + ".";
      if((i + 1 ) == level){
        temp = temp.substring(0, temp.length() - 1);
      }
      propertiesFilter[i] = temp;
    }
  }
}
