package com.bizunited.platform.user.service.local.repository;

import com.bizunited.platform.user.service.local.entity.OrganizationEntity;
import com.bizunited.platform.user.service.local.entity.PositionEntity;
import com.bizunited.platform.user.service.local.repository.internal.PositionRepositoryCustom;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.JpaSpecificationExecutor;
import org.springframework.data.jpa.repository.Modifying;
import org.springframework.data.jpa.repository.Query;
import org.springframework.data.repository.query.Param;
import org.springframework.stereotype.Repository;

import java.util.List;
import java.util.Set;

/**
 * PositionRepository
 *
 * @description:
 * @author: yanwe
 * @date: 17/Jan/2019 11:16
 */
@Repository("PositionRepository")
public interface PositionRepository
    extends JpaRepository<PositionEntity, String>, JpaSpecificationExecutor<PositionEntity>, PositionRepositoryCustom {

  /**
   * 根据编号查询
   *
   * @param code
   * @return
   */
  PositionEntity findByCode(@Param("code") String code);

  /**
   * 根据名称查询
   * @param name
   * @return
   */
  List<PositionEntity> findByName(@Param("name") String name);

  /**
   * 根据ID查询岗位的详细信息，包括相关组织、角色、用户等
   *
   * @param id
   * @return
   */
  @Query(
      "select distinct position from PositionEntity position "
          + "left join fetch position.organization "
          + "left join fetch position.users "
          + "left join fetch position.parent "
          + "where position.id = :id")
  PositionEntity findDetailsById(@Param("id") String id);

  /**
   * 根据Code查询岗位的详细信息，包括相关组织、角色、用户等
   *
   * @param code
   * @return
   */
  @Query(
      "select distinct position from PositionEntity position "
          + "left join fetch position.organization "
          + "left join fetch position.users "
          + "left join fetch position.positionLevel "
          + "where position.code = :code ")
  PositionEntity findDetailsByCode(@Param("code") String code);

  /**
   * 查询某一组织下所有岗位
   *
   * @param organization
   * @return
   */
  List<PositionEntity> findALlByOrAndOrganization(OrganizationEntity organization);

  /**
   * 查询某一用户与某一职位绑定关系
   * @param userId
   * @param positionId
   * @return
   */
  @Query(value = "SELECT COUNT(*) FROM engine_position_user_mapping WHERE user_id = :userId AND position_id = :positionId",nativeQuery = true)
  long countByUserAndPosition(@Param("userId") String userId, @Param("positionId") String positionId);

  /**
   * 绑定某一用户与某一职位关系
   *
   * @param userId
   * @param positionId
   */
  @Modifying
  @Query(
      value =
          "insert into engine_position_user_mapping(user_id,position_id) values (:userId,:positionId)",
      nativeQuery = true)
  void bindUser(@Param("userId") String userId, @Param("positionId") String positionId);

  /**
   * 解绑某一用户与某一职位关系
   *
   * @param userId
   * @param positionId
   */
  @Modifying
  @Query(
      value =
          "delete from engine_position_user_mapping where user_id = :userId and position_id = :positionId",
      nativeQuery = true)
  void unbindUser(@Param("userId") String userId, @Param("positionId") String positionId);

  /**
   * 解绑某一用户的所有职位绑定关系
   *
   * @param userId
   */
  @Modifying
  @Query(
      value = "delete from engine_position_user_mapping where user_id = :userId",
      nativeQuery = true)
  void unbindusers(@Param("userId") String userId);

  /**
   * 查询多个岗位信息
   * @param ids
   * @return
   */
  @Query("from PositionEntity m where m.id in :ids")
  public Set<PositionEntity> findByIds(@Param("ids") List<String> ids);

  /**
   * 根据用户ID查询岗位
   * @param userId
   */
  @Query("select p from PositionEntity p " +
          "inner join fetch p.users u " +
          "left join fetch p.children " +
          "where u.id = :userId")
  Set<PositionEntity> findDetailsByUserId(@Param("userId") String userId);

  /**
   * 根据用户ID查询岗位
   * @param userId
   */
  @Query("select p from PositionEntity p " +
      "inner join p.users u " +
      "where u.id = :userId order by p.createTime desc")
  Set<PositionEntity> findByUserId(@Param("userId") String userId);

  /**
   * 根据用户Account查询岗位
   * @param userAccount
   */
  @Query("select p from PositionEntity p " +
      "inner join p.users u " +
      "where u.account = :userAccount order by p.createTime desc")
  Set<PositionEntity> findByUserAccount(@Param("userAccount") String userAccount);

  /**
   * 查询某一用户是否已经绑定了职位
   * @param userId
   * @return
   */
  @Query(value = "SELECT COUNT(*) FROM engine_position_user_mapping WHERE user_id = :userId",nativeQuery = true)
  long countPositionByUserId(@Param("userId") String userId);

  /**
   * 查询某一职位是否已经与用户绑定
   * @param positionId
   * @return
   */
  @Query(value = "SELECT COUNT(*) FROM engine_position_user_mapping WHERE position_id = :positionId",nativeQuery = true)
  long countUserByPositionId(@Param("positionId") String positionId);

  /**
   * 根据岗位状态查询
   * @return
   */
  @Query("from PositionEntity m where m.tstatus =:tstatus order by m.createTime desc ")
  List<PositionEntity> findByTstatus(@Param("tstatus") int tstatus);

  /**
   * 查询职位名称最后一个
   * @param levelId
   * @return
   */
  @Query(value = "SELECT MAX(name) FROM engine_position WHERE position_level_id = :levelId",nativeQuery = true)
  String findMaxName(@Param("levelId") String levelId);

  /**
   * 根据组织编码相关信息（包括组织，职级，用户）
   * @param orgCode
   * @return
   */
  @Query("select p from PositionEntity p " +
          "left join fetch p.positionLevel " +
          "left join fetch p.users " +
          "left join fetch p.organization o " +
          "left join fetch p.children children " +
          "where o.code = :orgCode " )
  List<PositionEntity> findDetailsByOrgCode(@Param("orgCode") String orgCode);

  /**
   * 根据上级id查询职位信息（包括关联信息）
   * @param parentId
   * @return
   */
  @Query("select p from PositionEntity p " +
          "left join fetch p.users " +
          "left join fetch p.organization o " +
          "where p.parent.id = :parentId " )
  List<PositionEntity> findDetailsByParentId(@Param("parentId") String parentId);

  /**
   * 根据账户名查询岗位
   * @param account
   */
  @Query("select p from PositionEntity p " +
          "inner join p.users u " +
          "where u.account = :account order by p.createTime desc")
  Set<PositionEntity> findByAccount(@Param("account") String account);
  /**
   * 根据职级批量查询关联的职位
   * @param positionLevelIds
   * @return
   */
  @Query("select p from PositionEntity p " +
          "left join fetch p.positionLevel " +
          "left join fetch p.organization " +
          "left join fetch p.users " +
          "where p.positionLevel.id in :positionLevelIds")
  Set<PositionEntity> findByPositionLevelIds(@Param("positionLevelIds") String[] positionLevelIds);


  /**
   * 查询全部职位以及明细
   * //TODO 暂时用 之后替换为分页查询
   * @return
   */
  @Query("select distinct p from PositionEntity p " +
          "left join fetch p.users " +
          "left join fetch p.organization" +
          " left join fetch p.positionLevel")
  Set<PositionEntity> findAllDetail();
}
