package com.bizunited.platform.user.service.local.service.internal;

import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.core.annotations.NebulaServiceMethod;
import com.bizunited.platform.core.annotations.NebulaServiceMethod.ScopeType;
import com.bizunited.platform.core.annotations.ServiceMethodParam;
import com.bizunited.platform.core.service.invoke.InvokeParams;
import com.bizunited.platform.user.common.enums.RegionLevelEnum;
import com.bizunited.platform.user.common.service.region.AdministrativeRegionService;
import com.bizunited.platform.user.common.vo.AdministrativeRegionVo;
import com.bizunited.platform.user.service.local.entity.AdministrativeRegionEntity;
import com.bizunited.platform.user.service.local.repository.AdministrativeRegionRepository;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.redisson.api.RMap;
import org.redisson.api.RedissonClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import static com.bizunited.platform.common.constant.Constants.LEVEL;
import static com.bizunited.platform.common.constant.Constants.REDIS_KEY_ORG_REGION_LEVEL;


/**
 * AdministrativeRegionEntity业务模型的服务层接口实现
 *
 * @author saturn
 */
public class AdministrativeRegionServiceImpl implements AdministrativeRegionService {

  @Autowired
  private AdministrativeRegionRepository administrativeRegionRepository;

  @Autowired
  private RedissonClient redissonClient;

  /**
   * Kuiper表单引擎用于减少编码工作量的工具包
   */
  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  private static final String CHILDREN = "children";
  private static final String CHILDREN_CHILDREN = "children.children";

  @Override
  @Transactional
  public AdministrativeRegionVo create(AdministrativeRegionVo region) {
    this.createValidation(region);
    AdministrativeRegionEntity dbRegion = nebulaToolkitService.copyObjectByWhiteList(region, AdministrativeRegionEntity.class, HashSet.class, ArrayList.class, "parent");
    dbRegion.setDimensionalityCode("");
    this.initLevel(dbRegion);
    administrativeRegionRepository.save(dbRegion);
    return nebulaToolkitService.copyObjectByWhiteList(dbRegion, AdministrativeRegionVo.class, HashSet.class, ArrayList.class, "parent");
  }

  /**
   * 初始化层级
   * @param dbRegion
   * @return
   */
  private void initLevel(AdministrativeRegionEntity region) {
    int level = 1;
    AdministrativeRegionEntity parent = region.getParent();
    if(parent != null) {
      AdministrativeRegionEntity dbParent = administrativeRegionRepository.findById(parent.getId()).orElse(null);
      Validate.notNull(dbParent, "未找到上级区域：%s", parent.getId());
      level++;
      while ((dbParent = dbParent.getParent()) != null) {
        level++;
      }
    }
    region.setRegionLevel(level);
  }

  /**
   * 新增数据前校验数据
   * @param region
   */
  private void createValidation(AdministrativeRegionVo region) {
    Validate.notNull(region, "行政区域数据不能为空！");
    Validate.isTrue(region.getId() == null, "新增数据不能有主键ID");
    Validate.notBlank(region.getRegionCode(), "编码不能为空！");
    Validate.notBlank(region.getRegionName(), "名称不能为空！");
    Validate.notBlank(region.getLongitude(), "经度不能为空！");
    Validate.notBlank(region.getLatitude(), "纬度不能为空！");
    AdministrativeRegionVo parent = region.getParent();
    if(parent != null) {
      Validate.notBlank(parent.getId(), "上级区域ID不能为空");
      AdministrativeRegionEntity dbParent = administrativeRegionRepository.findById(parent.getId()).orElse(null);
      Validate.notNull(dbParent, "未找到上级区域：%s", parent.getId());
    }
    AdministrativeRegionEntity dbRegion = administrativeRegionRepository.findByRegionCode(region.getRegionCode());
    Validate.isTrue(dbRegion == null, "行政区域已存在: %s，请检查数据", region.getRegionCode());
  }

  @Override
  @NebulaServiceMethod(name = "AdministrativeRegionService.findDetailsById" , desc = "根据ID查询明细", scope = NebulaServiceMethod.ScopeType.READ, returnPropertiesFilterB = "..children,..orgs" )
  public AdministrativeRegionVo findDetailsById(@ServiceMethodParam(name = "id") String id) {
    // 这是主模型下的明细查询过程
    // 1、=======
    if (StringUtils.isBlank(id)) {
      return null;
    }
    AdministrativeRegionEntity current = this.administrativeRegionRepository.findDetailsById(id);
    if (current == null) {
      return null;
    }
    return nebulaToolkitService.copyObjectByWhiteList(current, AdministrativeRegionVo.class, HashSet.class, ArrayList.class, "..children");
  }

  @Override
  public AdministrativeRegionVo findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    AdministrativeRegionEntity administrativeRegionEntity = administrativeRegionRepository.findById(id).orElse(null);
    if(administrativeRegionEntity == null){
      return null;
    }
    return nebulaToolkitService.copyObjectByWhiteList(administrativeRegionEntity, AdministrativeRegionVo.class, HashSet.class, ArrayList.class,CHILDREN, CHILDREN_CHILDREN);
  }

  @Override
  @NebulaServiceMethod(name="AdministrativeRegionService.findByParentIdOrConditions" , desc="根据上级组织ID或多条件查询" , returnPropertiesFilterB = "..parent,..orgs" , scope=ScopeType.READ)
  public List<AdministrativeRegionVo> findByParentIdOrConditions(InvokeParams invokeParams) {
    invokeParams = ObjectUtils.defaultIfNull(invokeParams, new InvokeParams());
    Map<String, Object> params = invokeParams.getInvokeParams();
    // 清除空的值
    Set<String> keys = Sets.newHashSet("parentId", "regionName", "regionCode", "longitude", "latitude");
    params.entrySet().removeIf(entry -> !keys.contains(entry.getKey()) || entry.getValue() == null || (CharSequence.class.isAssignableFrom(entry.getValue().getClass()) && StringUtils.isBlank((CharSequence) entry.getValue())));
    String parentId = (String) params.get("parentId");
    params.remove("parentId");
    if(CollectionUtils.isEmpty(params)) {
      // 如果没有其他条件，则根据上级ID查询
      return this.findByParent(parentId);
    }
    return this.findByConditions(params);
  }

  @Override
  @NebulaServiceMethod(name="AdministrativeRegionService.findByConditions" , desc="多条件查询，返回树结构" , returnPropertiesFilterB = "..parent,..orgs" , scope=ScopeType.READ)
  public List<AdministrativeRegionVo> findByConditions(InvokeParams invokeParams) {
    invokeParams = ObjectUtils.defaultIfNull(invokeParams, new InvokeParams());
    Map<String, Object> params = invokeParams.getInvokeParams();
    return this.findByConditions(params);
  }

  /**
   * 组合查询，返回结果为一个树形结构。这个树形结构不包括以下
   * @param vo 查询条件，若为空则查询所有顶级区域
   * @return
   */
  @Override
  public List<AdministrativeRegionVo> findByConditions(Map<String, Object> conditions) {
    conditions = ObjectUtils.defaultIfNull(conditions, new HashMap<>());
    // 清除空的值
    Set<String> keys = Sets.newHashSet("parentId", "regionName", "regionCode", "longitude", "latitude");
    conditions.entrySet().removeIf(entry -> !keys.contains(entry.getKey()) || entry.getValue() == null || (CharSequence.class.isAssignableFrom(entry.getValue().getClass()) && StringUtils.isBlank((CharSequence) entry.getValue())));
    boolean emptyCondition = CollectionUtils.isEmpty(conditions);
    List<AdministrativeRegionEntity> regions;
    if(emptyCondition) {
      regions = administrativeRegionRepository.findAllDetails();
      if(CollectionUtils.isEmpty(regions)) {
        return Lists.newArrayList();
      }
    } else {
      regions = this.administrativeRegionRepository.findByConditions(conditions);
      if(CollectionUtils.isEmpty(regions)) {
        return Lists.newArrayList();
      }
      // 在有搜索条件的情况下，为了不返回多余的数据，先将查询的数据转换为非持久化的数据
      Collection<AdministrativeRegionEntity> collection = this.nebulaToolkitService.copyCollectionByBlankList(regions, AdministrativeRegionEntity.class, AdministrativeRegionEntity.class, LinkedHashSet.class, ArrayList.class, "..children", "..orgs");
      regions = Lists.newArrayList(collection);
      this.handleChildren(regions);
    }
    List<AdministrativeRegionEntity> regionTree = this.list2Tree(regions, emptyCondition);
    Collection<AdministrativeRegionVo> collection = this.nebulaToolkitService.copyCollectionByBlankList(regionTree, AdministrativeRegionEntity.class, AdministrativeRegionVo.class, LinkedHashSet.class, ArrayList.class , "..parent", "..orgs");
    this.handleEmptyChildren(collection);
    return collection.stream().sorted(Comparator.comparing(AdministrativeRegionVo::getId)).collect(Collectors.toList());
  }

  /**
   * 处理空子集，将空子集children设置为null
   * @param collection
   */
  private void handleEmptyChildren(Collection<AdministrativeRegionVo> collection) {
    for (AdministrativeRegionVo region : collection) {
      List<AdministrativeRegionVo> children = region.getChildren();
      if(CollectionUtils.isEmpty(children)) {
        region.setChildren(null);
      } else {
        this.handleEmptyChildren(children);
      }
    }
  }

  /**
   * 处理子集
   * @param regions
   */
  private void handleChildren(List<AdministrativeRegionEntity> regions) {
    for (AdministrativeRegionEntity region : regions) {
      AdministrativeRegionEntity parent = region.getParent();
      while (parent != null) {
        this.addChild(parent, region);
        region = parent;
        parent = region.getParent();
      }
    }
  }

  /**
   * 增加子集
   * @param region
   * @param child
   */
  private void addChild(AdministrativeRegionEntity region, AdministrativeRegionEntity child) {
    List<AdministrativeRegionEntity> children = region.getChildren();
    children = ObjectUtils.defaultIfNull(children, new ArrayList<>());
    children.add(child);
    region.setChildren(children);
  }

  /**
   * 根据层级查询,结果不包含任何父子关系
   *
   * @param regionLevel
   * @return
   */
  @Override
  public Set<AdministrativeRegionVo> findByRegionLevel(Integer regionLevel) {
    if (regionLevel == null) {
      return Sets.newHashSet();
    }
    List<AdministrativeRegionEntity> administrativeRegionEntities = this.administrativeRegionRepository.findByRegionLevel(regionLevel);
    if (CollectionUtils.isEmpty(administrativeRegionEntities)) {
      return Sets.newHashSet();
    }
    Collection<AdministrativeRegionVo> administrativeRegionVos = this.nebulaToolkitService.copyCollectionByWhiteList(administrativeRegionEntities, AdministrativeRegionEntity.class, AdministrativeRegionVo.class, LinkedHashSet.class, LinkedList.class);
    return Sets.newHashSet(administrativeRegionVos);
  }

  @Override
  @NebulaServiceMethod(name = "AdministrativeRegionService.findByOrgId", desc = "根据组织ID查询关联区域", returnPropertiesFilterB = "..children,..orgs", scope = NebulaServiceMethod.ScopeType.READ)
  public List<AdministrativeRegionVo> findByOrgId(@ServiceMethodParam(name = "id") String orgId) {
    if (StringUtils.isBlank(orgId)) {
      return Lists.newArrayList();
    }
    Integer level = this.findOrgMappingLevel();
    level = ObjectUtils.defaultIfNull(level, RegionLevelEnum.AREA.getLevel());
    List<AdministrativeRegionEntity> regions = administrativeRegionRepository.findByOrgIdAndLevel(orgId, level);
    Collection<AdministrativeRegionVo> collection = nebulaToolkitService.copyCollectionByBlankList(regions, AdministrativeRegionEntity.class, AdministrativeRegionVo.class, LinkedHashSet.class, ArrayList.class, "..children", "..orgs");
    return Lists.newArrayList(collection);
  }

  @Override
  @NebulaServiceMethod(name = "AdministrativeRegionService.findByParent", desc = "根据父ID查询子区域", returnPropertiesFilter = "", scope = NebulaServiceMethod.ScopeType.READ)
  public List<AdministrativeRegionVo> findByParent(@ServiceMethodParam(name = "parentId") String parentId) {
    List<AdministrativeRegionEntity> regions;
    if(StringUtils.isBlank(parentId)) {
      regions = administrativeRegionRepository.findByNullParent();
    } else {
      regions = administrativeRegionRepository.findByParentId(parentId);
    }
    if(CollectionUtils.isEmpty(regions)) {
      return Lists.newArrayList();
    }
    Collection<AdministrativeRegionVo> collection = nebulaToolkitService.copyCollectionByWhiteList(regions, AdministrativeRegionEntity.class, AdministrativeRegionVo.class, LinkedHashSet.class, ArrayList.class);
    return collection.stream().sorted(Comparator.comparing(AdministrativeRegionVo::getId)).collect(Collectors.toList());
  }

  @Override
  @NebulaServiceMethod(name = "AdministrativeRegionService.findByCode", desc = "根据编码查询行政区域", returnPropertiesFilter = "", scope = NebulaServiceMethod.ScopeType.READ)
  public AdministrativeRegionVo findByCode(@ServiceMethodParam(name = "code") String code) {
    if(StringUtils.isBlank(code)) {
      return null;
    }
    AdministrativeRegionEntity region = administrativeRegionRepository.findByRegionCode(code);
    if(region == null) {
      return null;
    }
    return nebulaToolkitService.copyObjectByWhiteList(region, AdministrativeRegionVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  @NebulaServiceMethod(name = "AdministrativeRegionService.updateOrgMappingLevel", desc = "更新组织机构关联行政区域的层级", returnPropertiesFilter = "", scope = ScopeType.WRITE)
  public Integer updateOrgMappingLevel(@ServiceMethodParam(name = "level") Integer level) {
    Validate.notNull(level, "关联层级不能为空");
    RegionLevelEnum regionLevelEnum = RegionLevelEnum.valueOfLevel(level);
    Validate.notNull(regionLevelEnum, "不支持的区域层级");
    RMap<Object, Object> map = redissonClient.getMap(REDIS_KEY_ORG_REGION_LEVEL);
    map.put(LEVEL, level);
    return level;
  }

  @Override
  @NebulaServiceMethod(name = "AdministrativeRegionService.findOrgMappingLevel", desc = "获取组织机构关联行政区域的层级", returnPropertiesFilter = "", scope = NebulaServiceMethod.ScopeType.READ)
  public Integer findOrgMappingLevel() {
    RMap<Object, Object> map = redissonClient.getMap(REDIS_KEY_ORG_REGION_LEVEL);
    return (Integer) map.get(LEVEL);
  }

  /**
   * 将list转化成树
   * @param list
   * @param emptyCondition
   * @return
   */
  private List<AdministrativeRegionEntity> list2Tree(List<AdministrativeRegionEntity> list, boolean emptyCondition) {
    if(CollectionUtils.isEmpty(list)) {
      return Lists.newArrayList();
    }
    Map<String, AdministrativeRegionEntity> rootMap = new HashMap<>(32);
    for (AdministrativeRegionEntity region : list) {
      AdministrativeRegionEntity parent = region.getParent();
      if(parent == null) {
        rootMap.put(region.getId(), region);
        continue;
      }
      // 如果有条件查询，则向上递归把最顶级加入到顶级集合中
      if(!emptyCondition) {
        while (parent.getParent() != null) {
          parent = parent.getParent();
        }
        rootMap.put(parent.getId(), parent);
      }
    }
    return Lists.newArrayList(rootMap.values());
  }

}
