package com.bizunited.platform.user.service.local.service.internal;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.common.constant.Constants;
import com.bizunited.platform.common.constant.RedisKeyConstants;
import com.bizunited.platform.common.enums.NormalStatusEnum;
import com.bizunited.platform.common.enums.RbacRelationEnum;
import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.common.service.redis.RedisMutexService;
import com.bizunited.platform.core.annotations.NebulaServiceMethod;
import com.bizunited.platform.core.annotations.ServiceMethodParam;
import com.bizunited.platform.core.service.invoke.InvokeParams;
import com.bizunited.platform.rbac.server.util.SecurityUtils;
import com.bizunited.platform.user.common.service.UserMappingService;
import com.bizunited.platform.user.common.service.organization.OrganizationService;
import com.bizunited.platform.user.common.service.position.PositionService;
import com.bizunited.platform.user.common.service.region.AdministrativeRegionService;
import com.bizunited.platform.user.common.vo.AdministrativeRegionVo;
import com.bizunited.platform.user.common.vo.OrganizationVo;
import com.bizunited.platform.user.common.vo.UserMappingVo;
import com.bizunited.platform.user.service.local.entity.AdministrativeRegionEntity;
import com.bizunited.platform.user.service.local.entity.OrganizationEntity;
import com.bizunited.platform.user.service.local.entity.PositionEntity;
import com.bizunited.platform.user.service.local.entity.UserEntity;
import com.bizunited.platform.user.service.local.repository.OrganizationRepository;
import com.bizunited.platform.user.service.local.repository.UserRepository;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.redisson.api.RMap;
import org.redisson.api.RedissonClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.domain.Sort;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.security.Principal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * @author yinwenjie ycx
 */
public class OrganizationServiceImpl implements OrganizationService {

  private static final String ERROR_MESS_OGS = "没有找到指定的组织机构";
  private static final String MESS_ORG_NAME = "orgName";
  private static final Logger LOGGER = LoggerFactory.getLogger(OrganizationServiceImpl.class);

  @Value("${user.dimensionalityCodeSize:3}")
  private Integer dimensionalityCodeSize;
  @Autowired
  private RedissonClient redissonClient;
  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private OrganizationRepository organizationRepository;
  @Autowired
  private UserRepository userRepository;
  @Autowired
  private UserMappingService userMappingService;
  @Autowired
  private RedisMutexService redisMutexService;
  @Autowired
  private PositionService positionService;
  @Autowired
  private AdministrativeRegionService administrativeRegionEntityService;


  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.OrganizationService#create(com.bizunited.platform.rbac.server.vo.OrganizationVo)
   */
  @Override
  @Transactional
  @NebulaServiceMethod(name="OrganizationService.create" , desc="组织机构创建服务源" , returnPropertiesFilter="" , scope= NebulaServiceMethod.ScopeType.WRITE)
  public OrganizationVo create(OrganizationVo organization) {
    Validate.notNull(organization, "组织机构信息不能为空，请检查");
    Validate.isTrue(organization.getId() == null, "添加组织机构时，不能传入组织机构id,请重新添加!!");
    String orgName = organization.getOrgName();
    Validate.notBlank(orgName, "组织机构名称不能为空，请检查!!");
    Validate.notBlank(organization.getDescription(), "组织机构描述不能为空，请检查!!");
    Validate.notNull(organization.getType(), "组织机构类型不能为空，请检查!!");
    String code = organization.getCode();
    Validate.notNull(code, "组织机构编号不能为空");
    OrganizationEntity oldOrg = organizationRepository.findByCode(code);
    Validate.isTrue(null == oldOrg, "存在重复的code,请重新输入组织机构编码!!");
    Validate.notNull(organization.getLevel(), "组织层级不能为空");
    if (organization.getTstatus() == null) {
      organization.setTstatus(NormalStatusEnum.ENABLE.getStatus());
    }

    // 转换后保存
    OrganizationEntity currentOrg = this.nebulaToolkitService.copyObjectByWhiteList(organization, OrganizationEntity.class, HashSet.class, ArrayList.class);
    //新增如果存在父级组织机构
    OrganizationVo organizationParent = organization.getParent();
    // 检查父级组织机构并设定
    if (organizationParent != null) {
      Validate.notBlank(organizationParent.getId(), "当前组织机构父级组织机构的id不能为空，请检查!!");
      Optional<OrganizationEntity> parentEntityOp = organizationRepository.findById(organizationParent.getId());
      OrganizationEntity parentEntity = parentEntityOp.orElse(null);
      Validate.notNull(parentEntity, "创建组织机构时，传入了指定的父级机构编号信息，但是并未在本地数据中找到这样的父级机构，请检查！！");
      Validate.isTrue(parentEntity.getTstatus() == 1, "创建组织机构时，传入了指定的父级机构编号信息，但是父级机构状态不正确，请检查！！");
      // 设定父级
      parentEntity.setLeaf(false);
      organizationRepository.save(parentEntity);
      currentOrg.setParent(parentEntity);
    }
    // 设置降维编码
    try {
      currentOrg.setDimensionalityCode(this.buildDimensionalityCode(organization));
      currentOrg.setLeaf(true);
      currentOrg.setCreateUser(SecurityUtils.getCurrentUser().getUserName());
      organizationRepository.save(currentOrg);
      organization.setId(currentOrg.getId());
    } catch (Exception e) {
      LOGGER.error(e.getMessage() , e);
      throw new IllegalArgumentException(e);
    }
    return organization;
  }

  /**
   * 生成层级
   * @param organization
   * @return
   */
  private Integer buildLevel(OrganizationVo organization) {
    Integer level = null;
    if(organization != null && organization.getParent() != null){
      OrganizationEntity parent = organizationRepository.findById(organization.getParent().getId()).orElse(null);
      Validate.notNull(parent, "未找到该组织上级，请检查");
      if(parent.getLevel() != null){
        level = parent.getLevel() + 1;
      }
    }else {
      level = 1;
    }
    return level;
  }

  /**
   * 生成降维编码
   * @param organization
   * @return
   */
  private String buildDimensionalityCode(OrganizationVo organization) {
    String dimensionalityCode = "";
    String temp = "";
    int num = 0;
    if(organization != null && organization.getParent() != null){
      OrganizationEntity parent = organizationRepository.findById(organization.getParent().getId()).orElse(null);
      Validate.notNull(parent, "未找到该组织上级，请检查");
      String maxChildrenDimensionalityCode = organizationRepository.findMaxChildrenDimensionalityCodeByParentId(parent.getId(), organization.getId());
      if(StringUtils.isNotBlank(maxChildrenDimensionalityCode)) {
        String tempStr = maxChildrenDimensionalityCode.substring(maxChildrenDimensionalityCode.length() - dimensionalityCodeSize);
        num = Integer.parseInt(tempStr);
      }
      String numLength = (num + 1) + "";
      temp = StringUtils.leftPad(temp, dimensionalityCodeSize, '0');
      dimensionalityCode = parent.getDimensionalityCode() + temp + numLength;
    }else {
      String maxDimensionalityCode = organizationRepository.findMaxDimensionalityCodeByLevel(1);
      if(StringUtils.isNotBlank(maxDimensionalityCode)){
        num = Integer.parseInt(maxDimensionalityCode);
      }
      String numLength = (num + 1) + "";
      if(numLength.length() < dimensionalityCodeSize){
        temp = StringUtils.leftPad(temp, dimensionalityCodeSize, '0');
        dimensionalityCode = temp + numLength;
      }
    }
    return dimensionalityCode;
  }


  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.OrganizationService#update(com.bizunited.platform.rbac.server.vo.OrganizationVo)
   */
  @Override
  @Transactional
  @NebulaServiceMethod(name="OrganizationService.update" , desc="组织机构更新服务源" , returnPropertiesFilter="" , scope= NebulaServiceMethod.ScopeType.WRITE)
  public OrganizationVo update(OrganizationVo organization) {
    Validate.notNull(organization, "组织机构信息不能为空，请检查");
    Validate.notBlank(organization.getId(), "修改组织机构时，id不能为空,请重新添加!!");
    OrganizationEntity currentOrg = this.organizationRepository.findById(organization.getId()).orElse(null);
    Validate.notNull(currentOrg, "未在数据层找到对应的组织结构信息!!");

    // 修改前验证和赋值
    String orgName = organization.getOrgName();
    Validate.notBlank(orgName, "组织机构名称不能为空，请检查!!");
    Validate.notBlank(organization.getDescription(), "组织机构描述不能为空，请检查!!");
    if (organization.getType() != null) {
      currentOrg.setType(organization.getType());
    }
    OrganizationVo parent = organization.getParent();
    this.rebindParent(currentOrg, parent);

    currentOrg.setOrgName(orgName);
    currentOrg.setDescription(organization.getDescription());
    currentOrg.setExtend1(organization.getExtend1());
    currentOrg.setExtend2(organization.getExtend2());
    currentOrg.setExtend3(organization.getExtend3());
    currentOrg.setExtend4(organization.getExtend4());
    currentOrg.setExtend5(organization.getExtend5());
    currentOrg.setExtend6(organization.getExtend6());
    currentOrg.setExtend7(organization.getExtend7());
    currentOrg.setExtend8(organization.getExtend8());
    currentOrg.setExtend9(organization.getExtend9());
    currentOrg.setExtend10(organization.getExtend10());
    if(organization.getTstatus() != null) {
      currentOrg.setTstatus(organization.getTstatus());
    }
    organizationRepository.save(currentOrg);

    // 转换后返回
    return this.nebulaToolkitService.copyObjectByWhiteList(currentOrg, OrganizationVo.class, HashSet.class, ArrayList.class);
  }

  /**
   * 处理上下级关系改变的leaf
   * @param oldParent
   * @param newParent
   */
  private void updateLeaf(OrganizationEntity oldParent, OrganizationEntity newParent) {
    if (oldParent != null && (newParent == null ||!oldParent.getId().equals(newParent.getId()))){
      if(CollectionUtils.isEmpty(oldParent.getChild())){
        oldParent.setLeaf(true);
        organizationRepository.save(oldParent);
      }
    }
    if(newParent != null){
      newParent.setLeaf(false);
      organizationRepository.save(newParent);
    }
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.OrganizationService#updateStatus(java.lang.String)
   */
  @Override
  @Transactional
  public OrganizationVo updateStatus(String orgId) {
    Validate.notBlank(orgId, "组织机构id不能为空，请检查");
    Optional<OrganizationEntity> op = organizationRepository.findById(orgId);
    OrganizationEntity organization = op.orElse(null);
    Validate.notNull(organization, "没有该组织机构，请检查!!");

    Integer status = organization.getTstatus();
    // 0:禁用,1:启用
    status = (status == NormalStatusEnum.ENABLE.getStatus() ?
            NormalStatusEnum.DISABLE.getStatus() : NormalStatusEnum.ENABLE.getStatus());
    organization.setTstatus(status);
    organizationRepository.save(organization);
    this.updateChildrenStatus(organization, status);
    // 转换后返回
    return this.nebulaToolkitService.copyObjectByWhiteList(organization, OrganizationVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  @Transactional
  public Set<OrganizationVo> enables(String[] orgIds){
    Set<OrganizationEntity> organizations = new HashSet<>();
    Validate.notEmpty(orgIds, "组织机构ids不能为空，请检查");
    for (String orgId : orgIds){
      Optional<OrganizationEntity> op = organizationRepository.findById(orgId);
      OrganizationEntity organization = op.orElse(null);
      Validate.notNull(organization, "没有该组织机构，请检查!!");
      // 0:禁用,1:启用
      Integer status =  NormalStatusEnum.ENABLE.getStatus();
      organization.setTstatus(status);
      organizations.add(organization);
      organizationRepository.save(organization);
    }
    return (Set<OrganizationVo>) this.nebulaToolkitService.copyCollectionByWhiteList(organizations, OrganizationEntity.class, OrganizationVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  @Transactional
  public Set<OrganizationVo> disables(String[] orgIds){
    Set<OrganizationEntity> organizations = new HashSet<>();
    Validate.notEmpty(orgIds, "组织机构ids不能为空，请检查");
    for (String orgId : orgIds){
      Optional<OrganizationEntity> op = organizationRepository.findById(orgId);
      OrganizationEntity organization = op.orElse(null);
      Validate.notNull(organization, "没有该组织机构，请检查!!");
      // 0:禁用,1:启用
      Integer status =  NormalStatusEnum.DISABLE.getStatus();
      organization.setTstatus(status);
      organizations.add(organization);
      organizationRepository.save(organization);
    }
    return (Set<OrganizationVo>) this.nebulaToolkitService.copyCollectionByWhiteList(organizations, OrganizationEntity.class, OrganizationVo.class, HashSet.class, ArrayList.class);
  }

  /**
   * 启用或禁用时，将子级也同时启用或禁用
   * @param organization
   * @param status
   */
  private void updateChildrenStatus(OrganizationEntity organization, Integer status) {
    Validate.notNull(organization, "未在数据层找到对应的组织结构信息!!");
    Set<OrganizationEntity> organizations = this.organizationRepository.findByParent(organization.getId());
    if (CollectionUtils.isEmpty(organizations)){
      return;
    }
    for (OrganizationEntity child:organizations) {
      child.setTstatus(status);
      this.organizationRepository.save(child);
      this.updateChildrenStatus(child, status);
    }
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.OrganizationService#bindParent(java.lang.String, java.lang.String)
   */
  @Override
  @Transactional
  public void bindParent(String currentOrgId, String parentOrgId) {
    Validate.isTrue(!StringUtils.equals(currentOrgId, parentOrgId), "当前组织机构不能与父级组织机构重复");

    // 验证子级组织机构的正确性
    Validate.notBlank(currentOrgId, "指定的组织机构必须传入!!");
    Validate.notBlank(parentOrgId, "指定的父级组织机构必须传入!!");
    Optional<OrganizationEntity> op = this.organizationRepository.findById(currentOrgId);
    Validate.isTrue(op.isPresent(), ERROR_MESS_OGS);
    Validate.isTrue(op.get().getParent() == null, "当前组织机构已经绑定父级，请先解绑后再重新绑定");

    // 验证父级组织机构的正确性
    Optional<OrganizationEntity> optional = this.organizationRepository.findById(parentOrgId);
    Validate.isTrue(optional.isPresent(), "没有找到指定的父级组织机构!!");
    Validate.isTrue(!StringUtils.equals(currentOrgId, parentOrgId), "绑定组织机构时，不能自己绑定自己!!");

    // 验证当前指定的父级之上所有的父级结构，并不存在当前要绑定的子级组织机构ID
    // 主要保证组织机构绑定时，不会出现循环结构
    OrganizationEntity cycleParent = null;
    String cycleParentId = parentOrgId;
    LinkedList<String> allParentIds = Lists.newLinkedList();
    if(optional.get().getParent() != null) {
      allParentIds.add(optional.get().getParent().getId());
    }
    int index = 0;
    while(index < Integer.MAX_VALUE) {
      // 如果条件成立，说明一定存在循环依赖的父级结构，抛出错误
      Validate.isTrue(!allParentIds.contains(currentOrgId) , "在进行父级节点绑定时，发现指定的父级节点出现了循环绑定结构，请检查!!");
      Validate.isTrue(!StringUtils.equals(currentOrgId, cycleParentId) , "在进行父级节点绑定时，发现指定的父级节和当前组织机构节点出现了循环绑定的结构，请检查!!");
      // 放入栈中，并进行上一级查询
      allParentIds.push(cycleParentId);
      cycleParent = this.organizationRepository.findByChildId(cycleParentId);
      if(cycleParent != null) {
        cycleParentId = cycleParent.getId();
        continue;
      }
      break;
    }
    this.organizationRepository.bindParent(currentOrgId, parentOrgId);
    this.organizationRepository.flush();
    //处理父级的leaf
    this.updateLeaf(op.orElse(null).getParent(), optional.orElse(null));
    try {
      redisMutexService.lock(RedisKeyConstants.LOCK_ORGANIZATION_PREFIX.concat(currentOrgId));
      //处理降维编码
      OrganizationEntity entity = op.orElse(null);
      entity.setParent(optional.orElse(null));
      OrganizationVo organizationVo = nebulaToolkitService.copyObjectByWhiteList(entity, OrganizationVo.class, HashSet.class, ArrayList.class, "parent");
      String dimensionalityCode = this.buildDimensionalityCode(organizationVo);
      entity.setDimensionalityCode(dimensionalityCode);
      //处理层级
      Integer level = this.buildLevel(organizationVo);
      entity.setLevel(level);
      organizationRepository.save(entity);
      //处理子级降维编码
      this.buildChildrenDimensionalityCode(entity);
    }catch (RuntimeException e){
      LOGGER.error(e.getMessage() , e);
      throw new RuntimeException(e);
    }finally {
      redisMutexService.unlock(RedisKeyConstants.LOCK_ORGANIZATION_PREFIX.concat(currentOrgId));
    }
  }

  /**
   * 处理子级的降维编码
   * @param organization
   */
  private void buildChildrenDimensionalityCode(OrganizationEntity organization) {
    Set<OrganizationEntity> children = organization.getChild();
    if(CollectionUtils.isEmpty(children)){
      return;
    }
    for(OrganizationEntity org : children){
      OrganizationVo organizationVo = nebulaToolkitService.copyObjectByWhiteList(org, OrganizationVo.class, HashSet.class, ArrayList.class, "parent");
      String dimensionalityCode = this.buildDimensionalityCode(organizationVo);
      org.setDimensionalityCode(dimensionalityCode);
      organizationRepository.save(org);
      this.buildChildrenDimensionalityCode(org);
    }
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.OrganizationService#unbindParent(java.lang.String)
   */
  @Override
  @Transactional
  public void unbindParent(String currentOrgId) {
    Validate.notBlank(currentOrgId, "进行绑定结构消除时，必须指定当前的组织机构信息!!");
    Optional<OrganizationEntity> op = this.organizationRepository.findById(currentOrgId);
    Validate.isTrue(op.isPresent(), ERROR_MESS_OGS);
    this.organizationRepository.unbindParent(currentOrgId);
    op.get().setParent(null);
    this.organizationRepository.flush();
    //处理父级的leaf
    this.updateLeaf(op.orElse(null).getParent(), null);
    try {
      redisMutexService.lock(RedisKeyConstants.LOCK_ORGANIZATION_PREFIX.concat(currentOrgId));
      //处理降维编码
      OrganizationEntity entity = op.orElse(null);
      entity.setParent(null);
      OrganizationVo organizationVo = nebulaToolkitService.copyObjectByWhiteList(entity, OrganizationVo.class, HashSet.class, ArrayList.class, "parent");
      String dimensionalityCode = this.buildDimensionalityCode(organizationVo);
      entity.setDimensionalityCode(dimensionalityCode);
      //处理层级
      Integer level = this.buildLevel(organizationVo);
      entity.setLevel(level);
      organizationRepository.save(entity);
      //处理子级降维编码
      this.buildChildrenDimensionalityCode(entity);
    }catch (RuntimeException e){
      LOGGER.error(e.getMessage() , e);
      throw new RuntimeException(e);
    }finally {
      redisMutexService.unlock(RedisKeyConstants.LOCK_ORGANIZATION_PREFIX.concat(currentOrgId));
    }
  }

  @Override
  @Transactional
  public void mergeOrgById(String id, String currentOrgId) {
    Validate.notBlank(currentOrgId, "传入替换组织机构为空，请检查！");
    Validate.notBlank(id, "传入被替换组织机构为空，请检查！");
    Validate.isTrue(!StringUtils.equals(currentOrgId, id), "不能合并自己，请检查!");
    OrganizationEntity currentEntity = organizationRepository.findDetailsById(currentOrgId);
    OrganizationEntity entity = organizationRepository.findDetailsById(id);
    Validate.notNull(currentEntity, "合并的组织不存在,请检查!");
    Validate.notNull(entity, "被合并的组织不存在,请检查!");
    //查询出B组织的所有下级组织，替换这些组织的上级组织为A组织
    Set<OrganizationEntity> organizationChild = entity.getChild();
    this.handleCircular(organizationChild, currentOrgId);
    if (!CollectionUtils.isEmpty(organizationChild)) {
      for (OrganizationEntity organizationEntity : organizationChild) {
        this.rebindParent(organizationEntity.getId(), currentOrgId);
      }
    }
    //查询出B组织下所有职位，替换这些职位的所属组织为A组织（不管B组织的下级组织的职位，只管B组织下的职位）
    Set<PositionEntity> positionEntities = entity.getPositions();
    if (!CollectionUtils.isEmpty(positionEntities)) {
      for (PositionEntity positionEntity : positionEntities) {
        this.positionService.reBindOrg(currentOrgId, positionEntity.getId());
      }
    }
    //查询出B组织下所有用户，替换这些用户的所属组织为A组织（不管B组织的下级组织的职位，只管B组织下的职位）
    Set<UserEntity> users = entity.getUsers();
    List<String> list = new ArrayList<>();
    if (!CollectionUtils.isEmpty(users)) {
      for (UserEntity userEntity : users){
        list.add(userEntity.getId());
      }
      String[] userIds = (String[]) list.toArray();
      this.rebindUsers(currentOrgId, userIds);
    }
    entity.setTstatus(0);
    entity.setLeaf(true);
    this.organizationRepository.save(entity);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.OrganizationService#reBindParent(java.lang.String)
   */
  @Override
  @Transactional
  public void rebindParent(String currentOrgId, String parentOrgId) {
    this.unbindParent(currentOrgId);
    if(StringUtils.isNotBlank(parentOrgId)) {
      this.bindParent(currentOrgId, parentOrgId);
    }
  }

  /**
   * 更新父级组织
   * @param org
   * @param newParent
   */
  private void rebindParent(OrganizationEntity org, OrganizationVo newParent) {
    OrganizationEntity oldParent = org.getParent();
    if(newParent != null) {
      Validate.notBlank(newParent.getId(), "组织机构的父级ID不能为空");
    }
    // 如果新旧父级都是null，则直接返回，不做处理
    if(oldParent == null && newParent == null) {
      return;
    }
    // 新旧父级是一样的
    if(oldParent != null && newParent != null && newParent.getId().equals(oldParent.getId())) {
      return;
    }
    String newParentId = newParent == null ? null : newParent.getId();
    this.rebindParent(org.getId(), newParentId);
  }


  /**
   * 验证绑定用户和组织机构
   * @param currentOrgId
   * @param userId
   * @param mappingVo
   */
  private void validateBindUser(UserMappingVo mappingVo, String currentOrgId, String userId){
    Optional<UserEntity> optional = userRepository.findById(userId);
    Validate.isTrue(optional.isPresent(), "没有该用户[%s],请检查!!", userId);
    if (mappingVo == null) {
      long count = organizationRepository.countByOrgIdAndUserId(currentOrgId, userId);
      Validate.isTrue(count == 0, "在批量绑定时,指定用户[%s]已经绑定了，不能重复绑定，请检查!!", userId);
    } else {
      int bindRelation = mappingVo.getUserOrgRelation();
      //如果用户和组织绑定关系为多对多,多对一，一对多，一对一分别控制判断
      if (bindRelation == RbacRelationEnum.MANY_TO_MANY.getRelation()) {
        long count = organizationRepository.countByOrgIdAndUserId(currentOrgId, userId);
        Validate.isTrue(count == 0, "在批量绑定时,用户和组织多对多，指定用户[%s]已经绑定了，不能重复绑定，请检查!!", userId);
      }
      if (bindRelation == RbacRelationEnum.MANY_TO_ONE.getRelation()) {
        long count = organizationRepository.countOrganizationByUserId(userId);
        Validate.isTrue(count == 0, "在批量绑定时,用户和组织多对一，指定用户[%s]已经绑定了组织，不能继续绑定，请检查!!", userId);
      }
      if (bindRelation == RbacRelationEnum.ONE_TO_MANY.getRelation()) {
        long count = organizationRepository.countUserByOrgId(currentOrgId);
        Validate.isTrue(count == 0, "在批量绑定时,用户和组织一对多，指定组织[%s]已经绑定了用户，不能继续绑定，请检查!!", currentOrgId);
      }
      if (bindRelation == RbacRelationEnum.ONE_TO_ONE.getRelation()) {
        long countOrganization = organizationRepository.countOrganizationByUserId(userId);
        Validate.isTrue(countOrganization == 0, "在批量绑定时,用户和组织一对一，指定用户[%s]已经绑定了组织，不能继续绑定，请检查!!", userId);
        long countUser = organizationRepository.countUserByOrgId(currentOrgId);
        Validate.isTrue(countUser == 0, "在批量绑定时,用户和组织一对一，指定组织[%s]已经绑定了用户，不能继续绑定，请检查!!", currentOrgId);
      }

    }
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.OrganizationService#bindUsers(java.lang.String, java.lang.String[])
   */
  @Override
  @Transactional
  public void bindUsers(String currentOrgId, String[] userIds) {
    Validate.notBlank(currentOrgId, "进行组织机构-人员绑定时，必须指定当前的组织机构信息!!");
    Optional<OrganizationEntity> op = this.organizationRepository.findById(currentOrgId);
    Validate.isTrue(op.isPresent(), ERROR_MESS_OGS);
    Validate.isTrue(userIds != null && userIds.length > 0, "进行组织机构-人员绑定时,至少传入一个人员编号信息!!");
    UserMappingVo mapping = userMappingService.findOne();
    for (String userId : userIds) {
      this.validateBindUser(mapping, currentOrgId, userId);
      this.organizationRepository.bindUsers(currentOrgId, userId);
    }
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.OrganizationService#unbindUser(java.lang.String, java.lang.String[])
   */
  @Override
  @Transactional
  public void unbindUsers(String currentOrgId, String[] userIds) {
    Validate.notBlank(currentOrgId, "进行组织机构-人员解绑时，必须指定当前的组织机构信息!!");
    Optional<OrganizationEntity> op = this.organizationRepository.findById(currentOrgId);
    Validate.isTrue(op.isPresent(), ERROR_MESS_OGS);
    Validate.notNull(userIds, "进行组织机构-人员解绑时，必须指定当前的人员信息!!");
    for (String userId : userIds) {
      Optional<UserEntity> optional = this.userRepository.findById(userId);
      Validate.isTrue(optional.isPresent(), "进行组织机构-人员解绑绑定时，未找到指定的人员信息【id=%s】" , userId);
      this.organizationRepository.unbindUser(currentOrgId, userId);
    }
  }

  @Override
  @Transactional
  public void rebindUsers(String currentOrgId, String[] userIds) {
    Validate.notBlank(currentOrgId, "进行组织机构-人员绑定时，必须指定当前的组织机构信息!!");
    if(userIds == null) {
      userIds = new String[0];
    }
    OrganizationEntity org = organizationRepository.findById(currentOrgId).orElse(null);
    Validate.notNull(org, ERROR_MESS_OGS);
    Set<String> newUserIds = Arrays.stream(userIds).collect(Collectors.toSet());
    Set<UserEntity> users = org.getUsers();
    if(users == null) {
      users = Sets.newHashSet();
    }
    Set<String> oldUserIds = users.stream().map(UserEntity::getId).collect(Collectors.toSet());
    Sets.SetView<String> insertUsers = Sets.difference(newUserIds, oldUserIds);
    Sets.SetView<String> deleteUsers = Sets.difference(oldUserIds, newUserIds);
    UserMappingVo mapping = userMappingService.findOne();
    for (String userId : deleteUsers) {
      organizationRepository.unbindUser(currentOrgId, userId);
    }
    for (String userId : insertUsers) {
      this.validateBindUser(mapping, currentOrgId, userId);
      this.organizationRepository.bindUsers(currentOrgId, userId);
    }
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.OrganizationService#findByNullParent()
   */
  @Override
  public List<OrganizationVo> findByNullParent() {
    Set<OrganizationEntity> organizationEntitys = this.organizationRepository.findByNullParent();
    if(organizationEntitys == null || organizationEntitys.isEmpty()) {
      return Lists.newArrayList();
    }

    Collection<OrganizationVo> orgVos = this.nebulaToolkitService.copyCollectionByWhiteList(organizationEntitys, OrganizationEntity.class, OrganizationVo.class, HashSet.class, ArrayList.class);
    return Lists.newArrayList(orgVos);
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.OrganizationService#findByParent(java.lang.String)
   */
  @Override
  public List<OrganizationVo> findByParent(String parentId) {
    if (StringUtils.isBlank(parentId)) {
      return Lists.newArrayList();
    }
    Set<OrganizationEntity> organizationEntitys = this.organizationRepository.findByParent(parentId);
    if(organizationEntitys == null || organizationEntitys.isEmpty()) {
      return Lists.newArrayList();
    }

    Collection<OrganizationVo> orgVos = this.nebulaToolkitService.copyCollectionByWhiteList(organizationEntitys, OrganizationEntity.class, OrganizationVo.class, HashSet.class, ArrayList.class);
    return Lists.newArrayList(orgVos);
  }

  @Override
  @NebulaServiceMethod(name = "OrganizationService.findTreeByConditions", desc = "根据条件查询组织机构树", returnPropertiesFilter = "parent", scope = NebulaServiceMethod.ScopeType.READ)
  public List<OrganizationVo> findTreeByConditions(InvokeParams invokeParams) {
    invokeParams = ObjectUtils.defaultIfNull(invokeParams, new InvokeParams());
    Map<String, Object> parameters = invokeParams.getInvokeParams();
    List<OrganizationEntity> orgs = organizationRepository.findAllByConditions(parameters);
    if(CollectionUtils.isEmpty(orgs)) {
      return Lists.newArrayList();
    }
    Collection<OrganizationVo> collection = nebulaToolkitService.copyCollectionByWhiteList(orgs, OrganizationEntity.class, OrganizationVo.class, HashSet.class, ArrayList.class, "parent");
    List<OrganizationVo> orgVos = Lists.newArrayList();
    // 用set集合做去重操作
    Set<String> orgIds = new HashSet<>();
    collection.forEach(o -> orgIds.add(o.getId()));
    orgVos.addAll(collection);
    // 将结果集的所有父级加到结果集中
    for (OrganizationEntity org : orgs) {
      OrganizationEntity parent = org;
      while ((parent = parent.getParent()) != null) {
        if(orgIds.add(parent.getId())) {
          OrganizationVo parentVo = nebulaToolkitService.copyObjectByWhiteList(parent, OrganizationVo.class, HashSet.class, ArrayList.class, "parent");
          orgVos.add(parentVo);
        }
      }
    }
    return orgVos;
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.OrganizationService#findByNullParentAndStatus(java.lang.Integer)
   */
  @Override
  @NebulaServiceMethod(name="OrganizationService.findByNullParentAndStatus" , desc="根据状态查询组织机构的最顶级数据" , returnPropertiesFilter="" , scope= NebulaServiceMethod.ScopeType.READ)
  public List<OrganizationVo> findByNullParentAndStatus(@ServiceMethodParam(name = "status") Integer status) {
    if (status == null) {
      return Collections.emptyList();
    }
    List<OrganizationEntity> organizationEntitys = this.organizationRepository.findByNullParentAndStatus(status);
    if(CollectionUtils.isEmpty(organizationEntitys)) {
      return Collections.emptyList();
    }

    Collection<OrganizationVo> orgVos = this.nebulaToolkitService.copyCollectionByWhiteList(organizationEntitys, OrganizationEntity.class, OrganizationVo.class, HashSet.class, ArrayList.class);
    return Lists.newArrayList(orgVos);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.OrganizationService#findByParentAndStatus(java.lang.String, java.lang.Integer)
   */
  @Override
  @NebulaServiceMethod(name="OrganizationService.findByParentAndStatus" , desc="根据父级ID和状态查询组织机构的最顶级数据" , returnPropertiesFilter="" , scope= NebulaServiceMethod.ScopeType.READ)
  public List<OrganizationVo> findByParentAndStatus(@ServiceMethodParam(name = "parentId") String parentId, @ServiceMethodParam(name = "status") Integer status) {
    if (StringUtils.isBlank(parentId) || status == null) {
      return Lists.newArrayList();
    }
    Set<OrganizationEntity> organizationEntitys = this.organizationRepository.findByParentAndStatus(parentId, status);
    if(organizationEntitys == null || organizationEntitys.isEmpty()) {
      return Lists.newArrayList();
    }

    Collection<OrganizationVo> orgVos = this.nebulaToolkitService.copyCollectionByWhiteList(organizationEntitys, OrganizationEntity.class, OrganizationVo.class, HashSet.class, ArrayList.class);
    return Lists.newArrayList(orgVos);
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.OrganizationService#findByOrgName(java.lang.String)
   */
  @Override
  public JSONArray findByOrgName(String orgName) {
    if (StringUtils.isBlank(orgName)) {
      return null;
    }
    /*
     * 这个方法需要递归进行，过程如下
     * 1、首先根据一般的查询语句，查询满足条件的一个或者多个组织机构
     * 2、最后进行这个组织机构所有父级节点的遍历
     * 在遍历过程中，将连续查询当前组织结构的父级信息，并将其遍历节点的所有子级节点置为null
     * */
    // 1、======
    Set<OrganizationEntity> currentOrgs = this.organizationRepository.findByOrgName(orgName);
    if (currentOrgs == null || currentOrgs.isEmpty()) {
      return null;
    }

    // 2、======
    JSONArray jsonObjects = new JSONArray();
    for (OrganizationEntity orgItem : currentOrgs) {
      JSONObject jsonObject = this.recursiveFindParent(orgItem);
      if (jsonObject != null) {
        jsonObjects.add(jsonObject);
      }
    }
    return jsonObjects;
  }

  /**
   * 根据组织机构名称实现模糊查询
   * @param orgName
   * @return
   */
  @Override
  public Set<OrganizationVo> findByOrgNameLike(String orgName) {
    Set<OrganizationEntity> entitySet = this.organizationRepository.findByOrgNameLike(orgName);
    if (CollectionUtils.isEmpty(entitySet)){
      return Sets.newHashSet();
    }
    Collection<OrganizationVo> organizationVos = this.nebulaToolkitService.copyCollectionByWhiteList(entitySet, OrganizationEntity.class, OrganizationVo.class, LinkedHashSet.class, ArrayList.class);

    return Sets.newHashSet(organizationVos);
  }
  /**
   * 对一个指定的组织机构节点，递归查询其下所有的父级节点信息
   * @param currentOrg
   */
  private JSONObject recursiveFindParent(OrganizationEntity currentOrg) {
    if (currentOrg == null || currentOrg.getTstatus() != 1) {
      return null;
    }
    JSONObject orgObject = new JSONObject();
    orgObject.put("id", currentOrg.getId());
    orgObject.put(MESS_ORG_NAME, currentOrg.getOrgName());
    orgObject.put("code", currentOrg.getCode());
    orgObject.put("createTime", currentOrg.getCreateTime());
    orgObject.put("sortIndex", currentOrg.getSortIndex());
    orgObject.put("status", currentOrg.getTstatus());
    orgObject.put("type", currentOrg.getType());

    // 查询父级结构
    OrganizationEntity parent = currentOrg.getParent();
    if (parent != null) {
      orgObject.put("parent", this.recursiveFindParent(parent));
    }
    return orgObject;
  }
  @Override
  public OrganizationVo findByCode(String code) {
    if(StringUtils.isBlank(code)) {
      return null;
    }
    OrganizationEntity org = organizationRepository.findByCode(code);
    if(org == null) {
      return null;
    }
    return nebulaToolkitService.copyObjectByWhiteList(org, OrganizationVo.class, HashSet.class, ArrayList.class);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.OrganizationService#findById(java.lang.String)
   */
  @Override
  @NebulaServiceMethod(name="OrganizationService.findDetailsById" , desc="根据组织机构ID查询明细服务源" , returnPropertiesFilter="parent,child,roles,users,positions" , scope= NebulaServiceMethod.ScopeType.READ)
  public OrganizationVo findDetailsById(@ServiceMethodParam(name = "id") String id) {
    Validate.notBlank(id, "组织机构id不能为空,请检查!!");
    OrganizationEntity currentOrg = organizationRepository.findDetailsById(id);
    if(currentOrg == null) {
      return null;
    }
    return this.nebulaToolkitService.copyObjectByWhiteList(currentOrg, OrganizationVo.class, LinkedHashSet.class, ArrayList.class, "parent","child","roles","users","positions");
  }

  @Override
  public Set<OrganizationVo> findByIds(List<String> ids) {
    if(CollectionUtils.isEmpty(ids)){
      return Sets.newHashSet();
    }
    Set<OrganizationEntity> orgs = organizationRepository.findByIds(ids);
    if(CollectionUtils.isEmpty(orgs)){
      return Sets.newHashSet();
    }
    Collection<OrganizationVo> ovos = this.nebulaToolkitService.copyCollectionByWhiteList(orgs,OrganizationEntity.class,OrganizationVo.class,HashSet.class,ArrayList.class);
    return Sets.newHashSet(ovos);
  }
  /**
   * 树结构查询-根据组织机构ID查询根节点以及所有子节点以树形结构返回
   * @param id
   * @return
   */
  @Override
  public OrganizationVo findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }

    OrganizationEntity organizationEntity = this.organizationRepository.findById(id).orElse(null);
    if(organizationEntity == null){
      return null;
    }
    return this.findOrgTree(organizationEntity);
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.OrganizationService#findByUserId(java.lang.String)
   */
  @Override
  public JSONArray findByUserId(String userId) {
    if (StringUtils.isBlank(userId)) {
      return null;
    }

    Set<OrganizationEntity> currentOrgs = this.organizationRepository.findByUserId(userId);
    if (currentOrgs == null || currentOrgs.isEmpty()) {
      return null;
    }

    // 递归构造父级（包括父级的Roles信息）
    JSONArray jsonObjects = new JSONArray();
    for (OrganizationEntity orgItem : currentOrgs) {
      JSONObject jsonObject = this.recursiveFindParent(orgItem);
      if (jsonObject != null) {
        jsonObjects.add(jsonObject);
      }
    }
    return jsonObjects;
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.OrganizationService#findUnbindOrgById(java.lang.String)
   */
  @Override
  public Object[] findUnbindOrgById(String currentOrgId) {
    /*
     * 1:查询出所有状态为启用的组织机构id与组织机构名称，并存入集合jsonArray中。
     * 2：根据当前组织机构id递归查询出所有子级组织机构的id与组织机构名称，并存入另外一个集合array中。
     * 3：检索出当前组织机构的父级组织机构id,并查询出父级组织机构名称，保存到array中。
     * 4：根据jsonArray移除掉所有array集合中的元素，就是可以绑定的组织机构节点
     * 5：返回可以绑定的组织机构id与组织机构名称
     */
    if (StringUtils.isBlank(currentOrgId)) {
      return new Object[0];
    }
    List<OrganizationEntity> allOrganizations = this.organizationRepository.findAllByStatus();
    JSONArray jsonArray = new JSONArray();
    if (null != allOrganizations && !allOrganizations.isEmpty()) {
      for (OrganizationEntity org : allOrganizations) {
        JSONObject orgAllObject = new JSONObject();
        orgAllObject.put("id", org.getId());
        orgAllObject.put(MESS_ORG_NAME, org.getOrgName());
        jsonArray.add(orgAllObject);
      }
    }
    LOGGER.debug("所有组织机构节点: {}", JSON.toJSONString(jsonArray));
    Optional<OrganizationEntity> op = this.organizationRepository.findById(currentOrgId);
    OrganizationEntity childOrg = op.orElse(null);
    if(childOrg == null){
      throw new IllegalArgumentException("未找到任何组织机构信息，请检查!!");
    }

    JSONArray array = this.findChildOrgs(childOrg, new JSONArray());
    //查询父级组织机构信息,并放入要移除的集合中
    if (null != childOrg.getParent() && StringUtils.isNotBlank(childOrg.getParent().getId())) {
      String parentOrgId = childOrg.getParent().getId();
      Optional<OrganizationEntity> optional = this.organizationRepository.findById(parentOrgId);
      if (optional.isPresent()) {
        JSONObject jsonObject = new JSONObject();
        OrganizationEntity parentOrg = optional.get();
        jsonObject.put("id", parentOrg.getId());
        jsonObject.put(MESS_ORG_NAME, parentOrg.getOrgName());
        array.add(jsonObject);
      }
    }
    LOGGER.debug("不允许绑定的组织机构节点(当前组织机构以及所有子级组织机构): {}", JSON.toJSONString(array));
    if (!array.isEmpty()) {
      jsonArray.removeAll(array);
    }
    LOGGER.debug("允许绑定的组织机构节点: {}", JSON.toJSONString(jsonArray));
    return jsonArray.toArray();
  }

  /**
   * 对一个指定的组织机构节点，递归查询其下所有的子级节点信息
   * @param currentOrg
   */
  private JSONArray findChildOrgs(OrganizationEntity currentOrg, JSONArray jsonArray) {
    if (currentOrg != null) {
      Optional<OrganizationEntity> op = organizationRepository.findById(currentOrg.getId());
      OrganizationEntity organization = op.orElse(null);
      Validate.notNull(organization, "没有可用的组织机构,请检查!!");

      JSONObject orgObject = new JSONObject();
      orgObject.put("id", organization.getId());
      orgObject.put(MESS_ORG_NAME, organization.getOrgName());
      jsonArray.add(orgObject);
      // 查询子级结构
      Set<OrganizationEntity> childs = organizationRepository.findByParent(organization.getId());
      if (childs != null && !childs.isEmpty()) {
        for (OrganizationEntity child : childs) {
          this.findChildOrgs(child, jsonArray);
        }
      }
    }
    return jsonArray;
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.OrganizationService#findOrgByUserId(java.lang.String)
   */
  @Override
  public Set<OrganizationVo> findOrgByUserId(String userId) {
    Validate.notBlank(userId,"传入的用户ID不能为空！");
    UserEntity user = userRepository.findDetailsById(userId);
    if(user==null){
      return Sets.newHashSet();
    }
    Set<OrganizationEntity> orgs = user.getOrgs();
    if(CollectionUtils.isEmpty(orgs)){
      return Sets.newHashSet();
    }
    Collection<OrganizationVo> vos = nebulaToolkitService.copyCollectionByWhiteList(orgs, OrganizationEntity.class, OrganizationVo.class, HashSet.class, ArrayList.class);
    return Sets.newHashSet(vos);
  }

  /**
   * 一级查询
   * @param currentOrg
   * @return
   */
  private OrganizationVo findOrgTree(OrganizationEntity currentOrg) {
    OrganizationVo organizationVo = this.nebulaToolkitService.copyObjectByWhiteList(currentOrg, OrganizationVo.class, HashSet.class, ArrayList.class );
    organizationVo.setChild(this.findChildren(organizationVo));
    return organizationVo;
  }

  /**
   * 子集查询-子集查找遍历，递归查询其下所有的子级节点信息
   * @param currentOrg
   * @return
   */
  private Set<OrganizationVo> findChildren(OrganizationVo currentOrg) {
    if (Objects.isNull(currentOrg)){
      return Sets.newHashSet();
    }
    Set<OrganizationVo> organizationVoSet = new HashSet<>();
    // 查询子级结构
    Set<OrganizationEntity> childs = organizationRepository.findByParent(currentOrg.getId());
    if (CollectionUtils.isEmpty(childs)) {
      return Sets.newHashSet();
    }
    for (OrganizationEntity child : childs) {
      OrganizationVo organizationVo = this.nebulaToolkitService.copyObjectByWhiteList(child, OrganizationVo.class, HashSet.class, ArrayList.class);
      organizationVoSet.add(organizationVo);
      organizationVo.setChild(this.findChildren(organizationVo));
    }
    return organizationVoSet;
  }

  @Override
  public OrganizationVo findByChild(String childId) {
    Validate.notBlank(childId, "子级组织ID不能为空");
    OrganizationEntity org = organizationRepository.findByChildId(childId);
    if(org == null) {
      return null;
    }
    return nebulaToolkitService.copyObjectByWhiteList(org, OrganizationVo.class, HashSet.class, ArrayList.class);
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.OrganizationService#findAll()
   */
  @Override
  public List<OrganizationVo> findAll() {
    List<OrganizationEntity> orgs = organizationRepository.findAll(Sort.by(Sort.Order.desc("createTime")));
    if(orgs == null) {
      return Collections.emptyList();
    }
    // 用非持久化对象返回，避免parent和child两个属性的过滤冲突
    Collection<OrganizationVo> result = nebulaToolkitService.copyCollectionByWhiteList(orgs, OrganizationEntity.class, OrganizationVo.class, LinkedHashSet.class, ArrayList.class, "parent");
    return new ArrayList<>(result);
  }

  @Override
  public OrganizationVo findMainOrgByUserId(String userId) {
    Validate.notBlank(userId, "用户ID不能为空");
    RMap<Object, Object> userMap = redissonClient.getMap(Constants.REDIS_KEY_RBAC_USER + userId);
    Object obj = userMap.get(Constants.MAIN_ORG);
    OrganizationVo mainOrg = null;
    if(obj != null) {
      // 从redis获取到设置的主组织机构，并验证获取的主组织机构在数据库中是否存在
      OrganizationVo org = JSON.parseObject((String) obj, OrganizationVo.class);
      OrganizationEntity organization = organizationRepository.findByIdAndUserId(org.getId(), userId);
      if(organization != null) {
        mainOrg = org;
      }
    }
    if(mainOrg == null) {
      // 没有从redis中获取到一个主组织机构，则将用户绑定的第一个组织机构作为主组织机构
      Set<OrganizationVo> orgs = this.findOrgByUserId(userId);
      if(!CollectionUtils.isEmpty(orgs)) {
        mainOrg = orgs.iterator().next();
        userMap.put(Constants.MAIN_ORG, JSON.toJSONString(mainOrg));
      }
    }
    return mainOrg;
  }

  @Override
  @Transactional
  public void updateUserMainOrg(String userId, String orgId) {
    Validate.notBlank(userId, "用户ID不能为空");
    Validate.notBlank(orgId, "组织机构ID不能为空");
    UserEntity user = userRepository.findById(userId).orElse(null);
    Validate.notNull(user, "用户不存在");
    OrganizationEntity org = organizationRepository.findByIdAndUserId(orgId, userId);
    Validate.notNull(org, "未找到用户关联的组织机构");
    RMap<Object, Object> userMap = redissonClient.getMap(Constants.REDIS_KEY_RBAC_USER + userId);
    OrganizationVo orgVo = nebulaToolkitService.copyObjectByWhiteList(org, OrganizationVo.class, HashSet.class, ArrayList.class);
    userMap.put(Constants.MAIN_ORG, JSON.toJSONString(orgVo));
  }

  /**
   * 根据登陆人关联的组织查询
   * @param type
   * @return
   */
  @Override
  public Set<OrganizationVo> findByPrincipal(Integer type, Principal principal) {
    Validate.notNull(type, "传入类型不能为空，请检查");
    String account = principal.getName();
    UserEntity user = userRepository.findByAccount(account);
    Validate.notNull(user, "未获取登陆人信息，请检查");
    Set<OrganizationVo> orgs = null;
    switch (type) {
      case 1:
        orgs = this.findParent(user);
        break;
      case 2:
        orgs = this.findChildren(user);
        break;
      case 3:
        orgs = this.findParentAndChildren(user);
        break;
      case 4:
        orgs = this.findOrg(user);
        break;
      default:
        break;
    }
    if(CollectionUtils.isEmpty(orgs)){
      return Sets.newHashSet();
    }
    return orgs;
  }

  /**
   * 根据登陆人关联的组织查询
   * @param user
   * @return
   */
  private Set<OrganizationVo> findOrg(UserEntity user) {
    Set<OrganizationEntity> orgs = organizationRepository.findByUserId(user.getId());
    Collection<OrganizationVo> result = nebulaToolkitService.copyCollectionByWhiteList(orgs, OrganizationEntity.class, OrganizationVo.class, LinkedHashSet.class, ArrayList.class);
    return Sets.newHashSet(result);
  }

  /**
   * 根据登陆人关联的组织查询上下级
   * @param user
   * @return
   */
  private Set<OrganizationVo> findParentAndChildren(UserEntity user) {
    Set<OrganizationVo> result = new HashSet<>();
    Set<OrganizationVo> removes = new HashSet<>();
    Map<String, OrganizationVo> childMap = new HashMap<>();
    Map<String, OrganizationVo> parentMap = new HashMap<>();
    Set<OrganizationVo> rootOrgs = this.findOrg(user);
    for(OrganizationVo rootOrg : rootOrgs) {
      rootOrg.setChild(this.findAllChildren(rootOrg, childMap));
      rootOrg.setParent(this.findAllParent(rootOrg, parentMap));
      result.add(rootOrg);
    }
    //过滤
    for(OrganizationVo item : result){
      if(childMap.containsKey(item.getId())){
        removes.add(item);
      }
    }
    for(OrganizationVo remove : removes){
      result.remove(remove);
    }
    return result;
  }

  /**
   * 根据登陆人关联的组织查询下级
   * @param user
   * @return
   */
  private Set<OrganizationVo> findChildren(UserEntity user) {
    Set<OrganizationVo> result = new HashSet<>();
    Set<OrganizationVo> removes = new HashSet<>();
    Map<String, OrganizationVo> childMap = new HashMap<>();
    Set<OrganizationVo> rootOrgs = this.findOrg(user);
    for(OrganizationVo rootOrg : rootOrgs) {
      rootOrg.setChild(this.findAllChildren(rootOrg, childMap));
      result.add(rootOrg);
    }
    //过滤
    for(OrganizationVo item : result){
      if(childMap.containsKey(item.getId())){
        removes.add(item);
      }
    }
    for(OrganizationVo remove : removes){
      result.remove(remove);
    }
    return result;
  }

  /**
   * 查询下级并将下级信息存入map
   * @param org
   * @return
   */
  private Set<OrganizationVo> findAllChildren(OrganizationVo org, Map<String, OrganizationVo> childMap) {
    Set<OrganizationVo> result = new HashSet<>();
    // 查询子级结构
    OrganizationEntity currentOrg = organizationRepository.findById(org.getId()).orElse(null);
    Set<OrganizationEntity> children = currentOrg.getChild();
    if(CollectionUtils.isEmpty(children)){
      return Sets.newHashSet();
    }
    for (OrganizationEntity child : children) {
      OrganizationVo organizationVo = this.nebulaToolkitService.copyObjectByWhiteList(child, OrganizationVo.class, HashSet.class, ArrayList.class);
      childMap.put(organizationVo.getId(), organizationVo);
      organizationVo.setChild(this.findAllChildren(organizationVo, childMap));
      result.add(organizationVo);
    }
    return result;
  }

  /**
   * 根据登陆人关联的组织查询上级
   * @param user
   * @return
   */
  private Set<OrganizationVo> findParent(UserEntity user) {
    Set<OrganizationVo> result = new HashSet<>();
    Set<OrganizationVo> removes = new HashSet<>();
    Map<String, OrganizationVo> parentMap = new HashMap<>();
    Set<OrganizationVo> rootOrgs = this.findOrg(user);
    for(OrganizationVo rootOrg : rootOrgs) {
      rootOrg.setParent(this.findAllParent(rootOrg, parentMap));
      result.add(rootOrg);
    }
    //过滤
    for(OrganizationVo item : result){
      if(parentMap.containsKey(item.getId())){
        removes.add(item);
      }
    }
    for(OrganizationVo remove : removes){
      result.remove(remove);
    }
    return result;
  }

  /**
   * 查询父级并将下级信息存入map
   * @param rootOrg
   * @param parentMap
   * @return
   */
  private OrganizationVo findAllParent(OrganizationVo rootOrg, Map<String, OrganizationVo> parentMap) {
    // 查询父级结构
    OrganizationEntity currentOrg = organizationRepository.findById(rootOrg.getId()).orElse(null);
    OrganizationEntity parent = currentOrg.getParent();
    if(parent == null){
      return null;
    }
    OrganizationVo organizationVo = this.nebulaToolkitService.copyObjectByWhiteList(parent, OrganizationVo.class, HashSet.class, ArrayList.class);
    parentMap.put(organizationVo.getId(), organizationVo);
    organizationVo.setParent(this.findAllParent(organizationVo, parentMap));
    return organizationVo;
  }

  /**
   * 根据登陆人关联的组织和输入层级查询控件
   * @param type
   * @param account
   * @param level
   * @return
   */
  @Override
  public Set<OrganizationVo> findByTypeAndLevel(Integer type, String account, Integer level) {
    Validate.notNull(type, "传入类型不能为空，请检查");
    Validate.notNull(account, "传入账号名不能为空，请检查");
    Validate.notNull(level, "传入层级不能为空，请检查");
    UserEntity user = userRepository.findByAccount(account);
    Validate.notNull(user, "未获取登陆人信息，请检查");
    Set<OrganizationVo> orgs = null;
    String[] propertiesFilter = new String[level];
    Set<OrganizationVo> newOrgs = new HashSet<>();
    switch (type) {
      case 1:
        orgs = this.findParent(user);
        this.bindPropertiesFilter(level, propertiesFilter, "parent");
        break;
      case 2:
        orgs = this.findChildren(user);
        this.bindPropertiesFilter(level, propertiesFilter, "child");
        break;
      default:
        break;
    }
    if(CollectionUtils.isEmpty(orgs)){
      return Sets.newHashSet();
    }
    for (OrganizationVo org : orgs) {
      OrganizationVo organizationVo = this.nebulaToolkitService.copyObjectByWhiteList(org, OrganizationVo.class, HashSet.class, ArrayList.class, propertiesFilter);
      newOrgs.add(organizationVo);
    }
    return newOrgs;
  }

  /**
   * 处理参数
   * @param level
   * @param propertiesFilter
   * @return
   */
  private void bindPropertiesFilter(Integer level, String[] propertiesFilter, String type) {
    String temp = "";
    for(int i = 0 ; i < level; i++){
      temp = temp + type + ".";
      if((i + 1 ) == level){
        temp = temp.substring(0, temp.length() - 1);
      }
      propertiesFilter[i] = temp;
    }
  }

  @Override
  @Transactional
  @NebulaServiceMethod(name = "OrganizationService.bindRegions", desc = "绑定指定组织机构的行政区域", returnPropertiesFilter = "", scope = NebulaServiceMethod.ScopeType.WRITE)
  public void bindRegions(@ServiceMethodParam(name = "orgId") String orgId, @ServiceMethodParam(name = "regionIds") String[] regionIds) {
    Validate.notBlank(orgId, "组织机构ID不能为空");
    Validate.notEmpty(regionIds, "行政区域ID集合不能为空");
    OrganizationEntity org = organizationRepository.findById(orgId).orElse(null);
    Validate.notNull(org, "组织机构不存在，请检查数据！");
    Set<AdministrativeRegionEntity> regions = ObjectUtils.defaultIfNull(org.getRegions(), Sets.newHashSet());
    Set<String> regionsSet = regions.stream().map(AdministrativeRegionEntity::getId).collect(Collectors.toSet());
    Integer orgMappingLevel = administrativeRegionEntityService.findOrgMappingLevel();
    Validate.notNull(orgMappingLevel, "组织机构关联区域层级为空，请检查配置!!");
    for (String regionId : regionIds) {
      if(regionsSet.contains(regionId)) {
        continue;
      }
      AdministrativeRegionVo region = administrativeRegionEntityService.findById(regionId);
      Validate.notNull(region, "区域不存在：%s", regionId);
      if(!orgMappingLevel.equals(region.getRegionLevel())) {
        continue;
      }
      organizationRepository.bindRegion(orgId, regionId);
      regionsSet.add(regionId);
    }
  }

  @Override
  @Transactional
  @NebulaServiceMethod(name = "OrganizationService.rebindRegions", desc = "重新绑定指定组织机构的行政区域", returnPropertiesFilter = "", scope = NebulaServiceMethod.ScopeType.WRITE)
  public void rebindRegions(@ServiceMethodParam(name = "orgId") String orgId, @ServiceMethodParam(name = "regionIds") String[] regionIds) {
    Validate.notBlank(orgId, "组织机构ID不能为空");
    organizationRepository.unbindRegionsByOrgId(orgId);
    if(ArrayUtils.isEmpty(regionIds)) {
      return;
    }
    this.bindRegions(orgId, regionIds);
  }

  @Override
  @Transactional
  @NebulaServiceMethod(name = "OrganizationService.unbindRegions", desc = "解绑指定组织机构的行政区域", returnPropertiesFilter = "", scope = NebulaServiceMethod.ScopeType.WRITE)
  public void unbindRegions(@ServiceMethodParam(name = "orgId") String orgId, @ServiceMethodParam(name = "regionIds") String[] regionIds) {
    Validate.notBlank(orgId, "组织机构ID不能为空");
    Validate.notEmpty(regionIds, "行政区域ID集合不能为空");
    OrganizationEntity org = organizationRepository.findById(orgId).orElse(null);
    Validate.notNull(org, "组织机构不存在，请检查数据！");
    Set<AdministrativeRegionEntity> regions = ObjectUtils.defaultIfNull(org.getRegions(), Sets.newHashSet());
    Set<String> regionsSet = regions.stream().map(AdministrativeRegionEntity::getId).collect(Collectors.toSet());
    Set<String> regionIdsSet = Sets.newHashSet(regionIds);
    for (String regionId : regionIdsSet) {
      if(!regionsSet.contains(regionId)) {
        continue;
      }
      organizationRepository.unbindRegion(orgId, regionId);
    }
  }

  /**
   * 组织机构-树结构查询-根据指定组织机构ids或者编码集合(必传任一)，查询这个组织机构以及这个组织机构对应的所有子级结构信息
   * @param ids
   * @param codes
   * @return
   */
  @Override
  public List<OrganizationVo> findChildrenByIdsOrCodes(String[] ids, String[] codes) {
    List<OrganizationVo> orgs = new ArrayList<>();
    if (ids != null && ids.length >0) {
      for(String id : ids){
        OrganizationVo org = this.findById(id);
        if(org != null) {
          orgs.add(org);
        }
      }
    }
    if (codes != null && codes.length >0){
      for(String code :codes) {
        OrganizationVo org = this.findChildrenByCode(code);
        if(org != null) {
          orgs.add(org);
        }
      }
    }

    if(CollectionUtils.isEmpty(orgs)){
      return Lists.newArrayList();
    }
    return orgs;
  }

  @Override
  @NebulaServiceMethod(name="OrganizationService.findByIdStruSelfAndDescendant" , desc="根据ID查询当前组织及当前组织的所有下级,以一维数组的格式返回数据" , returnPropertiesFilter="parent" , scope= NebulaServiceMethod.ScopeType.READ)
  public List<OrganizationVo> findByIdStruSelfAndDescendant(@ServiceMethodParam(name = "id") String id) {
    List<OrganizationEntity> orgs;
    if(StringUtils.isBlank(id)) {
      orgs = organizationRepository.findAll();
    } else {
      OrganizationEntity org = organizationRepository.findDetailsById(id);
      if(org == null) {
        return Lists.newArrayList();
      }
      orgs = Lists.newArrayList();
      orgs.add(org);
      orgs.addAll(this.findDescendant(org));
    }
    if(CollectionUtils.isEmpty(orgs)) {
      return Lists.newArrayList();
    }
    Collection<OrganizationVo> collection = nebulaToolkitService.copyCollectionByWhiteList(orgs, OrganizationEntity.class, OrganizationVo.class, LinkedHashSet.class, ArrayList.class, "parent");
    return Lists.newArrayList(collection);
  }

  @Override
  @NebulaServiceMethod(name="OrganizationService.findByLevel" , desc="根据组织层级查询组织机构" , returnPropertiesFilter="" , scope= NebulaServiceMethod.ScopeType.READ)
  public List<OrganizationVo> findByLevel(@ServiceMethodParam(name = "level") Integer level) {
    if(level == null) {
      return Lists.newArrayList();
    }
    List<OrganizationEntity> orgs = organizationRepository.findByLevel(level);
    if(CollectionUtils.isEmpty(orgs)) {
      return Lists.newArrayList();
    }
    Collection<OrganizationVo> collection = nebulaToolkitService.copyCollectionByWhiteList(orgs, OrganizationEntity.class, OrganizationVo.class, LinkedHashSet.class, ArrayList.class);
    return Lists.newArrayList(collection);
  }

  @Override
  @NebulaServiceMethod(name="OrganizationService.findParentsByLevel" , desc="根据层级查询上级组织机构" , returnPropertiesFilter="" , scope= NebulaServiceMethod.ScopeType.READ)
  public List<OrganizationVo> findParentsByLevel(@ServiceMethodParam(name = "level") Integer level) {
    if(level == null || level <= 1) {
      return Lists.newArrayList();
    }
    level--;
    return this.findByLevel(level);
  }

  /**
   * 查询组织机构的子孙
   * @param org
   * @return
   */
  private List<OrganizationEntity> findDescendant(OrganizationEntity org) {
    List<OrganizationEntity> orgs = new ArrayList<>();
    Set<OrganizationEntity> children = org.getChild();
    if(CollectionUtils.isEmpty(children)) {
      return Lists.newArrayList();
    }
    orgs.addAll(children);
    for (OrganizationEntity chid : children) {
      List<OrganizationEntity> descendant = this.findDescendant(chid);
      if(!CollectionUtils.isEmpty(descendant)) {
        orgs.addAll(descendant);
      }
    }
    return orgs;
  }

  /**
   * 树结构查询-根据组织机构编码查询根节点以及所有子节点以树形结构返回
   * @param code
   * @return
   */
  private OrganizationVo findChildrenByCode(String code) {
    if (StringUtils.isBlank(code)) {
      return null;
    }

    OrganizationEntity organizationEntity = this.organizationRepository.findByCode(code);
    if(organizationEntity == null){
      return null;
    }
    return this.findOrgTree(organizationEntity);
  }

  /**
   * 校验是否循环依赖
   * @param children
   * @param parentId
   */
  private void handleCircular(Set<OrganizationEntity> children, String parentId){
    if(CollectionUtils.isEmpty(children) || StringUtils.isBlank(parentId)){
      return;
    }
    for(OrganizationEntity child:children){
      Validate.isTrue(!StringUtils.equals(child.getId(), parentId),"形成循环依赖，更新失败");
      this.handleCircular(child.getChild(), parentId);
    }
  }
}
