package com.bizunited.platform.user.service.local.service.internal;

import com.bizunited.platform.common.enums.NormalStatusEnum;
import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.core.annotations.NebulaServiceMethod;
import com.bizunited.platform.core.annotations.ServiceMethodParam;
import com.bizunited.platform.core.service.invoke.InvokeParams;
import com.bizunited.platform.rbac.server.util.SecurityUtils;
import com.bizunited.platform.user.common.service.UserModelCodeGenerateService;
import com.bizunited.platform.user.common.service.position.PositionLevelEventListener;
import com.bizunited.platform.user.common.service.position.PositionLevelService;
import com.bizunited.platform.user.common.service.position.PositionService;
import com.bizunited.platform.user.common.vo.PositionLevelVo;
import com.bizunited.platform.user.common.vo.PositionVo;
import com.bizunited.platform.user.common.vo.UserVo;
import com.bizunited.platform.user.service.local.entity.PositionLevelEntity;
import com.bizunited.platform.user.service.local.repository.PositionLevelRepository;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * 职位级别服务实现
 * @author: weikangning
 * @Date: 2020/8/18 16:29
 */
@Service
public class PositionLevelServiceImpl implements PositionLevelService {

  @Autowired
  private PositionLevelRepository positionLevelRepository;
  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private PositionService positionService;
  @Autowired
  private UserModelCodeGenerateService userModelCodeGenerateService;


  /**
   * 已注册到系统中的职级事件监听器
   */
  @Autowired(required = false)
  private List<PositionLevelEventListener> positionLevelEventListeners;

  @Override
  @Transactional
  @NebulaServiceMethod(name = "PositionLevelService.create" , desc = "创建模型对象PositionLevelVo的职级", scope = NebulaServiceMethod.ScopeType.WRITE, returnPropertiesFilter = "" )
  public PositionLevelVo create(PositionLevelVo positionLevelVo) {
    this.createValidation(positionLevelVo);
    if(StringUtils.isBlank(positionLevelVo.getCreateUser()) || StringUtils.isBlank(positionLevelVo.getModifyUser())) {
      UserVo user = SecurityUtils.getCurrentUser();
      positionLevelVo.setCreateUser(user.getAccount());
      positionLevelVo.setModifyUser(user.getAccount());
    }
    Date date = new Date();
    //处理角色列表字段
    if(StringUtils.isNotBlank(positionLevelVo.getRoleCodeStr())){
      positionLevelVo.setRoleCodes(Arrays.asList(positionLevelVo.getRoleCodeStr().split(",")));
    }
    // 转换后进行保存
    PositionLevelEntity positionLevel = this.nebulaToolkitService.copyObjectByWhiteList(positionLevelVo, PositionLevelEntity.class, HashSet.class, ArrayList.class, "roles");
    positionLevel.setCreateTime(date);
    positionLevel.setModifyDate(date);
    positionLevel.setState(ObjectUtils.defaultIfNull(positionLevel.getState(), NormalStatusEnum.ENABLE.getStatus()));
    positionLevelRepository.save(positionLevel);
    positionLevelVo.setId(positionLevel.getId());
    if(!CollectionUtils.isEmpty(positionLevelEventListeners)) {
      for (PositionLevelEventListener eventListener : positionLevelEventListeners) {
        eventListener.onPositionLevelCreated(positionLevelVo);
      }
    }
    return positionLevelVo;
  }

  /**
   * 校验职级Vo
   * @param positionLevelVo
   */
  private void createValidation(PositionLevelVo positionLevelVo){
    Validate.notNull(positionLevelVo, "职位级别信息不能为空，请检查");
    Validate.isTrue(StringUtils.isBlank(positionLevelVo.getId()), "添加职位级别时，不能传入职位级别id,请重新添加!!");
    String name = positionLevelVo.getName();
    Validate.notBlank(name, "职位级别名称不能为空，请检查!!");
    if(StringUtils.isBlank(positionLevelVo.getCode())) {
      positionLevelVo.setCode(userModelCodeGenerateService.positionLevelCodeGenerate());
    }
    //验证长度
    Validate.isTrue(name == null || name.length() < 255 , "职级名称,在进行添加时填入值超过了限定长度(255)，请检查!");
    PositionLevelEntity currentPositionLevel = positionLevelRepository.findByCode(positionLevelVo.getCode());
    Validate.isTrue(currentPositionLevel == null , "存在重复的code,请重新输入职位级别编码!!");
    PositionLevelEntity byName = this.positionLevelRepository.findByName(name);
    Validate.isTrue(byName == null, "存在重复的名称,请重新输入职位级别名称!!");
  }

  @Override
  @Transactional
  @NebulaServiceMethod(name = "PositionLevelService.update" , desc = "修改模型PositionLevelVo的职级", scope = NebulaServiceMethod.ScopeType.WRITE, returnPropertiesFilter = "" )
  public PositionLevelVo update(PositionLevelVo positionLevelVo) {
    // 验证
    this.updateValidation(positionLevelVo);
    //处理角色列表字段
    if(StringUtils.isNotBlank(positionLevelVo.getRoleCodeStr())){
      positionLevelVo.setRoleCodes(Arrays.asList(positionLevelVo.getRoleCodeStr().split(",")));
    }
    String currentId = positionLevelVo.getId();
    PositionLevelEntity positionLevel = positionLevelRepository.findById(currentId).orElse(null);
    Validate.notNull(positionLevel, "未在数据层找到对应的职位级别信息");
    positionLevel.setName(positionLevelVo.getName());
    if(positionLevelVo.getState() != null) {
      positionLevel.setState(positionLevelVo.getState());
    }
    positionLevel.setExtend1(positionLevelVo.getExtend1());
    positionLevel.setExtend2(positionLevelVo.getExtend2());
    positionLevel.setExtend3(positionLevelVo.getExtend3());
    positionLevel.setExtend4(positionLevelVo.getExtend4());
    positionLevel.setExtend5(positionLevelVo.getExtend5());
    positionLevel.setExtend6(positionLevelVo.getExtend6());
    positionLevel.setExtend7(positionLevelVo.getExtend7());
    positionLevel.setExtend8(positionLevelVo.getExtend8());
    positionLevel.setExtend9(positionLevelVo.getExtend9());
    positionLevel.setExtend10(positionLevelVo.getExtend10());
    positionLevelRepository.save(positionLevel);
    PositionLevelVo dbPositionLevel = this.nebulaToolkitService.copyObjectByWhiteList(positionLevel, PositionLevelVo.class, HashSet.class, ArrayList.class);
    dbPositionLevel.setRoleCodes(positionLevelVo.getRoleCodes());
    if(!CollectionUtils.isEmpty(positionLevelEventListeners)) {
      for (PositionLevelEventListener eventListener : positionLevelEventListeners) {
        eventListener.onPositionLevelUpdated(dbPositionLevel);
      }
    }
    return dbPositionLevel;
  }


  /**
   * 更新校验
   * @param positionLevelVo
   */
  private void updateValidation(PositionLevelVo positionLevelVo){
    Validate.notNull(positionLevelVo, "职位级别信息不能为空，请检查");
    String id = positionLevelVo.getId();
    Validate.isTrue(!StringUtils.isBlank(id), "修改职位级别时，必须传入职位级别id,请重新修改!!");
    String name = positionLevelVo.getName();
    Validate.notBlank(name, "职位级别名称不能为空，请检查!!");
    PositionLevelEntity dbPositionLevel = this.positionLevelRepository.findByName(name);
    Validate.isTrue(dbPositionLevel == null || StringUtils.equals(dbPositionLevel.getId(), id), "名称为[%s]的职级已存在,请检查", name);
  }

  @Override
  @Transactional
  public void disableByIds(String[] positionLevelIds) {
    UserVo user = SecurityUtils.getCurrentUser();
    for (String positionLevelId : positionLevelIds) {
      PositionLevelEntity positionLevel = positionLevelRepository.findById(positionLevelId).orElse(null);
      Validate.notNull(positionLevel, "职位级别不存在:%s", positionLevelId);
      if(NormalStatusEnum.DISABLE.getStatus().equals(positionLevel.getState())) {
        continue;
      }
      positionLevel.setState(NormalStatusEnum.DISABLE.getStatus());
      positionLevel.setModifyUser(user.getAccount());
      positionLevel.setModifyDate(new Date());
      positionLevelRepository.save(positionLevel);
    }
  }

  @Override
  @Transactional
  public void enableByIds(String[] ids) {
    UserVo user = SecurityUtils.getCurrentUser();
    for (String id : ids) {
      PositionLevelEntity positionLevel = positionLevelRepository.findById(id).orElse(null);
      Validate.notNull(positionLevel, "职位级别不存在:%s", id);
      if(NormalStatusEnum.ENABLE.getStatus().equals(positionLevel.getState())) {
        continue;
      }
      positionLevel.setState(NormalStatusEnum.ENABLE.getStatus());
      positionLevel.setModifyUser(user.getAccount());
      positionLevel.setModifyDate(new Date());
      positionLevelRepository.save(positionLevel);
    }
  }

  @Override
  public PositionLevelVo findByCode(String code) {
    if (StringUtils.isBlank(code)) {
      return null;
    }
    PositionLevelEntity positionLevel = positionLevelRepository.findByCode(code);
    if (positionLevel == null) {
      return null;
    }
    return nebulaToolkitService.copyObjectByWhiteList(positionLevel, PositionLevelVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  @NebulaServiceMethod(name = "PositionLevelService.findById" , desc = "根据id查询职级", scope = NebulaServiceMethod.ScopeType.READ, returnPropertiesFilter = "" )
  public PositionLevelVo findById(@ServiceMethodParam(name = "id") String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    PositionLevelEntity positionLevel = positionLevelRepository.findById(id).orElse(null);
    PositionLevelVo positionLevelVo = nebulaToolkitService.copyObjectByWhiteList(positionLevel, PositionLevelVo.class, HashSet.class, ArrayList.class);
    return positionLevelVo;
  }

  @Override
  @Transactional
  public void deleteByIds(String[] ids) {
    Validate.notEmpty(ids,"需要删除的职级主键不能为空");
    Set<PositionVo> positionVos = this.positionService.findByPositionLevelIds(ids);
    // 删除职级前先解绑相关职位
    if(!CollectionUtils.isEmpty(positionVos)){
      positionVos.forEach(p -> this.positionService.unbindParent(p.getId()));
    }
    for (String id : ids) {
      PositionLevelEntity positionLevel = positionLevelRepository.findById(id).orElse(null);
      Validate.notNull(positionLevel, "职位级别不存在:%s", id);
      positionLevelRepository.delete(positionLevel);
      PositionLevelVo positionLevelVo = nebulaToolkitService.copyObjectByWhiteList(positionLevel, PositionLevelVo.class, HashSet.class, ArrayList.class, "roleIds", "positions");
      this.sendDeleteEvent(positionLevelVo);
    }
  }

  /**
   * 推送职级删除事件
   * @param positionLevel
   */
  private void sendDeleteEvent(PositionLevelVo positionLevel) {
    if(CollectionUtils.isEmpty(this.positionLevelEventListeners) || positionLevel == null){
      return;
    }
    this.positionLevelEventListeners.forEach(listener -> listener.onPositionLevelDeleted(positionLevel));
  }

  @Override
  @NebulaServiceMethod(name = "PositionLevelService.findAll" , desc = "查询所有职级列表(不包括已删除的)", scope = NebulaServiceMethod.ScopeType.READ, returnPropertiesFilter = "" )
  public List<PositionLevelVo> findAll() {
    List<PositionLevelEntity> positionLevelEntities = this.positionLevelRepository.findAll();
    if(CollectionUtils.isEmpty(positionLevelEntities)){
      return Lists.newArrayList();
    }
    Collection<PositionLevelVo> positionVos = nebulaToolkitService.copyCollectionByWhiteList(positionLevelEntities,
            PositionLevelEntity.class, PositionLevelVo.class, HashSet.class, ArrayList.class);
    if(CollectionUtils.isEmpty(positionVos)){
      return Lists.newArrayList();
    }
    return Lists.newArrayList(positionVos);
  }

  @Override
  @NebulaServiceMethod(name = "PositionLevelService.findByConditions" , desc = "条件查询职级列表", scope = NebulaServiceMethod.ScopeType.READ, returnPropertiesFilter = "" )
  public Page<PositionLevelVo> findByConditions(InvokeParams invokeParams, @ServiceMethodParam(name = "pageable") Pageable pageable) {
    // 如果当前没有设定分页信息，则默认第一页，每页50条数据
    pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    invokeParams = ObjectUtils.defaultIfNull(invokeParams, new InvokeParams());
    Map<String, Object> conditions = ObjectUtils.defaultIfNull(invokeParams.getInvokeParams(), Maps.newHashMap());
    Page<PositionLevelEntity> page = positionLevelRepository.findByConditions(pageable, conditions);
    // 分页信息需要自行转换
    List<PositionLevelEntity> positionLevels = page.getContent();
    if (positionLevels.isEmpty()) {
      return Page.empty(pageable);
    }
    Collection<PositionLevelVo> positionLevelVos = this.nebulaToolkitService.copyCollectionByWhiteList(positionLevels, PositionLevelEntity.class, PositionLevelVo.class, LinkedHashSet.class, ArrayList.class);
    return new PageImpl<>(new ArrayList<>(positionLevelVos), pageable, page.getTotalElements());
  }


}
