package com.bizunited.platform.user.service.local.service.internal;

import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.common.constant.Constants;
import com.bizunited.platform.common.constant.RedisKeyConstants;
import com.bizunited.platform.common.enums.NormalStatusEnum;
import com.bizunited.platform.common.enums.RbacRelationEnum;
import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.common.service.redis.RedisMutexService;
import com.bizunited.platform.core.annotations.NebulaServiceMethod;
import com.bizunited.platform.core.annotations.ServiceMethodParam;
import com.bizunited.platform.core.service.invoke.InvokeParams;
import com.bizunited.platform.user.common.service.UserMappingService;
import com.bizunited.platform.user.common.service.UserModelCodeGenerateService;
import com.bizunited.platform.user.common.service.organization.OrganizationService;
import com.bizunited.platform.user.common.service.position.PositionEventListener;
import com.bizunited.platform.user.common.service.position.PositionLevelService;
import com.bizunited.platform.user.common.service.position.PositionService;
import com.bizunited.platform.user.common.service.user.UserService;
import com.bizunited.platform.user.common.vo.OrganizationVo;
import com.bizunited.platform.user.common.vo.PositionLevelVo;
import com.bizunited.platform.user.common.vo.PositionSimpleVo;
import com.bizunited.platform.user.common.vo.PositionVo;
import com.bizunited.platform.user.common.vo.UserMappingVo;
import com.bizunited.platform.user.common.vo.UserVo;
import com.bizunited.platform.user.service.local.entity.OrganizationEntity;
import com.bizunited.platform.user.service.local.entity.PositionEntity;
import com.bizunited.platform.user.service.local.entity.UserEntity;
import com.bizunited.platform.user.service.local.repository.PositionRepository;
import com.bizunited.platform.user.service.local.repository.UserRepository;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.security.Principal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.transaction.Transactional;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.redisson.api.RedissonClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.util.CollectionUtils;

/**
 * 职位本地服务实现
 *
 * @author: yanwe
 * @date: 17/Jan/2019 15:01
 */
public class PositionServiceImpl implements PositionService {

  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(PositionServiceImpl.class);

  @Autowired 
  private PositionRepository positionRepository;
  @Autowired 
  private OrganizationService organizationService;
  @Autowired
  private UserService userService;
  @Autowired
  private UserMappingService userMappingService;
  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private RedissonClient redissonClient;
  @Autowired
  private UserRepository userRepository;
  @Autowired
  private PositionLevelService positionLevelService;
  @Autowired
  private RedisMutexService redisMutexService;
  @Autowired
  private UserModelCodeGenerateService userModelCodeGenerateService;
  @Autowired(required = false)
  private List<PositionEventListener> positionLevelEventListeners;

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.PositionService#create(com.bizunited.platform.rbac.server.vo.PositionVo)
   */
  @Override
  @Transactional
  @NebulaServiceMethod(name = "PositionService.create" , desc = "创建一条模型为PositionVo的职位信息", scope = NebulaServiceMethod.ScopeType.WRITE, returnPropertiesFilter = "" )
  public PositionVo create(PositionVo positionVo) {
    Validate.notNull(positionVo, "岗位信息不能为空，请检查");
    Validate.isTrue(StringUtils.isBlank(positionVo.getId()), "添加岗位时，不能传入岗位id,请重新添加!!");
    Validate.notBlank(positionVo.getName(), "岗位名称不能为空，请检查!!");
    if(StringUtils.isBlank(positionVo.getCode())) {
      positionVo.setCode(userModelCodeGenerateService.positionCodeGenerate());
    }
    positionVo.setCreateTime(new Date());
    PositionEntity currentPosition = positionRepository.findByCode(positionVo.getCode());
    Validate.isTrue(null == currentPosition,"存在重复的code,请重新输入岗位编码!!");
    if(positionVo.getParent() != null){
      Validate.notBlank(positionVo.getParent().getId(), "未找到该岗位的上级岗位的id，请检查！");
      PositionEntity parent = positionRepository.findById(positionVo.getParent().getId()).orElse(null);
      Validate.notNull(parent, "未找到该岗位的上级岗位，请检查！");
    }
    if(positionVo.getOrganization() != null) {
      Validate.notBlank(positionVo.getOrganization().getId(), "未找到该岗位绑定的组织id，请检查！");
      OrganizationVo organizationVo = this.organizationService.findById(positionVo.getOrganization().getId());
      Validate.notNull(organizationVo, "没有获取到该岗位绑定的组织，请检查！");
    }
    // 转换后进行保存
    PositionEntity positionEntity = this.nebulaToolkitService.copyObjectByWhiteList(positionVo, PositionEntity.class, HashSet.class, ArrayList.class, "parent", "organization");
    positionRepository.save(positionEntity);
    positionVo.setId(positionEntity.getId());
    if(!CollectionUtils.isEmpty(positionLevelEventListeners)) {
      for (PositionEventListener listener : positionLevelEventListeners) {
        listener.onPositionCreated(positionVo);
      }
    }
    return positionVo;
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.PositionService#update(com.bizunited.platform.rbac.server.vo.PositionVo)
   */
  @Override
  @Transactional
  @NebulaServiceMethod(name = "PositionService.update" , desc = "修改一条模型为PositionVo的职位信息", scope = NebulaServiceMethod.ScopeType.WRITE, returnPropertiesFilter = "" )
  public PositionVo update(PositionVo positionVo) {
    // 验证
    Validate.notNull(positionVo, "岗位信息不能为空，请检查");
    Validate.isTrue(!StringUtils.isBlank(positionVo.getId()), "修改岗位时，必须传入岗位id,请重新修改!!");
    Validate.notBlank(positionVo.getName(), "岗位名称不能为空，请检查!!");
    Optional<PositionEntity> op = positionRepository.findById(positionVo.getId());
    PositionEntity position = op.orElse(null);
    Validate.notNull(position, "未在数据层找到对应的岗位信息");
    if(positionVo.getParent() != null){
      String parentId = positionVo.getParent().getId();
      Validate.notBlank(parentId, "未找到该岗位的上级岗位的id，请检查！");
      Validate.isTrue(!positionVo.getId().equals(parentId), "禁止将该岗位本身设置为上级岗位");
      PositionEntity parent = positionRepository.findById(parentId).orElse(null);
      Validate.notNull(parent, "未找到该岗位的上级岗位，请检查！");
      position.setParent(parent);
      Set<String> positionStack = new HashSet<>();
      positionStack.add(parentId);
      this.validateCircular(position, positionStack);
    }else {
      position.setParent(null);
    }
    // 重新绑定组织
    if(positionVo.getOrganization() != null) {
      OrganizationVo organizationVo = this.organizationService.findById(positionVo.getOrganization().getId());
      Validate.notNull(organizationVo, "没有获取到所属组织");
      OrganizationEntity organizationEntity = new OrganizationEntity();
      organizationEntity.setId(organizationVo.getId());
      position.setOrganization(organizationEntity);
    } else {
      position.setOrganization(null);
    }
    // 赋值并保存
    position.setName(positionVo.getName());
    //修改扩展字段
    position.setExtend1(positionVo.getExtend1());
    position.setExtend2(positionVo.getExtend2());
    position.setExtend3(positionVo.getExtend3());
    position.setExtend4(positionVo.getExtend4());
    position.setExtend5(positionVo.getExtend5());
    position.setExtend6(positionVo.getExtend6());
    position.setExtend7(positionVo.getExtend7());
    position.setExtend8(positionVo.getExtend8());
    position.setExtend9(positionVo.getExtend9());
    position.setExtend10(positionVo.getExtend10());
    positionRepository.save(position);
    if(!CollectionUtils.isEmpty(positionLevelEventListeners)) {
      for (PositionEventListener listener : positionLevelEventListeners) {
        listener.onPositionUpdated(positionVo);
      }
    }
    // 转换后返回
    return this.nebulaToolkitService.copyObjectByWhiteList(position, PositionVo.class, HashSet.class, ArrayList.class);
  }

  /**
   *判断是否形成循环依赖
   * @param parent
   * @param positionStack
   */
  private void validateCircular(PositionEntity parent, Set<String> positionStack) {
    if(CollectionUtils.isEmpty(parent.getChildren())) {
      return;
    }
    for (PositionEntity positionEntity : parent.getChildren()) {
      Validate.isTrue(!positionStack.contains(positionEntity.getId()), "形成循环依赖，更新失败，请检查！");
      positionStack.add(positionEntity.getId());
      this.validateCircular(positionEntity, positionStack);
    }
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.PositionService#updateStatus(java.lang.String)
   */
  @Override
  @Transactional
  public PositionVo updateStatus(String positionId) {
    Validate.notBlank(positionId, "岗位id不能为空，请检查");
    Optional<PositionEntity> op = positionRepository.findById(positionId);
    PositionEntity position = op.orElse(null);
    Validate.notNull(position, "没有该岗位，请检查!!");
    Integer status = position.getTstatus();
    
    // 反转状态，0:禁用,1:启用
    status = (status == 1 ? 0 : 1);
    position.setTstatus(status);
    positionRepository.save(position);
    
    // 转换后返回
    return this.nebulaToolkitService.copyObjectByWhiteList(position, PositionVo.class, HashSet.class, ArrayList.class);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.PositionService#findById(java.lang.String)
   */
  @Override
  @NebulaServiceMethod(name = "PositionService.findDetailsById" , desc = "根据id查询职位详情", scope = NebulaServiceMethod.ScopeType.READ, returnPropertiesFilter = "organization,roles,users,parent" )
  public PositionVo findDetailsById(@ServiceMethodParam(name = "id") String positionId) {
    if (StringUtils.isBlank(positionId)) {
      return null;
    }
    PositionEntity positionEntity = positionRepository.findDetailsById(positionId);
    if(positionEntity == null) {
      return null;
    }

    // 转换后返回
    return this.nebulaToolkitService.copyObjectByWhiteList(positionEntity, PositionVo.class, LinkedHashSet.class, ArrayList.class, "organization" , "roles" , "users", "parent");
  }

  @Override
  @Transactional
  @NebulaServiceMethod(name = "PositionService.bindOrg" , desc = "职位绑定组织", scope = NebulaServiceMethod.ScopeType.WRITE, returnPropertiesFilter = "" )
  public void bindOrg(@ServiceMethodParam(name = "orgId") String orgId, @ServiceMethodParam(name = "positionId") String positionId) {
    Validate.notBlank(orgId, "组织机构ID不能为空!");
    Validate.notBlank(positionId, "岗位ID不能为空!");
    OrganizationVo organizationVo = this.organizationService.findById(orgId);
    Validate.notNull(organizationVo, "未找到指定的组织机构!");
    OrganizationEntity organization = nebulaToolkitService.copyObjectByWhiteList(organizationVo, OrganizationEntity.class, HashSet.class, ArrayList.class);
    Optional<PositionEntity> op = positionRepository.findById(positionId);
    PositionEntity position = op.orElse(null);
    Validate.notNull(position, "未找到指定的岗位!");
    Validate.isTrue(null == position.getOrganization(), "该岗位已绑定有组织机构，请检查！");
    position.setOrganization(organization);
    positionRepository.saveAndFlush(position);
  }
  
  @Override
  @Transactional
  @NebulaServiceMethod(name = "PositionService.unbindOrg" , desc = "职位解绑组织", scope = NebulaServiceMethod.ScopeType.WRITE, returnPropertiesFilter = "" )
  public void unbindOrg(@ServiceMethodParam(name = "positionId") String positionId) {
    Validate.notBlank(positionId, "岗位ID不能为空!");
    Optional<PositionEntity> op = positionRepository.findById(positionId);
    PositionEntity position = op.orElse(null);
    Validate.notNull(position, "未找到指定的岗位!");
    
    // 解除绑定关系
    position.setOrganization(null);
    positionRepository.saveAndFlush(position);
  }

  @Override
  @Transactional
  public void reBindOrg(String orgId, String positionId) {
    Validate.notBlank(orgId, "组织机构ID不能为空!");
    Validate.notBlank(positionId, "岗位ID不能为空!");
    OrganizationVo organizationVo = this.organizationService.findById(orgId);
    Validate.notNull(organizationVo, "未找到指定的组织机构");
    OrganizationEntity organization = nebulaToolkitService.copyObjectByWhiteList(organizationVo, OrganizationEntity.class, HashSet.class, ArrayList.class);
    Optional<PositionEntity> op = positionRepository.findById(positionId);
    PositionEntity position = op.orElse(null);
    Validate.notNull(position, "未找到指定的岗位!");
    
    // 重新设定指定岗位的组织机构信息
    position.setOrganization(organization);
    positionRepository.saveAndFlush(position);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.PositionService#bindUsersAndPosition(java.util.List, java.lang.String)
   */
  @Override
  @Transactional
  public void bindUsersAndPosition(List<String> userIds, String positionId) {
    Validate.isTrue(!CollectionUtils.isEmpty(userIds), "用户ID集合不能为空，请检查！");
    UserMappingVo mappingVo = userMappingService.findOne();
    userIds.stream().forEach(o -> bindUserAndPosition(mappingVo, o, positionId));
  }
  /**
   * 验证绑定用户和岗位
   * @param userId
   * @param positionId
   * @param mappingVo
   */
  private void validateBindUser(UserMappingVo mappingVo,String positionId, String userId) {
    if (mappingVo == null) {
      long count = positionRepository.countByUserAndPosition(userId, positionId);
      Validate.isTrue(count == 0, "在批量绑定时,指定用户[%s]已经绑定了，不能重复绑定，请检查!!", userId);
    } else {
      int bindRelation = mappingVo.getUserPositionRelation();
      //如果用户和岗位绑定关系为多对多,多对一，一对多，一对一分别控制判断
      if (bindRelation == RbacRelationEnum.MANY_TO_MANY.getRelation()) {
        long count = positionRepository.countByUserAndPosition(userId, positionId);
        Validate.isTrue(count == 0, "在批量绑定时,用户和岗位多对多，指定用户[%s]已经绑定了，不能重复绑定，请检查!!", userId);
      }
      if (bindRelation == RbacRelationEnum.MANY_TO_ONE.getRelation()) {
        long count = positionRepository.countPositionByUserId(userId);

        Validate.isTrue(count == 0, "在批量绑定时,用户和岗位多对一，指定用户[%s]已经绑定了岗位，不能继续绑定，请检查!!", userId);
      }
      if (bindRelation == RbacRelationEnum.ONE_TO_MANY.getRelation()) {
        long count = positionRepository.countUserByPositionId(positionId);
        Validate.isTrue(count == 0, "在批量绑定时,用户和岗位一对多，指定岗位[%s]已经绑定了用户，不能继续绑定，请检查!!", positionId);
      }
      if (bindRelation == RbacRelationEnum.ONE_TO_ONE.getRelation()) {
        long countUser = positionRepository.countUserByPositionId(positionId);
        Validate.isTrue(countUser == 0, "在批量绑定时,用户和岗位一对一，指定岗位[%s]已经绑定了用户，不能继续绑定，请检查!!", positionId);
        long countPosition = positionRepository.countPositionByUserId(userId);
        Validate.isTrue(countPosition == 0, "在批量绑定时,用户和岗位一对一，指定用户[%s]已经绑定了岗位，不能继续绑定，请检查!!", userId);
      }

    }
  }
  /**
   * 绑定用户和岗位
   * @param userId
   * @param positionId
   */
  private void bindUserAndPosition(UserMappingVo mappingVo,String userId, String positionId) {
    Validate.notBlank(userId, "用户ID不能为空!");
    Validate.notBlank(positionId, "岗位ID不能为空!");
    UserVo user = userService.findByUserId(userId);
    Validate.notNull(user, "未找到指定的用户!");

    Optional<PositionEntity> optional = positionRepository.findById(positionId);
    Validate.isTrue(optional.isPresent(), "未找到指定的岗位!");
    this.validateBindUser(mappingVo, positionId, userId);
    positionRepository.bindUser(userId, positionId);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.PositionService#unbindUsersAndPosition(java.util.List, java.lang.String)
   */
  @Override
  @Transactional
  public void unbindUsersAndPosition(List<String> userIds, String positionId) {
    Validate.notEmpty(userIds, "用户ID集合不能为空，请检查！");
    userIds.stream().forEach(o -> unbindUserAndPosition(o, positionId));
  }

  /**
   * 解绑用户和岗位
   * @param userId
   * @param positionId
   */
  private void unbindUserAndPosition(String userId, String positionId) {
    Validate.notBlank(userId, "用户ID不能为空!");
    Validate.notBlank(positionId, "岗位ID不能为空!");
    UserVo user = userService.findByUserId(userId);
    Validate.notNull(user, "未找到指定的用户!");
    PositionEntity position = positionRepository.findById(positionId).orElse(null);
    Validate.notNull(position, "未找到指定的岗位!");
    Validate.isTrue(0L != positionRepository.countByUserAndPosition(userId, positionId), "该用户与该岗位未有绑定关系，请检查！");
    //主岗位不能解绑
    PositionVo positionVo = null;
    String key = Constants.REDIS_KEY_RBAC_USER + userId;
    Map<String, String> mainPosition = redissonClient.getMap(key);
    if(mainPosition.containsKey(Constants.MAIN_POSITION)){
      positionVo = JSONObject.parseObject(mainPosition.get(Constants.MAIN_POSITION), PositionVo.class);
    }
    if(positionVo != null) {
      Validate.isTrue(!positionId.equals(positionVo.getId()), "[%s]是用户的主岗位不能解绑", position.getName());
    }
    positionRepository.unbindUser(userId, positionId);
    if(!CollectionUtils.isEmpty(positionLevelEventListeners)) {
      // 调用监听器，通知岗位解绑用户事件
      PositionVo dbPositionVo = nebulaToolkitService.copyObjectByWhiteList(position, PositionVo.class, LinkedHashSet.class, ArrayList.class);
      for (PositionEventListener positionLevelEventListener : positionLevelEventListeners) {
        positionLevelEventListener.onUnbindUser(dbPositionVo, user);
      }
    }
  }
  
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.PositionService#findByUserId(java.lang.String)
   */
  @Override
  public List<PositionVo> findByUserId(String userId) {
    if(StringUtils.isBlank(userId)) {
      return Lists.newArrayList();
    }
    Set<PositionEntity> positions = positionRepository.findByUserId(userId);
    if(CollectionUtils.isEmpty(positions)) {
      return Lists.newArrayList();
    }
    Collection<PositionVo> positionVos = this.nebulaToolkitService.copyCollectionByWhiteList(positions, PositionEntity.class, PositionVo.class, HashSet.class, ArrayList.class);
    return Lists.newArrayList(positionVos);
  }

  @Override
  public List<PositionVo> findByUserAccount(String userAccount) {
    if(StringUtils.isBlank(userAccount)) {
      return Lists.newArrayList();
    }
    Set<PositionEntity> positions = positionRepository.findByUserAccount(userAccount);
    if(CollectionUtils.isEmpty(positions)) {
      return Lists.newArrayList();
    }
    Collection<PositionVo> positionVos = this.nebulaToolkitService.copyCollectionByWhiteList(positions, PositionEntity.class, PositionVo.class, HashSet.class, ArrayList.class);
    return Lists.newArrayList(positionVos);
  }

  @Override
  public Set<PositionVo> findByIds(List<String> ids) {
    if(CollectionUtils.isEmpty(ids)){
      return Sets.newHashSet();
    }
    Set<PositionEntity> pos = positionRepository.findByIds(ids);
    if(CollectionUtils.isEmpty(pos)){
      return Sets.newHashSet();
    }
    Collection<PositionVo> pvo = this.nebulaToolkitService.copyCollectionByWhiteList(pos,PositionEntity.class,PositionVo.class,HashSet.class,ArrayList.class);
    return Sets.newHashSet(pvo);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.PositionService#findByConditions(java.lang.String, java.lang.String, java.lang.Integer, org.springframework.data.domain.Pageable)
   */
  @Override
  public Page<PositionVo> findByConditions(Map<String, Object> conditions, Pageable pageable) {
    // 如果当前没有设定分页信息，则默认第一页，每页50条数据
    if (pageable == null) {
      pageable = PageRequest.of(0, 50);
    }

    Page<PositionEntity> page = positionRepository.queryPage(pageable, conditions);
    List<PositionEntity> content = page.getContent();
    if(CollectionUtils.isEmpty(content)) {
      return Page.empty(pageable);
    }
    Collection<PositionVo> collection = this.nebulaToolkitService.copyCollectionByWhiteList(content, PositionEntity.class, PositionVo.class, LinkedHashSet.class, ArrayList.class, "parent", "organization", "positionLevel","users");
    return new PageImpl<>(Lists.newArrayList(collection), pageable, page.getTotalElements());
  }

  @Override
  @NebulaServiceMethod(name = "PositionService.findByConditions", desc = "条件查询职位列表", scope = NebulaServiceMethod.ScopeType.READ, returnPropertiesFilter = "positionLevel,parent,organization")
  public Page<PositionVo> findByConditions(@ServiceMethodParam(name = "pageable") Pageable pageable, InvokeParams invokeParams) {
    pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    invokeParams = ObjectUtils.defaultIfNull(invokeParams, new InvokeParams());
    Map<String, Object> params = invokeParams.getInvokeParams();
    return this.findByConditions(params, pageable);
  }

  @Override
  public PositionVo findByCode(String code) {
    if(StringUtils.isBlank(code)) {
      return null;
    }
    PositionEntity position = positionRepository.findByCode(code);
    if(position == null) {
      return null;
    }
    return nebulaToolkitService.copyObjectByWhiteList(position, PositionVo.class, HashSet.class, ArrayList.class);
  }

  /**
   * 切换岗位
   * @param positionId
   */
  @Override
  public UserVo changePosition(String positionId, String userId){
    Validate.notBlank(positionId, "切换岗位id不能为空");
    Validate.notBlank(userId, "用户id不能为空");

    UserVo user = userService.findByUserId(userId);
    Validate.notNull(user, "未获取到用户信息，请检查");

    PositionEntity positionEntity = positionRepository.findById(positionId).orElse(null);
    Validate.notNull(positionEntity, "未获取到岗位信息，请检查");
    PositionVo positionVo = nebulaToolkitService.copyObjectByWhiteList(positionEntity, PositionVo.class, HashSet.class, ArrayList.class);

    List<PositionVo> positionVos = this.findByUserId(userId);
    user.setPositions(Sets.newLinkedHashSet(positionVos));
    user.setMainPosition(positionId);
    Map<String, String> mainPosition = redissonClient.getMap(Constants.REDIS_KEY_RBAC_USER + userId);
    mainPosition.put(Constants.MAIN_POSITION, JSONObject.toJSON(positionVo).toString());
    return user;
  }

  /**
   * 查询Redis中主岗位，如果redis中不存在，则默认主岗位为数据库查询的第一条数据，并且将该数据存入redis中
   * @param userId
   * @return
   */
  @Override
  @NebulaServiceMethod(name = "PositionService.findMainPositionByUserId" , desc = "查询人员当前登录职位", scope = NebulaServiceMethod.ScopeType.READ, returnPropertiesFilter = "" )
  public PositionVo findMainPositionByUserId(@ServiceMethodParam(name = "userId") String userId){
    Validate.notBlank(userId, "传入的用户id不能为空");
    PositionVo positionVo = null;
    String key = Constants.REDIS_KEY_RBAC_USER + userId;
    Map<String, String> mainPosition = redissonClient.getMap(key);
    if(mainPosition.containsKey(Constants.MAIN_POSITION)){
      positionVo = JSONObject.parseObject(mainPosition.get(Constants.MAIN_POSITION), PositionVo.class);
    }
    if(positionVo == null){
      List<PositionVo> positionVoList = this.findByUserId(userId);
      if(CollectionUtils.isEmpty(positionVoList)){
        return null;
      }else {
        positionVo = positionVoList.get(0);
      }
      String json = JSONObject.toJSON(positionVo).toString();
      mainPosition.put(Constants.MAIN_POSITION, json);
    }
    return positionVo;
  }

  /**
   * 重新给岗位绑定用户关系
   * @param userIds
   * @param positionId
   */
  @Override
  @Transactional
  public void rebindUsers(String[] userIds, String positionId) {
    /***
     * 1.用户已绑定不再重复绑定
     * 2.用户未绑定进行绑定
     * 3.该岗位以前的用户进行解绑
     */
    Validate.notBlank(positionId, "岗位ID不能为空!");
    //已绑定过的用户id
    Set<UserVo> users = userService.findByPositionId(positionId);
    if(users == null) {
      users = Sets.newHashSet();
    }
    Set<String> bindedUserIds = users.stream().map(UserVo::getId).collect(Collectors.toSet());
    //传入的用户id
    Set<String> currentUserIds = new HashSet<>();
    if (userIds != null){
      for (String userId : userIds) {
        currentUserIds.add(userId);
      }
    }
    //解绑用户
    Set<String> needUnbindsUserIds = Sets.difference(bindedUserIds, currentUserIds);
    if (!CollectionUtils.isEmpty(needUnbindsUserIds)){
      for (String userId : needUnbindsUserIds) {
        if (StringUtils.isBlank(userId)){
          continue;
        }
        this.unbindUserAndPosition(userId, positionId);
      }
    }
    //绑定用户
    Set<String> needbindsUserIds = Sets.difference(currentUserIds, bindedUserIds);
    UserMappingVo mappingVo = userMappingService.findOne();
    if (!CollectionUtils.isEmpty(needbindsUserIds)){
      for (String userId : needbindsUserIds) {
        if (StringUtils.isBlank(userId)){
          continue;
        }
        this.bindUserAndPosition(mappingVo, userId, positionId);
      }
    }
  }

  /**
   * 查询所有岗位信息（根据创建时间排序）
   * @return
   */
  @Override
  public List<PositionVo> findAll(){
    Sort sort = Sort.by(Sort.Order.desc("createTime"));
    List<PositionEntity> positions = positionRepository.findAll(sort);
    if (CollectionUtils.isEmpty(positions)) {
      return Lists.newArrayList();
    }
    Collection<PositionVo> positionVos = nebulaToolkitService.copyCollectionByWhiteList(positions, PositionEntity.class, PositionVo.class, HashSet.class, ArrayList.class, "parent");
    return Lists.newArrayList(positionVos);
  }

  /**
   * 根据条件查询所有岗位
   * @param status
   * @return
   */
  @Override
  public List<PositionVo> findByStatus(Integer status) {
    List<PositionEntity> positions = positionRepository.findByTstatus(status);
    if(CollectionUtils.isEmpty(positions)) {
      return Lists.newArrayList();
    }
    Collection<PositionVo> collection = nebulaToolkitService.copyCollectionByWhiteList(positions, PositionEntity.class, PositionVo.class, HashSet.class, ArrayList.class);
    return Lists.newArrayList(collection);
  }

  /**
   * 根据登陆人关联的岗位查询
   * @param type
   * @return
   */
  @Override
  public Set<PositionVo> findByPrincipal(Integer type, Principal principal) {
    Validate.notNull(type, "传入类型不能为空，请检查");
    String account = principal.getName();
    UserEntity user = userRepository.findByAccount(account);
    Validate.notNull(user, "未获取登陆人信息，请检查");
    Set<PositionVo> positions = null;
    switch (type) {
      case 1:
        positions = this.findParent(user);
        break;
      case 2:
        positions = this.findChildren(user);
        break;
      case 3:
        positions = this.findParentAndChildren(user);
        break;
      case 4:
        positions = this.findPosition(user);
        break;
      default:
        break;
    }
    if(CollectionUtils.isEmpty(positions)){
      return Sets.newHashSet();
    }
    return positions;
  }

  /**
   * 根据登陆人关联的岗位查询上级
   * @param user
   * @return
   */
  private Set<PositionVo> findParent(UserEntity user) {
    Set<PositionVo> result = new HashSet<>();
    Set<PositionVo> removes = new HashSet<>();
    Map<String, PositionVo> parentMap = new HashMap<>();
    Set<PositionVo> rootPositions = this.findPosition(user);
    for(PositionVo rootPosition : rootPositions) {
      rootPosition.setParent(this.findAllParent(rootPosition, parentMap));
      result.add(rootPosition);
    }
    //过滤
    for(PositionVo item : result){
      if(parentMap.containsKey(item.getId())){
        removes.add(item);
      }
    }
    for(PositionVo remove : removes){
      result.remove(remove);
    }
    return result;
  }

  /**
   * 根据登陆人关联的岗位查询
   * @param user
   * @return
   */
  private Set<PositionVo> findPosition(UserEntity user) {
    Set<PositionEntity> positions = positionRepository.findByUserId(user.getId());
    Collection<PositionVo> result = nebulaToolkitService.copyCollectionByWhiteList(positions, PositionEntity.class, PositionVo.class, LinkedHashSet.class, ArrayList.class);
    return Sets.newHashSet(result);
  }

  /**
   * 查询父级并将下级信息存入map
   * @param rootPosition
   * @param parentMap
   * @return
   */
  private PositionVo findAllParent(PositionVo rootPosition, Map<String, PositionVo> parentMap) {
    // 查询父级结构
    PositionEntity currentPosition = positionRepository.findById(rootPosition.getId()).orElse(null);
    PositionEntity parent = currentPosition.getParent();
    if(parent == null){
      return null;
    }
    PositionVo positionVo = this.nebulaToolkitService.copyObjectByWhiteList(parent, PositionVo.class, HashSet.class, ArrayList.class);
    parentMap.put(positionVo.getId(), positionVo);
    positionVo.setParent(this.findAllParent(positionVo, parentMap));
    return positionVo;
  }

  /**
   * 根据登陆人关联的岗位查询下级
   * @param user
   * @return
   */
  private Set<PositionVo> findChildren(UserEntity user) {
    Set<PositionVo> result = new HashSet<>();
    Set<PositionVo> removes = new HashSet<>();
    Map<String, PositionVo> childMap = new HashMap<>();
    Set<PositionVo> rootPositions = this.findPosition(user);
    for(PositionVo rootPosition : rootPositions) {
      rootPosition.setChildren(this.findAllChildren(rootPosition, childMap));
      result.add(rootPosition);
    }
    //过滤
    for(PositionVo item : result){
      if(childMap.containsKey(item.getId())){
        removes.add(item);
      }
    }
    for(PositionVo remove : removes){
      result.remove(remove);
    }
    return result;
  }

  /**
   * 查询下级并将下级信息存入map
   * @param position
   * @return
   */
  private Set<PositionVo> findAllChildren(PositionVo position, Map<String, PositionVo> childMap) {
    Set<PositionVo> result = new HashSet<>();
    // 查询子级结构
    PositionEntity currentPosition = positionRepository.findById(position.getId()).orElse(null);
    Set<PositionEntity> children = currentPosition.getChildren();
    if(CollectionUtils.isEmpty(children)){
      return Sets.newHashSet();
    }
    for (PositionEntity child : children) {
      PositionVo positionVo = this.nebulaToolkitService.copyObjectByWhiteList(child, PositionVo.class, HashSet.class, ArrayList.class);
      childMap.put(positionVo.getId(), positionVo);
      positionVo.setChildren(this.findAllChildren(positionVo, childMap));
      result.add(positionVo);
    }
    return result;
  }

  /**
   * 根据登陆人关联的岗位查询上下级
   * @param user
   * @return
   */
  private Set<PositionVo> findParentAndChildren(UserEntity user) {
    Set<PositionVo> result = new HashSet<>();
    Set<PositionVo> removes = new HashSet<>();
    Map<String, PositionVo> childMap = new HashMap<>();
    Map<String, PositionVo> parentMap = new HashMap<>();
    Set<PositionVo> rootPositions = this.findPosition(user);
    for(PositionVo rootPosition : rootPositions) {
      rootPosition.setChildren(this.findAllChildren(rootPosition, childMap));
      rootPosition.setParent(this.findAllParent(rootPosition, parentMap));
      result.add(rootPosition);
    }
    //过滤
    for(PositionVo item : result){
      if(childMap.containsKey(item.getId())){
        removes.add(item);
      }
    }
    for(PositionVo remove : removes){
      result.remove(remove);
    }
    return result;
  }

  /**
   * 根据登陆人关联的岗位和输入层级查询控件
   * @param type
   * @param principal
   * @param level
   * @return
   */
  @Override
  public Set<PositionVo> findByTypeAndLevel(Integer type, Principal principal, Integer level) {
    Validate.notNull(type, "传入类型不能为空，请检查");
    Validate.notNull(level, "传入层级不能为空，请检查");
    String account = principal.getName();
    UserEntity user = userRepository.findByAccount(account);
    Validate.notNull(user, "未获取登陆人信息，请检查");
    Set<PositionVo> positions = null;
    String[] propertiesFilter = new String[level];
    Set<PositionVo> newPositions = new HashSet<>();
    switch (type) {
      case 1:
        positions = this.findParent(user);
        this.bindPropertiesFilter(level, propertiesFilter, "parent");
        break;
      case 2:
        positions = this.findChildren(user);
        this.bindPropertiesFilter(level, propertiesFilter, "children");
        break;
      default:
        break;
    }
    if(CollectionUtils.isEmpty(positions)){
      return Sets.newHashSet();
    }
    for (PositionVo position : positions) {
      PositionVo positionVo = this.nebulaToolkitService.copyObjectByWhiteList(position, PositionVo.class, HashSet.class, ArrayList.class, propertiesFilter);
      newPositions.add(positionVo);
    }
    return newPositions;
  }

  /**
   * 根据职级批量查询关联的职位
   * @param positionLevelIds 职级主键列表
   * @return
   */
  @Override
  public Set<PositionVo> findByPositionLevelIds(String[] positionLevelIds) {
    if(ArrayUtils.isEmpty(positionLevelIds)){
      return Sets.newHashSet();
    }
    Set<PositionEntity> positionEntities = this.positionRepository.findByPositionLevelIds(positionLevelIds);
    if(CollectionUtils.isEmpty(positionEntities)){
      return Sets.newHashSet();
    }
    Collection<PositionVo> positionVos = this.nebulaToolkitService.copyCollectionByWhiteList(positionEntities, PositionEntity.class, PositionVo.class, LinkedHashSet.class, LinkedList.class);
    return Sets.newHashSet(positionVos);
  }

  /**
   * 根据职位主键，批量禁用职位
   * @param ids
   */
  @Override
  @Transactional
  public void disableByIds(String[] ids) {
    Validate.notEmpty(ids,"职级主键不能为空");
    for(String id:ids){
      PositionEntity positionEntity = this.positionRepository.findById(id).orElse(null);
      Validate.notNull(positionEntity,"找不到主键对应的职位，请检查!");
      positionEntity.setTstatus(0);
      this.positionRepository.save(positionEntity);
    }
  }

  /**
   * 根据职位主键，批量启用职位
   * @param ids
   */
  @Override
  @Transactional
  public void enableByIds(String[] ids) {
    Validate.notEmpty(ids,"职位主键列表不能为空");
    for(String id:ids){
      PositionEntity positionEntity = this.positionRepository.findById(id).orElse(null);
      Validate.notNull(positionEntity,"找不到主键对应的职位，请检查!");
      positionEntity.setTstatus(1);
      this.positionRepository.save(positionEntity);
    }
  }

  @Override
  @Transactional
  public void reBindOrgCode(String[] codes, String orgCode) {
    Validate.notEmpty(codes, "岗位不能为空!");
    Validate.notBlank(orgCode, "组织机构不能为空!");
    OrganizationVo organizationVo = organizationService.findByCode(orgCode);
    Validate.notNull(organizationVo, "未找到指定的组织机构!");
    OrganizationEntity entity = new OrganizationEntity();
    entity.setId(organizationVo.getId());
    for (String code : codes){
      PositionEntity position = this.positionRepository.findByCode(code);
      Validate.notNull(position, "未找到指定[%s]的岗位!" ,code);
      // 重新设定指定岗位的组织机构信息
      position.setOrganization(entity);
      positionRepository.saveAndFlush(position);
    }
  }

  /**
   * 根据职级分页查询职位，带出用户拓展以及组织信息
   * @param pageable
   * @param conditions
   * @return
   */
  @Override
  public Page<PositionSimpleVo> findByPositionLevelAndConditions(Pageable pageable, Map<String, Object> conditions) {
    if(conditions == null){
      conditions = Maps.newHashMap();
    }
    return this.positionRepository.findByPositionLevelAndConditions(pageable, conditions);
  }

  @Override
  public List<PositionVo> findByUserIdStruDescendant(String userId) {
    Set<PositionEntity> positionEntities = this.positionRepository.findDetailsByUserId(userId);
    List<PositionEntity> resultList = Lists.newArrayList();
    if(CollectionUtils.isEmpty(positionEntities)){
      return Lists.newArrayList();
    }
    positionEntities.forEach(p -> this.findStruDescendant(p, resultList));
    if(CollectionUtils.isEmpty(resultList)){
      return Lists.newArrayList();
    }
    return Lists.newArrayList(nebulaToolkitService.copyCollectionByWhiteList(resultList, PositionEntity.class, PositionVo.class, HashSet.class, ArrayList.class, "users"));
  }

  @Override
  @Transactional
  public PositionVo bindParent(String id, String parentId) {
    Validate.notBlank(id, "当前职位id不能为空");
    Validate.notBlank(parentId, "需要绑定的父级职位id不能为空");
    PositionEntity positionEntity = this.positionRepository.findById(id).orElse(null);
    Validate.notNull(positionEntity, "没有获取到当前职位信息");
    PositionEntity parent = this.positionRepository.findById(parentId).orElse(null);
    Validate.notNull(parent, "没有获取到父级职位信息");
    positionEntity.setParent(parent);
    this.positionRepository.saveAndFlush(positionEntity);
    return nebulaToolkitService.copyObjectByWhiteList(positionEntity, PositionVo.class, HashSet.class, ArrayList.class,
            "organization", "users", "parent", "positionLevel");
  }

  /**
   * 查询某职位所有下级职位
   * @param positionEntity
   * @param resultList
   */
  private void findStruDescendant(PositionEntity positionEntity, List<PositionEntity> resultList) {
    if(positionEntity == null || resultList == null){
      return;
    }
    Set<PositionEntity> children = positionEntity.getChildren();
    if(CollectionUtils.isEmpty(children)){
      return;
    }
    children.forEach(p -> {
      resultList.add(p);
      this.findStruDescendant(p, resultList);
    });
  }

  @Override
  @Transactional
  public void unbindParent(String id) {
    Validate.notBlank(id, "职位ID不能为空");
    PositionEntity position = positionRepository.findById(id).orElse(null);
    position.setParent(null);
    positionRepository.save(position);
  }

  /**
   * 处理参数
   * @param level
   * @param propertiesFilter
   * @return
   */
  private void bindPropertiesFilter(Integer level, String[] propertiesFilter, String type) {
    String temp = "";
    for(int i = 0 ; i < level; i++){
      temp = temp + type + ".";
      if((i + 1 ) == level){
        temp = temp.substring(0, temp.length() - 1);
      }
      propertiesFilter[i] = temp;
    }
  }

  /**
   * 根据职级创建相关职位
   * @param levelCode
   * @return
   */
  @Override
  @Transactional
  public PositionVo createByPositionLevelCode(String levelCode) {
    Validate.notBlank(levelCode, "职级编码不能为空");
    PositionLevelVo positionLevel = positionLevelService.findByCode(levelCode);
    Validate.notNull(positionLevel, "职级信息不存在，请检查");
    PositionVo position = new PositionVo();
    position.setCode(userModelCodeGenerateService.positionCodeGenerate());
    position.setName(this.buildName(positionLevel));
    position.setCreateTime(new Date());
    position.setPositionLevel(positionLevel);
    position.setTstatus(NormalStatusEnum.ENABLE.getStatus());
    return this.create(position);
  }

  /**
   * 根据组织编码相关信息（包括组织，职级，用户，下级职位）
   * @param orgCode
   * @return
   */
  @Override
  @NebulaServiceMethod(name = "PositionService.findDetailsByOrgCode" , desc = "根据组织编码相关信息（包括组织，职级，用户，下级职位）", scope = NebulaServiceMethod.ScopeType.READ, returnPropertiesFilter = "positionLevel,users,organization,children" )
  public List<PositionVo> findDetailsByOrgCode(@ServiceMethodParam(name = "orgCode") String orgCode) {
    if(StringUtils.isBlank(orgCode)){
      return Lists.newArrayList();
    }
    List<PositionEntity> positions = positionRepository.findDetailsByOrgCode(orgCode);
    if(CollectionUtils.isEmpty(positions)){
      return Lists.newArrayList();
    }
    Collection<PositionVo> positionVos = nebulaToolkitService.copyCollectionByWhiteList(positions, PositionEntity.class, PositionVo.class, HashSet.class, ArrayList.class, "positionLevel", "users", "organization", "children");
    if(CollectionUtils.isEmpty(positionVos)){
      return Lists.newArrayList();
    }
    //拼接用于下拉选展示的名称
    positionVos.forEach(p -> this.buildViewName(p));
    return Lists.newArrayList(positionVos);
  }

  /**
   * 拼接用于下拉选展示的组织名称/职位名称/用户名称
   * @param positionVo
   */
  private void buildViewName(PositionVo positionVo) {
    OrganizationVo organizationVo = positionVo.getOrganization();
    Set<UserVo> userVos = positionVo.getUsers();
    StringBuilder viewName = new StringBuilder();
    if(organizationVo != null){
      //1、添加组织名
      viewName.append(organizationVo.getOrgName()).append("/");
    }
    //2、添加职位名
    viewName.append(positionVo.getName());
    if(!CollectionUtils.isEmpty(userVos)) {
      viewName.append("/");
      //3、添加用户名
      Iterator<UserVo> userVoIterator = userVos.iterator();
      userVoIterator.forEachRemaining(u -> viewName.append(u.getUserName()).append(userVoIterator.hasNext() ? "、" : ""));
    }
    positionVo.setViewName(viewName.toString());
  }

  /**
   * 根据编码查询职位的所有下级信息
   * @param code
   * @return
   */
  @Override
  public List<PositionVo> findDetailsByCode(String code) {
    if(StringUtils.isBlank(code)){
      return Lists.newArrayList();
    }
    PositionEntity parent = positionRepository.findByCode(code);
    if(parent == null){
      return Lists.newArrayList();
    }
    List<PositionEntity> positions = positionRepository.findDetailsByParentId(parent.getId());
    if(CollectionUtils.isEmpty(positions)){
      return Lists.newArrayList();
    }
    Collection<PositionVo> positionVos = nebulaToolkitService.copyCollectionByWhiteList(positions, PositionEntity.class, PositionVo.class, HashSet.class, ArrayList.class, "positionLevel", "users", "organization");
    return Lists.newArrayList(positionVos);
  }

  /**
   * 根据编码查询职位详情，包含组织信息、用户信息
   * @param code
   * @return
   */
  @Override
  public PositionVo findPositionDetailByCode(String code) {
    if(StringUtils.isBlank(code)){
      return null;
    }
    PositionEntity detailsByCode = this.positionRepository.findDetailsByCode(code);
    if(detailsByCode == null){
      return null;
    }
    return this.nebulaToolkitService.copyObjectByWhiteList(detailsByCode, PositionVo.class, LinkedHashSet.class, LinkedList.class,"organization","users","positionLevel");
  }

  /**
   * 自动创建职位名称
   * @param positionLevel
   * @return
   */
  private String buildName(PositionLevelVo positionLevel) {
    Validate.notNull(positionLevel, "职级信息不能为空");
    Validate.notBlank(positionLevel.getName(), "职级名称不能为空");
    String positionName = "";
    try {
      redisMutexService.lock(RedisKeyConstants.LOCK_POSITION_PREFIX.concat(positionLevel.getId()));
      String maxName = positionRepository.findMaxName(positionLevel.getId());
      int maxNum = 1;
      int item = 0;
      if(StringUtils.isNotBlank(maxName)){
        String regEx = "[^0-9]";
        Pattern p = Pattern.compile(regEx);
        Matcher m = p.matcher(maxName);
        String trim = m.replaceAll("").trim();
        if(StringUtils.isNotBlank(trim)){
          item = Integer.parseInt(trim);
        }
      }
      maxNum = item + 1;
      positionName = positionLevel.getName() + maxNum;
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      throw new IllegalArgumentException(e);
    } finally {
      redisMutexService.unlock(RedisKeyConstants.LOCK_POSITION_PREFIX.concat(positionLevel.getId()));
    }
    return positionName;
  }

}
