package com.bizunited.platform.user.service.local.service.internal;

import com.bizunited.platform.common.enums.NormalStatusEnum;
import com.bizunited.platform.common.enums.RbacRelationEnum;
import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.core.annotations.NebulaServiceMethod;
import com.bizunited.platform.core.annotations.ServiceMethodParam;
import com.bizunited.platform.user.common.service.UserMappingService;
import com.bizunited.platform.user.common.service.userGroup.UserGroupService;
import com.bizunited.platform.user.common.vo.UserGroupVo;
import com.bizunited.platform.user.common.vo.UserMappingVo;
import com.bizunited.platform.user.service.local.entity.UserEntity;
import com.bizunited.platform.user.service.local.entity.UserGroupEntity;
import com.bizunited.platform.user.service.local.repository.UserGroupRepository;
import com.bizunited.platform.user.service.local.repository.UserRepository;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

public class UserGroupServiceImpl implements UserGroupService {
  @Autowired
  private UserGroupRepository userGroupRepository;
  @Autowired
  private UserRepository userRepository;
  @Autowired
  private UserMappingService userMappingService;
  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;
  private static final String ERROR_GROUP_ID = "分组id不能为空，请检查";
  private static final String ERROR_NOT_FOUND = "没有该工作组，请检查!!";

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.UserGroupService#create(com.bizunited.platform.rbac.server.vo.UserGroupVo)
   */
  @Transactional
  @Override
  @NebulaServiceMethod(name = "UserGroupService.create" , desc = "创建一条模型为UserGroupVo的用户组信息", scope = NebulaServiceMethod.ScopeType.WRITE, returnPropertiesFilter = "" )
  public UserGroupVo create(UserGroupVo userGroup) {
    // 验证传入分组信息的正确性
    Validate.notNull(userGroup, "分组信息不能为空，请检查");
    Validate.isTrue(StringUtils.isBlank(userGroup.getId()), "添加分组时，不能传入分组id,请重新添加!!");
    String groupName = userGroup.getGroupName();
    Validate.notBlank(groupName, "分组名称不能为空，请检查!!");
    String groupCode = userGroup.getGroupCode();
    Validate.notBlank(groupCode, "分组编号不能为空,请检查");
    Validate.notBlank(userGroup.getGroupDescription(), "分组描述不能为空，请检查!!");
    Integer tstatus = ObjectUtils.defaultIfNull(userGroup.getTstatus(), NormalStatusEnum.ENABLE.getStatus());
    userGroup.setTstatus(tstatus);
    UserGroupEntity entity = userGroupRepository.findByGroupCode(groupCode);
    Validate.isTrue(null == entity, "该分组编号已经存在，请重新输入分组编号");
    // 转换后保存
    UserGroupEntity userGroupEntity = this.nebulaToolkitService.copyObjectByWhiteList(userGroup, UserGroupEntity.class, HashSet.class, ArrayList.class);
    userGroupRepository.save(userGroupEntity);
    userGroup.setId(userGroupEntity.getId());
    return userGroup;
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.UserGroupService#update(com.bizunited.platform.rbac.server.vo.UserGroupVo)
   */
  @Transactional
  @Override
  @NebulaServiceMethod(name = "UserGroupService.update" , desc = "修改一条模型为UserGroupVo的用户组信息", scope = NebulaServiceMethod.ScopeType.WRITE, returnPropertiesFilter = "" )
  public UserGroupVo update(UserGroupVo userGroup) {
    // 验证传入的用户信息
    Validate.notNull(userGroup, "分组信息不能为空，请检查");
    String groupId = userGroup.getId();
    Validate.notNull(groupId, ERROR_GROUP_ID);
    Optional<UserGroupEntity> op = userGroupRepository.findById(groupId);
    Validate.isTrue(op.isPresent(), "该分组不存在，请重新输入分组名");
    UserGroupEntity currentGroup = op.get();
    // 只有分组名称和描述信息能够修改
    if (!StringUtils.isBlank(userGroup.getGroupName())) {
      currentGroup.setGroupName(userGroup.getGroupName());
    }
    if (!StringUtils.isBlank(userGroup.getGroupDescription())) {
      currentGroup.setGroupDescription(userGroup.getGroupDescription());
    }
    userGroupRepository.save(currentGroup);

    // 转换后输出
    return this.nebulaToolkitService.copyObjectByWhiteList(currentGroup, UserGroupVo.class, HashSet.class, ArrayList.class);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.UserGroupService#updateStatus(java.lang.String)
   */
  @Transactional
  @Override
  public UserGroupVo updateStatus(String groupId) {
    Validate.notNull(groupId, ERROR_GROUP_ID);
    Optional<UserGroupEntity> op = userGroupRepository.findById(groupId);
    Validate.isTrue(op.isPresent(), ERROR_NOT_FOUND);
    UserGroupEntity userGroup = op.get();
    Integer status = userGroup.getTstatus();
    status = (status == 1 ? 0 : 1); //0:禁用,1:启用
    userGroup.setTstatus(status);
    userGroupRepository.save(userGroup);

    // 转换后输出
    return this.nebulaToolkitService.copyObjectByWhiteList(userGroup, UserGroupVo.class, HashSet.class, ArrayList.class);
  }

  /**
   * 验证绑定用户和分组
   * @param userId
   * @param groupId
   * @param mappingVo
   */
  private void validateBindUser(UserMappingVo mappingVo, String groupId, String userId) {
    Optional<UserEntity> optional = userRepository.findById(userId);
    Validate.isTrue(optional.isPresent(), "没有该用户[%s],请检查!!", userId);
    if (mappingVo == null) {
      long count = userGroupRepository.countUserGroup(groupId, userId);
      Validate.isTrue(count == 0, "在批量绑定时,指定用户[%s]已经绑定了，不能重复绑定，请检查!!", userId);
    } else {
      int bindRelation = mappingVo.getUserGroupRelation();
      //如果用户和用户分组绑定关系为多对多,多对一，一对多，一对一分别控制判断
      if (bindRelation == RbacRelationEnum.MANY_TO_MANY.getRelation()) {
        long count = userGroupRepository.countUserGroup(groupId, userId);
        Validate.isTrue(count == 0, "在批量绑定时,用户和分组多对多，指定用户[%s]已经绑定了，不能重复绑定，请检查!!", userId);
      }
      if (bindRelation == RbacRelationEnum.MANY_TO_ONE.getRelation()) {
        long count = userGroupRepository.countGroupByUserId(userId);
        Validate.isTrue(count == 0, "在批量绑定时,用户和分组多对一，指定用户[%s]已经绑定了分组，不能继续绑定，请检查!!", userId);
      }
      if (bindRelation == RbacRelationEnum.ONE_TO_MANY.getRelation()) {
        long count = userGroupRepository.countUserByGroupId(groupId);
        Validate.isTrue(count == 0, "在批量绑定时,用户和分组一对多，指定分组[%s]已经绑定了用户，不能继续绑定，请检查!!", groupId);
      }
      if (bindRelation == RbacRelationEnum.ONE_TO_ONE.getRelation()) {
        long countGroup = userGroupRepository.countGroupByUserId(userId);
        Validate.isTrue(countGroup == 0, "在批量绑定时,用户和分组一对一，指定用户[%s]已经绑定了分组，不能继续绑定，请检查!!", userId);
        long countUser = userGroupRepository.countUserByGroupId(groupId);
        Validate.isTrue(countUser == 0, "在批量绑定时,用户和分组一对一，指定分组[%s]已经绑定了用户，不能继续绑定，请检查!!", groupId);
      }

    }
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.UserGroupService#bindUser(java.lang.String, java.lang.String[])
   */
  @Override
  @Transactional
  public void bindUser(String groupId, String[] userIds) {
    Validate.notNull(groupId, ERROR_GROUP_ID);
    Validate.isTrue(userIds != null && userIds.length > 0, "用户id不能为空（至少需要传入一个），请检查");
    Optional<UserGroupEntity> op = userGroupRepository.findById(groupId);
    Validate.isTrue(op.isPresent(), ERROR_NOT_FOUND);
    UserMappingVo mappingVo = userMappingService.findOne();
    for (String userId : userIds) {
      this.validateBindUser(mappingVo, groupId, userId);
      userGroupRepository.bindUser(groupId, userId);
    }
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.UserGroupService#unBindUser(java.lang.String, java.lang.String[])
   */
  @Override
  @Transactional
  public void unBindUser(String groupId, String[] userIds) {
    Validate.notNull(groupId, ERROR_GROUP_ID);
    Validate.isTrue(userIds != null && userIds.length > 0, "用户id不能为空（至少需要传入一个），请检查");
    Optional<UserGroupEntity> op = userGroupRepository.findById(groupId);
    Validate.isTrue(op.isPresent(), ERROR_NOT_FOUND);
    for (String userId : userIds) {
      Optional<UserEntity> optional = userRepository.findById(userId);
      Validate.isTrue(optional.isPresent(), "没有该用户,请检查!!");
      userGroupRepository.unBindUser(groupId, userId);
    }
  }

  /**
   * 用户与用户组的重新绑定
   * @param groupId
   * @param userIds
   */
  @Override
  @Transactional
  public void reBindUser(String groupId, String[] userIds) {
    Validate.notBlank(groupId, "用户与用户组的重新绑定时，用户组Id不能为空");
    if(userIds == null){
      userIds = new String[]{};
    }
  
    UserGroupEntity userGroup = userGroupRepository.findById(groupId).orElse(null);
    Validate.notNull(userGroup, ERROR_NOT_FOUND);
  
    Set<String> existUserIds = userGroupRepository.findByGroupId(groupId);
    Set<String> newUserIds;
    if (userIds.length == 0) {
      newUserIds = new HashSet<>();
    }else{
      newUserIds = new HashSet<>(Arrays.asList(userIds));
    }
    Set<String> needDelete = Sets.difference(existUserIds, newUserIds);
    Set<String> needInsert = Sets.difference(newUserIds, existUserIds);
    //解绑
    for(String userId : needDelete){
      userGroupRepository.unBindUser(groupId, userId);
    }
    //绑定
    UserMappingVo mappingVo = userMappingService.findOne();
    for(String userId : needInsert){
      this.validateBindUser(mappingVo, groupId, userId);
      userGroupRepository.bindUser(groupId, userId);
    }
  }

  @Override
  public UserGroupVo findByCode(String groupCode) {
    if (StringUtils.isBlank(groupCode)) {
      return null;
    }
    UserGroupEntity entity = userGroupRepository.findByGroupCode(groupCode);
    if (entity == null) {
      return null;
    }
    return nebulaToolkitService.copyObjectByWhiteList(entity, UserGroupVo.class, HashSet.class, ArrayList.class);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.UserGroupService#findById(java.lang.String)
   */
  @Override
  @NebulaServiceMethod(name = "UserGroupService.findDetailsById" , desc = "根据id查询用户组详情", scope = NebulaServiceMethod.ScopeType.READ, returnPropertiesFilter = "roles,users" )
  public UserGroupVo findDetailsById(@ServiceMethodParam(name = "id") String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    UserGroupEntity userGroupEntity = userGroupRepository.findDetailsById(id);
    if (userGroupEntity == null) {
      return null;
    }

    // 转换后输出
    UserGroupVo userGroupVo = this.nebulaToolkitService.copyObjectByWhiteList(userGroupEntity, UserGroupVo.class, LinkedHashSet.class, ArrayList.class, "roles", "users");
    return userGroupVo;
  }

  @Override
  public Set<UserGroupVo> findByIds(List<String> ids) {
    if (CollectionUtils.isEmpty(ids)) {
      return Sets.newHashSet();
    }
    Set<UserGroupEntity> groups = userGroupRepository.findByIds(ids);
    if (CollectionUtils.isEmpty(groups)) {
      return Sets.newHashSet();
    }
    Collection<UserGroupVo> ugroups = this.nebulaToolkitService.copyCollectionByWhiteList(groups, UserGroupEntity.class, UserGroupVo.class, HashSet.class, ArrayList.class);
    return Sets.newHashSet(ugroups);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.UserGroupService#findByUserId(java.lang.String)
   */
  @Override
  public Set<UserGroupVo> findByUserId(String userId) {
    if (StringUtils.isBlank(userId)) {
      return Sets.newHashSet();
    }
    Set<UserGroupEntity> userGroups = userGroupRepository.findByUserId(userId);
    if (CollectionUtils.isEmpty(userGroups)) {
      return Sets.newHashSet();
    }
    Collection<UserGroupVo> groups = this.nebulaToolkitService.copyCollectionByWhiteList(userGroups, UserGroupEntity.class, UserGroupVo.class, HashSet.class, ArrayList.class, "users", "roles");
    return Sets.newLinkedHashSet(groups);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.UserGroupService#findByCondition(java.lang.String, java.lang.Integer, org.springframework.data.domain.Pageable)
   */
  @Override
  @NebulaServiceMethod(name = "UserGroupService.findByCondition" , desc = "条件查询用户组列表", scope = NebulaServiceMethod.ScopeType.READ, returnPropertiesFilter = "" )
  public Page<UserGroupVo> findByCondition(@ServiceMethodParam(name = "groupName") String groupName, @ServiceMethodParam(name = "groupCode") String groupCode,
                                           @ServiceMethodParam(name = "tstatus") Integer status, @ServiceMethodParam(name = "pageable") Pageable pageable) {
    Map<String, Object> conditions = new HashMap<>();
    if (StringUtils.isNotBlank(groupName)) {
      conditions.put("groupName", groupName);
    }
    if (StringUtils.isNotBlank(groupCode)) {
      conditions.put("groupCode", groupCode);
    }
    if (status != null) {
      conditions.put("status", status);
    }
    // 如果当前没有设定分页信息，则默认第一页，每页50条数据
    if (pageable == null) {
      pageable = PageRequest.of(0, 50);
    }
    Page<UserGroupEntity> userGroupPage = userGroupRepository.queryPage(pageable, conditions);

    // 分页信息需要自行转换
    List<UserGroupEntity> userGroupEntitys = userGroupPage.getContent();
    Page<UserGroupVo> userGroupVoPage;
    if (!userGroupEntitys.isEmpty()) {
      Collection<UserGroupVo> userGroupVos = this.nebulaToolkitService.copyCollectionByWhiteList(userGroupEntitys, UserGroupEntity.class, UserGroupVo.class, LinkedHashSet.class, ArrayList.class);
      userGroupVoPage = new PageImpl<>(new ArrayList<>(userGroupVos), pageable, userGroupPage.getTotalElements());
    } else {
      userGroupVoPage = new PageImpl<>(Lists.newArrayList(), pageable, 0l);
    }
    return userGroupVoPage;
  }

  @Override
  public UserGroupVo findByGroupName(String groupName) {
    if (StringUtils.isBlank(groupName)) {
      return null;
    }
    UserGroupEntity userGroup = userGroupRepository.findByGroupName(groupName);
    if (userGroup == null) {
      return null;
    }
    return nebulaToolkitService.copyObjectByWhiteList(userGroup, UserGroupVo.class, HashSet.class, ArrayList.class);
  }
  
  @Override
  public List<UserGroupVo> findByGroupCodes(List<String> groupCodes) {
    if(CollectionUtils.isEmpty(groupCodes)){
      return Lists.newArrayList();
    }
    List<UserGroupEntity> userGroups = userGroupRepository.findByGroupCodes(groupCodes);
    if(CollectionUtils.isEmpty(userGroups)){
      return Lists.newArrayList();
    }
    Collection<UserGroupVo> collection = nebulaToolkitService.copyCollectionByWhiteList(userGroups, UserGroupEntity.class, UserGroupVo.class, HashSet.class, ArrayList.class);
    return Lists.newArrayList(collection);
  }
  
  @Override
  public List<UserGroupVo> findAll() {
    Sort sort = Sort.by(Sort.Order.desc("createTime"));
    List<UserGroupEntity> entitySet = userGroupRepository.findAll(sort);
    if (CollectionUtils.isEmpty(entitySet)) {
      return Lists.newArrayList();
    }
    Collection<UserGroupVo> groups = nebulaToolkitService.copyCollectionByWhiteList(entitySet, UserGroupEntity.class, UserGroupVo.class, HashSet.class, ArrayList.class, "users", "roles");
    return Lists.newArrayList(groups);
  }
}
