package com.bizunited.platform.user.service.local.service.internal;

import com.bizunited.platform.common.constant.Constants;
import com.bizunited.platform.common.enums.NormalStatusEnum;
import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.common.util.JsonUtils;
import com.bizunited.platform.core.annotations.NebulaServiceMethod;
import com.bizunited.platform.core.annotations.ServiceMethodParam;
import com.bizunited.platform.core.service.invoke.InvokeParams;
import com.bizunited.platform.user.common.service.organization.OrganizationService;
import com.bizunited.platform.user.common.service.position.PositionService;
import com.bizunited.platform.user.common.service.user.UserEventListener;
import com.bizunited.platform.user.common.service.user.UserService;
import com.bizunited.platform.user.common.vo.OrganizationVo;
import com.bizunited.platform.user.common.vo.PositionVo;
import com.bizunited.platform.user.common.vo.UserVo;
import com.bizunited.platform.user.service.local.entity.UserEntity;
import com.bizunited.platform.user.service.local.repository.UserRepository;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Lazy;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 基本用户信息服务的默认实现
 * @author yinwenjie
 */
public class UserServiceImpl implements UserService {
  @Autowired
  private UserRepository userRepository;
  @Autowired
  @Qualifier("passwordEncoder")
  @Lazy
  private PasswordEncoder passwordEncoder;
  @Autowired
  private PositionService positionService;
  @Autowired
  private OrganizationService organizationService;
  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;
  /**
   * 可能已注册到系统中的用户事件监听器
   */
  @Autowired(required = false)
  private List<UserEventListener> localUserEventListeners;
  
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.UserService#create(com.bizunited.platform.rbac.server.vo.UserVo)
   */
  @Override
  @Transactional
  @NebulaServiceMethod(name="UserService.create" , desc="创建一个新的UserVo模型对象（包括了可能的第三方系统调用、复杂逻辑处理等）" , returnPropertiesFilter="" , scope= NebulaServiceMethod.ScopeType.WRITE)
  public UserVo create(UserVo userVo) {
    Validate.notNull(userVo , "必须传入要添加的用户信息!");
    String userName = userVo.getUserName();
    String account = userVo.getAccount();
    Validate.notBlank(userName , "用户真实姓名必须传入!");
    Validate.notBlank(account , "用户账号信息（可能是手机号）必须传入!");
    // 验证唯一性
    UserEntity currentUser = this.userRepository.findByAccount(account);
    Validate.isTrue(currentUser == null , "该账号信息在本地系统中已经被使用，请重新填写!");
    // 检测可能填写的电话信息
    String phone = userVo.getPhone();
    if(!StringUtils.isBlank(phone)) {
      currentUser = this.userRepository.findByPhone(phone);
      Validate.isTrue(currentUser == null , "该手机信息在本例系统中已经被使用，请重新填写!");
    } else {
      userVo.setPhone(null);
    }
    userVo.setCreateTime(new Date());
    // 如果前端没有传状态则默认启用
    userVo.setUseStatus(ObjectUtils.defaultIfNull(userVo.getUseStatus(), NormalStatusEnum.ENABLE.getStatus()));
    // 验证初始填入的密码信息
    String password = userVo.getPassword();
    Validate.notBlank(password , "密码信息必须传入!");
    userVo.setPassword(passwordEncoder.encode(password));
    // 开始添加
    currentUser = this.nebulaToolkitService.copyObjectByWhiteList(userVo, UserEntity.class, HashSet.class, ArrayList.class);
    this.userRepository.saveAndFlush(currentUser);
    // 触发监听器，监听器将会帮助依赖用户模块的相关组建监听用户模块的动态
    // 在其中可以完成诸如用户绑定角色这样的扩展动作
    if(localUserEventListeners != null) {
      for (UserEventListener eventListener : localUserEventListeners) {
        eventListener.onUserCreated(userVo);
      }
    }
    
    userVo.setId(currentUser.getId());
    return userVo;
  }

  /**
   * 调用用户更新监听器
   * @param user
   */
  private void doUpdateNotify(UserVo user) {
    if(localUserEventListeners != null) {
      for (UserEventListener eventListener : localUserEventListeners) {
        eventListener.onUserUpgraded(user);
      }
    }
  }
  
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.UserService#update(com.bizunited.platform.rbac.server.vo.UserVo)
   */
  @Override
  @Transactional
  @NebulaServiceMethod(name="UserService.update" , desc="修改一个新的UserVo模型对象（包括了可能的第三方系统调用、复杂逻辑处理等）" , returnPropertiesFilter="" , scope= NebulaServiceMethod.ScopeType.WRITE)
  public UserVo update(UserVo user) {
    Validate.notNull(user , "必须传入要修改的用户信息!");
    String userId = user.getId();
    String userName = user.getUserName();
    String account = user.getAccount();
    Validate.notBlank(userId , "修改时，用户数据的编号信息必须传入!");
    Validate.notBlank(userName , "用户真实姓名必须传入!");
    Validate.notBlank(account , "用户账号信息（一般是手机号）必须传入!");
    // 验证新的账户的唯一性
    UserEntity currentUser = this.userRepository.findByAccount(account);
    Validate.isTrue(currentUser == null || StringUtils.equals(currentUser.getId(), userId), "该账号信息已经被使用，请重新填写!");
    // 检测可能填写的电话信息
    String phone = user.getPhone();
    if(!StringUtils.isBlank(phone)) {
      currentUser = this.userRepository.findByPhone(phone);
      Validate.isTrue(currentUser == null || StringUtils.equals(currentUser.getId(), userId), "指定的电话信息已被使用，请检查!!");
    }

    // 开始更新(这个方法只能更新这些信息，其他信息即使传入也不更新)
    Optional<UserEntity> op = this.userRepository.findById(userId);
    UserEntity oldUser = op.orElse(null);
    Validate.notNull(oldUser , "未找到原始的用户信息，请检查!!");
    oldUser.setEntryTime(user.getEntryTime());
    oldUser.setPhone(phone);
    oldUser.setUserHead(user.getUserHead());
    oldUser.setUserName(userName);
    oldUser.setGender(user.getGender());
    if(user.getUseStatus() != null) {
      oldUser.setUseStatus(user.getUseStatus());
    }
    //修改扩展字段
    oldUser.setExtend1(user.getExtend1());
    oldUser.setExtend2(user.getExtend2());
    oldUser.setExtend3(user.getExtend3());
    oldUser.setExtend4(user.getExtend4());
    oldUser.setExtend5(user.getExtend5());
    oldUser.setExtend6(user.getExtend6());
    oldUser.setExtend7(user.getExtend7());
    oldUser.setExtend8(user.getExtend8());
    oldUser.setExtend9(user.getExtend9());
    oldUser.setExtend10(user.getExtend10());
    this.userRepository.save(oldUser);
    this.doUpdateNotify(user);
    return user;
  }
  
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.UserService#updatePassword(java.lang.String, java.lang.String)
   */
  @Override
  @Transactional
  @NebulaServiceMethod(name="UserService.updatePassword" , desc="修改密码" , returnPropertiesFilter="" , scope= NebulaServiceMethod.ScopeType.WRITE)
  public UserVo updatePassword(@ServiceMethodParam(name = "userId") String userId, @ServiceMethodParam(name = "newPassword") String newPassword) {
    Validate.notBlank(userId, "修改密码时 ,用户编号不能为空!!");
    Validate.notBlank(newPassword, "修改密码时 ,新的密码不能为空!!");
    Optional<UserEntity> op = this.userRepository.findById(userId);
    UserEntity user = op.orElse(null);
    Validate.notNull(user , "指定的用户没有被找到，请检查!!");
    return this.updatePassword(user, newPassword);
  }
  
  /**
   * 私有方法无需进行边界校验
   * @param user
   * @param newPassword
   * @return
   */
  private UserVo updatePassword(UserEntity user , String newPassword) {
    user.setPassword(passwordEncoder.encode(newPassword));
    userRepository.saveAndFlush(user);
    UserVo currentUserVo = this.nebulaToolkitService.copyObjectByWhiteList(user, UserVo.class, HashSet.class, ArrayList.class);
    this.doUpdateNotify(currentUserVo);
    return currentUserVo;
  }
  
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.UserService#updatePassword(java.lang.String, java.lang.String, java.lang.String)
   */
  @Override
  @Transactional
  public UserVo updatePassword(String userId, String newPassword, String oldPassword) {
    Validate.notBlank(userId, "修改密码时 ,用户编号不能为空!!");
    Validate.notBlank(newPassword, "修改密码时 ,新的密码不能为空!!");
    Validate.notBlank(oldPassword, "修改密码时 ,原有的密码不能为空!!");

    // 对原来的密码进行加密后判断
    Optional<UserEntity> op = this.userRepository.findById(userId);
    UserEntity user = op.orElse(null);
    Validate.notNull(user , "指定的用户没有被找到，请检查!!");
    Validate.isTrue(passwordEncoder.matches(oldPassword, user.getPassword()), "原始密码输入错误，请检查!!");
    return this.updatePassword(user, newPassword);
  }
  
  @Override
  @Transactional
  public void updateLastloginTime(String account) {
    Validate.notBlank(account, "修改最后登录时间时 ,用户账号不能为空!!");
    Date lastloginTime = new Date();
    UserVo cuser = this.findByAccount(account);
    Validate.notNull(cuser, "未在本地系统中发现指定的用户信息，请检查!!");

    this.userRepository.updateLastloginTime(account, lastloginTime);
    this.userRepository.flush();
    this.doUpdateNotify(cuser);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.UserService#enabled(java.lang.String)
   */
  @Override
  @Transactional
  public UserVo enabled(String account) {
    Validate.notBlank(account, "用户账号必须传入!!");
    UserVo user = this.findByAccount(account);
    Validate.notNull(user , "未找到需要修改的用户信息，请检查!!");
    user.setUseStatus(1);
    
    // 转换成Entity
    UserEntity ruser = this.nebulaToolkitService.copyObjectByWhiteList(user, UserEntity.class, HashSet.class, ArrayList.class, new String[] {});
    userRepository.saveAndFlush(ruser);
    this.doUpdateNotify(user);
    return user;
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.UserService#disable(java.lang.String)
   */
  @Override
  @Transactional
  public UserVo disable(String account) {
    Validate.notBlank(account, "用户账号必须传入!!");
    UserVo user = this.findByAccount(account);
    Validate.notNull(user , "未找到需要修改的用户信息，请检查!!");
    user.setUseStatus(0);
    
    // 转换成Entity
    UserEntity ruser = this.nebulaToolkitService.copyObjectByWhiteList(user, UserEntity.class, HashSet.class, ArrayList.class, new String[] {});
    userRepository.saveAndFlush(ruser);
    this.doUpdateNotify(user);
    return user;
  }
  
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.UserService#findById(java.lang.String)
   */
  @Override
  @NebulaServiceMethod(name="UserService.findDetailsById" , desc="根据id查询明细记录" , returnPropertiesFilter="positions,groups,orgs" , scope= NebulaServiceMethod.ScopeType.READ)
  public UserVo findDetailsById(@ServiceMethodParam(name = "id") String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    // 用户详细信息，但是不包括用户角色，因为用户角色需要单独查
    UserEntity currentUser = userRepository.findDetailsById(id);
    if(currentUser == null) {
      return null;
    }
    UserVo userVo = this.nebulaToolkitService.copyObjectByWhiteList(currentUser, UserVo.class, HashSet.class, ArrayList.class, "positions","orgs","groups");
    PositionVo mainPosition = positionService.findMainPositionByUserId(id);
    if(mainPosition != null) {
      userVo.setMainPosition(mainPosition.getId());
    }
    OrganizationVo mainOrg = organizationService.findMainOrgByUserId(id);
    if(mainOrg != null) {
      userVo.setMainOrg(mainOrg.getId());
    }
    return userVo;
  }
  
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.UserService#findByAccount(java.lang.String)
   */
  @Override
  public UserVo findByAccount(String account) {
    if (StringUtils.isBlank(account)) {
      return null;
    }
    // 用户详细信息，但是不包括用户角色，等其它关联信息
    UserEntity currentUser = this.userRepository.findDetailsByAccount(account);
    if(currentUser == null) {
      return null;
    }
    PositionVo mainPosition = positionService.findMainPositionByUserId(currentUser.getId());
    UserVo userVo = this.nebulaToolkitService.copyObjectByWhiteList(currentUser, UserVo.class, HashSet.class, ArrayList.class, "positions", "orgs","positions.organization");
    if(mainPosition != null) {
      userVo.setMainPosition(mainPosition.getId());
    }
    OrganizationVo mainOrg = organizationService.findMainOrgByUserId(currentUser.getId());
    if(mainOrg != null) {
      userVo.setMainOrg(mainOrg.getId());
    }

    if(mainPosition != null && mainOrg == null){
      PositionVo positionVo = positionService.findDetailsById(mainPosition.getId());
      if(positionVo != null && positionVo.getOrganization() != null) {
        userVo.setMainOrg(positionVo.getOrganization().getId());
      }
    }
    return userVo;
  }

  @Override
  public List<UserVo> findByAccountLikeOrNameLike(String param) {
    List<UserEntity> accountLike = this.userRepository.findByAccountLike(param);
    List<UserEntity> nameLike = this.userRepository.findByUserNameLike(param);
    List<UserEntity> total = new ArrayList<>();
    total.addAll(accountLike);
    total.addAll(nameLike);
    return total.stream().distinct().map(o -> this.nebulaToolkitService.copyObjectByWhiteList(o, UserVo.class, HashSet.class, ArrayList.class)).collect(Collectors.toList());
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.UserService#findByUserId(java.lang.String)
   */
  @Override
  public UserVo findByUserId(String userId) {
    Validate.notBlank(userId,"传入的用户ID不能为空！");
    UserEntity user= userRepository.findDetailsById(userId);
    if(user==null){
      return null;
    }
    return this.nebulaToolkitService.copyObjectByWhiteList(user, UserVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  public Set<UserVo> findByIds(List<String> ids) {
    if(CollectionUtils.isEmpty(ids)){
      return Sets.newHashSet();
    }
    Set<UserEntity> users = userRepository.findByIds(ids);
    if(CollectionUtils.isEmpty(users)){
      return Sets.newHashSet();
    }
    return (Set<UserVo>)this.nebulaToolkitService.copyCollectionByWhiteList(users,UserEntity.class,UserVo.class,HashSet.class,ArrayList.class);
  }

  @Override
  public Set<UserVo> findByOrgId(String orgId) {
    if(StringUtils.isBlank(orgId)) {
      return Sets.newHashSet();
    }
    Set<UserEntity> users = userRepository.findByOrgId(orgId);
    if(CollectionUtils.isEmpty(users)) {
      return Sets.newHashSet();
    }
    Collection<UserVo> collection = nebulaToolkitService.copyCollectionByWhiteList(users, UserEntity.class, UserVo.class, HashSet.class, ArrayList.class);
    return Sets.newLinkedHashSet(collection);
  }

  @Override
  public UserVo findByPhone(String phone) {
    if(StringUtils.isBlank(phone)) {
      return null;
    }

    UserEntity user= this.userRepository.findByPhone(phone);
    if(user==null){
      return null;
    }
    return this.nebulaToolkitService.copyObjectByWhiteList(user, UserVo.class, HashSet.class, ArrayList.class);
  }

  /**
   * 根据岗位id查询绑定的用户
   * @param positionId
   * @return
   */
  @Override
  public Set<UserVo> findByPositionId(String positionId) {
    if(StringUtils.isBlank(positionId)) {
      return null;
    }
    Set<UserEntity> users = this.userRepository.findByPositionId(positionId);
    if(CollectionUtils.isEmpty(users)) {
      return Sets.newHashSet();
    }
    Collection<UserVo> collection = nebulaToolkitService.copyCollectionByWhiteList(users, UserEntity.class, UserVo.class, HashSet.class, ArrayList.class);
    return Sets.newHashSet(collection);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.UserService#findByConditions(java.lang.String, java.lang.String, java.lang.Integer, org.springframework.data.domain.Pageable)
   */
  @Override
  public Page<UserVo> findByConditions(UserVo user, Pageable pageable) {
    // Page需要自己转
    Page<UserEntity> userPage = this.userRepository.findByConditions(pageable, user);
    List<UserEntity> users = userPage.getContent();
    Page<UserVo> page;
    if (!users.isEmpty()) {
      Collection<UserVo> userVos = this.nebulaToolkitService.copyCollectionByWhiteList(users, UserEntity.class, UserVo.class, LinkedHashSet.class, ArrayList.class);
      page = new PageImpl<>(new ArrayList<>(userVos), pageable, userPage.getTotalElements());
    } else {
      page = new PageImpl<>(Lists.newArrayList(), pageable, 0l);
    }
    return page;
  }

  @Override
  @NebulaServiceMethod(name = "UserService.findByConditions", desc = "条件查询用户列表", returnPropertiesFilter = "positions,groups,orgs,roles", scope = NebulaServiceMethod.ScopeType.READ)
  public Page<UserVo> findByConditions(@ServiceMethodParam(name = "pageable") Pageable pageable, InvokeParams invokeParams) {
    pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    invokeParams = ObjectUtils.defaultIfNull(invokeParams, new InvokeParams());
    Map<String, Object> params = invokeParams.getInvokeParams();
    UserVo user = JsonUtils.convert(params, UserVo.class);
    return this.findByConditions(user, pageable);
  }

  @Override
  public Boolean isDefaultPassword(String account) {
    Validate.notBlank(account,"判断用户密码是否默认密码时，账号信息必须传入");
    UserVo user = this.findByAccount(account);
    Validate.notNull(user,"根据用户账号【%s】，未能查询到用户信息",account);
    return passwordEncoder.matches(Constants.DEFAULT_PASSWORD,user.getPassword());
  }

  @Override
  public List<UserVo> findByAccountAndTypeStruDescendant(String account, Integer type) {
    if(StringUtils.isBlank(account)){
      return Lists.newArrayList();
    }
    //type为1时通过职位的方式查询下属，为2时通过组织的方式查询下属，为0时通过职位、组织的方式查询
    if(Objects.equals(type, new Integer(1)) || Objects.equals(type, new Integer(0))){
      //1、通过职位的方式查询所有下属
      return this.findByUserAccountAndPositionStrDescendant(account);
    }
    //2、暂未实现组织的方式查询下属
    return Lists.newArrayList();
  }

  /**
   * 通过职位的方式，查询某用户的所有下属
   * 当查询方式是职位时，调用此方法
   * @param account
   */
  private List<UserVo> findByUserAccountAndPositionStrDescendant(String account) {
    if(StringUtils.isBlank(account)){
      return Lists.newArrayList();
    }
    UserEntity userEntity = this.userRepository.findByAccount(account);
    if(userEntity == null){
      return Lists.newArrayList();
    }
    //获取该用户的所有下属职位
    List<PositionVo> positionVos = this.positionService.findByUserIdStruDescendant(userEntity.getId());
    if(CollectionUtils.isEmpty(positionVos)){
      return Lists.newArrayList();
    }
    Set<String> positionIds = Sets.newHashSet();
    positionVos.forEach(p -> positionIds.add(p.getId()));
    List<UserEntity> userEntities = this.userRepository.findByPositionIds(positionIds);
    if(CollectionUtils.isEmpty(userEntities)){
      return Lists.newArrayList();
    }
    return Lists.newArrayList(nebulaToolkitService.copyCollectionByWhiteList(userEntities, UserEntity.class, UserVo.class, HashSet.class, ArrayList.class));
  }
}
