package com.bizunited.platform.venus.web.controller;

import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.venus.common.service.document.DocumentService;
import com.bizunited.platform.venus.common.vo.DocumentVo;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.security.Principal;
import java.util.List;
import java.util.Set;
import java.util.UUID;

/**
 * 文档服务控制层
 * @author: pangdajin
 * @date: 2020-04-07 16:59
 **/
@RestController
@RequestMapping(value = "/v1/venus/documents")
public class DocumentController extends BaseController {

  private static final Logger LOGGER = LoggerFactory.getLogger(DocumentController.class);

  @Autowired
  private DocumentService documentService;

  /**
   * 新建文档
   * @param documentVo
   * @return
   */
  @ApiOperation("新建文档")
  @PostMapping("")
  public ResponseModel create(@RequestBody
                            @ApiParam(name = "documentVo", value = "新建文档时必须至少包括文档标题、文档排序等信息，如果有文档父节点id则必须传入") DocumentVo documentVo){
    try {
      Principal principal = this.getPrincipal();
      DocumentVo result = documentService.create(documentVo, principal);
      return this.buildHttpResultW(result);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 修改文档
   * @param documentVo
   * @return
   */
  @ApiOperation("修改文档")
  @PatchMapping("")
  public ResponseModel update(@RequestBody
         @ApiParam(name = "documentVo", value = "保存时必须至少包括当前文档id、文档标题、文档排序等信息，如果有文档父节点id则必须传入") DocumentVo documentVo){
    try {
      Principal principal = this.getPrincipal();
      DocumentVo result = documentService.update(documentVo, principal);
      return this.buildHttpResultW(result);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 获取树
   * @return
   */
  @ApiOperation("获取树")
  @GetMapping("findTree")
  public ResponseModel findTree(){
    try {
      Set<DocumentVo> tree = documentService.findTree();
      return this.buildHttpResult(tree);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
  /**
   * 删除文档，该文档下若有子文档则一并删除
   * @param id
   * @return
   */
  @ApiOperation("删除文档，该文档下若有子文档则一并删除")
  @DeleteMapping("/deleteById")
  public ResponseModel deleteById(@RequestParam("id")
                                    @ApiParam(value = "文档主键id", required = true) String id){
    try {
      documentService.deleteById(id);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 读取内容
   * @param id
   * @return
   */
  @ApiOperation("读取内容")
  @GetMapping("/findDetailsById")
  public ResponseModel findDetailsById(@RequestParam("id")
                                           @ApiParam(value = "主键", required = true) String id){
    try {
      DocumentVo content = documentService.findDetailsById(id);
      return this.buildHttpResultW(content);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据关键字查询文档，支持模糊
   * @return
   */
  @GetMapping("findByKeyword")
  @ApiOperation(value = "根据关键字查询文档")
  public ResponseModel findByKeyword(@RequestParam(required = false) @ApiParam("搜索关键字") String keyword) {
    try {
      List<DocumentVo> documents = documentService.findByKeyword(keyword);
      return buildHttpResult(documents);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 导出文档数据
   */
  @GetMapping("export")
  @ApiOperation("导出文档数据")
  public void export(HttpServletRequest request, HttpServletResponse response) {
    try {
      byte[] result = this.documentService.export();
      if(result == null || result.length == 0) {
        throw new IllegalArgumentException("没有任何下载信息，很大几率原因是导出逻辑出现错误!!");
      }
      // 随机生成下载内容的文件名（zip文件）
      String fileName = StringUtils.join(UUID.randomUUID().toString() , ".zip");
      this.writeResponseFile(request, response, result, fileName);
    } catch (RuntimeException e) {
      super.printResponseModel(super.buildHttpResultForException(e), response);
    }
  }

  /**
   * 导入文档数据
   */
  @PostMapping("importDoc")
  @ApiOperation("导入文档数据")
  public ResponseModel importDoc(@RequestParam @ApiParam("压缩文件") MultipartFile file) {
    try {
      documentService.importDoc(file);
      return buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

}
