package com.bizunited.nebula.europa.database.register.local.service;

import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import com.bizunited.nebula.europa.database.register.sdk.constant.Constant;
import com.bizunited.nebula.europa.database.register.sdk.service.DataviewRegister;
import com.bizunited.nebula.europa.database.sdk.constant.DataSourceConstant;
import com.bizunited.nebula.europa.database.sdk.strategy.AbstractDatabaseExecutionStrategy;
import com.bizunited.nebula.europa.database.sdk.vo.DatabaseViewVo;

import com.bizunited.nebula.europa.sdk.context.execute.RequestParameter;
import com.bizunited.nebula.europa.sdk.context.matedata.MetaData;
import com.bizunited.nebula.europa.sdk.service.strategy.ExecutionStrategy;
import com.bizunited.nebula.europa.sdk.service.strategy.PreExecutionStrategy;
import com.bizunited.nebula.europa.sdk.vo.AbstractView;
import com.bizunited.nebula.europa.sdk.vo.EuropaInfoVo;

/**
 * 针对指定的数据源和SQL，进行数据视图的执行(注册形式的数据源)</br>
 * 注意数据视图的执行策略优先于数据视图的拦截器功能执行，所以这里只需要处理最原始的SQL查询信息即可
 * @author yinwenjie
 */
@Component("DatabaseRegisterExecutionStrategy")
public class DatabaseRegisterExecutionStrategy extends AbstractDatabaseExecutionStrategy implements ExecutionStrategy {
  @Autowired
  @Qualifier("databasePreRegisterExecutionStrategy")
  private PreExecutionStrategy databasePreExecutionStrategy;
  @Autowired(required = false)
  private List<DataviewRegister> dataviewRegisters;
  
  @Override
  public boolean isDefault() {
    // 这个注册性质的执行方式就是默认的执行方式
    return true;
  }

  @Override
  public boolean validate(EuropaInfoVo europaInfoVo) {
    // 首先，只要是sourceType为database的数据视图，都可以由本数据视图执行策略进行执行
    AbstractView abstractView = europaInfoVo.getView();
    if(abstractView == null || !(abstractView instanceof DatabaseViewVo)) {
      return false;
    }
    DatabaseViewVo databaseViewVo = (DatabaseViewVo)abstractView;
    return StringUtils.equals(databaseViewVo.getSouceType(), Constant.DATA_SOURCE_TYPE);
  }

  @Override
  public MetaData buildQueryMetaData(EuropaInfoVo europaInfoVo, RequestParameter requestParameter) {
    /*
     * 这里构造元数据的方法，与DatabasePreExecutionStrategy中构建元数据结构的方法工作过程一致
     * */
    DatabaseViewVo databaseView = (DatabaseViewVo)europaInfoVo.getView();
    Validate.notNull(databaseView , "执行数据视图时，未发现指定的数据视图【数据库执行】扩展信息，请检查!!");
    // 正式开始构建元数据
    String dataSourceCode = databaseView.getDatasouceCode();
    String registerCode = databaseView.getRegisterCode();
    if(StringUtils.isBlank(dataSourceCode)) {
      dataSourceCode = DataSourceConstant.DEFAULT_DATASOURCE_CODE;
    }
    String contentJson = "{\"dataSourceCode\":\"" + dataSourceCode + "\" , \"dataviewCode\":\"" + registerCode + "\"}";
    MetaData queryMetaData = this.databasePreExecutionStrategy.preExecution(Constant.DATA_SOURCE_TYPE, contentJson);
    Validate.notNull(queryMetaData , "执行数据视图时，元数据结构未构建成功，请检查!!");
    return queryMetaData;
  }

  @Override
  public String sourceSql(EuropaInfoVo europaInfoVo) {
    DatabaseViewVo databaseView = (DatabaseViewVo)europaInfoVo.getView();
    String registerCode = databaseView.getRegisterCode();
    Validate.isTrue(!CollectionUtils.isEmpty(dataviewRegisters) , "执行数据视图时，未发现任何注册的数据视图模板DataViewRegister，请检查!!");
    for (DataviewRegister dataviewRegister : dataviewRegisters) {
      if(StringUtils.equals(dataviewRegister.code(), registerCode)) {
        return dataviewRegister.buildSql();
      }
    }
    throw new IllegalArgumentException("执行数据视图时，未发现任何注册的数据视图模板DataViewRegister，请检查!!");
  }
}
