package com.bizunited.platform.core.controller;

import com.alibaba.fastjson.JSONArray;
import com.bizunited.platform.core.controller.model.ResponseModel;
import com.bizunited.platform.rbac.server.service.OrganizationService;
import com.bizunited.platform.rbac.server.vo.OrganizationVo;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;
import java.util.Set;

@RestController
@RequestMapping(value = "/v1/nebula/orgs")
public class OrganizationController extends BaseController {

  @Autowired
  private OrganizationService organizationService;
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(OrganizationController.class);

  @ApiOperation(value = "添加组织机构")
  @RequestMapping(value = "", method = RequestMethod.POST)
  public ResponseModel create(@ApiParam(name = "组织机构相关信息") @RequestBody OrganizationVo organization) {
    try {
      OrganizationVo organizationVo = this.organizationService.create(organization);
      return this.buildHttpResultW(organizationVo);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "修改组织机构")
  @RequestMapping(value = "/update", method = RequestMethod.PATCH)
  public ResponseModel update(@ApiParam(name = "组织机构相关信息") @RequestBody OrganizationVo organizationEntity) {
    try {
      OrganizationVo organizationVo = this.organizationService.update(organizationEntity);
      return this.buildHttpResultW(organizationVo);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "修改组织机构状态---(启用、禁用)")
  @RequestMapping(value = "/updateStatus/{orgId}", method = RequestMethod.PATCH)
  public ResponseModel updateStatus(@PathVariable("orgId") String orgId) {
    LOGGER.debug("组织机构id:{}", orgId);
    try {
      OrganizationVo organization = this.organizationService.updateStatus(orgId);
      return this.buildHttpResultW(organization);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "对一个指定的currentOrgId进行其父级组织机构parentOrgId的绑定操作（原有currentOrgId的父级信息将会被取消）")
  @RequestMapping(value = "/bindParent", method = RequestMethod.PATCH)
  public ResponseModel bindParent(@RequestParam(name = "currentOrgId") @ApiParam(name = "currentOrgId", value = "当前的组织结构编号") String currentOrgId, @RequestParam(name = "parentOrgId") @ApiParam(name = "parentOrgId", value = "当前需要绑定的父级组织机构编号") String parentOrgId) {
    try {
      this.organizationService.bindParent(currentOrgId, parentOrgId);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "对一个指定的currentOrgId取消其当前父级关系的绑定信息")
  @RequestMapping(value = "/unbindParent", method = RequestMethod.PATCH)
  public ResponseModel unbindParent(@RequestParam(name = "currentOrgId") @ApiParam(name = "currentOrgId", value = "当前的组织结构编号") String currentOrgId) {
    try {
      this.organizationService.unbindParent(currentOrgId);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "将指定组织机构（currentOrgId）和指定的一个/多个用户数据编号（userId）形成绑定关系——一个用户只能属于一个组织机构", notes = "注意，目前一个人员可以和多个组织机构进行绑定，但是呢一个人员最多只能和一个组织机构绑定一次")
  @RequestMapping(value = "/bindUsers", method = RequestMethod.PATCH)
  public ResponseModel bindUsers(@RequestParam(name = "currentOrgId") @ApiParam(name = "currentOrgId", value = "当前的组织结构编号") String currentOrgId, @RequestParam(name = "userIds") @ApiParam(name = "userIds", value = "指定的一个/多个用户数据编号（userId）") String[] userIds) {
    try {
      this.organizationService.bindUsers(currentOrgId, userIds);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "解除指定用户和指定组织机构的绑定关系")
  @RequestMapping(value = "/unbindUsers", method = RequestMethod.PATCH)
  public ResponseModel unbindUsers(@RequestParam(name = "currentOrgId") @ApiParam(name = "currentOrgId", value = "指定的组织机构数据库编号") String currentOrgId, @RequestParam(name = "userId") @ApiParam(name = "userId", value = "当前需要操作的人员id信息") String[] userId) {
    try {
      this.organizationService.unbindUser(currentOrgId, userId);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "将指定组织机构（currentOrgId）和指定的一个/多个角色形成绑定关系——一个角色只能属于一个组织机构")
  @RequestMapping(value = "/bindRoles", method = RequestMethod.PATCH)
  public ResponseModel bindRoles(@RequestParam(name = "currentOrgId") @ApiParam(name = "currentOrgId", value = "当前的组织结构编号") String currentOrgId, @RequestParam(name = "roleIds") @ApiParam(name = "roleIds", value = "指定的一个/多个用户数据编号（roleId）") String[] roleIds) {
    try {
      this.organizationService.bindRoles(currentOrgId, roleIds);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "解除指定角色和任何组织机构的绑定关系")
  @RequestMapping(value = "/unbindRoles", method = RequestMethod.PATCH)
  public ResponseModel unBindRoles(@RequestParam(name = "roleId") @ApiParam(name = "roleId", value = "当前需要解绑的角色id") String[] roleId) {
    try {
      this.organizationService.unbindRole(roleId);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "查询目前没有指定父级组织机构的所有组织机构信息，无论其状态是否正常（status无论什么值）", notes = "注意：(查询的信息不包括任何关联信息)")
  @RequestMapping(value = "/findByNullParent", method = RequestMethod.GET)
  public ResponseModel findByNullParent() {
    try {
      List<OrganizationVo> resutls = this.organizationService.findByNullParent();
      return this.buildHttpResultW(resutls);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "查询指定父级组织机构下的所有子级组织机构信息，无论其状态是否正常（status无论什么值）", notes = "注意：(查询的信息不包括任何关联信息)")
  @RequestMapping(value = "/findByParent", method = RequestMethod.GET)
  public ResponseModel findByParent(@RequestParam(name = "parentId") @ApiParam(name = "parentId", value = "父级组织机构id") String parentId) {
    try {
      List<OrganizationVo> resutls = this.organizationService.findByParent(parentId);
      return this.buildHttpResultW(resutls);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "查询目前没有指定父级组织机构的所有组织机构信息，并且依据其状态进行查询", notes = "注意：(查询的信息不包括任何关联信息)")
  @RequestMapping(value = "/findByNullParentAndStatus", method = RequestMethod.GET)
  public ResponseModel findByNullParentAndStatus(@RequestParam(name = "status") @ApiParam(name = "status", value = "状态信息（1:正常；0禁用/异常）") Integer status) {
    try {
      List<OrganizationVo> resutls = this.organizationService.findByNullParentAndStatus(status);
      return this.buildHttpResultW(resutls);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "查询指定父级组织机构下的所有子级组织机构信息，并且依据其状态进行查询", notes = "注意：(查询的信息不包括任何关联信息)")
  @RequestMapping(value = "/findByParentAndStatus", method = RequestMethod.GET)
  public ResponseModel findByParentAndStatus(@RequestParam(name = "parentId") @ApiParam(name = "parentId", value = "父级组织机构id") String parentId, @RequestParam(name = "status") @ApiParam(name = "status", value = "状态信息（1:正常；0禁用/异常）") Integer status) {
    try {
      List<OrganizationVo> resutls = this.organizationService.findByParentAndStatus(parentId, status);
      return this.buildHttpResultW(resutls);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "根据组织机构名称，查询这个组织机构以及这个组织（这些组织机构）对应的所有父级、子级结构信息")
  @RequestMapping(value = "/findByOrgName", method = RequestMethod.GET)
  public ResponseModel findByOrgName(@RequestParam(name = "orgName") @ApiParam(name = "orgName", value = "组织机构名称信息") String orgName) {
    try {
      JSONArray resutls = this.organizationService.findByOrgName(orgName);
      return this.buildHttpResultW(resutls);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "查询指定组织下的详细信息，无论其状态是否正常（status无论什么值）", notes = "注意：包括关联的用户信息")
  @RequestMapping(value = "/findDetailsById", method = RequestMethod.GET)
  public ResponseModel findDetailsById(@RequestParam(name = "id") @ApiParam(name = "id", value = "组织机构id") String id) {
    try {
      OrganizationVo organization = this.organizationService.findDetailsById(id);
      return this.buildHttpResultW(organization, "parent","child","users","roles");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "根据指定组织机构，查询这个组织机构以及这个组织（这些组织机构）对应的所有子级结构信息")
  @RequestMapping(value = "/findUnbindOrgById", method = RequestMethod.GET)
  public ResponseModel findUnbindOrgById(@RequestParam(name = "currentOrgId") @ApiParam(name = "currentOrgId", value = "当前组织机构") String currentOrgId) {
    try {
      Object[] resutls = this.organizationService.findUnbindOrgById(currentOrgId);
      return this.buildHttpResultW(resutls);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "查询所有的组织机构信息，无论其状态是否正常（status无论什么值）", notes = "注意：(查询的信息不包括任何关联信息)")
  @RequestMapping(value = "/findAll", method = RequestMethod.GET)
  public ResponseModel findAll() {
    try {
      List<OrganizationVo> resutls = this.organizationService.findAll();
      return this.buildHttpResultW(resutls, "parent");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "预制控件查询，查询用户组织机构相关信息")
  @RequestMapping(value = "/findOrgByUserId",method = RequestMethod.GET)
  public ResponseModel findOrgByUserId(@ApiParam(name = "userId", value = "用户id信息") @RequestParam("userId")String userId){
    try {
      Set<OrganizationVo> org = this.organizationService.findOrgByUserId(userId);
      return this.buildHttpResultW(org);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "查询用户组织机构相关信息")
  @RequestMapping(value = "/findByIds",method = RequestMethod.POST)
  public ResponseModel findByIds(@RequestBody List<String> ids){
    try {
      Set<OrganizationVo> org = this.organizationService.findByIds(ids);
      return this.buildHttpResultW(org);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 组织机构-树结构查询-根据指定组织机构ID，查询这个组织机构以及这个组织机构对应的所有子级结构信息
   * @param id 当前组织机构Id
   * @return
   */
  @ApiOperation("组织机构-树结构查询-根据指定组织机构ID，" +
          "查询这个组织机构以及这个组织机构对应的所有子级结构信息")
  @GetMapping("/findById")
  public ResponseModel findById(@RequestParam(value = "id") @ApiParam(name = "id", value = "当前组织机构Id") String id){
    try {
      OrganizationVo result = this.organizationService.findById(id);
      return this.buildHttpResult(result);
    }catch (RuntimeException e){
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 组织机构-查询所有根节点组织机构
   * @return
   */
  @ApiOperation("组织机构-查询所有根节点组织机构")
  @GetMapping("/findRootOrg")
  public ResponseModel findRootOrg(){
    try {
      List<OrganizationVo> organization = this.organizationService.findRootOrg();
      return this.buildHttpResultW(organization);
    }catch (RuntimeException e){
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 组织机构-根据组织机构名称实现模糊查询
   * @param orgName 组织机构名称
   * @return
   */
  @ApiOperation("组织机构-根据组织机构名称实现模糊查询")
  @GetMapping("/findByOrgNameLike")
  public ResponseModel findByOrgNameLike(@RequestParam(name = "orgName") @ApiParam(name = "orgName", value = "组织机构名称") String orgName){
    try {
      Set<OrganizationVo> organization = this.organizationService.findByOrgNameLike(orgName);
      return this.buildHttpResultW(organization);
    }catch (RuntimeException e){
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据组织编码查询
   * @param code
   * @return
   */
  @GetMapping("findByCode")
  @ApiOperation("根据组织编码查询")
  public ResponseModel findByCode(@RequestParam @ApiParam("组织编码") String code) {
    try {
      OrganizationVo org = organizationService.findByCode(code);
      return buildHttpResultW(org);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }
}
