package com.bizunited.platform.core.controller;

import com.bizunited.platform.core.controller.model.ResponseModel;
import com.bizunited.platform.rbac.server.service.PositionService;
import com.bizunited.platform.rbac.server.vo.PositionVo;
import com.bizunited.platform.rbac.server.vo.UserVo;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.Arrays;
import java.util.List;
import java.util.Set;

/**
 * PositionController
 *
 * @description:
 * @author: yanwe
 * @date: 18/Jan/2019 10:00
 */
@RestController
@RequestMapping(value = "/v1/nebula/position")
public class PositionController extends BaseController {
  private static final Logger LOGGER = LoggerFactory.getLogger(PositionController.class);
  @Autowired 
  private PositionService positionService;
  
  @ApiOperation(value = "按照条件搜索用户，不支持模糊查询")
  @RequestMapping(value = "/findByConditions", method = RequestMethod.GET)
  public ResponseModel findByConditions(
      @ApiParam(name = "name", value = "名称") String name,
      @ApiParam(name = "code", value = "编码") String code,
      @ApiParam(name = "tstatus", value = "状态") Integer tstatus,
      @PageableDefault(value = 50) Pageable pageable) {
    try {
      Page<PositionVo> page = positionService.findByConditions(code, name, tstatus, pageable);
      return this.buildHttpResultW(page, "organization", "users", "roles", "parent", "children");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "根据ID查询岗位的详细信息，包括相关组织、角色、用户等")
  @RequestMapping(value = "/find/{positionId}", method = RequestMethod.GET)
  public ResponseModel findDetailsById(@PathVariable("positionId") String positionId) {
    try {
      PositionVo positionVo = positionService.findDetailsById(positionId);
      return buildHttpResultW(positionVo, "organization", "users", "roles");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "添加岗位")
  @RequestMapping(value = "", method = RequestMethod.POST)
  public ResponseModel create(@ApiParam(name = "岗位相关信息") @RequestBody PositionVo position) {
    try {
      PositionVo positionVo = positionService.create(position);
      return buildHttpResultW(positionVo);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "修改岗位")
  @RequestMapping(value = "update", method = RequestMethod.PATCH)
  public ResponseModel update(@ApiParam(name = "岗位相关信息") @RequestBody PositionVo position) {
    try {
      PositionVo positionVo = positionService.update(position);
      return buildHttpResultW(positionVo);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "修改岗位状态（启用禁用）")
  @RequestMapping(value = "updateStatus/{positionId}", method = RequestMethod.PATCH)
  public ResponseModel updateStatus(@PathVariable("positionId") String positionId) {
    try {
      PositionVo positionVo = positionService.updateStatus(positionId);
      return buildHttpResultW(positionVo);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "绑定指定的岗位和指定的组织机构")
  @RequestMapping(value = "/bindOrgAndPosition", method = RequestMethod.PATCH)
  public ResponseModel bindOrgAndPosition(
      @RequestParam(name = "positionId") @ApiParam(name = "positionId", value = "指定的岗位ID")
          String positionId,
      @RequestParam(name = "orgId") @ApiParam(name = "orgId", value = "指定的组织机构（orgId）")
          String orgId) {
    try {
      positionService.bindOrgAndPosition(orgId, positionId);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "解绑指定的岗位和指定的组织机构")
  @RequestMapping(value = "/unbindOrgAndPosition", method = RequestMethod.PATCH)
  public ResponseModel unbindOrgAndPosition(
      @RequestParam(name = "positionId") @ApiParam(name = "positionId", value = "指定的岗位ID")
          String positionId,
      @RequestParam(name = "orgId") @ApiParam(name = "orgId", value = "指定的组织机构（orgId）")
          String orgId) {
    try {
      positionService.unbindOrgAndPosition(orgId, positionId);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "将指定一个岗位和指定的多个角色形成绑定关系")
  @RequestMapping(value = "/bindRolesAndPosition", method = RequestMethod.PATCH)
  public ResponseModel bindRolesAndPosition(
      @RequestParam(name = "positionId") @ApiParam(name = "positionId", value = "当前的岗位ID")
          String positionId,
      @RequestParam(name = "roleIds") @ApiParam(name = "roleIds", value = "指定的一个/多个角色数据编号（roleId）")
          String[] roleIds) {
    try {

      positionService.bindRolesAndPosition(Arrays.asList(roleIds), positionId);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "解绑多个角色与某一指定的岗位")
  @RequestMapping(value = "/unbindRolesAndPosition", method = RequestMethod.PATCH)
  public ResponseModel unbindRolesAndPosition(
      @RequestParam(name = "positionId") @ApiParam(name = "positionId", value = "指定的岗位ID")
          String positionId,
      @RequestParam(name = "roleIds") @ApiParam(name = "roleIds", value = "指定的角色ID")
          String[] roleIds) {
    try {
      positionService.unbindRolesAndPosition(Arrays.asList(roleIds), positionId);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "将指定一个岗位和指定的多个用户形成绑定关系")
  @RequestMapping(value = "/bindUsersAndPosition", method = RequestMethod.PATCH)
  public ResponseModel bindUsers(
      @RequestParam(name = "positionId") @ApiParam(name = "positionId", value = "当前的岗位ID")
          String positionId,
      @RequestParam(name = "userIds") @ApiParam(name = "userIds", value = "指定的一个/多个用户数据编号（userId）")
          String[] userIds) {
    try {
      positionService.bindUsersAndPosition(Arrays.asList(userIds), positionId);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "解绑多个用户与某一指定的岗位")
  @RequestMapping(value = "/unbindUsersAndPosition", method = RequestMethod.PATCH)
  public ResponseModel unbindUsersAndPosition(
      @RequestParam(name = "positionId") @ApiParam(name = "positionId", value = "指定的岗位ID")
          String positionId,
      @RequestParam(name = "userIds") @ApiParam(name = "userIds", value = "指定的用户ID")
          String[] userIds) {
    try {
      positionService.unbindUsersAndPosition(Arrays.asList(userIds), positionId);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "预制控件查询，查询用户岗位相关信息")
  @RequestMapping(value = "/findByUserId",method = RequestMethod.GET)
  public ResponseModel findByUserId(@ApiParam(name = "userId", value = "用户id") @RequestParam("userId")String userId){
    try {
      List<PositionVo> position = this.positionService.findByUserId(userId);
      return this.buildHttpResultW(position);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "查询多个用户岗位相关信息")
  @RequestMapping(value = "/findByIds",method = RequestMethod.POST)
  public ResponseModel findByIds(@RequestBody List<String> ids){
    try {
      Set<PositionVo> position = this.positionService.findByIds(ids);
      return this.buildHttpResultW(position);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据岗位编码查询
   * @param code
   * @return
   */
  @GetMapping("findByCode")
  @ApiOperation("根据岗位编码查询")
  public ResponseModel findByCode(@RequestParam @ApiParam("岗位编码") String code) {
    try {
      PositionVo position = positionService.findByCode(code);
      return buildHttpResultW(position);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 切换岗位
   * @param positionId
   * @param userId
   * @return
   */
  @PostMapping("changePosition")
  @ApiOperation("切换岗位")
  public ResponseModel changePosition(@RequestParam(name = "positionId") @ApiParam(name = "positionId", value = "岗位id", required=true) String positionId, @RequestParam(name = "userId") @ApiParam(name = "userId", value = "指定的用户ID", required=true) String userId){
    try{
      UserVo userVo = this.positionService.changePosition(positionId, userId);
      return buildHttpResultW(userVo, "roles", "positions");
    }catch (RuntimeException e){
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }
}
