package com.bizunited.platform.core.repository;

import com.bizunited.platform.core.entity.OrganizationEntity;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.JpaSpecificationExecutor;
import org.springframework.data.jpa.repository.Modifying;
import org.springframework.data.jpa.repository.Query;
import org.springframework.data.repository.query.Param;
import org.springframework.stereotype.Repository;

import java.util.List;
import java.util.Set;

/**
 * @author yinwenjie ycx
 */
@Repository("OrganizationRepository")
public interface OrganizationRepository extends JpaRepository<OrganizationEntity, String>, JpaSpecificationExecutor<OrganizationEntity> {

  /**
   * 对一个指定的currentOrgId进行其父级组织机构parentOrgId的绑定操作（原有currentOrgId的父级信息将会被取消）
   * @param currentOrgId
   * @param parentOrgId
   */
  @Modifying
  @Query(value = "update engine_org set parent_id = :parentOrgId where id = :currentOrgId", nativeQuery = true)
  public void bindParent(@Param("currentOrgId") String currentOrgId, @Param("parentOrgId") String parentOrgId);

  /**
   * 对一个指定的currentOrgId取消其当前父级关系的绑定信息
   * @param currentOrgId
   */
  @Modifying
  @Query(value = "update engine_org set parent_id = null where id = :currentOrgId", nativeQuery = true)
  public void unbindParent(@Param("currentOrgId") String currentOrgId);

  /**
   * 将指定组织机构（currentOrgId）和指定的用户数据编号（userId）形成绑定关系——一个用户只能属于一个组织机构
   * @param currentOrgId
   * @param userId
   */
  @Modifying
  @Query(value = "insert into engine_org_user_mapping (org_id , user_id) values (:currentOrgId , :userId)", nativeQuery = true)
  public void bindUsers(@Param("currentOrgId") String currentOrgId, @Param("userId") String userId);

  /**
   * 解除指定用户和任何组织机构的绑定关系
   * @param userId
   */
  @Modifying
  @Query(value = "delete from engine_org_user_mapping where user_id = :userId", nativeQuery = true)
  public void unbindUser(@Param("userId") String userId);

  /**
   * 解除指定用户和指定组织机构的绑定关系
   */
  @Modifying
  @Query(value = "delete from engine_org_user_mapping where user_id = :userId and org_id = :currentOrgId", nativeQuery = true)
  public void unbindUser(@Param("currentOrgId") String currentOrgId, @Param("userId") String userId);

  /**
   * 绑定组织机构与角色
   * @param currentOrgId
   * @param roleId
   */
  @Modifying
  @Query(value = "insert into engine_org_role_mapping (org_id , role_id) values (:currentOrgId , :roleId)", nativeQuery = true)
  public void bindRole(@Param("currentOrgId") String currentOrgId, @Param("roleId") String roleId);

  /**
   * 解除角色与组织机构的绑定
   * @param roleId
   */
  @Modifying
  @Query(value = "delete from engine_org_role_mapping where role_id = :roleId", nativeQuery = true)
  public void unbindRole(@Param("roleId") String roleId);
  
  /**
   * 解除指定用户和指定组织机构的绑定关系
   * @param currentOrgId 指定的组织机构
   * @param roleId 指定的角色id
   */
  @Modifying
  @Query(value = "delete from engine_org_role_mapping where role_id = :roleId and org_id = :currentOrgId", nativeQuery = true)
  public void unbindRole(@Param("currentOrgId") String currentOrgId ,@Param("roleId") String roleId);

  /**
   * 查询目前没有指定父级组织机构的所有组织机构信息，无论其状态是否正常（status无论什么值）<br>
   * 注意：(查询的信息不包括任何关联信息)
   * @return
   */
  @Query("select distinct org from OrganizationEntity org "
    +"where org.parent is null order by org.sortIndex")
  public List<OrganizationEntity> findByNullParent();

  /**
   * 查询指定父级组织机构下的所有子级组织机构信息，无论其状态是否正常（status无论什么值）<br>
   * 注意：(查询的信息不包括任何关联信息)
   * @return
   */
  @Query("from OrganizationEntity org where org.parent.id = :parentId order by org.sortIndex")
  public List<OrganizationEntity> findByParent(@Param("parentId") String parentId);

  /**
   * 查询目前没有指定父级组织机构的所有组织机构信息，并且依据其状态进行查询<br>
   * 注意：(查询的信息不包括任何关联信息)
   * @return
   */
  @Query("from OrganizationEntity org where org.parent is null and org.tstatus = :status order by org.sortIndex")
  public List<OrganizationEntity> findByNullParentAndStatus(@Param("status") Integer status);

  /**
   * 查询指定父级组织机构下的所有子级组织机构信息，并且依据其状态进行查询<br>
   * 注意：(查询的信息不包括任何关联信息)
   * @return
   */
  @Query("from OrganizationEntity org where org.parent.id = :parentId and org.tstatus = :status order by org.sortIndex")
  public List<OrganizationEntity> findByParentAndStatus(@Param("parentId") String parentId, @Param("status") Integer status);

  @Query("from OrganizationEntity org left join fetch org.parent pp where org.orgName = :orgName")
  public List<OrganizationEntity> findByOrgName(@Param("orgName") String orgName);

  /**
   * 根据组织机构编号查询是否重复
   * @param code 组织机构编号
   * @return
   */
  @Query("from OrganizationEntity org  where org.code = :code")
  OrganizationEntity findByCode(@Param("code") String code);

  /**
   * 查询组织机构的详细信息，包括绑定的用户信息
   * @param id
   * @return
   */
  @Query("select distinct org from OrganizationEntity org "
   + "left join fetch org.parent "
   + "left join fetch org.child "
   + "left join fetch org.roles "
   + "left join fetch org.users "
   + "left join fetch org.positions "
   + "where org.id = :id")
  OrganizationEntity findDetailsById(@Param("id") String id);

  /**
   * 按照指定的人员数据库编号，查询这个人员已经绑定的一个或者多个角色的基本信息
   * @param userId
   * @return
   */
  @Query("select distinct org from OrganizationEntity org left join fetch org.users u "
      + " left join fetch org.roles r "
      + " where u.id = :userId order by org.sortIndex")
  Set<OrganizationEntity> findByUserId(@Param("userId") String userId);

  /**
   * 查询组织机构与角色是否绑定
   * @param currentOrgId 当前组织机构id
   * @param roleId       角色id
   * @return
   */
  @Query(value = "SELECT COUNT(*) FROM engine_org_role_mapping WHERE org_id = :currentOrgId AND role_id = :roleId", nativeQuery = true)
  long countByOrgAndRoleId(@Param("currentOrgId") String currentOrgId, @Param("roleId") String roleId);

  @Query(value = "SELECT COUNT(*) FROM engine_org_user_mapping WHERE org_id = :currentOrgId AND user_id = :userId", nativeQuery = true)
  long countByOrgIdAndUserId(@Param("currentOrgId") String currentOrgId, @Param("userId") String userId);

  @Query(value = "select org from OrganizationEntity org where org.tstatus = 1")
  public List<OrganizationEntity> findAllByStatus();
  
  @Query(value = "from OrganizationEntity org left join fetch org.parent order by org.createTime desc ")
  public List<OrganizationEntity> findAll();

  @Query("from OrganizationEntity m where m.id in :ids")
  public Set<OrganizationEntity> findByIds(@Param("ids") List<String> ids);

  /**
   * 组织机构-查询所有根节点
   * @return
   */
  @Query("from OrganizationEntity m where m.parent=null order by m.createTime desc")
  List<OrganizationEntity> findRootOrg();

  /**
   * 模糊查询-查询这个组织机构以及这个组织对应的所有子级结构信息
   * @param orgName
   * @return
   */
  @Query(value = "from OrganizationEntity org where orgName like CONCAT(CONCAT('%',:orgName),'%')")
  Set<OrganizationEntity> findByOrgNameLike(@Param("orgName") String orgName);

  /**
   * 查询用户是否已经绑定组织机构
   * @param userId       用户id
   * @return
   */
  @Query(value = "SELECT COUNT(*) FROM engine_org_user_mapping WHERE user_id = :userId", nativeQuery = true)
  long countOrganizationByUserId(@Param("userId") String userId);

  /**
   * 查询组织机构是否已经绑定用户
   * @param  currentOrgId    组织机构id
   * @return
   */
  @Query(value = "SELECT COUNT(*) FROM engine_org_user_mapping WHERE org_id = :currentOrgId", nativeQuery = true)
  long countUserByOrgId(@Param("currentOrgId") String currentOrgId);

  /**
   * 查询角色是否已经绑定组织机构
   * @param roleId       角色id
   * @return
   */
  @Query(value = "SELECT COUNT(*) FROM engine_org_role_mapping WHERE role_id = :roleId", nativeQuery = true)
  long countOrganizationByRoleId(@Param("roleId") String roleId);

  /**
   * 查询组织机构是否已经绑定角色
   * @param currentOrgId 当前组织机构id
   * @return
   */
  @Query(value = "SELECT COUNT(*) FROM engine_org_role_mapping WHERE org_id = :currentOrgId", nativeQuery = true)
  long countRoleByOrgId(@Param("currentOrgId") String currentOrgId);

  /**
   * 根据子级ID查询
   * @param childId
   * @return
   */
  @Query("select o from OrganizationEntity o inner join o.child c where c.id = :childId")
  OrganizationEntity findByChildId(@Param("childId") String childId);
  
}
