package com.bizunited.platform.core.repository;

import com.bizunited.platform.core.entity.UserGroupEntity;
import com.bizunited.platform.core.repository.internal.UserGroupRepositoryCustom;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.JpaSpecificationExecutor;
import org.springframework.data.jpa.repository.Modifying;
import org.springframework.data.jpa.repository.Query;
import org.springframework.data.repository.query.Param;
import org.springframework.stereotype.Repository;

import java.util.List;
import java.util.Set;

@Repository(value = "UserGroupRepository")
public interface UserGroupRepository extends JpaRepository<UserGroupEntity, String>, JpaSpecificationExecutor<UserGroupEntity>, UserGroupRepositoryCustom {

  /**
   * 新增用户组与用户关联
   * @param groupId 用户组id
   * @param userId  用户id
   * @return
   */
  @Modifying
  @Query(value = "INSERT INTO engine_group_user_mapping (group_id,user_id) VALUES (:groupId,:userId)", nativeQuery = true)
  int bindUser(@Param("groupId") String groupId, @Param("userId") String userId);

  /**
   * 删除用户组与用户关联
   *
   * @param groupId 用户组id
   * @param userId  用户id
   * @return
   */
  @Modifying
  @Query(value = "DELETE FROM engine_group_user_mapping WHERE group_id = :groupId AND user_id = :userId", nativeQuery = true)
  int unBindUser(@Param("groupId") String groupId, @Param("userId") String userId);

  /**
   * 新增用户组与角色关联
   *
   * @param groupId 用户组id
   * @param roleId  角色id
   * @return
   */
  @Modifying
  @Query(value = "INSERT INTO engine_group_role_mapping (group_id,role_id) VALUES (:groupId,:roleId)", nativeQuery = true)
  int bindRole(@Param("groupId") String groupId, @Param("roleId") String roleId);

  /**
   * 删除用户组与用户关联
   *
   * @param groupId 用户组id
   * @param roleId  角色id
   * @return
   */
  @Modifying
  @Query(value = "DELETE FROM engine_group_role_mapping WHERE group_id = :groupId AND role_id = :roleId", nativeQuery = true)
  int unBindRole(@Param("groupId") String groupId, @Param("roleId") String roleId);

  /**
   * 判定是否存在重复
   * @param groupId 用户组id
   * @param userId  用户id
   * @return
   */
  @Query(value = "SELECT COUNT(*) FROM engine_group_user_mapping WHERE group_id = :groupId AND user_id = :userId", nativeQuery = true)
  long countUserGroup(@Param("groupId") String groupId, @Param("userId") String userId);

  /**
   * 判定是否存在重复
   * @param groupId 用户组id
   * @param roleId  角色id
   * @return
   */
  @Query(value = "SELECT COUNT(*) FROM engine_group_role_mapping WHERE group_id = :groupId AND role_id = :roleId", nativeQuery = true)
  long countRoleGroup(@Param("groupId") String groupId, @Param("roleId") String roleId);
  /**
   * 根据用户组名称查询是否存在该用户组
   * @param groupName 用户组名称
   * @return
   */
  @Query("select ug from UserGroupEntity ug where ug.groupName = :groupName")
  UserGroupEntity findByGroupName(@Param("groupName") String groupName);


  /**
   * 查询用户组详情
   * @param id 用户组id
   * @return
   */
  @Query(value = "select ug from UserGroupEntity ug " 
   + "left join fetch ug.users u "
   + "left join fetch ug.roles r " + "where ug.id = :id")
  Set<UserGroupEntity> findDetailsById(@Param("id") String id);

  /**
   * 按照指定的人员数据库编号，查询人员已经绑定的一个或者多个用户组信息。<p>
   * 注意：用户组信息中包括了用户组已经绑定的角色信息。
   * @param userId 指定的用户数据库编号
   * @return
   */
  @Query("select distinct ug from UserGroupEntity ug left join fetch ug.users users left join fetch ug.roles where users.id = :userId")
  Set<UserGroupEntity> findByUserId(@Param("userId") String userId);

  /**
   * 查询多个用户组信息
   * @param ids
   * @return
   */
  @Query("from UserGroupEntity m where m.id in :ids")
  Set<UserGroupEntity> findByIds(@Param("ids") List<String> ids);

  /**
   * 判定是否一个用户已经绑定分组
   * @param userId  用户id
   * @return
   */
  @Query(value = "SELECT COUNT(*) FROM engine_group_user_mapping WHERE user_id = :userId ", nativeQuery = true)
  long countGroupByUserId(@Param("userId") String userId);

  /**
   * 判定是否一个分组已经绑定用户
   * @param groupId  用户组id
   * @return
   */
  @Query(value = "SELECT COUNT(*) FROM engine_group_user_mapping WHERE group_id = :groupId ", nativeQuery = true)
  long countUserByGroupId(@Param("groupId") String groupId);

  /**
   * 判定是否一个角色已经绑定分组
   * @param roleId  角色id
   * @return
   */
  @Query(value = "SELECT COUNT(*) FROM engine_group_role_mapping WHERE role_id = :roleId ", nativeQuery = true)
  long countGroupByRoleId(@Param("roleId") String roleId);

  /**
   * 判定是否一个分组已经绑定角色
   * @param groupId 用户组id
   * @return
   */
  @Query(value = "SELECT COUNT(*) FROM engine_group_role_mapping WHERE group_id = :groupId ", nativeQuery = true)
  long countRoleByGroupId(@Param("groupId") String groupId);
}
