package com.bizunited.platform.core.repository.dataview;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.bizunited.platform.core.common.enums.SQLCorrelationEnum;
import com.bizunited.platform.core.entity.DataSourceEntity;
import com.bizunited.platform.core.entity.DataViewEntity;
import com.bizunited.platform.core.entity.DataViewFieldEntity;
import com.bizunited.platform.core.entity.DataViewSystemEntity;
import com.bizunited.platform.core.repository.dataview.analysis.AbstractSqlAnalysis;
import com.bizunited.platform.core.repository.dataview.analysis.Constants;
import com.bizunited.platform.core.repository.dataview.analysis.SqlAnalysis;
import com.bizunited.platform.core.repository.dynamic.DynamicDataSourceManager;
import com.bizunited.platform.core.service.dataview.model.SQLParamModel;
import com.bizunited.platform.core.service.invoke.model.InvokeOperations;
import com.bizunited.platform.core.service.invoke.model.InvokeParams;
import org.apache.commons.lang3.RegExUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.query.internal.NativeQueryImpl;
import org.hibernate.transform.Transformers;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.orm.hibernate5.SessionFactoryUtils;
import org.springframework.stereotype.Repository;
import org.springframework.util.CollectionUtils;

import javax.persistence.EntityManager;
import javax.persistence.EntityManagerFactory;
import javax.persistence.PersistenceContext;
import javax.persistence.Query;
import javax.sql.DataSource;
import java.math.BigInteger;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

@Repository("DataViewRepositoryCustomImpl")
public class DataViewRepositoryCustomImpl implements DataViewRepositoryCustom{
  @Autowired
  private DynamicDataSourceManager dynamicDataSourceManager;
  @Autowired
  @PersistenceContext
  private EntityManager entityManager;
  private static final String SYSTEM_PARAMS_PATTERN = "\\{\\:[a-zA-Z]{1}\\w*\\}";
  private static final String HIBERNATE_DIALECT = "hibernate.dialect";
  private static final String ERROR_METADATA = "未能获取到数据库类型元数据信息，请检查!!";
  private static final String ERROR_DATAVIEW = "数据视图不能为空！";
  private static final String ERROR_RESULTSET = "ResultSet资源异常:";
  private static final String ERROR_SQL = "原始SQL不能为空！";
  private static final String ERROR_SQL_EXECUTE = "SQL执行异常，请检查!!详情：";
  private static final String ERROR_SESSION_FACTORY = "未能获取到第三方的sessionFactory信息，请检查!!";

  @Override
  public JSONArray executeSQLTop1(SqlAnalysis sqlAnalysis) {
    return this.executeSQL(sqlAnalysis);
  }
  
  @Override
  public JSONArray executeSQL(SqlAnalysis sqlAnalysis) {
    sqlAnalysis.analysis();
    Map<SQLCorrelationEnum,Object> result = sqlAnalysis.getResult();

    EntityManagerFactory emf = entityManager.getEntityManagerFactory();
    Map<String, Object> emfProperties = emf.getProperties();
    String driverType = emfProperties.get(HIBERNATE_DIALECT).toString();
    Validate.notBlank(driverType, ERROR_METADATA);
    String sql = result.get(SQLCorrelationEnum.RESULT_SQL).toString();

    Query query = entityManager.createNativeQuery(sql);
    sqlAnalysis.processSQLParams(query);
    query.unwrap(NativeQueryImpl.class).setResultTransformer(Transformers.ALIAS_TO_ENTITY_MAP);
    List<?> list = query.getResultList();
    return JSON.parseArray(JSON.toJSONString(list,SerializerFeature.WriteMapNullValue));
  }

  @Override
  public Page<JSONObject> executeSQLPageAble(SqlAnalysis sqlAnalysis) {
    sqlAnalysis.analysis();
    Map<SQLCorrelationEnum,Object> result = sqlAnalysis.getResult();

    EntityManagerFactory emf = entityManager.getEntityManagerFactory();
    Map<String, Object> emfProperties = emf.getProperties();
    String driverType = emfProperties.get(HIBERNATE_DIALECT).toString();
    Validate.notBlank(driverType, ERROR_METADATA);
    String sql = result.get(SQLCorrelationEnum.RESULT_SQL).toString();

    String countSql = result.getOrDefault(SQLCorrelationEnum.PRE_ORDER_PAGE_SQL, sql).toString();

    Query query = entityManager.createNativeQuery(sql);
    sqlAnalysis.processSQLParams(query);
    Query countQuery = entityManager.createNativeQuery(countSql);
    sqlAnalysis.processSQLParams(countQuery);
    query.unwrap(NativeQueryImpl.class).setResultTransformer(Transformers.ALIAS_TO_ENTITY_MAP);
    List<?> list = query.getResultList();
    List<JSONObject> jsonList = new ArrayList<>();
    if(!CollectionUtils.isEmpty(list)) {
      for(Object o : list) {
        JSONObject json = JSON.parseObject(JSON.toJSONString(o,SerializerFeature.WriteMapNullValue));
        jsonList.add(json);
      }
    }

    // 查询总数量信息
    long count = ((BigInteger)countQuery.getResultList().get(0)).longValue();
    return new PageImpl<>(jsonList, sqlAnalysis.getPageable(), count);
  }

  @Override
  public DataViewEntity executeResource(DataViewEntity dataView) {
    Validate.notNull(dataView,ERROR_DATAVIEW);
    Validate.notBlank(dataView.getSourceSql(), ERROR_SQL);
    Session session = (Session) entityManager.getDelegate();
    String sourceSql = dataView.getSourceSql();
    this.replaceSysParam(dataView);
    session.doWork(connection -> {
      Set<DataViewFieldEntity> fields = new LinkedHashSet<>();
      try(
          Statement statement = connection.createStatement();
          ResultSet result = statement.executeQuery(dataView.getSourceSql())
      ){
        ResultSetMetaData metaData = result.getMetaData();
        //遍历结果集中每一个字段，得到字段信息并存入fields
        Set<String> fieldNames = new HashSet<>();
        //用户参数与系统参数也不能重名
        Set<String> sysParamsStrs = dataView.getSystemFilters().stream().map(DataViewSystemEntity::getParamName).collect(Collectors.toSet());
        fieldNames.addAll(sysParamsStrs);
        for(int i=1;i<=metaData.getColumnCount();i++){
          DataViewFieldEntity field = new DataViewFieldEntity();
          field.setFieldName(metaData.getColumnLabel(i));
          //如果查询出的字段集中含有相同的字段名，那么报错提醒用户
          Validate.isTrue(fieldNames.add(metaData.getColumnLabel(i)),"返回字段集或与系统参数中含有相同字段名：%s,请为其取别名，或做其他调整。" , metaData.getColumnLabel(i));
          field.setFieldType(metaData.getColumnClassName(i));
          field.setPhysical(!StringUtils.isEmpty(metaData.getTableName(i)));
          field.setSchemaName(metaData.getSchemaName(i));
          field.setTargetTable(metaData.getTableName(i));
          field.setTargetField(metaData.getColumnName(i));
          field.setSortIndex(i);
          fields.add(field);
        }
        fieldNames.clear();
      }
      dataView.setFields(fields);
    });
    dataView.setSourceSql(sourceSql);
    return dataView;
  }

  @Override
  public void checkTable(String tableName) {
    Validate.notBlank(tableName,"输入数据表名不能为空，请检查！");
    Session session = (Session) entityManager.getDelegate();
    session.doWork(connection -> {
      DatabaseMetaData md = connection.getMetaData();
      try(ResultSet rs = md.getTables(null, null, tableName, null)){
        Validate.isTrue(rs.next(),"所设置基础表在主数据源不存在，请检查！");
      }
    });
  }


  private void closeResultSet(ResultSet results){
    try{
      results.close();
    }catch (Exception e){
      throw new IllegalArgumentException("SQL查询结果集资源关闭异常！！");
    }
  }

  //======================以下是第三方========================

  @Override
  public JSONArray executeSQLTop1(String dataSourceCode, SqlAnalysis sqlAnalysis) {
    return this.executeSQLHandle(dataSourceCode, sqlAnalysis, 1);
  }

  @Override
  public JSONArray executeSQL(String dataSourceCode , SqlAnalysis sqlAnalysis , DataViewEntity dataView) {
    // TODO 这里的dataView入参没有使用？
    return this.executeSQLHandle(dataSourceCode, sqlAnalysis, 2);
  }
  
  public JSONArray executeSQLHandle(String dataSourceCode , SqlAnalysis sqlAnalysis , Integer type) {
    sqlAnalysis.analysis();
    Map<SQLCorrelationEnum,Object> result = sqlAnalysis.getResult();

    SessionFactory sessionFactory = this.dynamicDataSourceManager.getCurrentSessionFactory(dataSourceCode);
    Validate.notNull(sessionFactory, ERROR_SESSION_FACTORY);

    try(Connection conn = SessionFactoryUtils.getDataSource(sessionFactory).getConnection();
        PreparedStatement st = conn.prepareStatement(result.get(SQLCorrelationEnum.RESULT_SQL).toString(), ResultSet.TYPE_SCROLL_INSENSITIVE, ResultSet.CONCUR_READ_ONLY)){
      sqlAnalysis.processSQLParams(st);
      return this.processResultSet(st,type);
    }catch (Exception e) {
      throw new IllegalArgumentException(ERROR_SQL_EXECUTE+e.getMessage());
    }
  }

  @Override
  public Page<JSONObject> executeSQLPageAble(String dataSourceCode, SqlAnalysis sqlAnalysis, Pageable pageable) {
    sqlAnalysis.analysis();
    Map<SQLCorrelationEnum,Object> result = sqlAnalysis.getResult();

    SessionFactory sessionFactory = this.dynamicDataSourceManager.getCurrentSessionFactory(dataSourceCode);
    Validate.notNull(sessionFactory, ERROR_SESSION_FACTORY);

    int count = 0;
    //先执行count语句
    try(Connection conn = SessionFactoryUtils.getDataSource(sessionFactory).getConnection();
        PreparedStatement countStatement = conn.prepareStatement(result.get(SQLCorrelationEnum.PRE_ORDER_PAGE_SQL).toString())){
      sqlAnalysis.processSQLParams(countStatement);
      count = this.processCountResultSet(countStatement);
    }catch (Exception e) {
      throw new IllegalArgumentException(ERROR_SQL_EXECUTE+e.getMessage());
    }

    //再执行分页语句
    try(Connection conn = SessionFactoryUtils.getDataSource(sessionFactory).getConnection();
        PreparedStatement st = conn.prepareStatement(result.get(SQLCorrelationEnum.RESULT_SQL).toString())){
      sqlAnalysis.processSQLParams(st);
      return this.processResultSet(st,pageable,count);
    }catch (Exception e) {
      throw new IllegalArgumentException(ERROR_SQL_EXECUTE+e.getMessage());
    }
  }


  @Override
  public DataViewEntity executeResource(String dataSourceCode, DataViewEntity dataView) {
    Validate.notNull(dataView,ERROR_DATAVIEW);
    Validate.notBlank(dataView.getSourceSql(), ERROR_SQL);
    SessionFactory sessionFactory = this.dynamicDataSourceManager.getCurrentSessionFactory(dataSourceCode);
    Validate.notNull(sessionFactory,"未找到该数据源！");
    String sourceSql = dataView.getSourceSql();
    Set<DataViewFieldEntity> fields = new LinkedHashSet<>();
    DataSource dataSource = SessionFactoryUtils.getDataSource(sessionFactory);
    DataViewEntity currentDataView = this.replaceSysParam(dataView);
    try(Connection connection = dataSource.getConnection();
        Statement statement = connection.createStatement();
        ResultSet result = statement.executeQuery(currentDataView.getSourceSql())) {
      ResultSetMetaData metaData = result.getMetaData();
      //遍历结果集中每一个字段，得到字段信息并存入fields
      Set<String> fieldNames = new HashSet<>();
      //用户参数与系统参数也不能重名
      Set<String> sysParamsStrs = currentDataView.getSystemFilters().stream().map(DataViewSystemEntity::getParamName).collect(Collectors.toSet());
      fieldNames.addAll(sysParamsStrs);
      for(int i=1;i<=metaData.getColumnCount();i++){
        DataViewFieldEntity field = new DataViewFieldEntity();
        field.setFieldName(metaData.getColumnLabel(i));
        //如果查询出的字段集中含有相同的字段名，那么报错提醒用户
        Validate.isTrue(fieldNames.add(metaData.getColumnLabel(i)),"返回字段集或与系统参数中含有相同字段名：%s,请为其取别名，或做其他调整。" , metaData.getColumnLabel(i));
        field.setFieldType(metaData.getColumnClassName(i));
        field.setPhysical(!StringUtils.isEmpty(metaData.getTableName(i)));
        field.setSchemaName(metaData.getSchemaName(i));
        field.setTargetTable(metaData.getTableName(i));
        field.setTargetField(metaData.getColumnName(i));
        field.setSortIndex(i);
        fields.add(field);
      }
      fieldNames.clear();
      currentDataView.setFields(fields);
    }catch (SQLException e){
      throw new IllegalArgumentException(ERROR_SQL_EXECUTE+e.getMessage());
    }
    currentDataView.setSourceSql(sourceSql);
    return currentDataView;
  }

  @Override
  public void checkTable(String dataSourceCode, String tableName) {
    Validate.notBlank(tableName,"输入数据表名不能为空，请检查！");
    Validate.notBlank(dataSourceCode,"输入第三方数据源编码不能为空，请检查！");
    SessionFactory sessionFactory = this.dynamicDataSourceManager.getCurrentSessionFactory(dataSourceCode);
    try (Connection connection = SessionFactoryUtils.getDataSource(sessionFactory).getConnection();){
      DatabaseMetaData md = connection.getMetaData();
      try(ResultSet rs = md.getTables(null, null, tableName, null);){
        Validate.isTrue(rs.next(),"所设置第三方数据源中基础表不存在，请检查！");
      }
    } catch (Exception e){
      throw new IllegalArgumentException(ERROR_SQL_EXECUTE+e.getMessage());
    }
  }


  /**
   * 分析SQL，获取其中的系统参数，并将系统参数值变为空字符串
   * 替换查询SQL中的系统参数,
   * @param dataView
   * @return
   */
  private DataViewEntity replaceSysParam(DataViewEntity  dataView){
    Validate.notNull(dataView,ERROR_DATAVIEW);
    Validate.notBlank(dataView.getSourceSql(), ERROR_SQL);
    String sql = dataView.getSourceSql();
    Set<DataViewSystemEntity> systemEntities = new LinkedHashSet<>();
    Set<String> sysParams = this.extractMessageByRegular(sql);
    if(!CollectionUtils.isEmpty(sysParams)){
      for(String param : sysParams){
        //取出系统参数字段
        DataViewSystemEntity dataViewSystem = new DataViewSystemEntity();
        dataViewSystem.setParamName(param);
        systemEntities.add(dataViewSystem);
        //将原SQL字段设置为空
        String replaceStr = "{:" + param + "}";
        // 这里加上括号是因为不确定前面的操作符是=或in
        sql = sql.replace(replaceStr,"('')");
      }
    }
    dataView.setSystemFilters(systemEntities);
    dataView.setSourceSql(sql);
    return dataView;
  }

  /**
   * 正则获取特定格式的系统参数
   * @param msg
   * @return
   */
  private Set<String> extractMessageByRegular(String msg){
    Set<String> list=new LinkedHashSet<>();
    Pattern p = Pattern.compile(SYSTEM_PARAMS_PATTERN);
    Matcher m = p.matcher(msg);
    while(m.find()){
      //截取符合规则的系统参数字符串
      list.add(m.group().substring(2, m.group().length()-1));
    }
    return list;
  }

  /**
   * 执行分页的count语句
   * @param st
   * @return
   */
  private int processCountResultSet(PreparedStatement st) {
    int count = 0;
    //执行分页count语句执行
    try(
        ResultSet results = st.executeQuery()){
      if(results.next()) {
        count = results.getInt(1);
      }
      this.closeResultSet(results);
    }catch (Exception e){
      throw new IllegalArgumentException("CountResultSet资源异常:" + e.getMessage());
    }
    return count;
  }


  private Page<JSONObject> processResultSet(PreparedStatement st,Pageable pageable,int count){
    Page<JSONObject> page;
    List<JSONObject> list = new ArrayList<>();

    //执行分页语句执行
    try(ResultSet results = st.executeQuery()){
      ResultSetMetaData  meta;
      while(results.next()) {
        meta = results.getMetaData();
        JSONObject jsonObj = new JSONObject();
        for(int i = 1 ; i <= meta.getColumnCount() ; i++) {
          jsonObj.put(meta.getColumnLabel(i), results.getObject(i));
        }
        list.add(jsonObj);
      }
      page = new PageImpl<>(list, pageable, count);
      this.closeResultSet(results);
    }catch (Exception e){
      throw new IllegalArgumentException(ERROR_RESULTSET + e.getMessage());
    }
    return page;
  }

  private JSONArray processResultSet(PreparedStatement st,Integer type){
    JSONArray jsonArr = new JSONArray();
    switch (type){
      case 1 :
        try(ResultSet results = st.executeQuery()){
          ResultSetMetaData  meta;
          //只取第一条数据
          if(results.first()) {
            meta = results.getMetaData();
            JSONObject jsonObj = new JSONObject();
            for(int i = 1 ; i <= meta.getColumnCount() ; i++) {
              jsonObj.put(meta.getColumnLabel(i), results.getObject(i));
            }
            jsonArr.add(jsonObj);
          }
        }catch (Exception e){
          throw new IllegalArgumentException(ERROR_RESULTSET + e.getMessage());
        }
        break;
      case 2:
        try(ResultSet results = st.executeQuery()){
          ResultSetMetaData  meta;
          while(results.next()) {
            meta = results.getMetaData();
            JSONObject jsonObj = new JSONObject();
            for(int i = 1 ; i <= meta.getColumnCount() ; i++) {
              jsonObj.put(meta.getColumnLabel(i), results.getObject(i));
            }
            jsonArr.add(jsonObj);
          }
        }catch (Exception e){
          throw new IllegalArgumentException(ERROR_RESULTSET + e.getMessage());
        }
        break;
      default: 
        break;
    }
    return jsonArr;
  }


  /**
   * 当数据有预制值信息时，该方法用于预制值回显的逻辑调用
   * 注：组合了第三方的数据视图或主数据视图的判断
   * @param dataView
   * @param invokeParams
   * @param presets
   * @param primaryKey
   * @param primaryValue
   * @return
   */
  @Override
  public JSONArray executeSQLEcho(DataViewEntity dataView, InvokeParams invokeParams,Map<String,Object> presets,String primaryKey, JSONArray primaryValue) {
    String sql = dataView.getSourceSql();
    Validate.notBlank(sql , "未能获取到视图SQL信息，请检查!!");
    Validate.notBlank(primaryKey,"传入的primaryKey信息不能为空");
    Validate.notNull(primaryValue,"传入的primaryValue信息不能为空");
    if(primaryValue.isEmpty()){
      return new JSONArray();
    }
    DataSourceEntity dataSource = dataView.getDataSource();
    Set<DataViewFieldEntity> fields = dataView.getFields();
    Validate.isTrue(!CollectionUtils.isEmpty(fields), "视图信息中，未能获取到视图的字段信息，请检查!!");


    String sourceSQL = dataView.getSourceSql();
    Set<DataViewSystemEntity> systemFilters = dataView.getSystemFilters();
    Pattern pattern = Pattern.compile(Constants.SYSTEM_PARAM_PATTERN);
    Integer count = 1;
    List<SQLParamModel> values = new ArrayList<>();
    if(!CollectionUtils.isEmpty(systemFilters)){
      for(int i = 0 ; i < systemFilters.size() ; i++){
        //匹配系统参数传参
        Matcher m = pattern.matcher(sourceSQL);
        String re = null;
        if(m.find()){
          //截取符合规则的系统参数字符串
          re = m.group().substring(2, m.group().length()-1);
        }
        Validate.notBlank(re,"根据正则表达式，没有获取到系统参数信息");
        String systemParamName = re;
        Optional<DataViewSystemEntity> op = systemFilters.stream().filter(e -> StringUtils.equals(e.getParamName(),systemParamName)).findFirst();
        DataViewSystemEntity systemEntity = op.orElse(null);
        Validate.notNull(systemEntity,"做系统参数分析时，未能查询到系统参数信息，请检查！！");
        Validate.isTrue(systemEntity.getParamSourceType() != null,"做系统参数分析时，未能获取到参数来源信息，请检查！！");

        String searchStr = "\\{\\:" + re + "\\}";
        String replacement = "?";
        if(dataSource == null) {
          replacement = replacement + count;
        }
        //根据上面的信息，辨别参数来源
        switch (systemEntity.getParamSourceType()){
          case 1 : //外部传入
            InvokeOperations model = invokeParams.get(systemParamName);
            Validate.notNull(model, "外部传入的参数不能为空，请检查!!");
            Object compareValue = model.getCompareValue();
            Validate.notNull(compareValue, "外部传入的参数值不能为空，请检查!!");
            //处理SQL
            sourceSQL = RegExUtils.replaceFirst(sourceSQL, searchStr, replacement);
            //开始构造参数
            SQLParamModel paramModel = new SQLParamModel();
            paramModel.setValue(compareValue);
            paramModel.setParamName(systemEntity.getParamName());
            paramModel.setParamType(systemEntity.getParamType());
            paramModel.setTransferType(1);
            paramModel.setMappingType(model.getMappingType());
            paramModel.setIndex(count);
            values.add(paramModel);
            break;
          case 2 : //固定值
            //处理SQL
            sourceSQL = RegExUtils.replaceFirst(sourceSQL, searchStr, replacement);
            //开始构造参数
            SQLParamModel paramModel2 = new SQLParamModel();
            paramModel2.setValue(systemEntity.getParamValue());
            paramModel2.setParamName(systemEntity.getParamName());
            paramModel2.setParamType(systemEntity.getParamType());
            paramModel2.setTransferType(2);
            paramModel2.setMappingType(systemEntity.getParamType());
            paramModel2.setIndex(count);
            values.add(paramModel2);
            break;
          case 3 : //预置值
            //处理SQL
            sourceSQL = RegExUtils.replaceFirst(sourceSQL, searchStr, replacement);
            //开始构造参数
            SQLParamModel paramModel3 = new SQLParamModel();
            paramModel3.setValue(presets.get(systemEntity.getParamName() + "|DataViewSystemEntity"));
            paramModel3.setParamName(systemEntity.getParamName());
            paramModel3.setParamKey(systemEntity.getParamKey());
            paramModel3.setTransferType(3);
            paramModel3.setSource(DataViewSystemEntity.class);
            paramModel3.setIndex(count);
            values.add(paramModel3);
            break;
          default: 
            break;
        }
        count++;
      }
    }

    StringBuilder executeSQL = new StringBuilder("select * from ("+ sourceSQL +") tb where 1=1 ");
    executeSQL.append(" and ").append(primaryKey).append(" in (");
    for(int index = 0 ; index < primaryValue.size() ; index++){
      String o = primaryValue.getString(index);
      executeSQL.append("?");
      if(dataSource == null) {
        executeSQL.append(count);
      }
      executeSQL.append(",");
      SQLParamModel model = new SQLParamModel();
      model.setValue(o);
      model.setParamName(primaryKey);
      model.setTransferType(2);
      model.setSource(DataViewSystemEntity.class);
      model.setIndex(count++);
      values.add(model);
    }
    executeSQL.replace(executeSQL.toString().length() - 1 , executeSQL.toString().length(),"");
    executeSQL.append(") ");

    JSONArray jsonArr;
    if(dataSource == null) {
      //主数据源
      jsonArr = this.executeSQLEcho(executeSQL.toString(), values);
    }else {
      //第三方数据源
      jsonArr = this.executeSQLEcho(dataSource.getCode(),executeSQL.toString(), values);
    }
    return jsonArr;
  }


  /**
   * 当数据有预制值信息时，该方法用于预制值回显的逻辑调用（只包含SQL执行）
   * 注：适用于JPA的方式
   * @param sql
   * @param models
   * @return
   */
  @Override
  public JSONArray executeSQLEcho(String sql , List<SQLParamModel> models){
    EntityManagerFactory emf = entityManager.getEntityManagerFactory();
    Map<String, Object> emfProperties = emf.getProperties();
    String driverType = emfProperties.get(HIBERNATE_DIALECT).toString();
    Validate.notBlank(driverType, ERROR_METADATA);

    Query query = entityManager.createNativeQuery(sql);
    for(SQLParamModel model : models){
      AbstractSqlAnalysis.matchJPAParamType(query,model.getValue(),model);
    }

    query.unwrap(NativeQueryImpl.class).setResultTransformer(Transformers.ALIAS_TO_ENTITY_MAP);
    List<?> list = query.getResultList();
    return JSON.parseArray(JSON.toJSONString(list,SerializerFeature.WriteMapNullValue));
  }


  /**
   * 当数据有预制值信息时，该方法用于预制值回显的逻辑调用（只包含SQL执行）
   * 注：适用于第三方的数据视图方式
   * @param dataSourceCode
   * @param sql
   * @param models
   * @return
   */
  @Override
  public JSONArray executeSQLEcho(String dataSourceCode,String sql , List<SQLParamModel> models){
    SessionFactory sessionFactory = this.dynamicDataSourceManager.getCurrentSessionFactory(dataSourceCode);
    Validate.notNull(sessionFactory, ERROR_SESSION_FACTORY);

    try(Connection conn = SessionFactoryUtils.getDataSource(sessionFactory).getConnection();
        PreparedStatement st = conn.prepareStatement(sql)){
      for(SQLParamModel model : models){
        AbstractSqlAnalysis.matchPreparedStatementParamType(st,model.getValue(),model);
      }
      return this.processResultSet(st,2);
    }catch (Exception e) {
      throw new IllegalArgumentException(ERROR_SQL_EXECUTE+e.getMessage());
    }
  }
}
