package com.bizunited.platform.core.repository.dataview.analysis;

import com.bizunited.platform.core.common.enums.SQLCorrelationEnum;
import com.bizunited.platform.core.entity.DataViewAuthHorizontalEntity;
import com.bizunited.platform.core.entity.DataViewAuthHorizontalRelationEntity;
import com.bizunited.platform.core.entity.DataViewFieldEntity;
import com.bizunited.platform.core.service.dataview.model.SQLParamModel;
import com.bizunited.platform.core.service.invoke.model.InvokeParams;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * 用于数据视图之横向数据权限条件的分析以及构造SQL工具类
 */
class SQLAuthHorizontalAnalysis {
  /**
   * 通过私有构造函数，让该类暂时不对外开发
   * 因为后期可能会对该类进行重新设计
   */
  private SQLAuthHorizontalAnalysis() {
  }

  /**
   * 横向数据权限 参数筛选条件的分析及构造
   * 默认Prestatement的构建方式(?传参)
   *
   * @param auths
   * @param map
   * @return
   */
  static void buildSQLAuthHorizoncal(Set<DataViewAuthHorizontalEntity> auths, Map<SQLCorrelationEnum, Object> map) {
    if (CollectionUtils.isEmpty(auths)) {
      return;
    }

    //开始处理数据视图的横向权限逻辑
    for (DataViewAuthHorizontalEntity e : auths) {
      DataViewFieldEntity field = e.getField();
      Validate.notNull(field, "横向权限中没有指定权限字段信息，请检查!!");
      processAuths(e, field, map);
    }
  }


  /**
   * 目前横向权限分为组织机构，用户组，角色，岗位，用户5种，但是再处理SQL与参数上，暂时分为两种情况
   * 1.对于IN或NOT IN的处理
   * 2.对于EQ或NEQ的处理
   *
   * @param auth
   * @param field
   * @param map
   */
  private static void processAuths(DataViewAuthHorizontalEntity auth, DataViewFieldEntity field, Map<SQLCorrelationEnum, Object> map) {
    if (StringUtils.equalsIgnoreCase("EQ", auth.getOprtType()) || StringUtils.equalsIgnoreCase("NEQ", auth.getOprtType())) {
      analysisEqConditions(auth, field, map);
    } else {
      analysisInConditions(auth, field, map);
    }
  }


  /**
   * 解析IN 或 NOT IN 的情况，并存储值
   * 支持：预制值和固定值
   *
   * @param auth
   * @param field
   * @param map
   */
  @SuppressWarnings({"unchecked", "rawtypes"})
  private static void analysisInConditions(DataViewAuthHorizontalEntity auth, DataViewFieldEntity field, Map<SQLCorrelationEnum, Object> map) {
    List<Object> values = new ArrayList<>();
    if(auth.getParamSourceType() == 2) {
      //固定值情况
      Set<DataViewAuthHorizontalRelationEntity> authRelations = auth.getAuthRelations();
      for (DataViewAuthHorizontalRelationEntity authRelation : authRelations) {
        values.add(authRelation.getAuthRelationIds());
      }
    } else {
      //预制值情况
      Map<String, Object> presets = (Map<String, Object>) map.get(SQLCorrelationEnum.PRESETS);
      String key = auth.getField().getFieldName() + "|DataViewAuthHorizontalEntity";
      Object object = presets.get(key);
      if(Iterable.class.isAssignableFrom(object.getClass())) {
        Iterable iterable = (Iterable) object;
        Iterator iterator = iterable.iterator();
        while (iterator.hasNext()) {
          Object next = iterator.next();
          values.add(next);
        }
      } else {
        values.add(object);
      }
    }
    List<String> holders = new ArrayList<>();
    for (Object value : values) {
      addConditionValue(auth, map, value.toString());
      holders.add(Constants.PLACE_HOLDER);
    }
    String inPattern = StringUtils.join(holders, ",");
    StringBuilder newVal = new StringBuilder();
    newVal.append(" and ")
        .append(Constants.ALIAS_STRING)
        .append(".")
        .append(field.getFieldName())
        .append(" ")
        .append(getOprtTypeValue(auth.getOprtType()))
        .append(" (")
        .append(inPattern)
        .append(")");
    addConditionSQL(newVal.toString(), map);
  }


  /**
   * 解析等于或不等于的情况，并存储值
   * 支持：预制值和固定值
   *
   * @param auth
   * @param field
   * @param map
   */
  @SuppressWarnings("unchecked")
  private static void analysisEqConditions(DataViewAuthHorizontalEntity auth, DataViewFieldEntity field, Map<SQLCorrelationEnum, Object> map) {
    String newVal = String.format(" and %s.%s %s %s", Constants.ALIAS_STRING, field.getFieldName(), getOprtTypeValue(auth.getOprtType()), Constants.PLACE_HOLDER);
    addConditionSQL(newVal, map);
    if (auth.getParamSourceType() == 3) {//预制值情况
      Map<String, Object> presets = (Map<String, Object>) map.get(SQLCorrelationEnum.PRESETS);
      String key = auth.getField().getFieldName() + "|DataViewAuthHorizontalEntity";
      Object object = presets.get(key);
      addConditionValue(auth, map, object);
      return;
    }
    //固定值情况
    Set<DataViewAuthHorizontalRelationEntity> authRelations = auth.getAuthRelations();
    Validate.notEmpty(authRelations, "有横向权限未设置固定值");
    for (DataViewAuthHorizontalRelationEntity e : authRelations) {
      addConditionValue(auth, map, e.getAuthRelationIds());
    }
  }

  /**
   * 存储各个解析步骤中的参数值
   * 设置参数值：这里分为3种情况
   * 1.当参数传递设置为固定值或者外部传入时，但是权限关系集元素为1个
   * 2.当参数传递设置为固定值或者外部传入时，但是权限关系集元素为多个
   * 3.当参数传递设置为预制值时,authRelations为空
   *
   * @param auth
   * @param map
   * @param idValue 指的是关系权限集DataViewAuthHorizontalRelationEntity中的authRelationIds属性
   */
  @SuppressWarnings("unchecked")
  private static void addConditionValue(DataViewAuthHorizontalEntity auth, Map<SQLCorrelationEnum, Object> map, Object idValue) {
    SQLParamModel spm = new SQLParamModel();
    spm.setParamName(auth.getField().getFieldName());
    spm.setTransferType(auth.getParamSourceType());
    spm.setParamType(auth.getParamType());
    spm.setParamKey(auth.getParamKey());
    //这里设置参数的下标计数
    setCounter(spm, map);
    spm.setValue(idValue);
    spm.setSource(DataViewAuthHorizontalEntity.class);
    Set<SQLParamModel> values = (LinkedHashSet<SQLParamModel>) map.get(SQLCorrelationEnum.AUTH_HORIZONTAL_VALUES);
    values.add(spm);
  }

  /**
   * 存储各个解析步骤中返回的sql
   *
   * @param conditionSQL
   * @param map
   */
  private static void addConditionSQL(String conditionSQL, Map<SQLCorrelationEnum, Object> map) {
    String oldVal = map.remove(SQLCorrelationEnum.RESULT_SQL).toString();
    String newVal = oldVal + conditionSQL;
    map.put(SQLCorrelationEnum.RESULT_SQL, newVal);
  }

  private static void setCounter(SQLParamModel spm, Map<SQLCorrelationEnum, Object> map) {
    int count = (int) map.remove(SQLCorrelationEnum.COUNTER);
    spm.setIndex(count++);
    map.put(SQLCorrelationEnum.COUNTER, count);
  }

  private static String getOprtTypeValue(String oprtType) {
    return InvokeParams.OprtCharType.getOprtValue(oprtType);
  }
}
