package com.bizunited.platform.core.repository.dataview.analysis;

import com.bizunited.platform.core.common.enums.SQLCorrelationEnum;
import com.bizunited.platform.core.entity.DataViewSystemEntity;
import com.bizunited.platform.core.service.dataview.model.SQLParamModel;
import com.bizunited.platform.core.service.invoke.model.InvokeOperations;
import com.bizunited.platform.core.service.invoke.model.InvokeParams;
import org.apache.commons.lang3.RegExUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static com.bizunited.platform.core.common.constant.Constants.DEFAULT_QUERY_COVER_CODE;

/**
 * 用于数据视图系统参数部分的参数处理工具类
 * 系统参数只涉及参数值的处理，没有SQL的拼接之类
 * 系统参数传参方式原有：1.外部传入 2.固定值 3.预制值
 * 系统参数书写格式为 {:xxx}
 */
class SQLSystemParamAnalysis {
  /**
   * 通过私有构造函数，让该类暂时不对外开发
   * 因为后期可能会对该类进行重新设计
   */
  private SQLSystemParamAnalysis() {
  }


  @SuppressWarnings("unchecked")
  static void buildSystemParamAnalysis(Set<DataViewSystemEntity> systemFilters, InvokeParams invokeParams, Map<SQLCorrelationEnum, Object> map, String sourceSQL) {
    Validate.isTrue(!CollectionUtils.isEmpty(map), "SQL分析时，上下文信息不能为空，请检查!!");
    if (CollectionUtils.isEmpty(systemFilters)) {
      map.put(SQLCorrelationEnum.RESULT_SQL, sourceSQL);
      return;
    }

    Pattern pattern = Pattern.compile(Constants.SYSTEM_PARAM_PATTERN);
    for (int i = 0; i < systemFilters.size(); i++) {
      //匹配系统参数传参
      Matcher m = pattern.matcher(sourceSQL);
      String re = null;
      if (m.find()) {
        //截取符合规则的系统参数字符串
        re = m.group().substring(2, m.group().length() - 1);
      }
      Validate.notBlank(re, "根据正则表达式，没有获取到系统参数信息");
      String systemParamName = re;
      Optional<DataViewSystemEntity> op = systemFilters.stream().filter(e -> StringUtils.equals(e.getParamName(), systemParamName)).findFirst();
      DataViewSystemEntity systemEntity = op.orElse(null);
      Validate.notNull(systemEntity, "做系统参数分析时，未能查询到系统参数信息，请检查！！");
      Validate.isTrue(systemEntity.getParamSourceType() != null, "做系统参数分析时，未能获取到参数来源信息，请检查！！");
      Set<SQLParamModel> values = (LinkedHashSet<SQLParamModel>) map.get(SQLCorrelationEnum.SYSTEM_PARAM_VALUES);

      String searchStr = "\\{\\:" + systemParamName + "\\}";
      //根据上面的信息，辨别参数来源
      switch (systemEntity.getParamSourceType()) {
        case 1: //外部传入
          InvokeOperations model = invokeParams.get(systemParamName);
          Validate.notNull(model, "外部传入的参数不能为空，请检查!!");
          Object compareValue = model.getCompareValue();
          Validate.notNull(compareValue, "外部传入的参数值不能为空，请检查!!");
          //处理SQL
          sourceSQL = RegExUtils.replaceFirst(sourceSQL, searchStr, Constants.PLACE_HOLDER);
          //开始构造参数
          SQLParamModel paramModel = new SQLParamModel();
          paramModel.setValue(compareValue);
          paramModel.setParamName(systemEntity.getParamName());
          paramModel.setParamType(systemEntity.getParamType());
          paramModel.setTransferType(1);
          paramModel.setMappingType(model.getMappingType());
          setCouter(paramModel, map);
          values.add(paramModel);
          break;
        case 2: //固定值
          //处理SQL
          sourceSQL = RegExUtils.replaceFirst(sourceSQL, searchStr, Constants.PLACE_HOLDER);
          //开始构造参数
          SQLParamModel paramModel2 = new SQLParamModel();
          paramModel2.setValue(systemEntity.getParamValue());
          paramModel2.setParamName(systemEntity.getParamName());
          paramModel2.setParamType(systemEntity.getParamType());
          paramModel2.setTransferType(2);
          paramModel2.setMappingType(systemEntity.getParamType());
          setCouter(paramModel2, map);
          values.add(paramModel2);
          break;
        case 3: //预置值
          // 如果是预置值，则要考虑预置值是集合的情况，如果预置值是一个集合，则遍历集合构造基础数据类型的sql参数，并用小括号包围占位符
          // 如果是预置值，构建sql时，应当使用 in 操作符，如：select * from a where a.orgId in {:orgIds}
          Map<String, Object> presetMap = (Map<String, Object>) map.get(SQLCorrelationEnum.PRESETS);
          String key = systemEntity.getParamName() + "|DataViewSystemEntity";
          Object value = presetMap.get(key);
          List<Object> paramValues = new ArrayList<>();
          boolean isIterable = false;
          if(value != null && Iterable.class.isAssignableFrom(value.getClass())) {
            isIterable = true;
            Iterable<?> iterable = (Iterable<?>) value;
            Iterator<?> iterator = iterable.iterator();
            if(!iterator.hasNext()) {
              paramValues.add(DEFAULT_QUERY_COVER_CODE);
            }
            while (iterator.hasNext()) {
              Object next = iterator.next();
              paramValues.add(next);
            }
          } else {
            paramValues.add(value);
          }
          List<String> holders = new ArrayList<>();
          //开始构造参数
          for (Object paramValue : paramValues) {
            holders.add(Constants.PLACE_HOLDER);
            SQLParamModel paramModel3 = new SQLParamModel();
            paramModel3.setValue(paramValue);
            paramModel3.setParamName(systemEntity.getParamName());
            paramModel3.setParamKey(systemEntity.getParamName());
            paramModel3.setTransferType(3);
            paramModel3.setSource(DataViewSystemEntity.class);
            setCouter(paramModel3, map);
            values.add(paramModel3);
          }
          String placeHolder = StringUtils.join(holders, ",");
          if(isIterable) {
            placeHolder = "(" + placeHolder + ")";
          }
          //处理SQL
          sourceSQL = RegExUtils.replaceFirst(sourceSQL, searchStr, placeHolder);
          break;
        default:
          break;
      }
    }
    map.put(SQLCorrelationEnum.RESULT_SQL, sourceSQL);
  }

  private static void setCouter(SQLParamModel spm, Map<SQLCorrelationEnum, Object> map) {
    int count = (int) map.remove(SQLCorrelationEnum.COUNTER);
    spm.setIndex(count++);
    map.put(SQLCorrelationEnum.COUNTER, count);
  }
}
