package com.bizunited.platform.core.service.dataview.internal;

import com.bizunited.platform.core.entity.DataViewEntity;
import com.bizunited.platform.core.entity.DataViewFieldEntity;
import com.bizunited.platform.core.entity.DataViewFilterEntity;
import com.bizunited.platform.core.repository.dataview.DataViewFilterRepository;
import com.bizunited.platform.core.service.dataview.DataViewFilterService;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.List;
import java.util.Optional;
import java.util.Set;

/**
 * DataViewFilterEntity业务模型的服务层接口实现
 * @author saturn
 */
@Service("DataViewFilterEntityServiceImpl")
public class DataViewFilterServiceImpl implements DataViewFilterService {
  @Autowired
  private DataViewFilterRepository dataViewFilterEntityRepository;
  /**
   * 在创建一个新的DataViewFilterEntity模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  public void createValidation(DataViewFilterEntity dataViewFilterEntity) {
    Validate.notNull(dataViewFilterEntity , "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(dataViewFilterEntity.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    dataViewFilterEntity.setId(null);
    Validate.notBlank(dataViewFilterEntity.getOpType(), "操作符类型不能为空！");
    Validate.notNull(dataViewFilterEntity.getSortIndex(), "字段排序不能为空！");
    Validate.notNull(dataViewFilterEntity.getNullable(), "是否可以不进行选择不能为空！");
    Validate.notBlank(dataViewFilterEntity.getDisplayName(), "显示中文信息不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况） 
    Validate.isTrue(dataViewFilterEntity.getOpType() == null || dataViewFilterEntity.getOpType().length() < 255 , "显示中文信息,填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(dataViewFilterEntity.getDisplayName() == null || dataViewFilterEntity.getDisplayName().length() < 64 , "显示中文信息,填入值超过了限定长度(64)，请检查!");
  }
  @Transactional
  @Override
  public DataViewFilterEntity create(DataViewFilterEntity dataViewFilterEntity , boolean ignoreValidate) { 
    if(!ignoreValidate) {
      this.createValidation(dataViewFilterEntity);
    }
    
    
    // ===============================
    //  和业务有关的验证填写在这个区域    
    // ===============================
    
    this.dataViewFilterEntityRepository.save(dataViewFilterEntity);
    
    return dataViewFilterEntity;
  }
  @Transactional
  @Override
  public DataViewFilterEntity update(DataViewFilterEntity dataViewFilterEntity) { 
    this.updateValidation(dataViewFilterEntity);
    // ===================基本信息
    String currentId = dataViewFilterEntity.getId();
    Optional<DataViewFilterEntity> op = this.dataViewFilterEntityRepository.findById(currentId);
    DataViewFilterEntity currentDataViewFilterEntity = op.orElse(null);
    Validate.notNull(currentDataViewFilterEntity,"未发现指定的原始模型对象信息");

    // 开始重新赋值，首先是一般对象，updateable = true形式的
    currentDataViewFilterEntity.setOpType(dataViewFilterEntity.getOpType());
    currentDataViewFilterEntity.setSortIndex(dataViewFilterEntity.getSortIndex());
    currentDataViewFilterEntity.setNullable(dataViewFilterEntity.getNullable());
    currentDataViewFilterEntity.setDisplayName(dataViewFilterEntity.getDisplayName());
    
    this.dataViewFilterEntityRepository.save(currentDataViewFilterEntity);
    // =============
    // ManyToMany 结构的关联关系，需要开发人员自行处理
    // =============
    return currentDataViewFilterEntity;
  }
  /**
   * 在更新一个已有的DataViewFilterEntity模型对象之前，检查对象各属性的正确性，其id属性必须有值
   */
  public void updateValidation(DataViewFilterEntity dataViewFilterEntity) { 
    Validate.isTrue(!StringUtils.isBlank(dataViewFilterEntity.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");
    
    // 基础信息判断，基本属性，需要满足not null 且 updateable == true
    Validate.notBlank(dataViewFilterEntity.getOpType(),  "操作符类型不能为空！");
    Validate.notNull(dataViewFilterEntity.getSortIndex(), "字段排序不能为空！");
    Validate.notNull(dataViewFilterEntity.getNullable(), "是否可以不进行选择不能为空！");
    Validate.notBlank(dataViewFilterEntity.getDisplayName(), "显示中文信息不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(dataViewFilterEntity.getOpType() == null || dataViewFilterEntity.getOpType().length() < 255 , "显示中文信息,填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(dataViewFilterEntity.getDisplayName() == null || dataViewFilterEntity.getDisplayName().length() < 64 , "显示中文信息,填入值超过了限定长度(64)，请检查!");
  } 
  @Override
  public List<DataViewFilterEntity> findByDataView(String dataViewId) { 
    if(StringUtils.isBlank(dataViewId)) { 
      return Lists.newArrayList();
    }
    return this.dataViewFilterEntityRepository.findByDataView(dataViewId);
  }
  @Override
  public List<DataViewFilterEntity> findByField(String fieldId) { 
    if(StringUtils.isBlank(fieldId)) { 
      return Lists.newArrayList();
    }
    return this.dataViewFilterEntityRepository.findByField(fieldId);
  }
  @Override
  public DataViewFilterEntity findDetailsById(String id) { 
    if(StringUtils.isBlank(id)) { 
      return null;
    }
    return this.dataViewFilterEntityRepository.findDetailsById(id);
  }
  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id , "进行删除时，必须给定主键信息!!");
    Optional<DataViewFilterEntity> op = this.dataViewFilterEntityRepository.findById(id);
    op.ifPresent(dataViewFilterEntity -> this.dataViewFilterEntityRepository.delete(dataViewFilterEntity));
  }
  @Override
  @Transactional
  public void deleteByField(String fieldId) {
    Validate.notBlank(fieldId , "进行删除时，必须给定Field主键信息!!");
    List<DataViewFilterEntity> filters = this.dataViewFilterEntityRepository.findByField(fieldId);
    if(!CollectionUtils.isEmpty(filters)) { 
      for(DataViewFilterEntity filter : filters) {
        this.dataViewFilterEntityRepository.deleteById(filter.getId());
      }
    }
  }

  @Override
  @Transactional
  public void create(Set<DataViewFilterEntity> filters, Set<DataViewFieldEntity> fields, DataViewEntity dataViewEntity) {
    Validate.notNull(dataViewEntity, "数据视图为空，请检查！");
    if(!CollectionUtils.isEmpty(filters)) {
      for(DataViewFilterEntity filter : filters) {
        this.createValidation(filter);
        filter.setDataView(dataViewEntity);
      }
    }

    if(filters != null){
      //这里需要把字段id信息设置到filter中
      filters.forEach( filter -> {
        Optional<DataViewFieldEntity> optional = fields.stream().filter(field -> StringUtils.equals(field.getFieldName(), filter.getField().getFieldName())).findFirst();
        Validate.isTrue(optional.isPresent(), "过滤字段中，未发现字段信息，请检查!!");
        filter.setField(optional.get());
      });
      this.dataViewFilterEntityRepository.saveAll(filters);
    }
  }
} 
