package com.bizunited.platform.core.service.internal;

import com.bizunited.platform.core.common.utils.ChineseCharUtil;
import com.bizunited.platform.core.entity.DictCategoryEntity;
import com.bizunited.platform.core.entity.UserEntity;
import com.bizunited.platform.core.repository.DictCategoryRepository;
import com.bizunited.platform.core.service.DictCategoryService;
import com.bizunited.platform.rbac.server.service.UserService;
import com.bizunited.platform.rbac.server.vo.UserVo;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.util.Date;
import java.util.Optional;
import java.util.Set;

/**
 * DictCategoryServiceImpl
 *
 * @description:
 * @author: yanwe
 * @date: 11/Jun/2019 17:34
 */

@Service("DictCategoryServiceImpl")
public class DictCategoryServiceImpl implements DictCategoryService {

  @Autowired
  private DictCategoryRepository dictCategoryRepository;
  @Autowired
  private UserService userService;

  @Override
  @Transactional
  public DictCategoryEntity create(DictCategoryEntity entity, String parentId) {
    Validate.notNull(entity,"字典分类不能为空！");
    Validate.isTrue(StringUtils.isBlank(entity.getId()),"新增时不能填入Id!");
    Validate.notBlank(entity.getCateName(),"字典分类名称不能为空！");
    Validate.notBlank(entity.getCateCode(),"字典分类编码不能为空！");
    Validate.isTrue(!ChineseCharUtil.hasChinese(entity.getCateCode()),"字典分类编码不能含有中文！");
    DictCategoryEntity existCode = dictCategoryRepository.findByCateCode(entity.getCateCode());
    Validate.isTrue(null == existCode,"字典分类编码重复，请检查！");
    //获取当前登录账户
    UserEntity userEntity = this.getAccount();
    //其他参数
    entity.setCreateUser(userEntity);
    entity.setCreateTime(new Date());
    entity.setModifyUser(userEntity);
    entity.setModifyTime(new Date());
    // 父级
    if(StringUtils.isNotBlank(parentId)){
      Optional<DictCategoryEntity> op = dictCategoryRepository.findById(parentId);
      DictCategoryEntity parent = op.orElse(null);
      Validate.notNull(parent,"未查询到传入父类字典分类！");
      entity.setParentCategory(parent);
    }
    return dictCategoryRepository.save(entity);
  }

  @Override
  @Transactional
  public DictCategoryEntity update(DictCategoryEntity entity) {
    Validate.notNull(entity,"字典分类不能为空！");
    Validate.notBlank(entity.getId(),"新增时不能填入Id!");
    Optional<DictCategoryEntity> op = dictCategoryRepository.findById(entity.getId());
    DictCategoryEntity existId = op.orElse(null);
    Validate.notNull(existId,"未查询到当前字典分类！");
    Validate.notBlank(entity.getCateName(),"字典分类名称不能为空！");
    //获取当前登录账户
    UserEntity userEntity = this.getAccount();
    //修改参数
    existId.setCateName(entity.getCateName());
    existId.setCateDesc(entity.getCateDesc());
    existId.setModifyUser(userEntity);
    existId.setModifyTime(new Date());
    return dictCategoryRepository.save(existId);
  }

  @Override
  public void bind(String childId, String parentId) {
    Validate.notBlank(childId,"子级字典分类ID不能为空！");
    Optional<DictCategoryEntity> op = dictCategoryRepository.findById(childId);
    DictCategoryEntity child = op.orElse(null);
    Validate.notNull(child,"未查询到子级字典分类！");

    child.setParentCategory(null);
    if(StringUtils.isNotEmpty(parentId)){
      Validate.notBlank(parentId,"父级字典分类ID不能为空！");
      Optional<DictCategoryEntity> optional = dictCategoryRepository.findById(parentId);
      Validate.isTrue(optional.isPresent(),"未查询到父级字典分类！");
      DictCategoryEntity parent = optional.get();
      child.setParentCategory(parent);
    }
    dictCategoryRepository.saveAndFlush(child);
  }

  /**
   * 获取当前登录账户
   * @return
   */
  private UserEntity getAccount(){
    //获取当前登录账户
    SecurityContext securityContext = SecurityContextHolder.getContext();
    Validate.notNull(securityContext , "未发现任何用户权限信息!!");
    Authentication authentication = securityContext.getAuthentication();
    Validate.notNull(authentication , "未发现任何用户登录信息!!");
    UserVo userVo = userService.findByAccount(authentication.getName());
    Validate.notNull(userVo,"未获取到当前登录账号！");
    UserEntity userEntity = new UserEntity();
    userEntity.setId(userVo.getId());
    return userEntity;
  }

  @Override
  public DictCategoryEntity findById(String id) {
    if(StringUtils.isBlank(id)) {
      return null;
    }
    Optional<DictCategoryEntity> op = dictCategoryRepository.findById(id);
    return op.orElse(null);
  }

  @Override
  public DictCategoryEntity findDetailsById(String id) {
    if(StringUtils.isBlank(id)) {
      return null;
    }
    return this.dictCategoryRepository.findDetailsById(id);
  }

  @Override
  public DictCategoryEntity findByCode(String code) {
    if(StringUtils.isBlank(code)) {
      return null;
    }
    return this.dictCategoryRepository.findByCateCode(code);
  }

  @Override
  public Set<DictCategoryEntity> findAll() {
    return dictCategoryRepository.findByTree();
  }

}
