package com.bizunited.platform.core.service.internal;

import com.bizunited.platform.core.entity.DictEntity;
import com.bizunited.platform.core.entity.DictItemEntity;
import com.bizunited.platform.core.repository.DictItemRepository;
import com.bizunited.platform.core.repository.DictRepository;
import com.bizunited.platform.core.service.DictItemService;
import com.bizunited.platform.core.service.DictService;
import com.bizunited.platform.core.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

@Service("DictItemServiceImpl")
public class DictItemServiceImpl implements DictItemService {

  private static final String ERROR_MESS = "字典明细ID不能为空！";
  @Autowired 
  private DictRepository dictRepository;
  @Autowired 
  private DictItemRepository dictItemRepository;
  @Autowired
  private DictService dictService;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;

  @Override
  public List<DictItemEntity> findItemsByCode(String dictCode) {
    DictEntity dictEntity = dictRepository.findByDictCode(dictCode);
    if (dictEntity == null) {
      return Lists.newArrayList();
    }
    return dictItemRepository.findByDictEntityOrderByDictSort(dictEntity);
  }

  @Override
  @Transactional
  public Set<DictItemEntity> save(String dictCode, Set<DictItemEntity> items) {
    DictEntity dict = dictService.findByDictCode(dictCode);
    Validate.notNull(dict, "未找到该编码匹配字典！");
    List<DictItemEntity> dictItemsList = dictItemRepository.findByDictEntity(dict);
    //参数校验
    this.saveValidation(dictCode, items, dictItemsList);
    // TODO: 2020/1/10 目前的业务只有新增的情况，更新删除待后期开发(如果后期业务需要)
    //可能的更新 更新Value对应的key(显示字段)
    Set<DictItemEntity> needInsertDictItems = new HashSet<>();
    Set<DictItemEntity> needUpdateDictItems = new HashSet<>();
    Set<DictItemEntity> needDeleteDictItems = new HashSet<>();
    nebulaToolkitService.collectionDiscrepancy(items, dictItemsList, DictItemEntity::getDictValue, needDeleteDictItems, needUpdateDictItems, needInsertDictItems);

    int itemNum;
    if (CollectionUtils.isEmpty(needUpdateDictItems)){
      itemNum = items.size() + dictItemsList.size();
    }else {
      itemNum = items.size() - needUpdateDictItems.size() + dictItemsList.size();
    }
    dict.setItemNum(itemNum);
    dict.setDictItemEntities(null);
    dictRepository.saveAndFlush(dict);
    dictItemRepository.flush();

    needInsertDictItems.stream().forEach(dictItem -> {
      dictItem.setId(null);
      dictItem.setCreateTime(new Date());
      dictItem.setDictSort(dictItem.getDictSort() == null ? 1 : dictItem.getDictSort());
      dictItem.setDictEntity(dict);
      dictItem.setDictItemStatus(dictItem.getDictItemStatus() == null ? Boolean.TRUE : dictItem.getDictItemStatus());
    });
    dictItemRepository.saveAll(needInsertDictItems);
    return needInsertDictItems;
  }

  @Override
  @Transactional
  public DictItemEntity create(String dictCode, DictItemEntity dictItem) {
    //参数校验
    this.createValidation(dictCode, dictItem);

    DictEntity dict = dictService.findByDictCode(dictCode);
    dict.setItemNum(dict.getItemNum() + 1);
    dictRepository.save(dict);

    dictItem.setCreateTime(new Date());
    dictItem.setDictSort(dictItem.getDictSort() == null ? 1 : dictItem.getDictSort());
    dictItem.setDictEntity(dict);
    dictItem.setDictItemStatus(dictItem.getDictItemStatus() == null ? Boolean.TRUE : dictItem.getDictItemStatus());
    return dictItemRepository.save(dictItem);
  }

  /**
   * 新增-参数校验
   * @param dictCode
   * @param dictItem
   */
  private void createValidation(String dictCode, DictItemEntity dictItem) {
    Validate.notBlank(dictCode, "字典编码不能为空，请检查");
    DictEntity dict = dictService.findByDictCode(dictCode);
    Validate.notNull(dict, "未找到该编码匹配字典！");

    Validate.notNull(dictItem, "保存对象不存在!");
    Validate.isTrue(dictItem.getId() == null, "保存数据ID必须为空!");
    Validate.notBlank(dictItem.getDictKey(), "字典名称不能为空，请检查!");
    Validate.notBlank(dictItem.getDictValue(), "字典值不能为空，请检查!");

    long countByDictKey = dictItemRepository.countByDictKeyAndDictId(dictItem.getDictKey(), dict.getId());
    Validate.isTrue(countByDictKey == 0L, "字典名称重复，请检查!");
    long countByDictValue = dictItemRepository.countByDictValueAndDictId(dictItem.getDictValue(), dict.getId());
    Validate.isTrue(countByDictValue == 0L, "字典值重复，请检查!");
  }


  @Override
  @Transactional
  public DictItemEntity enable(String dictItemId) {
    Validate.notBlank(dictItemId, ERROR_MESS);
    DictItemEntity dictItemEntity = dictItemRepository.findDetailById(dictItemId);
    Validate.notNull(dictItemEntity, ERROR_MESS);
    dictItemEntity.setDictItemStatus(true);
    return dictItemRepository.save(dictItemEntity);
  }

  @Override
  @Transactional
  public DictItemEntity disable(String dictItemId) {
    Validate.notBlank(dictItemId, ERROR_MESS);
    DictItemEntity dictItemEntity = dictItemRepository.findDetailById(dictItemId);
    Validate.notNull(dictItemEntity, ERROR_MESS);
    dictItemEntity.setDictItemStatus(false);
    return dictItemRepository.save(dictItemEntity);
  }


  /**
   * 参数校验
   * 重复性验证：value只需要验证传递过来的是否重复，key还需要验证与数据库里是否重复
   * @param dictCode
   * @param items 前端传递过来的
   * @param dictItemsList 数据库里原有的
   */
  private void saveValidation(String dictCode, Set<DictItemEntity> items, List<DictItemEntity> dictItemsList) {
    Validate.notBlank(dictCode, "字典编码不能为空，请检查");
    Collection<DictItemEntity> dictItemCollection = nebulaToolkitService.copyCollectionByWhiteList(dictItemsList, DictItemEntity.class, DictItemEntity.class, HashSet.class, ArrayList.class);
    List<DictItemEntity> oldItems = Lists.newArrayList(dictItemCollection);

    Set<DictItemEntity> needInsertItems = new HashSet<>();
    needInsertItems.addAll(items);
    Map<String, DictItemEntity>  oldItemsMap = oldItems.stream().collect(Collectors.toMap(DictItemEntity::getDictValue, e -> e));

    items.forEach(item -> {
      //能找到就是更新
      DictItemEntity dictItemEntity = oldItemsMap.get(item.getDictValue());
      if (null != dictItemEntity){
        //移除需要更新的字典明细项
        needInsertItems.remove(item);
        dictItemEntity.setDictKey(item.getDictKey());
      }
    });

    Set<String> dictItemValues = new HashSet<>();
    Set<String> dictItemKeys = new HashSet<>();

    items.forEach(item -> {
      Validate.notNull(item, "保存对象不存在!");
      Validate.notBlank(item.getDictKey(), "字典名称不能为空，请检查!");
      Validate.notBlank(item.getDictValue(), "字典值不能为空，请检查!");
      //判断前端传递过来的字典Value和字典Key是否重复
      Validate.isTrue(!dictItemValues.contains(item.getDictValue()), "字典值重复，请检查!");
      Validate.isTrue(!dictItemKeys.contains(item.getDictKey()), "字典名称重复，请检查!");
      dictItemValues.add(item.getDictValue());
      dictItemKeys.add(item.getDictKey());
    });

    needInsertItems.addAll(oldItems);
    Set<String> dbDictItemKeys = new HashSet<>();
    needInsertItems.forEach(item -> {
      Validate.isTrue(!dbDictItemKeys.contains(item.getDictKey()), "字典名称重复，请检查!");
      dbDictItemKeys.add(item.getDictKey());
    });
  }
}
