package com.bizunited.platform.core.service.internal;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.core.common.utils.ChineseCharUtil;
import com.bizunited.platform.core.entity.RemoteServiceAddressEntity;
import com.bizunited.platform.core.repository.RemoteServiceAddressRepository;
import com.bizunited.platform.core.service.RemoteServiceAddressService;
import com.bizunited.platform.core.service.RemoteServiceService;
import com.google.common.collect.Sets;
import feign.Client;
import feign.Request;
import feign.Request.HttpMethod;
import feign.Response;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

/**
 * RemoteServiceAddressServiceImpl
 *
 * @description:
 * @author: yanwe
 * @date: 06/May/2019 10:23
 */
@Service("RemoteServiceAddressServiceImpl")
public class RemoteServiceAddressServiceImpl implements RemoteServiceAddressService {


  private static final String REQUEST_METHOD = "RequestMethod";
  private static final String INTERFACE = "Interface";
  private static final String HEAD = "Head";
  private static final String REQUEST_PARAMS = "RequestParams";
  private static final String REQUEST_BODY = "RequestBody";
  private static final String ADDRESS_E_ERROR = "入参远程服务实体JSON不能为空";
  private static final String ADDRESS_P_ERROR = "入参远程服务参数JSON不能为空";
  private static final String ADDRESS_C_ERROR = "参数执行转化时发生错误，请检查！";
  private static final String ADDRESS_R_ERROR = "传入远端调用源不存在，请检查！";

  @Autowired
  private RemoteServiceAddressRepository  remoteServiceAddressRepository;
  @Autowired
  private RemoteServiceService remoteServiceService;
  
  @Override
  public Page<RemoteServiceAddressEntity> findByConditions(String code, String alias, Pageable pageable) {
    Validate.notNull(pageable, "查询分页信息不能为空!");
    Map<String, Object> conditions = new HashMap<>();
    if (!StringUtils.isBlank(code)) {
      conditions.put("code", code);
    }
    if (!StringUtils.isBlank(alias)) {
      conditions.put("alias", alias);
    }
    return remoteServiceAddressRepository.findByConditions(pageable, conditions);
  }

  @Override
  public void valid(String entityJson, String testJsonStr) {
    Validate.notBlank(entityJson,ADDRESS_E_ERROR);
    this.validJson(entityJson, testJsonStr);
  }
  
  /**
   * 验证并根据entityJson的结构生成一个RemoteServiceAddressEntity类的实例
   * @param entityJson
   * @param jsonStr
   * @return
   */
  private RemoteServiceAddressEntity validJson(String entityJson, String jsonStr) {
    Validate.notBlank(jsonStr,ADDRESS_P_ERROR);
    RemoteServiceAddressEntity entity;
    JSONObject testJson;
    try{
      entity = JSON.parseObject(entityJson,RemoteServiceAddressEntity.class);
      testJson = JSON.parseObject(jsonStr);
    }catch (RuntimeException e){
      throw new IllegalArgumentException(ADDRESS_C_ERROR,e);
    }
    this.validEntity(entity, true);
    this.validConnection(entity, testJson);
    return entity;
  }

  @Override
  @Transactional
  public RemoteServiceAddressEntity create(String entityJson, String testJsonStr) {
    Validate.notBlank(entityJson,ADDRESS_E_ERROR);
    RemoteServiceAddressEntity entity = this.validJson(entityJson, testJsonStr);
    entity.setCreateDate(new Date());
    entity.setAddressStatus(2);
    return remoteServiceAddressRepository.save(entity);
  }

  @Override
  @Transactional
  public RemoteServiceAddressEntity create(RemoteServiceAddressEntity remoteServiceAddressEntity) {
    this.validEntity(remoteServiceAddressEntity,true);
    return this.remoteServiceAddressRepository.save(remoteServiceAddressEntity);
  }

  @Override
  @Transactional
  public RemoteServiceAddressEntity update(String entityJson, String testJsonStr) {
    Validate.notBlank(entityJson,ADDRESS_E_ERROR);
    Validate.notBlank(testJsonStr,ADDRESS_P_ERROR);
    RemoteServiceAddressEntity entity;
    JSONObject testJson;
    try{
      entity = JSON.parseObject(entityJson,RemoteServiceAddressEntity.class);
      testJson = JSON.parseObject(testJsonStr);
    }catch (Exception e){
      throw new IllegalArgumentException(ADDRESS_C_ERROR,e);
    }
    this.validEntity(entity, false);
    this.validConnection(entity, testJson);
    // 修改
    Optional<RemoteServiceAddressEntity> op = remoteServiceAddressRepository.findById(entity.getId());
    RemoteServiceAddressEntity existEntity = op.orElse(null);
    Validate.notNull(existEntity,"未能查询到远程服务信息!!");

    existEntity.setProtocolType(entity.getProtocolType());
    existEntity.setConnectionType(entity.getConnectionType());
    existEntity.setAddress(entity.getAddress());
    existEntity.setAddressDesc(entity.getAddressDesc());
    return remoteServiceAddressRepository.save(existEntity);
  }

  @Override
  @Transactional
  public RemoteServiceAddressEntity enable(String addressId) {
    Validate.notBlank(addressId, "传入远端调用源ID不能为空!");
    Optional<RemoteServiceAddressEntity> op = remoteServiceAddressRepository.findById(addressId);
    RemoteServiceAddressEntity existEntity = op.orElse(null);
    Validate.notNull(existEntity,ADDRESS_R_ERROR);

    Validate.isTrue(existEntity.getAddressStatus() == 2, "启用的远端调用源状态必须为未启用，请检查！");
    existEntity.setAddressStatus(1);
    return remoteServiceAddressRepository.save(existEntity);
  }

  @Override
  @Transactional
  public RemoteServiceAddressEntity disable(String addressId) {
    Validate.notBlank(addressId, "传入远端调用源ID不能为空!");
    Optional<RemoteServiceAddressEntity> op = remoteServiceAddressRepository.findById(addressId);
    RemoteServiceAddressEntity existEntity = op.orElse(null);
    Validate.notNull(existEntity,ADDRESS_R_ERROR);

    Validate.isTrue(existEntity.getAddressStatus() == 1, "禁用的远端调用源状态必须为启用，请检查！");
    existEntity.setAddressStatus(0);
    //禁用远端地址的同时，禁用它下面的所有服务
    remoteServiceService.updateRemoteService(addressId, 0);
    return remoteServiceAddressRepository.save(existEntity);
  }

  @Override
  public RemoteServiceAddressEntity findByCode(String code) {
    return remoteServiceAddressRepository.findByCode(code);
  }

  /**
   * 新增/修改验证远端源管理数据
   *
   * @param entity
   */
  private void validEntity(RemoteServiceAddressEntity entity, Boolean isCreate) {
    Validate.notNull(entity, "传入远端地址数据不能为空！");
    Validate.notBlank(entity.getCode(), "远端调用源编码不能为空，请检查！");
    Validate.notBlank(entity.getAlias(), "远端调用源别名不能为空，请检查！");
    Validate.isTrue(!ChineseCharUtil.hasChinese(entity.getAlias()), "新增远端数据源别名不能含有中文！");
    Validate.isTrue(!ChineseCharUtil.hasChinese(entity.getCode()), "新增远端数据源编码不能含有中文！");
    Validate.notNull(entity.getProtocolType(), "远端调用源协议类型不能为空，请检查！");
    Validate.notNull(entity.getConnectionType(), "远端调用源连接类型不能为空，请检查！");
    Validate.notNull(entity.getAddress(), "远端调用源连接地址不能为空，请检查！");
    Validate.isTrue(!entity.getAddress().endsWith("/"), "连接地址不能以 / 号结尾，请移除！");
    Validate.notNull(entity.getAddressStatus(), "远端调用源状态不能为空，请检查！");
    // 新增
    if (Boolean.TRUE.equals(isCreate)) {
      // 验重
      RemoteServiceAddressEntity existCodes =
          remoteServiceAddressRepository.findByCode(entity.getCode());
      RemoteServiceAddressEntity existAliases =
          remoteServiceAddressRepository.findByAlias(entity.getAlias());
      Validate.isTrue(StringUtils.isBlank(entity.getId()), "新增的远端调用源ID必须为空！");
      Validate.isTrue(null == existAliases, "新增远端数据源别名重复，请更换别名！");
      Validate.isTrue(null == existCodes, "新增远端数据源编码重复，请更换编码！");
    } else {
      // 修改
      Validate.notBlank(entity.getId(), "修改的远端调用源ID不能为空！");
      Optional<RemoteServiceAddressEntity> op = remoteServiceAddressRepository.findById(entity.getId());
      RemoteServiceAddressEntity existEntity = op.orElse(null);
      Validate.notNull(existEntity,ADDRESS_R_ERROR);

      Validate.isTrue(existEntity.getAddressStatus() == 2, "只能对未启用状态的远程调用源做修改！");
      Validate.isTrue(existEntity.getCode().equals(entity.getCode()), "修改时不能修改远端调用源编号！");
      Validate.isTrue(existEntity.getAlias().equals(entity.getAlias()), "修改时不能修改远端调用源别名！");
    }
  }

  /*
   * <code>
   * 测试接口数据格式：
   * {
   *  "RequestMethod":"POST"    ---测试接口方法，POST/GET
   *  "Interface":"/uuuu",      ---测试接口（如果远端源类型为自定义，则必填，如果为spring cloud形式，则无需填写）
   *  "Head": {
   *    "a":"b",
   *    ...
   *  },
   *  "RequestParams": {
   *    "a":"b",
   *    ...
   *  },
   *  "RequestBody": {...}      ---测试请求体，格式为JSON
   *  }
   *  </code>
   **/

  /**
   * 验证新增远端数据源测试数据
   *
   * @param testJson
   */
  private void validConnection(RemoteServiceAddressEntity entity, JSONObject testJson) {
    Validate.notNull(testJson, "测试接口数据不能为空！");
    Validate.notBlank(testJson.getString(REQUEST_METHOD), "测试接口方法不能为空！");
    // 如果有接口地址，必须以'/'开始
    if (StringUtils.isNotEmpty(testJson.getString(INTERFACE))) {
      Validate.isTrue(testJson.getString(INTERFACE).startsWith("/"), "接口地址若不为空，必须以/号开始");
    }
    Client.Default defaultClient = new Client.Default(null, null);
    // 建立Request
    // 方法
    String method = testJson.getString(REQUEST_METHOD);
    // 路径
    String url =
        StringUtils.isEmpty(testJson.getString(INTERFACE))
            ? entity.getAddress()
            : entity.getAddress() + testJson.getString(INTERFACE);
    // HEAD
    Map<String, Collection<String>> headers = new HashMap<>();
    if (null != testJson.getJSONObject(HEAD)) {
      JSONObject heads = testJson.getJSONObject(HEAD);
      heads.keySet().forEach(o -> headers.put(o, Sets.newHashSet(heads.getString(o))));
    }
    // requestParams
    if (null != testJson.getJSONObject(REQUEST_PARAMS)) {
      JSONObject requestParams = testJson.getJSONObject(REQUEST_PARAMS);
      Set<String> paramSet = new HashSet<>();
      for(String key : requestParams.keySet()){
        paramSet.add(String.format("%s=%s",key,requestParams.getString(key)));
      }
      String paraUrl = String.format("?%s",StringUtils.join(paramSet,"&"));
      url = url.concat(paraUrl);
    }
    // requestBody
    byte[] body = new byte[] {};
    JSONObject bodyJson = testJson.getJSONObject(REQUEST_BODY);
    if (null != bodyJson) {
      String bodyStr = bodyJson.toJSONString();
      body = bodyStr.getBytes(StandardCharsets.UTF_8);
    }


    Request.Options options = new Request.Options(18000, 18000);
    HttpMethod httpMethod = HttpMethod.valueOf(method);
    Request request = Request.create(httpMethod, url, headers, body, Charset.defaultCharset());
    try {
      Response response = defaultClient.execute(request, options);
      Validate.isTrue(response.status() == HttpStatus.OK.value(), String.format("连接失败，返回编码：%s,原因：%s", response.status(), response.reason()));
    } catch (IOException e) {
      throw new IllegalArgumentException(e.getMessage() , e);
    }
  }
}
