package com.bizunited.platform.core.service.internal;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.core.entity.RemoteServiceEntity;
import com.bizunited.platform.core.repository.RemoteServiceRepository;
import com.bizunited.platform.core.service.RemoteServiceService;
import com.bizunited.platform.core.service.file.NebulaFileService;
import com.google.common.collect.Sets;
import feign.Client;
import feign.Request;
import feign.Request.HttpMethod;
import feign.Response;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Random;
import java.util.Set;
import java.util.UUID;

/**
 * RemoteServiceServiceImpl
 * @description: 远端服务源的业务过程实现
 * @author: yanwe
 * @date: 05/May/2019 11:27
 */
@Service("RemoteServiceServiceImpl")
public class RemoteServiceServiceImpl implements RemoteServiceService {
  /*
   * 远端服务JSON格式
   * {
   * //HEAD
   * "Head":[
   * {
   *   "key":"xxx", ---key名称
   *   "desc":"xxx", ---中文描述
   *   "nullable": true ---是否必传
   * },...
   * ],
   *
   * //HeadValue
   * "HeadValue":{
   *   "xxx":"abc",
   *   "yyy":"edf",
   *   ...
   * }
   *
   * //RequestParam
   * "RequestParam":[
   *    {
   *      "fieldName":"xxx", ---key名称
   *      "desc":"xxx", ---中文描述
   *      "nullable": true ---是否必传
   *    },...
   * ],
   *
   * //RequestParamValue
   * "RequestParamValue":
   * {
   *    "xxx":"abc",
   *    "yyy":"edf",
   *    ...
   * }
   *
   * //RequestBody
   * "RequestBody":[
   *
   *    // 一般字段
   *    {
   *        "fieldName":"xxx",
   *        "type":"文本", ---字段类型只能为{文本，数字，时间，布尔值，Object,ObjectArray}
   *        "desc":"xxx", --- 中文说明
   *        "key":"1-1-3"
   *        "nullable": true --- 是否必填
   *    },
   *
   *    // Object/ObjectArray
   *    {
   *        "fieldName":"xxx",
   *        "type":"Object", ---字段类型只能为{文本，数字，时间，布尔值，Object,ObjectArray}
   *        "desc":"xxx", --- 中文说明
   *        "nullable": true --- 是否必填
   *        "key":"1-1-3"
   *        "content": [ --- 如果类型为Object/ObjectArray，则需要有content向下包容
   *            {
   *                "fieldName":"xxx",
   *                "type":"文本",
   *                "desc":"xxx",
   *                "nullable": true
   *            },...
   *        ]
   *    },...
   * ],
   *
   *
   * // RequestBodyValue
   * "RequestBodyValue":{...json...}
   *
   * //ResponseBody
   * "ResponseBody":[
   *
   *  {
   *    "fieldName":"xxx",
   *    "type":"文本", ---字段类型只能为{文本，数字，时间，布尔值，Object,ObjectArray,分页总数据，分页单页数据，分页当前页码，分页最后页码}
   *    "desc":"xxx", --- 中文说明
   *    "nullable": true, --- 是否必填
   *    "selected":true
   *  },
   *
   *  {
   *    "fieldName":"xxx",
   *    "type":"Object",
   *    "desc":"xxx", "nullable":
   *    "selected":true
   *    true "content": { --- 如果类型为Object，则需要有content向下包容
   *      {
   *        "fieldName":"xxx",
   *        "type":"文本",
   *        "desc":"xxx",
   *        "nullable": true
   *      },...
   *      }
   *      },...
   *    ]
   * }
   */
  private static final String JSON_FILE_SUFFIX = "json";
  private static final String REQUEST_PARAM = "RequestParam";
  private static final String REQUEST_PARAM_VALUE = "RequestParamValue";
  private static final String HEAD = "Head";
  private static final String HEAD_VALUE = "HeadValue";
  private static final String REQUEST_BODY = "RequestBody";
  private static final String REQUEST_BODY_VALUE = "RequestBodyValue";
  private static final String RESPONSE_BODY = "ResponseBody";
  private static final String NULLABLE = "nullable";
  private static final String FIELD_NAME = "fieldName";
  private static final String TYPE = "type";
  private static final String DESC = "desc";
  private static final String CONTENT = "content";
  private static final String OBJECT = "Object";
  private static final String OBJECT_ARRAY = "ObjectArray";
  private static final String KEY = "key";
  private static final String SERVICE_E_ERROR = "入参远程服务实体JSON不能为空";
  private static final String SERVICE_P_ERROR = "入参远程服务参数JSON不能为空";
  private static final String SERVICE_C_ERROR = "参数执行转化时发生错误，请检查！";
  private static final String SERVICE_S_ERROR = "远端服务源不能为空!";
  
  private static final HashSet<String> REQUEST_TYPES = Sets.newHashSet("文本", "数字", "时间", "布尔值", OBJECT, OBJECT_ARRAY);
  private static final HashSet<String> RESPONSE_TYPES = Sets.newHashSet("文本", "数字", "时间", "布尔值", OBJECT, OBJECT_ARRAY, "分页总数据", "分页单页数据", "分页当前页码", "分页最后页码","错误信息");
  private static final HashSet<String> REQUEST_METHOD = Sets.newHashSet("POST", "GET", "PATCH", "DELETE", "PUT", "COPY", "HEAD");

  @Autowired private RemoteServiceRepository remoteServiceRepository;
  @Autowired private NebulaFileService nebulaFileService;

  @Override
  public JSON valid(String entityJson, String jsonStr) {
    Validate.notBlank(entityJson,SERVICE_E_ERROR);
    Validate.notBlank(jsonStr,SERVICE_P_ERROR);
    RemoteServiceEntity entity;
    JSONObject json;
    try {
      entity = JSON.parseObject(entityJson, RemoteServiceEntity.class);
      json = JSON.parseObject(jsonStr);
    } catch (Exception e) {
      throw new IllegalArgumentException(SERVICE_C_ERROR, e);
    }
    // 验证JSON
    this.validJson(json);
    // 验证连接情况,获取返还的结果
    JSONObject result  = this.validConnection(entity, json);
    //如果是测试连接情况，则返还结果的数据格式
    return this.resolveJson(result, "");
  }

  @Override
  @Transactional(rollbackOn = Exception.class)
  public RemoteServiceEntity create(String entityJson, String jsonStr) {
    Validate.notBlank(entityJson,SERVICE_E_ERROR);
    Validate.notBlank(jsonStr,SERVICE_P_ERROR);
    RemoteServiceEntity entity;
    JSONObject json;
    try {
      entity = JSON.parseObject(entityJson, RemoteServiceEntity.class);
      json = JSON.parseObject(jsonStr);
    } catch (Exception e) {
      throw new IllegalArgumentException(SERVICE_C_ERROR, e);
    }
    // 验证实体
    this.validEntity(entity, true);
    RemoteServiceEntity exsit = this.remoteServiceRepository.findByCode(entity.getCode());
    Validate.isTrue(exsit == null , "指定的业务编号已存在，请进行更换!!");
    // 验证JSON
    this.validJson(json);
    // 验证连接情况
    this.validConnection(entity, json);
    // 保存JSON到文件路径
    Date nowDate = new Date();
    String folderName = new SimpleDateFormat("yyyyMMdd").format(nowDate);
    String uuid = UUID.randomUUID().toString();
    String fileRename = uuid + "." + JSON_FILE_SUFFIX;
    String relativePath = StringUtils.join("/remoteServiceJson/", folderName, "/", (new Random().nextInt(100) % 10));
    byte[] jsonContent;
    jsonContent = json.toJSONString().getBytes(StandardCharsets.UTF_8);
    nebulaFileService.saveFile(relativePath, fileRename, fileRename, jsonContent);
    // 保存JSON后保存实体
    entity.setJsonRelativePath(relativePath);
    entity.setJsonName(fileRename);
    entity.setRemoteServiceStatus(1);
    return this.remoteServiceRepository.save(entity);
  }

  @Override
  @Transactional
  public RemoteServiceEntity create(RemoteServiceEntity remoteServiceEntity) {
    this.validEntity(remoteServiceEntity,true);
    return this.remoteServiceRepository.save(remoteServiceEntity);
  }

  @Override
  @Transactional
  public RemoteServiceEntity update(String entityJson, String jsonStr) {
    Validate.notBlank(entityJson,SERVICE_E_ERROR);
    Validate.notBlank(jsonStr,SERVICE_P_ERROR);
    RemoteServiceEntity entity;
    JSONObject json;
    try {
      entity = JSON.parseObject(entityJson, RemoteServiceEntity.class);
      json = JSON.parseObject(jsonStr);
    } catch (Exception e) {
      throw new IllegalArgumentException(SERVICE_C_ERROR, e);
    }
    // 验证实体
    this.validEntity(entity, false);
    // 验证JSON
    this.validJson(json);
    // 验证连接情况
    this.validConnection(entity, json);
    // 保存JSON到文件路径
    Date nowDate = new Date();
    String folderName = new SimpleDateFormat("yyyyMMdd").format(nowDate);
    String uuid = UUID.randomUUID().toString();
    String fileRename = uuid + "." + JSON_FILE_SUFFIX;
    String relativePath = StringUtils.join("/remoteServiceJson/", folderName, "/", (new Random().nextInt(100) % 10));
    byte[] jsonContent;
    jsonContent = json.toJSONString().getBytes(StandardCharsets.UTF_8);
    nebulaFileService.saveFile(relativePath, fileRename, fileRename, jsonContent);
    Optional<RemoteServiceEntity> op = remoteServiceRepository.findById(entity.getId());
    RemoteServiceEntity exist = op.orElse(null);
    Validate.notNull(exist,SERVICE_S_ERROR);

    exist.setRemoteServiceType(entity.getRemoteServiceType());
    exist.setAcceptContentType(entity.getAcceptContentType());
    exist.setRemoteServiceAddress(entity.getRemoteServiceAddress());
    exist.setRequestURL(entity.getRequestURL());
    exist.setRemoteServiceDesc(entity.getRemoteServiceDesc());
    exist.setRequestMethod(entity.getRequestMethod());
    exist.setRequestAccept(entity.getRequestAccept());
    exist.setAcceptContentType(entity.getAcceptContentType());
    // 保存JSON后保存实体
    exist.setJsonRelativePath(relativePath);
    exist.setJsonName(fileRename);
    return this.remoteServiceRepository.save(exist);
  }

  @Override
  @Transactional
  public RemoteServiceEntity disable(String remoteServiceId) {
    Validate.notBlank(remoteServiceId, "远程调用源id不能为空");
    Optional<RemoteServiceEntity> op = remoteServiceRepository.findById(remoteServiceId);
    RemoteServiceEntity remoteServiceEntity = op.orElse(null);
    Validate.notNull(remoteServiceEntity,SERVICE_S_ERROR);

    remoteServiceEntity.setRemoteServiceStatus(0);
    return remoteServiceRepository.save(remoteServiceEntity);
  }

  @Override
  @Transactional
  public RemoteServiceEntity enable(String id) {
    Validate.notBlank(id, "远程调用源id不能为空");
    Optional<RemoteServiceEntity> op = remoteServiceRepository.findById(id);
    RemoteServiceEntity remoteServiceEntity = op.orElse(null);
    remoteServiceEntity.setRemoteServiceStatus(1);
    return remoteServiceRepository.save(remoteServiceEntity);
  }

  @Override
  @Transactional
  public void updateRemoteService(String remoteServiceAddressId, Integer status) {
    Validate.notBlank(remoteServiceAddressId, "远端地址id不能为空！");
    Validate.notNull(status, "修改的状态值不能为空");
    Set<RemoteServiceEntity> remoteServiceEntities = remoteServiceRepository.findByRemoteServiceAddressId(remoteServiceAddressId);
    if (!CollectionUtils.isEmpty(remoteServiceEntities)) {
      remoteServiceEntities.forEach(rs -> {
        rs.setRemoteServiceStatus(status);
        remoteServiceRepository.save(rs);
      });
    }

  }

  @Override
  public JSONArray importJson(String jsonStr) {
    Validate.notBlank(jsonStr, "输入JSON不能为空！");
    JSONObject json = JSON.parseObject(jsonStr);
    return this.jsonToFormat(json, new JSONArray(), "", 1);
  }

  @Override
  public JSON execute(String code, String jsonStr) {
    Validate.notBlank(code, "远端服务源ID不能为空！");
    RemoteServiceEntity entity = remoteServiceRepository.findByCode(code);
    Validate.notNull(entity, SERVICE_S_ERROR);

    JSONObject json;
    try {
      json = JSON.parseObject(jsonStr);
    } catch (Exception e) {
      throw new IllegalArgumentException(SERVICE_C_ERROR, e);
    }
    // 验证JSON
    // 验证连接情况,获取返还的结果
    JSONObject result = this.validConnection(entity, json);
    // 获取之前的保存的返回结构
    byte[] scriptFile = nebulaFileService.readFileContent(entity.getJsonRelativePath(), entity.getJsonName());
    Validate.notNull(scriptFile, "远端服务保存的数据格式文件为空！");
    // 从返还结果中，过滤保存的字段
    return result;
  }

  @Override
  public JSON findCompleteByCode(String code) {
    Validate.notBlank(code,"远端调用源业务号不能为空！");
    RemoteServiceEntity entity = remoteServiceRepository.findByCode(code);
    Validate.notNull(entity,SERVICE_S_ERROR);
  
    byte[] params = nebulaFileService.readFileContent(entity.getJsonRelativePath(),entity.getJsonName());
    Validate.notNull(params,"远端调用源存储的参数格式数据不能为空！");
    JSONObject paramsJson = JSON.parseObject(new String(params));
    Validate.notNull(paramsJson,"远端调用源存储的参数转化json不能为空!");
    
    //处理head信息，如果 head 中有而 headValue中没有，则补充到headValue中
    JSONArray headArray = paramsJson.getJSONArray(HEAD);
    JSONObject headValue = paramsJson.getJSONObject(HEAD_VALUE) == null? new JSONObject():paramsJson.getJSONObject(HEAD_VALUE);
    if(null!=headArray){
      for(Object o : headArray){
        JSONObject oJson = (JSONObject)o;
        if(StringUtils.isBlank(headValue.getString(oJson.getString(FIELD_NAME)))){
          headValue.put(oJson.getString(FIELD_NAME),"");
        }
      }
    }
    paramsJson.put(HEAD_VALUE,headValue);
  
    //处理 requestParam 信息，如果 requestParam 中有而 requestParamValue中没有，则补充到 requestParamValue中
    JSONArray requestParamArray = paramsJson.getJSONArray(REQUEST_PARAM);
    JSONObject requestParamValue = paramsJson.getJSONObject(REQUEST_PARAM_VALUE) == null? new JSONObject():paramsJson.getJSONObject(REQUEST_PARAM_VALUE);
    if(null!=requestParamArray){
      for(Object o : requestParamArray){
        JSONObject oJson = (JSONObject)o;
        if(StringUtils.isBlank(requestParamValue.getString(oJson.getString(FIELD_NAME)))){
          requestParamValue.put(oJson.getString(FIELD_NAME),"");
        }
      }
    }
    paramsJson.put(REQUEST_PARAM_VALUE,requestParamValue);
  
    return paramsJson;
  }

  @Override
  public String findJsonByCode(String code) {
    Validate.notBlank(code, "远端调用源业务号不能为空！");
    RemoteServiceEntity entity = remoteServiceRepository.findByCode(code);
    Validate.notNull(entity,SERVICE_S_ERROR);

    byte[] bytes = nebulaFileService.readFileContent(entity.getJsonRelativePath(), entity.getJsonName());
    String json;
    json = new String(bytes, StandardCharsets.UTF_8);
    return json;
  }

  @Override
  public RemoteServiceEntity findByCode(String code) {
    return remoteServiceRepository.findByCode(code);
  }

  @Override
  public Page<RemoteServiceEntity> findByConditions(Integer remoteServiceType, String addressCode,
      String addressAlias, String requestMethod,Integer remoteServiceStatus, Pageable pageable) {
    Validate.notNull(pageable, "查询分页信息不能为空!");
    Map<String, Object> conditions = new HashMap<>();
    if (null != remoteServiceType) {
      conditions.put("remoteServiceType", remoteServiceType);
    }
    if (StringUtils.isNotBlank(addressCode)) {
      conditions.put("addressCode", addressCode);
    }
    if (StringUtils.isNotBlank(addressAlias)) {
      conditions.put("addressAlias", addressAlias);
    }
    if (StringUtils.isNotBlank(requestMethod)) {
      conditions.put("requestMethod", requestMethod);
    }
    if (null != remoteServiceStatus) {
      conditions.put("remoteServiceStatus", remoteServiceStatus);
    }
    return this.remoteServiceRepository.findByConditions(pageable, conditions);
  }

  @Override
  public Set<RemoteServiceEntity> findDetailsByIds(String[] remoteServiceIds) {
    if(remoteServiceIds == null || remoteServiceIds.length == 0) {
      return Sets.newHashSet();
    }
    
    return this.remoteServiceRepository.findDetailsByIds(remoteServiceIds);
  }

  @Override
  public int countByIds(String[] ids) {
    if(ids == null || ids.length == 0) {
      return 0;
    }
    
    return this.remoteServiceRepository.countByIds(ids);
  }

  /**
   * 实体验证
   * @param entity
   */
  private void validEntity(RemoteServiceEntity entity, Boolean isCreate) {
    Validate.notNull(entity, "远端服务源不能为空！");
    String code = entity.getCode();
    Validate.notBlank(code, "远端服务源服务业务编号不能为空！");
    Validate.notBlank(entity.getRequestURL(), "远端服务源服务地址不能为空！");
    Validate.notBlank(entity.getRequestMethod(), "远端服务源调用方式不能为空！");
    Validate.isTrue(REQUEST_METHOD.contains(entity.getRequestMethod()), "远端服务源调用方式不合要求，请检查！");
    Validate.notBlank(entity.getRequestAccept(), "远端服务源接受的回应内容类型不能为空！");
    Validate.notBlank(entity.getAcceptContentType(), "远端服务源接受内容类型设定不能为空！");
    // 默认为http类型
    entity.setRemoteServiceType(
        null == entity.getRemoteServiceType() ? 0 : entity.getRemoteServiceType());
    // 如果为自定义调用，则需要检查ADDRESS
    if (entity.getRemoteServiceType() == 0) {
      Validate.notNull(entity.getRemoteServiceAddress(), "远程调用服务所属远程调用源为空,请检查！");
      Validate.notBlank(entity.getRemoteServiceAddress().getId(), "远程调用服务所属远程调用源ID为空,请检查！");
    }
    if (Boolean.TRUE.equals(isCreate)) {
      Validate.isTrue(StringUtils.isBlank(entity.getId()), "新增时，不能传入id!");
    } else {
      Validate.isTrue(!StringUtils.isBlank(entity.getId()), "修改时，必须传入id!");
      Optional<RemoteServiceEntity> op = remoteServiceRepository.findById(entity.getId());
      Validate.isTrue(op.isPresent(),"更新时未发现已存数据！");
    }
  }

  /**
   * 新增JSON验证
   *
   * @param json
   */
  private void validJson(JSONObject json) {
    // RequestParam 验证请求参数
    if (null != json.get(HEAD)) {
      JSONArray head = json.getJSONArray(HEAD);
      head.forEach(o -> {
        JSONObject jsonObject = JSON.parseObject(o.toString());
        Validate.notBlank(jsonObject.getString(FIELD_NAME), "HEAD参数缺少FIELD NAME名称！");
        Validate.notNull(jsonObject.get(NULLABLE), String.format("HEAD参数%s缺少必填参数！", jsonObject.getString(FIELD_NAME)));
      });
    }
    // RequestParam 验证请求参数
    if (null != json.get(REQUEST_PARAM)) {
      JSONArray requstParam = json.getJSONArray(REQUEST_PARAM);
      requstParam.forEach(o -> {
        JSONObject jsonObject = JSON.parseObject(o.toString());
        Validate.notBlank(jsonObject.getString(FIELD_NAME), "请求参数缺少KEY名称！");
        Validate.notNull(jsonObject.get(NULLABLE), String.format("请求参数%s缺少必填参数！", jsonObject.getString(FIELD_NAME)));
      });
    }
    // RequestBody 验证请求体,此处修改，
    if (null != json.get(REQUEST_BODY)) {
      JSONArray requestBody = json.getJSONArray(REQUEST_BODY);
      if (null != requestBody) {
        for (Object o : requestBody) {
          JSONObject obj = JSON.parseObject(o.toString());
          this.validObjectJson(obj, true);
        }
      }

    }

    // ResponseBody 验证返回体
    if (null != json.get(RESPONSE_BODY)) {
      JSONArray responseBody = json.getJSONArray(RESPONSE_BODY);
      if (null != responseBody) {
        for (Object o : responseBody) {
          JSONObject obj = JSON.parseObject(o.toString());
          this.validObjectJson(obj, false);
        }
      }
    }
  }

  /**
   * 递归遍历一个与前端商定的OBJECT JSON，验证格式
   *
   * @param json
   * @param isRequest
   */
  private void validObjectJson(JSONObject json, Boolean isRequest) {
    Validate.notBlank(json.getString(FIELD_NAME), "有参数缺少参数名，请检查！");
    Validate.notNull(
        json.getBoolean(NULLABLE), String.format("参数%s未确认必填，请检查！", json.getString(FIELD_NAME)));
    Validate.notBlank(
        json.getString(TYPE), String.format("参数%s未确认类型，请检查！", json.getString(FIELD_NAME)));
    // 请求与返回的TYPE类型不一样
    if (Boolean.TRUE.equals(isRequest)) {
      Validate.isTrue(
          REQUEST_TYPES.contains(json.getString(TYPE)),
          String.format("参数%s不匹配规定请求类型，请检查！", json.getString(FIELD_NAME)));
    } else {
      Validate.isTrue(
          RESPONSE_TYPES.contains(json.getString(TYPE)),
          String.format("参数%s不匹配规定返回类型，请检查！", json.getString(FIELD_NAME)));
    }
    // 如果为 OBJECT/ OBJECTJSON , 则递归遍历子项
    if (json.getString(TYPE).equals(OBJECT) || json.getString(TYPE).equals(OBJECT_ARRAY)) {
      Validate.notNull(json.get(CONTENT), "设置为OBJECT的参数必须有子内容，请检查！");
      JSONArray contentJSon = json.getJSONArray(CONTENT);
      for (Object o : contentJSon) {
        JSONObject oJson = JSON.parseObject(o.toString());
        this.validObjectJson(oJson, isRequest);
      }
    }
  }

  /**
   * 根据用户输入json格式，输出模板示例
   *
   * @param json 输入json
   * @param result 返回结果
   * @param location 父级树地址
   * @param index 起始排序
   * @return
   */
  private JSONArray jsonToFormat(
      JSONObject json, JSONArray result, String location, Integer index) {
    Set<String> keys = json.keySet();
    for (String key : keys) {
      Object obj = json.get(key);
      JSONObject thisJson = new JSONObject();
      // 通用处理
      thisJson.put(FIELD_NAME, key);
      thisJson.put(DESC, "");
      String thisLocation =
          StringUtils.isEmpty(location)
              ? index.toString()
              : String.format("%s-%s", location, index.toString());
      thisJson.put(KEY, thisLocation);
      thisJson.put(NULLABLE, true);
      // 根据当前值结构，确定TYPE
      if (obj instanceof JSONObject) {
        // 如果当前为json object 结构，则递归遍历下一层
        thisJson.put(TYPE, OBJECT);
        thisJson.put(
            CONTENT, this.jsonToFormat((JSONObject) obj, new JSONArray(), thisLocation, 1));
      } else if (obj instanceof JSONArray) {
        // 如果当前为json array 结构，则递归遍历下一层，且只取array的第一项分析结构
        JSONArray thisJsonArry = (JSONArray) obj;
        Object child = thisJsonArry.get(0);
        thisJson.put(TYPE, OBJECT_ARRAY);
        if (child instanceof JSONObject) {
          thisJson.put(
              CONTENT, this.jsonToFormat((JSONObject) child, new JSONArray(), thisLocation, 1));
        }
      } else if (obj instanceof Boolean) {
        thisJson.put(TYPE, "布尔值");
      } else if (obj instanceof Number) {
        thisJson.put(TYPE, "数字");
      } else if (obj instanceof Date) {
        thisJson.put(TYPE, "时间");
      } else {
        thisJson.put(TYPE, "文本");
      }
      result.add(thisJson);
      index = index + 1;
    }
    return result;
  }

  /**
   * 验证连接情况,得到返回的json
   *
   * @param entity
   * @param json
   */
  private JSONObject validConnection(RemoteServiceEntity entity, JSONObject json) {
    Client.Default defaultClient = new Client.Default(null, null);
    // 建立Request
    // 路径
    String url = entity.getRemoteServiceAddress().getAddress().concat(entity.getRequestURL());
    // HEAD
    Map<String, Collection<String>> headers = new HashMap<>();
    if (null != json.getJSONObject(HEAD_VALUE)) {
      JSONObject heads = json.getJSONObject(HEAD_VALUE);
      heads.keySet().forEach(o -> headers.put(o, Sets.newHashSet(heads.getString(o))));
    }
    // RequestParams
    if (null != json.getJSONObject(REQUEST_PARAM_VALUE)) {
      JSONObject requestParams = json.getJSONObject(REQUEST_PARAM_VALUE);
      Set<String> paramSet = new HashSet<>();
      for (String key : requestParams.keySet()) {
        paramSet.add(String.format("%s=%s", key, requestParams.getString(key)));
      }
      String paraUrl = String.format("?%s", StringUtils.join(paramSet, "&"));
      url = url.concat(paraUrl);
    }
    // RequestBody
    byte[] body = new byte[] {};
    try {
      JSONObject bodyJson = json.getJSONObject(REQUEST_BODY_VALUE);
      if (null != bodyJson) {
        String bodyStr = bodyJson.toJSONString();
        body = bodyStr.getBytes(StandardCharsets.UTF_8);
      }
    } catch (Exception e) {
      throw new IllegalArgumentException("请求体执行转化时发生错误，请检查！", e);
    }

    Request.Options options = new Request.Options(18000, 18000);
    HttpMethod httpMethod = HttpMethod.valueOf(entity.getRequestMethod());
    Request request = Request.create(httpMethod, url, headers, body, Charset.defaultCharset());
    try {
      Response response = defaultClient.execute(request, options);
      Validate.isTrue(response.status() == HttpStatus.OK.value(), String.format("连接失败，返回编码：%s,原因：%s", response.status(), response.reason()));
      String resultJson = IOUtils.toString(response.body().asInputStream(), StandardCharsets.UTF_8);
      return JSON.parseObject(resultJson);
    } catch (IOException e) {
      throw new IllegalArgumentException(e.getMessage(), e);
    }
  }

  /**
   * 解析http接口返还的值，得到数据结构
   *
   * @param sourceJson
   * @return
   */
  private JSONArray resolveJson(JSONObject sourceJson, String location) {
    JSONArray resultJson = new JSONArray();
    if (sourceJson == null) {
      return resultJson;
    }
    Set<String> keys = sourceJson.keySet();
    if (!CollectionUtils.isEmpty(keys)) {
      int index = 1;
      for (String key : keys) {
        String newLocation = StringUtils.isEmpty(location) ? String.format("%s", index) : String.format("%s-%s", location, index);
        JSONObject result = new JSONObject();
        //通用参数
        result.put(KEY,newLocation);
        result.put(FIELD_NAME,key);
        result.put(DESC,"");
        result.put(NULLABLE,true);
        if (sourceJson.get(key) instanceof JSONObject) {
          // 判断 json 的值是否还为 jsonobject, 如果是，则继续解析
          result.put(TYPE, OBJECT);
          // 递归调用此方法
          result.put(CONTENT, this.resolveJson(sourceJson.getJSONObject(key), newLocation));
          resultJson.add(result);
        } else if (sourceJson.get(key) instanceof JSONArray) {
          // 判断 json 的值是否还为 jsonArray, 如果是，则继续解析
          result.put(TYPE, OBJECT_ARRAY);
          // 判断array里面是否符合json格式，如果符合，则解析第一项
          JSONArray arrayJson = sourceJson.getJSONArray(key);
          if (arrayJson.get(0) instanceof JSONObject) {
            // 递归调用此方法
            result.put(CONTENT, this.resolveJson((JSONObject) arrayJson.get(0), newLocation));
          }
          resultJson.add(result);
        } else {
          //寻常参数
          result.put(TYPE, this.getReusltType(sourceJson.get(key)));
          resultJson.add(result);
        }
        index = index + 1;
      }
    }
    return resultJson;
  }

  /**
   * 预获取格式
   *
   * @param obj
   * @return
   */
  private String getReusltType(Object obj) {
    if (obj instanceof Number) {
      return "数字";
    }
    if (obj instanceof Date) {
      return "时间";
    }
    if (obj instanceof Boolean) {
      return "布尔值";
    }
    if (obj instanceof JSONObject) {
      return OBJECT;
    }
    if (obj instanceof JSONArray) {
      return OBJECT_ARRAY;
    }
    if (obj instanceof String) {
      return "文本";
    }
    return "文本";
  }
}
