package com.bizunited.platform.core.service.internal;

import com.bizunited.platform.core.entity.ScriptEntity;
import com.bizunited.platform.core.entity.UserEntity;
import com.bizunited.platform.core.repository.script.ScriptRepository;
import com.bizunited.platform.core.service.ScriptService;
import com.bizunited.platform.core.service.file.NebulaFileService;
import com.bizunited.platform.core.service.invoke.InvokeProxy;
import com.bizunited.platform.core.service.invoke.InvokeProxyException;
import com.bizunited.platform.core.service.script.model.ScriptInputParamsModel;
import com.bizunited.platform.core.service.script.persistent.PeristentGroovyClassService;
import com.bizunited.platform.core.service.script.persistent.SimplePersistentGroovyServiceFactory;
import com.bizunited.platform.rbac.server.service.OrganizationService;
import com.bizunited.platform.rbac.server.service.PositionService;
import com.bizunited.platform.rbac.server.service.RoleService;
import com.bizunited.platform.rbac.server.service.UserService;
import com.bizunited.platform.rbac.server.service.redis.RedisMutexService;
import com.bizunited.platform.rbac.server.vo.UserVo;
import groovy.lang.Binding;
import groovy.lang.GroovyShell;
import groovy.lang.Script;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.codehaus.groovy.control.CompilationFailedException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.nio.charset.StandardCharsets;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Random;
import java.util.UUID;

/**
 * ScriptServiceImpl
 *
 * @description:
 * @author: yanwe
 * @date: 27/May/2019 15:50
 */
@Service("ScriptServiceImpl")
public class ScriptServiceImpl implements ScriptService {

  private static final Logger LOGGER = LoggerFactory.getLogger(ScriptServiceImpl.class);

  @Autowired private ApplicationContext applicationContext;

  @Autowired private NebulaFileService nebulaFileService;

  @Autowired private ScriptRepository scriptRepository;
  @Autowired(required = false)
  private RedisMutexService redisMutexService;

  @Autowired private RoleService roleService;
  @Autowired private PositionService positionService;
  @Autowired private OrganizationService organizationService;

  @Autowired private UserService userService;
  @Autowired private SimplePersistentGroovyServiceFactory simplePersistentGroovyServiceFactory;

  @Override
  public Page<ScriptEntity> findByConditions(Pageable pageable, String name, String language) {
    Validate.notNull(pageable, "分页信息不能为空");
    Map<String, Object> conditions = new HashMap<>();
    if (StringUtils.isNotBlank(name)) {
      conditions.put("name", name);
    }
    if (StringUtils.isNotBlank(language)) {
      conditions.put("language", language);
    }
    return scriptRepository.queryPage(pageable, conditions);
  }

  @Override
  public ScriptEntity findById(String scriptId) {
    if (StringUtils.isBlank(scriptId)) {
      return null;
    }
    Optional<ScriptEntity> scriptEntity = scriptRepository.findById(scriptId);
    return scriptEntity.isPresent() ? scriptEntity.get() : null;
  }

  @Override
  public ScriptEntity findByName(String name) {
    if (StringUtils.isBlank(name)) {
      return null;
    }
    return scriptRepository.findByName(name);
  }

  @Override
  public String findContentByName(String scriptName) {
    if (StringUtils.isBlank(scriptName)) {
      return null;
    }
    ScriptEntity currentScript = this.findByName(scriptName);
    if(currentScript == null) {
      return null;
    }

    return this.findContentById(currentScript.getId());
  }

  @Override
  public String findContentById(String scriptId) {
    if (StringUtils.isBlank(scriptId)) {
      return null;
    }
    Optional<ScriptEntity> scriptEntity = scriptRepository.findById(scriptId);
    if (!scriptEntity.isPresent()) {
      return null;
    }
    byte[] content = nebulaFileService.readFileContent(scriptEntity.get().getFileCode(), scriptEntity.get().getFileName());
    return new String(content, StandardCharsets.UTF_8);
  }

  @Override
  @Transactional
  public ScriptEntity update(ScriptEntity scriptEntity, String scriptContent) {

    Validate.notNull(scriptEntity, "脚本实体不能为空！");
    Validate.notBlank(scriptContent, "脚本内容不能为空！");
    Validate.notNull(scriptEntity.getLanguage(), "脚本语言不能为空");
    Validate.notBlank(scriptEntity.getName(), "脚本名称不能为空！");
    Validate.notBlank(scriptEntity.getId(), "脚本ID不能为空");
    Optional<ScriptEntity> op = scriptRepository.findById(scriptEntity.getId());
    ScriptEntity existEntity = op.orElse(null);
    Validate.notNull(existEntity, "查询脚本不能为空");

    // 获取账号信息
    SecurityContext securityContext = SecurityContextHolder.getContext();
    Validate.notNull(securityContext, "未发现任何用户权限信息!!");
    Authentication authentication = securityContext.getAuthentication();
    Validate.notNull(authentication, "未发现任何用户登录信息!!");
    UserVo modifyUserVo = userService.findByAccount(authentication.getName());
    Validate.notNull(modifyUserVo, "未找到当前用户信息");
    UserEntity modifyUser = new UserEntity();
    modifyUser.setId(modifyUserVo.getId());

    // 保存脚本到文件路径
    String[] fileinfo = this.saveScriptContent(scriptContent);
    // 保存文件路径
    existEntity.setFileName(fileinfo[0]);
    existEntity.setFileCode(fileinfo[1]);
    // 最后修改时间
    existEntity.setModifyTime(new Date());
    // 最后修改人
    existEntity.setCreateUser(modifyUser);
    // 脚本语言
    existEntity.setLanguage(scriptEntity.getLanguage());
    // 脚本名称
    existEntity.setName(scriptEntity.getName());
    //描述信息
    existEntity.setDescription(scriptEntity.getDescription());

    try{
      return scriptRepository.save(existEntity);
    } finally {
      //更新脚本内容后，需要移除之前脚本在缓冲中数据
      PeristentGroovyClassService peristentGroovyClassService = simplePersistentGroovyServiceFactory.createPeristentGroovyClassService();
      peristentGroovyClassService.delete(existEntity.getId());
    }

  }
  
  /**
   * 重复的代码，进行私有方法封装，返回的一个数组，第一个值是文件的重命名名称、第二个值是文件相对路径
   * @param scriptContent
   * @return
   */
  private String[] saveScriptContent(String scriptContent) {
    Date nowDate = new Date();
    String folderName = new SimpleDateFormat("yyyyMMdd").format(nowDate);
    String uuid = UUID.randomUUID().toString();
    String fileRename = uuid + ".txt";
    String relativePath = StringUtils.join("/groovyScript/", folderName, "/", (new Random().nextInt(100) % 10));
    byte[] scriptContentByte = scriptContent.getBytes(StandardCharsets.UTF_8);
    nebulaFileService.saveFile(relativePath, fileRename, fileRename, scriptContentByte);
    
    return new String[] {fileRename , relativePath};
  }

  @Override
  @Transactional
  public ScriptEntity create(ScriptEntity scriptEntity, String scriptContent) {
    Validate.notNull(scriptEntity, "脚本实体不能为空！");
    Validate.notBlank(scriptContent, "脚本内容不能为空！");
    Validate.notNull(scriptEntity.getLanguage(), "脚本语言不能为空");
    Validate.notBlank(scriptEntity.getName(), "脚本名称不能为空！");
    ScriptEntity existEntity = scriptRepository.findByName(scriptEntity.getName());
    Validate.isTrue(null == existEntity, "脚本名称重复，请检查！");
    // 获取账号信息
    SecurityContext securityContext = SecurityContextHolder.getContext();
    Validate.notNull(securityContext, "未发现任何用户权限信息!!");
    Authentication authentication = securityContext.getAuthentication();
    Validate.notNull(authentication, "未发现任何用户登录信息!!");
    UserVo creatorVo = userService.findByAccount(authentication.getName());
    Validate.notNull(creatorVo, "未找到当前用户信息");
    UserEntity creator = new UserEntity();
    creator.setId(creatorVo.getId());

    // 保存脚本到文件路径
    String[] fileinfo = this.saveScriptContent(scriptContent);
    // 保存文件路径
    scriptEntity.setFileName(fileinfo[0]);
    scriptEntity.setFileCode(fileinfo[1]);
    // 创建者
    scriptEntity.setCreateUser(creator);
    // 生成时间
    scriptEntity.setCreateTime(new Date());
    // 更新时间
    scriptEntity.setModifyTime(new Date());
    return scriptRepository.save(scriptEntity);
  }

  @SuppressWarnings("unchecked")
  @Override
  public Map<String, Object> invoke(String scriptId, Map<String, Object> params) throws InvokeProxyException {
    /*
     * 1、验证scriptId对应脚本的存在性
     * 2、取得已准备号的scriptShell。首先从cache中取得，如果没有再从磁盘上取得。
     * 3、以下信息需要作为script的全局变量进行加载：
     *    a、有params传入的各种外部参数
     *    b、spring的上下文，applicationContext，变量名ctx
     *    c、经常会使用的service、例如userService。
     *  4、开始执行，并输出script中的所有全局变量——以Map的方式输出。 可指定一个变量，例如returnKey，作为整个调用过程的返回值。
     */
    // 1、===
    Validate.notBlank(scriptId, "脚本ID不能为空！");
    Optional<ScriptEntity> op = scriptRepository.findById(scriptId);
    ScriptEntity scriptEntity = op.orElse(null);
    Validate.notNull(scriptEntity, "脚本不能为空");
    // 2、===
    // 缓存系统
    // 尝试从内存中获取，如果没有再从磁盘读取
    PeristentGroovyClassService peristentGroovyClassService = simplePersistentGroovyServiceFactory.createPeristentGroovyClassService();
    Script groovyScript = (Script) peristentGroovyClassService.findByClassName(scriptId);
    // 如果没有，则从磁盘中获取
    if (groovyScript == null) {
      byte[] scriptContent = nebulaFileService.readFileContent(scriptEntity.getFileCode(), scriptEntity.getFileName());
      String groovyStr = new String(scriptContent);
      GroovyShell shell = new GroovyShell(this.applicationContext.getClassLoader());
      try {
        groovyScript = shell.parse(groovyStr);
        peristentGroovyClassService.save(scriptEntity.getId(), groovyScript);
      } catch (CompilationFailedException e) {
        LOGGER.error(e.getMessage(), e);
        throw new IllegalArgumentException("编译groovy脚本时发生错误，请检查！");
      }
    }
    // 3、===
    Binding binding = new Binding();
    // 绑定添加默认方法参数
    binding.setVariable("userService", userService);
    binding.setVariable("roleService", roleService);
    binding.setVariable("positionService", positionService);
    binding.setVariable("organizationService", organizationService);
    if(redisMutexService != null) {
      binding.setVariable("redisMutexService", redisMutexService);
    }
    binding.setVariable("ctx", applicationContext);
    // 绑定传入参数
    if (!CollectionUtils.isEmpty(params)) {
      for (Map.Entry<String,Object> item : params.entrySet()) {
        binding.setVariable(item.getKey(), item.getValue());
      }
    }
    groovyScript.setBinding(binding);
    groovyScript.run();
    // 4、===
    Map<String, Object> result = new HashMap<>();
    // 将其他参数写入输出
    Map<String,Object> bindingParams= binding.getVariables();
    if(!CollectionUtils.isEmpty(bindingParams)){
      for(String key: bindingParams.keySet()){
        result.put(key,binding.getVariable(key));
      }
    }
    return result;
  }

  @Override
  public Object invoke(ScriptInputParamsModel model) throws InvokeProxyException {
    Validate.notNull(model, "脚本参数model不能为空！");
    Validate.notEmpty(model.getScriptIds(), "脚本id不能为空！");
    InvokeProxy groovyScriptServicableProxy = this.applicationContext.getBean("GroovyProxy", InvokeProxy.class);
    Validate.notNull(groovyScriptServicableProxy, "执行groovy脚本时未获取到调用代理器！");
    Map<String, Object> context = new HashMap<>();
    context.put("scriptInputParamsModel", model);
    return groovyScriptServicableProxy.doHandle(context);
  }
}
