package com.bizunited.platform.core.service.internal;

import com.bizunited.platform.core.entity.ServicableMethodEntity;
import com.bizunited.platform.core.entity.ServicableMethodPropertyEntity;
import com.bizunited.platform.core.repository.ServicableMethodPropertyRepository;
import com.bizunited.platform.core.repository.ServicableMethodRepository;
import com.bizunited.platform.core.service.serviceable.ServicableMethodPropertyService;
import com.bizunited.platform.core.service.serviceable.ServicableMethodService;
import com.bizunited.platform.core.service.invoke.InvokeProxyException;
import com.bizunited.platform.core.service.invoke.InvokeProxy;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import javax.transaction.Transactional;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * kuiper表单引擎服务源管理、调用的相关服务实现
 * @author yinwenjie
 */
@Service("KuiperServiceMethodServiceImpl")
public class ServicableMethodServiceImpl implements ServicableMethodService {
  @Autowired
  private ServicableMethodRepository servicableMethodRepository;
  @Autowired
  private ServicableMethodPropertyRepository servicableMethodPropertyRepository;
  @Autowired
  private ServicableMethodPropertyService servicableMethodPropertyService;
  @Autowired
  private ApplicationContext applicationContext;
  
  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.KuiperServiceMethodService#create(com.bizunited.platform.kuiper.starter.entity.KuiperServiceMethodEntity)
   */
  @Override
  @Transactional
  public void create(ServicableMethodEntity serviceMethod) {
    // 验证
    String name = serviceMethod.getName();
    Validate.notBlank(name , "未发现KuiperServiceMethod中的name属性，请检查!!");
    String desc = serviceMethod.getDescription();
    Validate.notBlank(desc , "未发现KuiperServiceMethod中的desc属性，请检查!!");
    String methodName = serviceMethod.getMethodName();
    Validate.notBlank(methodName , "未发现方法信息，请检查!!");
    String interfaceName = serviceMethod.getInterfaceName();
    Validate.notBlank(interfaceName , "未发现接口方法定义信息，请检查!!");
    String returnClassName = serviceMethod.getReturnClassName();
    Validate.notBlank(returnClassName , "未发现接口方法定义返回信息，请检查!!");
    String simpleMethodName = serviceMethod.getSimpleMethodName();
    Validate.notBlank(simpleMethodName , "未发现方法简称信息，请检查!!");

    // 进行添加
    this.servicableMethodRepository.saveAndFlush(serviceMethod);
    if(serviceMethod.getProperties() != null) {
      // 进行明细信息的添加
      List<ServicableMethodPropertyEntity> properties = new ArrayList<>(serviceMethod.getProperties());
      if(!properties.isEmpty()) {
        for (ServicableMethodPropertyEntity serviceMethodProperty : properties) {
          serviceMethodProperty.setServiceMethod(serviceMethod);
          servicableMethodPropertyService.create(serviceMethodProperty);
        }
      }
    }
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.KuiperServiceMethodService#deleteServiceMethodAndProperties()
   */
  @Override
  @Transactional
  public void deleteServiceMethodAndProperties() {
    // 包括删除服务暴露的属性明细和基本信息
    servicableMethodPropertyRepository.deleteAll();
    servicableMethodRepository.deleteAll();
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.ServicableMethodService#findByConditions(org.springframework.data.domain.Pageable, java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String)
   */
  @Override
  public Page<ServicableMethodEntity> findByConditions(Pageable pageable, String name, String description , String interfaceName, String simpleMethodName, String returnClassName , String propertyClassName , String usedScope) {
    Validate.notBlank(usedScope, "服务源的应用范围不能为空，请检查!!");
    
    Map<String, Object> conditions = new HashMap<>();
    
    conditions.put("usedScope", usedScope);
    if (StringUtils.isNotBlank(name)) {
      conditions.put("name", name);
    }
    if (StringUtils.isNotBlank(description)) {
      conditions.put("description", description);
    }
    if (StringUtils.isNotBlank(interfaceName)) {
      conditions.put("interfaceName", interfaceName);
    }
    if (StringUtils.isNotBlank(simpleMethodName)) {
      conditions.put("simpleMethodName", simpleMethodName);
    }
    if(StringUtils.isNotBlank(returnClassName)) {
      conditions.put("returnClassName", returnClassName);
    }
    if(StringUtils.isNotBlank(propertyClassName)) {
      conditions.put("propertyClassName", propertyClassName);
    }
    
    // 如果当前没有设定分页信息，则默认第一页，每页50条数据
    if(pageable == null) {
      pageable = PageRequest.of(0, 50);
    }
    
    return this.servicableMethodRepository.queryPage(pageable, conditions);
  }
  
  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.ServicableMethodService#findDetailsByName(java.lang.String)
   */
  @Override
  public ServicableMethodEntity findDetailsByName(String name) {
    Validate.notBlank(name , "服务源名称不能为空，请检查!!");
    return servicableMethodRepository.queryDetailsByName(name);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.ServicableMethodService#findByName(java.lang.String)
   */
  @Override
  public ServicableMethodEntity findByName(String name) {
    Validate.notBlank(name , "服务源名称不能为空，请检查!!");
    ServicableMethodEntity entity = servicableMethodRepository.queryByName(name);
    Validate.notNull(entity, "根据服务name值，未获取到服务信息，请检查!!");
    return entity;
  }
  
  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.ServicableMethodService#writeInvoke(java.util.Map)
   */
  @Override
  public Object writeInvoke(Map<String, Object> params) throws InvokeProxyException {
    InvokeProxy writeServicableProxy = this.applicationContext.getBean("WriteServicableProxy", InvokeProxy.class);
    Object result = null;
    if(params != null && !params.isEmpty()) {
      result = writeServicableProxy.doHandle(params);
    }
    return result;
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.ServicableMethodService#readInvoke(java.util.Map)
   */
  @Override
  public Object readInvoke(Map<String, Object> params) throws InvokeProxyException {
    InvokeProxy writeServicableProxy = this.applicationContext.getBean("ReadServicableProxy", InvokeProxy.class);
    Object result = null;
    if(params != null && !params.isEmpty()) {
      result = writeServicableProxy.doHandle(params);
    }
    return result;
  }  
}