package com.bizunited.platform.core.service.internal;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.core.entity.OrdinaryFileEntity;
import com.bizunited.platform.core.entity.SystemThemeEntity;
import com.bizunited.platform.core.repository.SystemThemeRepository;
import com.bizunited.platform.core.service.SystemThemeService;
import com.bizunited.platform.core.service.file.NebulaFileService;
import com.bizunited.platform.core.service.image.FileUpdateService;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.web.multipart.MultipartFile;

import javax.transaction.Transactional;
import java.nio.charset.StandardCharsets;
import java.security.Principal;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

//TODO 暂时这一版数据始终只有一条，用于全局
@Service("SystemThemeServiceImpl")
public class SystemThemeServiceImpl implements SystemThemeService {

  private static final Logger LOGGER = LoggerFactory.getLogger(SystemThemeServiceImpl.class);
  private static final String THEME_FILE_NAME = "theme.json";
  private static final String MESS_FIELD_NAME = "fileName";
  private static final String MESS_RELATIVE_LOCAL = "relativeLocal";
  private static final String MESS_THEME = "theme";
  @Autowired
  private FileUpdateService fileUpdateService;
  @Autowired
  private NebulaFileService nebulaFileService;
  @Autowired
  private SystemThemeRepository systemThemeRepository;

  @Override
  @Transactional
  public OrdinaryFileEntity uploadLogo(MultipartFile file, Principal principal) {
    String account = this.validatePrincipal(principal);
    Validate.notNull(file,"logo图片数据不能为空，请检查!!");
    List<OrdinaryFileEntity> logos = fileUpdateService.fileImageUpload("logo",account,null,new MultipartFile[]{file});
    Validate.notEmpty(logos,"logo文件上传失败，请检查!!");
    Validate.isTrue(!logos.isEmpty(),"logo文件上传失败，请检查!!");
    return logos.get(0);
  }

  @Override
  @Transactional
  public SystemThemeEntity save(JSONObject theme, Principal principal) {
    String account = this.validatePrincipal(principal);
    Validate.notEmpty(theme,"系统主题数据不能为空，请检查!!");

    List<SystemThemeEntity> themes = systemThemeRepository.findAll();
    Validate.isTrue(themes == null || themes.size() <= 1,"系统主题信息有多条，请检查!!");

    Map<String,String> themePath = this.resolveThemePath(theme);
    Validate.notNull(themePath,"解析前端传入的json文件路径信息有误，请检查!!");
    SystemThemeEntity entity;
    Date now = new Date();
    if(themes == null || themes.isEmpty()){
      //目前包含：logo，色板，布局框架，列表等
      Map<String,String> themePathMap = this.saveThemeFile(null,theme);
      Validate.notEmpty(themePathMap,"主题json文件保存成功，但未能获取到json文件的路径信息，请检查!!");

      entity = new SystemThemeEntity();
      entity.setCommitUser(account);
      entity.setUpdateUser(account);
      entity.setCreateTime(now);
      entity.setUpdateTime(now);
      entity.setFileName(themePathMap.get(MESS_FIELD_NAME));
      entity.setOriginalName(THEME_FILE_NAME);
      entity.setRelativeLocal(themePathMap.get(MESS_RELATIVE_LOCAL));
    }else{
      //保证数据库只有一条数据
      entity = themes.get(0);
      Validate.isTrue(StringUtils.equals(entity.getFileName(),themePath.get(MESS_FIELD_NAME)),"theme传入的fileName不一致，请检查!!");
      Validate.isTrue(StringUtils.equals(entity.getRelativeLocal(),themePath.get(MESS_RELATIVE_LOCAL)),"theme传入的relativeLocal不一致，请检查!!");
      Validate.notNull(entity,"根据主题信息，未能查询到主题相关信息，请检查!!");
      entity.setUpdateTime(now);
      entity.setUpdateUser(account);
      Map<String,String> themePathMap = this.saveThemeFile(entity,theme);
      Validate.notEmpty(themePathMap,"主题json文件保存成功，但未能获取到json文件的路径信息，请检查!!");
    }
    systemThemeRepository.save(entity);
    return entity;
  }


  @Override
  public JSONObject findTheme() {
    //1.======查询主题信息
    List<SystemThemeEntity> themes = systemThemeRepository.findAll();
    if(CollectionUtils.isEmpty(themes)){
      return null;
    }
    Validate.isTrue(themes.size() <= 1,"系统主题信息有多条，请检查!!");
    SystemThemeEntity theme = themes.get(0);

    //2.======解析主题数据，返回json格式
    byte[] content = nebulaFileService.readFileContent(theme.getRelativeLocal(),theme.getFileName());
    if(ArrayUtils.isEmpty(content)) {
      LOGGER.warn("未能发现系统主题信息数据，请检查!!");
      return null;
    }
    String jsonStr = new String(content, StandardCharsets.UTF_8);
    return JSON.parseObject(jsonStr);
  }

  /**
   * 验证用户信息
   * @param principal
   */
  private String validatePrincipal(Principal principal){
    Validate.notNull(principal,"未能获取到用户信息，请检查!!");
    String account = principal.getName();
    Validate.notBlank(account,"未能获取到用户账户信息，请检查!!");
    return account;
  }

  /**
   * 解析前端传入的theme配置路径信息
   * @return
   */
  private Map<String,String> resolveThemePath(JSONObject theme){
    Validate.notNull(theme,"系统主题数据不能为空，请检查!!");
    Map<String,String> themePathMap = new HashMap<>();
    JSONObject themeJson = theme.getJSONObject(MESS_THEME);
    if(themeJson == null || themeJson.size() == 0){
      return themePathMap;
    }
    String relativeLocal = themeJson.getString(MESS_RELATIVE_LOCAL);
    String fileName = themeJson.getString(MESS_FIELD_NAME);
    Validate.notBlank(relativeLocal,"系统主题中，给定的相对路径不能为空，请检查!!");
    Validate.notBlank(fileName,"系统主题中，给定的文件名不能为空，请检查!!");
    themePathMap.put(MESS_FIELD_NAME,fileName);
    themePathMap.put(MESS_RELATIVE_LOCAL,relativeLocal);
    return themePathMap;
  }


  private Map<String,String> saveThemeFile(SystemThemeEntity themeEntity,JSONObject theme){
    Validate.notEmpty(theme,"传入的主题json数据不能为空，请检查!!");
    //目前包含：logo，色板，布局框架，列表等
    byte[] content = null;


    String folderName = new SimpleDateFormat("yyyyMMdd").format(new Date());
    String relativePath = null;
    String renameFile = null;
    if(themeEntity == null){
      relativePath = "/theme/" + folderName;
      renameFile = UUID.randomUUID().toString()+".json";
    }else{
      relativePath = themeEntity.getRelativeLocal();
      renameFile = themeEntity.getFileName();
    }
    JSONObject themeJson = theme.getJSONObject(MESS_THEME);
    if(themeJson == null || themeJson.size() == 0){
      themeJson = new JSONObject();
    }
    themeJson.put(MESS_FIELD_NAME,renameFile);
    themeJson.put(MESS_RELATIVE_LOCAL,relativePath);
    theme.put(MESS_THEME,themeJson);
    try{
      content = theme.toJSONString().getBytes("utf-8");
    }catch (Exception e){
      throw new IllegalArgumentException("系统主题数据转码失败，请重试!");
    }
    nebulaFileService.saveFile(relativePath,THEME_FILE_NAME, renameFile,content);
    Map<String,String> themePathMap = new HashMap<>();
    themePathMap.put(MESS_RELATIVE_LOCAL,relativePath);
    themePathMap.put(MESS_FIELD_NAME,renameFile);
    return themePathMap;
  }
}
