package com.bizunited.platform.core.service.migrate.internal;

import com.bizunited.platform.core.entity.CodeRuleEntity;
import com.bizunited.platform.core.entity.DataSourceEntity;
import com.bizunited.platform.core.entity.DataViewAuthHorizontalEntity;
import com.bizunited.platform.core.entity.DataViewAuthVerticalEntity;
import com.bizunited.platform.core.entity.DataViewEntity;
import com.bizunited.platform.core.entity.DataViewGroupEntity;
import com.bizunited.platform.core.entity.DictCategoryEntity;
import com.bizunited.platform.core.entity.DictEntity;
import com.bizunited.platform.core.entity.EnvironmentVariableEntity;
import com.bizunited.platform.core.entity.RemoteServiceAddressEntity;
import com.bizunited.platform.core.entity.RemoteServiceEntity;
import com.bizunited.platform.core.entity.ScriptEntity;
import com.bizunited.platform.core.service.CodeRuleService;
import com.bizunited.platform.core.service.DictCategoryService;
import com.bizunited.platform.core.service.DictService;
import com.bizunited.platform.core.service.EnvironmentVariableService;
import com.bizunited.platform.core.service.NebulaToolkitService;
import com.bizunited.platform.core.service.RemoteServiceService;
import com.bizunited.platform.core.service.dataview.DataViewAuthHorizontalService;
import com.bizunited.platform.core.service.dataview.DataViewAuthVerticalService;
import com.bizunited.platform.core.service.dataview.DataViewService;
import com.bizunited.platform.core.service.file.NebulaFileService;
import com.bizunited.platform.core.service.migrate.MigrateExportService;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectOutputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.Set;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

/**
 * 基础工具套件中重要数据的迁出服务实现
 * @author yinwenjie
 */
@Service("migrateExportService")
public class MigrateExportServiceImpl implements MigrateExportService {
  @Autowired
  private DataViewService dataViewService;
  /**
   * 数据视图横向权限
   */
  @Autowired
  private DataViewAuthHorizontalService dataViewAuthHorizontalService;
  /**
   * 数据视图纵向权限
   */
  @Autowired
  private DataViewAuthVerticalService dataViewAuthVerticalService;
  @Autowired
  private CodeRuleService codeRuleService;
  @Autowired
  private DictService dictService;
  @Autowired
  private DictCategoryService dictCategoryService;
  @Autowired
  private EnvironmentVariableService environmentVariableService;
  @Autowired
  private RemoteServiceService remoteServiceService;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private NebulaFileService nebulaFileService;
  private static final Logger LOGGER = LoggerFactory.getLogger(MigrateExportServiceImpl.class);
  
  @Override
  public byte[] export(String[] dataviewIds, String[] codeRuleIds, String[] dictIds, String[] envIds, String[] remoteServiceIds) {
    boolean hasDateView = false;
    boolean hasCodeRule = false;
    boolean hasDict = false;
    boolean hasEnv = false;
    boolean hasRemoteService = false;
    if(dataviewIds != null && dataviewIds.length > 0) {
      Validate.isTrue(this.dataViewService.countByIds(dataviewIds) == dataviewIds.length , "指定导出的数据视图信息与数据库存储的信息不符，请检查");
      hasDateView = true;
    }
    if(codeRuleIds != null && codeRuleIds.length > 0) {
      Validate.isTrue(this.codeRuleService.countByIds(codeRuleIds) == codeRuleIds.length , "指定导出的业务编码规则信息与数据库存储的信息不符，请检查");
      hasCodeRule = true;
    }
    if(dictIds != null && dictIds.length > 0) {
      Validate.isTrue(this.dictService.countByIds(dictIds) == dictIds.length , "指定导出的数据字典信息与数据库存储的信息不符，请检查");
      hasDict = true;
    }
    if(envIds != null && envIds.length > 0) {
      Validate.isTrue(this.environmentVariableService.countByIds(envIds) == envIds.length , "指定导出的环境变量信息与数据库存储的信息不符，请检查");
      hasEnv = true;
    }
    if(remoteServiceIds != null && remoteServiceIds.length > 0) {
      Validate.isTrue(this.remoteServiceService.countByIds(remoteServiceIds) == remoteServiceIds.length , "指定导出的远端服务源信息与数据库存储的信息不符，请检查");
      hasRemoteService = true;
    }
    if(!hasDateView && !hasCodeRule && !hasDict && !hasEnv && !hasRemoteService) {
      throw new IllegalArgumentException("在进行基本数据导出时，至少需要选择一种数据集（数据视图、业务编码规则、数据字典、全局环境变量、远端方法源）");
    }

    /*
     * .在完成边界校验后，需要完成以下工作：
     * 1、生成压缩文件和压缩流，以便后续的所有处理过程使用；
     * 2、首先进行可能的数据视图的迁出处理（详见私有方法中的详细描述）
     * 3、然后进行可能的业务编码规则的迁出处理（详见私有方法中的详细描述）
     * 4、然后进行可能的数据字典的迁出处理（详见私有方法中的详细描述）
     * 5、再接下来进行可能的全局环境变量的迁出处理（详见私有方法中的详细描述）
     * 6、最后进行可能的远端调用源信息的的迁出处理（详见私有方法中的详细描述）
     * */
    // 1、========
    byte[] fullZipBytes = null;
    try (ByteArrayOutputStream fullZipBis = new ByteArrayOutputStream();
        ZipOutputStream zipf = new ZipOutputStream(fullZipBis);) {
      // 2、=====
      if(hasDateView) {
        this.handleDateView(zipf, dataviewIds);
      }
      // 3、======
      if(hasCodeRule) {
        this.handleCodeRule(zipf, codeRuleIds);
      }
      // 4、======
      if(hasDict) {
        this.handleDict(zipf, dictIds);
      }
      // 5、======
      if(hasEnv) {
        this.handleEnv(zipf, envIds);
      }
      // 6、======
      if(hasRemoteService) {
        this.handleRemoteService(zipf, remoteServiceIds);
      }
      zipf.finish();
      fullZipBytes = fullZipBis.toByteArray();
    } catch(IOException e) {
      throw new IllegalArgumentException(e.getMessage());
    }
    return fullZipBytes;
  }
  
  /**
   * 该私有方法用于进行数据视图的迁出
   * @param zipf
   * @param dataviewIds
   */
  private void handleDateView(ZipOutputStream zipf , String[] dataviewIds) {
    /*
     * 序列化格式如下：
     * 数据视图迁出信息（文件名dataview.in）：
     * 数据源数量|数据分组数量|数据视图数量|数据权限-横向权限数量|数据权限-纵向权限数量|
     * 第一个不重复的数据源（只有基本信息，不带任何关联信息）|………………|
     * 第一个不重复的数据分组（基本信息和关联的数据源信息）|………………|
     * 第一个不重复的数据视图（基本信息、关联的数据源、关联的数据分组、关联的字段、数据权限等信息）|…………|
     * 第一个不重复的横向权限（基本信息和关联信息）|………………|
     * 第一个不重复的纵向权限（基本信息和关联信息）|………………|
     * 
     * 处理过程为：
     * 1、首先依次查询出所有的数据视图的详情、包括所属分组、所属数据库配置、字段、数据权限等信息
     * 2、分别记录数据分组的不重复集合和不重复的数据库配置信息
     * 3、创建对象流，以便后续过程进行对象序列化时使用
     * 4、写入不重复的数据源（只有基本信息，不带任何关联信息）
     * 5、写入不重复的数据分组（基本信息和关联的数据库id）
     * 6、写入不重复的数据视图（基本信息、关联的数据源、关联的数据分组、关联的字段、数据权限等信息）
     * 7、写入不重复的横向权限（基本信息和关联信息）——如果有的话
     * 8、写入不重复的纵向权限（基本信息和关联信息）——如果有的话
     * 9、创建一个名叫dataview.in的zip文件entry，并将对象流写入文件、将文件写入压缩流
     * */
    // 1、===========
    Set<DataSourceEntity> allJpaDataSources = Sets.newHashSet();
    Set<DataViewGroupEntity> allJpaDataViewGroups = Sets.newHashSet();
    Set<DataViewEntity> allJpaDataViews = this.dataViewService.findDetailsByIds(dataviewIds);
    Set<DataViewAuthHorizontalEntity> allDataViewAuthHs = this.dataViewAuthHorizontalService.findDetailsByDataViewIds(dataviewIds);
    Set<DataViewAuthVerticalEntity> allDataViewAuthVs = this.dataViewAuthVerticalService.findDetailsByDataViewIds(dataviewIds);
    
    // 2、==========（由于allJpaDataViews集合是带有JPA特性的集合，所以使用JPA对象的引用特性进行重复对象过滤）
    for (DataViewEntity jpaDataViewItem : allJpaDataViews) {
      if(jpaDataViewItem.getDataSource() != null) {
        allJpaDataSources.add(jpaDataViewItem.getDataSource());
      }
      allJpaDataViewGroups.add(jpaDataViewItem.getDataViewGroup());
    }
    // 接着将所有数据脱离JPA对象特性
    Collection<DataSourceEntity> allDataSources = this.nebulaToolkitService.copyCollectionByWhiteList(allJpaDataSources, DataSourceEntity.class, DataSourceEntity.class, HashSet.class, ArrayList.class);
    Collection<DataViewGroupEntity> allDataViewGroups = this.nebulaToolkitService.copyCollectionByWhiteList(allJpaDataViewGroups, DataViewGroupEntity.class, DataViewGroupEntity.class, HashSet.class, ArrayList.class, "dataSource");
    Collection<DataViewEntity> allDataViews = this.nebulaToolkitService.copyCollectionByWhiteList(allJpaDataViews, DataViewEntity.class, DataViewEntity.class, HashSet.class, ArrayList.class, "dataSource","dataViewGroup","fields","filters","filters.field","systemFilters");
    int dataSourcesSize = allDataSources.size();
    int dataViewGroupsSize = allDataViewGroups.size();
    int dataViewsSize = allDataViews.size();
    int dataViewAuthHsSize = allDataViewAuthHs == null?0:allDataViewAuthHs.size();
    int dataViewAuthVsSize = allDataViewAuthVs == null?0:allDataViewAuthVs.size();
    
    // 3、============
    try (ByteArrayOutputStream bos = new ByteArrayOutputStream();
        ObjectOutputStream oos = new ObjectOutputStream(bos);) {
      oos.writeInt(dataSourcesSize);
      oos.writeInt(dataViewGroupsSize);
      oos.writeInt(dataViewsSize);
      oos.writeInt(dataViewAuthHsSize);
      oos.writeInt(dataViewAuthVsSize);
      
      // 4、===========
      for (DataSourceEntity dataSourceItem : allDataSources) {
        oos.writeObject(dataSourceItem);
      }
      // 5、==========
      for (DataViewGroupEntity dataViewGroupItem : allDataViewGroups) {
        oos.writeObject(dataViewGroupItem);
      }
      // 6、==========
      for (DataViewEntity dataViewitem : allDataViews) {
        oos.writeObject(dataViewitem);
      }
      // 7、==========
      if(allDataViewAuthHs != null) {
        for (DataViewAuthHorizontalEntity hItem : allDataViewAuthHs) {
          oos.writeObject(hItem);
        }
      }
      // 8、==========
      if(allDataViewAuthVs != null) {
        for (DataViewAuthVerticalEntity vItem : allDataViewAuthVs) {
          oos.writeObject(vItem);
        }
      }
      
      // 9、=========
      String zipFileName = "dataview.in";
      byte[] bosBytes = bos.toByteArray();
      this.writeZipFile(zipf, zipFileName, bosBytes);
    } catch(IOException e) {
      throw new IllegalArgumentException(e.getMessage());
    }
  }
  
  /**
   * 该私有方法用于对指定的业务编码规则进行迁出
   * @param zipf
   * @param codeRuleIds
   */
  private void handleCodeRule(ZipOutputStream zipf , String[] codeRuleIds) {
    /*
     * 序列化格式如下：
     * 业务编码生成规则信息（文件名codeRule.in）
     * 脚本文件数量|业务编码规则数量|
     * 第一个不重复的脚本信息（基本信息和附件信息）|………………|
     * 第一个不重复的业务编码规则信息（基本信息，关联的脚本信息）|………………|
     * 
     * 处理过程包括：
     * 1、首先进行业务编码规则的基本信息和关联信息
     * 2、从业务编码规则中过滤出不重复的脚本信息
     * 3、创建对象流，以便后续过程进行对象序列化时使用
     * 4、写入（序列化）不重复的脚本信息（基本信息和附件信息）
     * 5、写入（序列化）不重复的业务编码规则信息（基本信息，关联的脚本信息）
     * 6、创建一个名叫dataview.in的zip文件entry，并将对象流写入文件、将文件写入压缩流
     * */
    // 1、==========
    Set<ScriptEntity> allJpaScripts = Sets.newHashSet();
    Set<CodeRuleEntity> allJpaCodeRules = this.codeRuleService.findDetailsByIds(codeRuleIds);
    
    // 2、==========（由于allJpaDataViews集合是带有JPA特性的集合，所以使用JPA对象的引用特性进行重复对象过滤）
    for (CodeRuleEntity codeRuleItem : allJpaCodeRules) {
      ScriptEntity currentScript = codeRuleItem.getScript();
      Validate.notNull(currentScript , "当前编码脚本存在问题，没有关联任何脚本信息，不能进行导出");
      allJpaScripts.add(currentScript);
    }
    
    // 接着将所有数据脱离JPA对象特性
    Collection<ScriptEntity> allScripts = this.nebulaToolkitService.copyCollectionByWhiteList(allJpaScripts, ScriptEntity.class, ScriptEntity.class, HashSet.class, ArrayList.class);
    Collection<CodeRuleEntity> allCodeRules = this.nebulaToolkitService.copyCollectionByWhiteList(allJpaCodeRules, CodeRuleEntity.class, CodeRuleEntity.class, HashSet.class, ArrayList.class, "script");
    int scriptsSize = allScripts.size();
    int codeRulesSize = allCodeRules.size();
    
    // 3、==========
    try (ByteArrayOutputStream bos = new ByteArrayOutputStream();
        ObjectOutputStream oos = new ObjectOutputStream(bos);) {
      oos.writeInt(scriptsSize);
      oos.writeInt(codeRulesSize);
      
      // 4、===========
      for (ScriptEntity scriptItem : allScripts) {
        oos.writeObject(scriptItem);
        // 将动态脚本关联的脚本附件写入压缩文件
        String relativePath = scriptItem.getFileCode();
        String fileName = scriptItem.getFileName();
        this.writeZipFile(zipf, relativePath, fileName);
      }
      // 5、==========
      for (CodeRuleEntity codeRuleItem : allCodeRules) {
        oos.writeObject(codeRuleItem);
      }
      // 6、=========
      String zipFileName = "codeRule.in";
      byte[] bosBytes = bos.toByteArray();
      this.writeZipFile(zipf, zipFileName, bosBytes);
    } catch(IOException e) {
      LOGGER.error(e.getMessage() , e);
      throw new IllegalArgumentException(e.getMessage());
    }
  }
  
  /**
   * 该私有方法用于对指定的数据字典进行迁出
   * @param zipf
   * @param dictIds
   */
  private void handleDict(ZipOutputStream zipf , String[] dictIds) {
    /*
     * 序列化格式如下：
     * 数据字典信息（文件名dict.in）
     * 数据字典分类数量|数据字典信息|
     * 第一个不重复且关联依赖已经在之前列出的数据字典分类信息中出现过的分类信息（基本信息，和关联的父级分类信息）|………………|
     * 第一个不重复的字典信息（基本信息、关联的字段分类信息、关联的字典明细信息）|………………|
     * 
     * 处理过程如下：
     * 1、首先进行数据字典的基本信息和关联信息的查询
     * 2、过滤出重复的涉及的所有父级字典分类信息，作为分类信息排序的起始节点
     * 3、利用双端队列的特点，对涉及到的所有字典分类信息进行排序（字段分类信息是一个层级为N的树结构，开始排序时，我们已知的是这个树的所有叶子节点）
     * , 那么排序要形成的结构，就是保证在读取时，当前读取到的分类信息的所有父级结构已经在之前被读取到了
     * 4、创建对象流，以便后续过程进行对象序列化时使用
     * 5、写入（序列化）不重复的字典分类信息（基本信息）
     * 6、写入（序列化）不重复的字典信息（基本信息，关联的字典分类信息、字典字段信息）
     * 7、创建一个名叫dict.in的zip文件entry，并将对象流写入文件、将文件写入压缩流
     * */
    // 1、========
    // 这里是等一下处理的字典分类的子页节点
    Set<DictCategoryEntity> allJpaLeafDictCategorys = Sets.newHashSet();
    Set<DictEntity> allJpaDicts = this.dictService.findDetailsByIds(dictIds);
    // 2、========
    for (DictEntity dictEntity : allJpaDicts) {
      allJpaLeafDictCategorys.add(dictEntity.getCategory());
    }
    Collection<DictCategoryEntity> allLeafDictCategorys = this.nebulaToolkitService.copyCollectionByWhiteList(allJpaLeafDictCategorys, DictCategoryEntity.class, DictCategoryEntity.class, HashSet.class, ArrayList.class);
    /*
     * 3、========
     * .那么我们最终要形成的链条是
     * .第一个字典分类的根节点<->第一个字典分类的下层父节<->…………<->第一个字典分类
     * <-> 第二个字典分类的最顶层父节点（且没有在之前出现过）<->…………<->第二个字典分类
     * <-> ………………………… <-> 最后一个字典分类
     * ,另外注意：这里每做一次递归，都会进行一次中拷，性能比较差，但是迁移功能使用度很低，所以性能问题的优先级不高
     * */
    LinkedList<DictCategoryEntity> allDictCategorys = Lists.newLinkedList();
    for (DictCategoryEntity leafDictCategoryItem : allLeafDictCategorys) {
      LinkedList<DictCategoryEntity> stackDictCategorys = Lists.newLinkedList();
      String leafDictId = leafDictCategoryItem.getId();
      DictCategoryEntity current = this.dictCategoryService.findDetailsById(leafDictId);
      current = this.nebulaToolkitService.copyObjectByWhiteList(current, DictCategoryEntity.class, HashSet.class, ArrayList.class, "parentCategory");
      // 添加当前叶子节点（为什么不适用leafDictCategoryItem直接添加呢？因为里面没有父级节点的信息）
      stackDictCategorys.addFirst(current);
      DictCategoryEntity parentDictCategory = current.getParentCategory();
      if(parentDictCategory == null) {
        allDictCategorys.addAll(stackDictCategorys);
        continue;
      }
      
      // 如果条件成立，说明要进行记录，记录是按照栈的特性进行
      final String parentDictCategoryIdf = parentDictCategory.getId();
      while(parentDictCategory != null && !allDictCategorys.stream().filter(item -> StringUtils.equals(item.getId(), parentDictCategoryIdf)).findAny().isPresent()) {
        final String parentDictCategoryId = parentDictCategory.getId();
        // 为什么还要查询一次呢？因为里面可能没有父级节点的信息
        parentDictCategory = this.dictCategoryService.findDetailsById(parentDictCategoryId);
        parentDictCategory = this.nebulaToolkitService.copyObjectByWhiteList(parentDictCategory, DictCategoryEntity.class, HashSet.class, ArrayList.class, "parentCategory");
        stackDictCategorys.addFirst(parentDictCategory);
        parentDictCategory = parentDictCategory.getParentCategory();
      }
      // 将本次循环得到的各级字典分类节点放入结果集合，
      allDictCategorys.addAll(stackDictCategorys);
    }
    // 接着将所有数据脱离JPA对象特性
    Collection<DictEntity> allDicts = this.nebulaToolkitService.copyCollectionByWhiteList(allJpaDicts, DictEntity.class, DictEntity.class, HashSet.class, ArrayList.class, "category" , "dictItemEntities");
    int dictCategorysSize = allDictCategorys.size();
    int dictsSize = allDicts.size();
    
    // 4、========
    try (ByteArrayOutputStream bos = new ByteArrayOutputStream();
        ObjectOutputStream oos = new ObjectOutputStream(bos);) {
      oos.writeInt(dictCategorysSize);
      oos.writeInt(dictsSize);
      
      // 5、==========
      for (DictCategoryEntity dictCategoryItem : allDictCategorys) {
        oos.writeObject(dictCategoryItem);
      }
      // 6、==========
      for (DictEntity dictItem : allDicts) {
        oos.writeObject(dictItem);
      }
      // 7、=========
      String zipFileName = "dict.in";
      byte[] bosBytes = bos.toByteArray();
      this.writeZipFile(zipf, zipFileName, bosBytes);
    } catch(IOException e) {
      throw new IllegalArgumentException(e.getMessage());
    }
  }
  
  /**
   * 该私有方法用于对指定的全局环境变量进行迁出
   * @param zipf
   * @param envIds
   */
  private void handleEnv(ZipOutputStream zipf , String[] envIds) {
    /*
     * 序列化格式如下：
     * 环境变量信息（文件名env.in）
     * 环境变量数量|
     * 第一个不重复的环境变量信息（只有基本信息，不带任何关联信息）|………………|
     * 
     * 处理过程包括：
     * 1、首先进行环境变量信息的基本信息和关联信息的查询
     * 2、创建对象流，以便后续过程进行对象序列化时使用
     * 3、写入（序列化）不重复的环境变量信息
     * 4、创建一个名叫env.in的zip文件entry，并将对象流写入文件、将文件写入压缩流
     * */
    
    // 1、==========
    Set<EnvironmentVariableEntity> allJpaEnvs = this.environmentVariableService.findDetailsByIds(envIds);
    // 接着将所有数据脱离JPA对象特性（注意：这里不需要关联序列化创建者或修改者信息了）
    Collection<EnvironmentVariableEntity> allEnvs = this.nebulaToolkitService.copyCollectionByWhiteList(allJpaEnvs, EnvironmentVariableEntity.class, EnvironmentVariableEntity.class, HashSet.class, ArrayList.class);
    int envsSize = allEnvs.size();
    
    // 2、==========
    try (ByteArrayOutputStream bos = new ByteArrayOutputStream();
        ObjectOutputStream oos = new ObjectOutputStream(bos);) {
      oos.writeInt(envsSize);
      // 3、========
      for (EnvironmentVariableEntity envItem : allEnvs) {
        oos.writeObject(envItem);
      }
      // 4、=========
      String zipFileName = "env.in";
      byte[] bosBytes = bos.toByteArray();
      this.writeZipFile(zipf, zipFileName, bosBytes);
    } catch(IOException e) {
      LOGGER.error(e.getMessage() , e);
      throw new IllegalArgumentException(e.getMessage());
    }
  }
  
  /**
   * 该私有方法用于对指定的远端服务源进行迁出
   * @param zipf
   * @param remoteServiceIds
   */
  private void handleRemoteService(ZipOutputStream zipf , String[] remoteServiceIds) {
    /*
     * 序列化格式如下：
     * 远端服务源（文件名：remote.in）
     * 远端服务地址数量|远端服务源数量|
     * 第一个不重复的远端服务地址信息（只有基本信息，不带任何关联信息）|………………|
     * 第一个不重复的远端服务源信息（基本信息，关联的服务地址信息和附件信息）|………………|
    *  
     * 处理过程包括：
    * 1、首先进行指定的远端服务源信息的详细信息查询
    * 2、从业务编码规则中过滤出不重复的服务源地址信息（可以利用JPA特点）
    * 3、创建对象流，以便后续过程进行对象序列化时使用
    * 4、写入（序列化）不重复的服务源地址信息（基本信息）
    * 5、写入（序列化）不重复的远端服务源信息（基本信息、关联信息和附件信息）
    * 6、创建一个名叫remote.in的zip文件entry，并将对象流写入文件、将文件写入压缩流
    * */
    
    // 1、==========
    Set<RemoteServiceAddressEntity> allJpaRemoteAddresses = Sets.newHashSet();
    Set<RemoteServiceEntity> allJpaRemoteServices = this.remoteServiceService.findDetailsByIds(remoteServiceIds);
    // 2、==========（由于allJpaRemoteServices集合是带有JPA特性的集合，所以使用JPA对象的引用特性进行重复对象过滤）
    for (RemoteServiceEntity remoteServiceItem : allJpaRemoteServices) {
      allJpaRemoteAddresses.add(remoteServiceItem.getRemoteServiceAddress());
    }
    // 接着将所有数据脱离JPA对象特性
    Collection<RemoteServiceAddressEntity> allRemoteAddresses = this.nebulaToolkitService.copyCollectionByWhiteList(allJpaRemoteAddresses, RemoteServiceAddressEntity.class, RemoteServiceAddressEntity.class, HashSet.class, ArrayList.class);
    Collection<RemoteServiceEntity> allRemoteServices = this.nebulaToolkitService.copyCollectionByWhiteList(allJpaRemoteServices, RemoteServiceEntity.class, RemoteServiceEntity.class, HashSet.class, ArrayList.class, "remoteServiceAddress");
    int remoteAddressSize = allRemoteAddresses.size();
    int remoteServicesSize = allRemoteServices.size();
    
    // 3、=========
    try (ByteArrayOutputStream bos = new ByteArrayOutputStream();
        ObjectOutputStream oos = new ObjectOutputStream(bos);) {
      oos.writeInt(remoteAddressSize);
      oos.writeInt(remoteServicesSize);
      
      // 4、========
      for (RemoteServiceAddressEntity allRemoteAddressItem : allRemoteAddresses) {
        oos.writeObject(allRemoteAddressItem);
      }
      // 5、========
      for (RemoteServiceEntity allRemoteServiceItem : allRemoteServices) {
        oos.writeObject(allRemoteServiceItem);
        // 附件信息在这里写入
        String relativePath = allRemoteServiceItem.getJsonRelativePath();
        String jsonFileName = allRemoteServiceItem.getJsonName();
        Validate.isTrue(StringUtils.isNotBlank(relativePath) && StringUtils.isNotBlank(jsonFileName)
          , "在迁出远端调用服务信息[%s]发现错误的附件描述地址，请检查数据" , allRemoteServiceItem.getId());
        this.writeZipFile(zipf, relativePath, jsonFileName);
      }
      // 6、=========
      String zipFileName = "remote.in";
      byte[] bosBytes = bos.toByteArray();
      this.writeZipFile(zipf, zipFileName, bosBytes);
    } catch(IOException e) {
      LOGGER.error(e.getMessage() , e);
      throw new IllegalArgumentException(e.getMessage());
    }
  }
  
  /**
   * 将指定的二进制内容，写入zip压缩包中对应的文件路径下
   * @param zipf zip压缩流
   * @param zipFileName zip压缩包中文件的相对路径
   * @param bosBytes 写入的压缩内容
   * @throws IOException
   */
  private void writeZipFile(ZipOutputStream zipf , String zipFileName , byte[] bosBytes) throws IOException {
    ZipEntry zipEntry = new ZipEntry(zipFileName);
    zipf.putNextEntry(zipEntry);
    int maxLen = 9060;
    byte[] fileContents = new byte[maxLen];
    int realLen = 0;
    try (ByteArrayInputStream bis = new ByteArrayInputStream(bosBytes);) {
      while((realLen = bis.read(fileContents, 0, maxLen)) != -1) {
        zipf.write(fileContents, 0, realLen);
      }
    }
  }
  
  /**
   * 该私有方法将指定的服务器端的某一个完成路径压入zip压缩流
   * @param zipf
   * @param fullFilePath
   * @throws IOException
   */
  private void writeZipFile(ZipOutputStream zipf , String relativePath , String fileName) throws IOException {
    byte[] templateLayoutContents = this.nebulaFileService.readFileContent(relativePath, fileName);
    String relativeFilePath = StringUtils.join(relativePath , "/" , fileName);
    // 最头部出现的“/”或者“\\”需要去掉
    if(relativeFilePath.indexOf('/') != -1 || relativeFilePath.indexOf('\\') != -1) {
      relativeFilePath = relativeFilePath.substring(1, relativeFilePath.length());
    }
    
    // 这样layoutFis才能完整关闭
    try(InputStream layoutcis = new ByteArrayInputStream(templateLayoutContents);) {
      ZipEntry zipEntry = new ZipEntry(relativeFilePath);
      zipf.putNextEntry(zipEntry);
      // 写入压缩文件内容
      int maxLen = 9060;
      byte[] fileContents = new byte[maxLen];
      int realLen = 0;
      while((realLen = layoutcis.read(fileContents, 0, maxLen)) != -1) {
        zipf.write(fileContents, 0, realLen);
      }
    } 
  }
}
