package com.bizunited.platform.core.service.migrate.internal;

import com.bizunited.platform.core.entity.CodeRuleEntity;
import com.bizunited.platform.core.entity.DataSourceEntity;
import com.bizunited.platform.core.entity.DataViewAuthHorizontalEntity;
import com.bizunited.platform.core.entity.DataViewAuthHorizontalRelationEntity;
import com.bizunited.platform.core.entity.DataViewAuthVerticalEntity;
import com.bizunited.platform.core.entity.DataViewAuthVerticalRelationEntity;
import com.bizunited.platform.core.entity.DataViewEntity;
import com.bizunited.platform.core.entity.DataViewFieldEntity;
import com.bizunited.platform.core.entity.DataViewFilterEntity;
import com.bizunited.platform.core.entity.DataViewGroupEntity;
import com.bizunited.platform.core.entity.DataViewSystemEntity;
import com.bizunited.platform.core.entity.DictCategoryEntity;
import com.bizunited.platform.core.entity.DictEntity;
import com.bizunited.platform.core.entity.EnvironmentVariableEntity;
import com.bizunited.platform.core.entity.MigrateConfigAnalysisEntity;
import com.bizunited.platform.core.entity.MigrateImportEntity;
import com.bizunited.platform.core.entity.OrdinaryFileEntity;
import com.bizunited.platform.core.entity.RemoteServiceAddressEntity;
import com.bizunited.platform.core.entity.RemoteServiceEntity;
import com.bizunited.platform.core.entity.ScriptEntity;
import com.bizunited.platform.core.entity.UserEntity;
import com.bizunited.platform.core.repository.dynamic.DynamicDataSourceManager;
import com.bizunited.platform.core.repository.migrate.MigrateConfigAnalysisRepository;
import com.bizunited.platform.core.repository.migrate.MigrateImportRepository;
import com.bizunited.platform.core.service.CodeRuleService;
import com.bizunited.platform.core.service.DictCategoryService;
import com.bizunited.platform.core.service.DictService;
import com.bizunited.platform.core.service.EnvironmentVariableService;
import com.bizunited.platform.core.service.RemoteServiceAddressService;
import com.bizunited.platform.core.service.RemoteServiceService;
import com.bizunited.platform.core.service.ScriptService;
import com.bizunited.platform.core.service.dataview.DataSourceService;
import com.bizunited.platform.core.service.dataview.DataViewAuthHorizontalService;
import com.bizunited.platform.core.service.dataview.DataViewAuthVerticalService;
import com.bizunited.platform.core.service.dataview.DataViewFieldService;
import com.bizunited.platform.core.service.dataview.DataViewGroupService;
import com.bizunited.platform.core.service.dataview.DataViewService;
import com.bizunited.platform.core.service.file.NebulaFileService;
import com.bizunited.platform.core.service.image.FileUpdateService;
import com.bizunited.platform.core.service.migrate.MigrateImportService;
import com.bizunited.platform.rbac.server.service.UserService;
import com.bizunited.platform.rbac.server.vo.UserVo;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.web.multipart.MultipartFile;

import javax.transaction.Transactional;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.security.Principal;
import java.util.Base64;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

/**
 * MigrateImportServiceImpl
 *
 * @description: 数据信息迁入的查询，分析，执行。包括基础配置信息与表单信息
 * @author: yanwe
 * @date: 03/Sep/2019 15:47
 */
@Service("MigrateImportService")
public class MigrateImportServiceImpl implements MigrateImportService {

  @Value("${nebula.file.fileRoot}")
  private String fileRoot;

  @Autowired private MigrateImportRepository migrateImportRepository;
  @Autowired private MigrateConfigAnalysisRepository migrateConfigAnalysisRepository;
  @Autowired private UserService userService;
  @Autowired private FileUpdateService fileUpdateService;
  @Autowired private NebulaFileService nebulaFileService;
  @Autowired private DataSourceService dataSourceService;
  @Autowired private DynamicDataSourceManager dynamicDataSourceManager;
  @Autowired private DataViewService dataViewService;
  @Autowired private DataViewGroupService dataViewGroupService;
  @Autowired private DataViewAuthHorizontalService dataViewAuthHorizontalService;
  @Autowired private DataViewAuthVerticalService dataViewAuthVerticalService;
  @Autowired private DataViewFieldService dataViewFieldService;
  @Autowired private ScriptService scriptService;
  @Autowired private CodeRuleService codeRuleService;
  @Autowired private DictCategoryService dictCategoryService;
  @Autowired private DictService dictService;
  @Autowired private EnvironmentVariableService environmentVariableService;
  @Autowired private RemoteServiceAddressService remoteServiceAddressService;
  @Autowired private RemoteServiceService remoteServiceService;

  private static final String DATA_VIEW_FILENAME = "dataview.in";
  private static final String CODE_RULE_FILENAME = "codeRule.in";
  private static final String DICT_FILENAME = "dict.in";
  private static final String ENV_FILENAME = "env.in";
  private static final String REMOTE_FILENAME = "remote.in";
  private static final String ERROR_MESS = "数据视图横向权限关联的数据视图不能为空！";

  @Override
  public Page<MigrateImportEntity> findByCondition(Map<String, Object> conditions, Pageable pageable) {
    Validate.notNull(conditions, "查询条件不能为空！");
    Validate.notNull(pageable, "分页信息不能为空！");
    return migrateImportRepository.findByConditions(pageable, conditions);
  }

  @Override
  @Transactional
  public MigrateImportEntity upload(String code, String migrateDesc, Principal currentUser, MultipartFile file) {
    Validate.notNull(file, "上传文件不能为空！");
    Validate.notBlank(code,"上次编码不能为空！");
    Validate.notNull(currentUser,"当前登录信息不能为空！");
    // 构造记录信息
    MigrateImportEntity migrateImportEntity = new MigrateImportEntity();
    // 保存编码
    migrateImportEntity.setCode(code);
    // 上传时间
    migrateImportEntity.setCreateTime(new Date());
    // 上传时还未执行
    migrateImportEntity.setExecuted(false);
    // 上传人
    migrateImportEntity.setCreator(this.getCurrentUser(currentUser.getName()));
    // 上传文件原名
    migrateImportEntity.setOriginalFileName(file.getOriginalFilename());
    // 保存上传文件
    MultipartFile[] files = new MultipartFile[] {file};
    List<OrdinaryFileEntity> ordinaryFileEntities =
        fileUpdateService.fileUpload("migrate", currentUser.getName(), null, files);
    Validate.notEmpty(files, "保存上传文件失败，请检查！");
    OrdinaryFileEntity fileEntity = ordinaryFileEntities.get(0);
    Validate.notNull(fileEntity,"上次文件保存返回结果不能为空！");
    migrateImportEntity.setFileName(fileEntity.getFileName());
    migrateImportEntity.setRelativeLocal(fileEntity.getRelativeLocal());
    migrateImportEntity.setMigrateDesc(migrateDesc);
    return this.create(migrateImportEntity);
  }

  @Override
  @Transactional
  public MigrateImportEntity create(MigrateImportEntity migrateImportEntity) {
    Validate.notBlank(migrateImportEntity.getCode(), "新增迁入文件的编号不能为空！");
    MigrateImportEntity existCode = migrateImportRepository.findByCode(migrateImportEntity.getCode());
    Validate.isTrue(null == existCode, "迁入文件编号已存在，请检查！");
    Validate.notNull(migrateImportEntity.getCreateTime(), "保存记录时，导入时间不能为空！");
    Validate.notBlank(migrateImportEntity.getRelativeLocal(), "迁入文件在本地路径不能为空！");
    Validate.notBlank(migrateImportEntity.getFileName(), "迁入文件重命名后的文件名字不能为空！");
    Validate.notNull(migrateImportEntity.getCreator(), "迁入文件上传人不能为空！");
    Validate.notNull(migrateImportEntity.getCreateTime(), "迁入文件上传时间不能为空！");
    Validate.notNull(migrateImportEntity.getExecuted(), "迁入文件任务是否被执行不能为空！");
    return migrateImportRepository.save(migrateImportEntity);
  }

  @Override
  @Transactional
  public MigrateConfigAnalysisEntity analysisConfig(String migrateImportId, Principal currentUser) {
    /*
     * 分析迁入的基础配置信息逻辑
     * 1.根据输入的ID，获取文件
     * 2.分析迁入的数据视图（详见私有方法中的详细描述）
     * 3.分析可能迁入的业务规则编码（详见私有方法中的详细描述)
     * 4.分析可能迁入的数据字典（详见私有方法中的详细描述)
     * 5.分析可能的全局环境变量（详见私有方法中的详细描述)
     * 6.分析可能的远端调用源（详见私有方法中的详细描述)
     */
    Validate.notBlank(migrateImportId, "迁入上传ID不能为空！");
    Validate.notNull(migrateImportId, "迁入上传ID不能为空！");
    Optional<MigrateImportEntity> importEntityOp = migrateImportRepository.findById(migrateImportId);
    MigrateImportEntity importEntity = importEntityOp.orElse(null);
    Validate.notNull(importEntity, "上传记录信息不能为空！");
    MigrateConfigAnalysisEntity result = new MigrateConfigAnalysisEntity();
    String fileName = importEntity.getFileName();
    String relativeLocal = importEntity.getRelativeLocal();
    byte[] fileBytes = nebulaFileService.readFileContent(relativeLocal, fileName);
    // 首先保存 BYTE[] 到本地文件
    File localFile = this.saveByteToLocalFile(fileBytes);
    try (ZipFile zipFile = new ZipFile(localFile)) {
      // 分析各项，具体逻辑在各自私有方法中
      this.analysisDataView(zipFile, result);
      this.analysisCodeRule(zipFile, result);
      this.analysisDict(zipFile, result);
      this.analysisEnv(zipFile, result);
      this.analysisRemote(zipFile, result);
      // 保存关联上传，分析人，分析时间
      result.setMigrateImportEntity(importEntity);
      result.setCreateTime(new Date());
      result.setExecutor(this.getCurrentUser(currentUser.getName()));
      result = migrateConfigAnalysisRepository.save(result);
    } catch (IOException | ClassNotFoundException e) {
      throw new IllegalArgumentException("解析迁入基础配置数据信息失败，原因：" + e.getMessage());
    }
    return result;
  }

  @Override
  @Transactional
  public MigrateImportEntity execute(String migrateImportId, Principal currentUser) {
    Validate.notBlank(migrateImportId, "迁移文件ID不能为空！");
    Optional<MigrateImportEntity> importEntityOp = migrateImportRepository.findById(migrateImportId);
    MigrateImportEntity importEntity = importEntityOp.orElse(null);
    Validate.notNull(importEntity, "上传记录信息不能为空！");
    String fileName = importEntity.getFileName();
    String relativeLocal = importEntity.getRelativeLocal();
    byte[] fileBytes = nebulaFileService.readFileContent(relativeLocal, fileName);
    // 首先保存 BYTE[] 到本地文件
    File localFile = this.saveByteToLocalFile(fileBytes);
    try (ZipFile zipFile = new ZipFile(localFile)) {
      // 执行各项，具体逻辑在各自私有方法中
      this.executeDataView(zipFile, currentUser);
      this.executeCodeRule(zipFile);
      this.executeDict(zipFile);
      this.executeEnv(zipFile, currentUser);
      this.executeRemoteService(zipFile);
      // 保存执行人，执行时间
      importEntity.setExecuted(true);
      importEntity.setExecutor(this.getCurrentUser(currentUser.getName()));
      importEntity.setExecuteTime(new Date());
    } catch (IOException | ClassNotFoundException e) {
      throw new IllegalArgumentException("执行迁入基础配置数据信息失败，原因：" + e.getMessage());
    }
    return migrateImportRepository.save(importEntity);
  }

  @Override
  public Set<MigrateConfigAnalysisEntity> findConfigAnalysisByImport(String importId) {
    if (StringUtils.isBlank(importId)) {
      return Collections.emptySet();
    }
    return migrateConfigAnalysisRepository.findByImportId(importId);
  }

  /**
   * 获取当前登录账户id，注入到新的UserEntity中
   *
   * @param account
   * @return
   */
  private UserEntity getCurrentUser(String account) {
    UserVo userVo = userService.findByAccount(account);
    Validate.notNull(userVo, "上传模板信息时未找到当前用户信息，请检查！");
    UserEntity userEntity = new UserEntity();
    userEntity.setId(userVo.getId());
    return userEntity;
  }

  /**
   * 验证数据视图分析相关
   * @param zipFile
   * @param result
   * @throws IOException
   * @throws ClassNotFoundException
   */
  private void analysisDataView(ZipFile zipFile, MigrateConfigAnalysisEntity result) throws IOException, ClassNotFoundException {
    ZipEntry dataViewEntry = zipFile.getEntry(DATA_VIEW_FILENAME);
    if (dataViewEntry == null) {
      return;
    }
    InputStream is = zipFile.getInputStream(dataViewEntry);
    ObjectInputStream ois = new ObjectInputStream(is);
    // 获取数据源数量，数据分组数量，数据视图数量，数据权限-横向权限数量，数据权限-纵向权限数量
    // 此处读取ZIP包内数据格式，参考 MigrateExportServiceImpl 类中handleDateView 方法
    Integer dataSourceNum = ois.readInt();
    Integer dataGroupNum = ois.readInt();
    Integer dataViewNum = ois.readInt();
    Integer dataAuthHorNum = ois.readInt();
    Integer dataAuthVerNum = ois.readInt();
    result.setDataviewSourceNum(dataSourceNum);
    result.setDataviewGroupNum(dataGroupNum);
    result.setDataviewNum(dataViewNum);
    result.setDataviewAuthHorNum(dataAuthHorNum);
    result.setDataviewAuthVerNum(dataAuthVerNum);
    // 验证数据源,主要验证编码是否有重复
    for (int i = 0; dataSourceNum != null && i < dataSourceNum; i++) {
      DataSourceEntity sourceEntity = (DataSourceEntity) ois.readObject();
      if (sourceEntity == null) {
        continue;
      }
      DataSourceEntity existSource = dataSourceService.findDetailsByCode(sourceEntity.getCode());
      if (existSource == null) {
        result.setDataviewSourceCreate(result.getDataviewSourceCreate() + 1);
      }
    }
    // 验证数据分组，主要验证编码是否有重复
    for (int i = 0; dataGroupNum != null && i < dataGroupNum; i++) {
      DataViewGroupEntity groupEntity = (DataViewGroupEntity) ois.readObject();
      if (groupEntity == null) {
        continue;
      }
      DataViewGroupEntity existGroup = dataViewGroupService.findByCode(groupEntity.getCode());
      if (existGroup == null) {
        result.setDataviewGroupCreate(result.getDataviewGroupCreate() + 1);
      }
    }
    // 验证数据视图，主要验证编码是否有重复
    for (int i = 0; dataViewNum != null && i < dataViewNum; i++) {
      DataViewEntity dataView = (DataViewEntity) ois.readObject();
      if (dataView == null) {
        continue;
      }
      DataViewEntity existDataview = dataViewService.findByCode(dataView.getCode());
      if (existDataview == null) {
        result.setDataviewCreate(result.getDataviewCreate() + 1);
      }
    }
    // 验证数据视图横向权限，因为没有编码，主要验证是否存在
    for (int i = 0; dataAuthHorNum != null && i < dataAuthHorNum; i++) {
      DataViewAuthHorizontalEntity horizontalEntity =
          (DataViewAuthHorizontalEntity) ois.readObject();
      if (horizontalEntity == null) {
        continue;
      }
      result.setDataviewAuthHorCreate(result.getDataviewAuthHorCreate() + 1);
    }
    // 验证数据视图纵向权限，因为没有编码，主要验证是否存在
    for (int i = 0; dataAuthVerNum != null && i < dataAuthVerNum; i++) {
      DataViewAuthVerticalEntity verticalEntity = (DataViewAuthVerticalEntity) ois.readObject();
      if (verticalEntity == null) {
        continue;
      }
      result.setDataviewAuthVerCreate(result.getDataviewAuthVerCreate() + 1);
    }
  }

  /**
   * 验证脚本与业务编码
   *
   * @param zipFile
   * @param result
   * @throws IOException
   * @throws ClassNotFoundException
   */
  private void analysisCodeRule(ZipFile zipFile, MigrateConfigAnalysisEntity result)
      throws IOException, ClassNotFoundException {
    ZipEntry codeRuleEntry = zipFile.getEntry(CODE_RULE_FILENAME);
    if (codeRuleEntry == null) {
      return;
    }
    InputStream is = zipFile.getInputStream(codeRuleEntry);
    ObjectInputStream ois = new ObjectInputStream(is);
    // 获取脚本数量，业务编码规则数量
    Integer scriptNum = ois.readInt();
    Integer codeRuleNum = ois.readInt();
    result.setScriptNum(scriptNum);
    result.setCodeRuleNum(codeRuleNum);
    // 验证脚本，主要根据名称验证
    for (int i = 0; scriptNum != null && i < scriptNum; i++) {
      ScriptEntity scriptEntity = (ScriptEntity) ois.readObject();
      if (scriptEntity == null) {
        continue;
      }
      ScriptEntity existName = scriptService.findByName(scriptEntity.getName());
      if (existName == null) {
        result.setScriptCreate(result.getScriptCreate() + 1);
      }
    }
    // 验证业务编码，主要根据名称验证
    for (int i = 0; codeRuleNum != null && i < codeRuleNum; i++) {
      CodeRuleEntity codeRuleEntity = (CodeRuleEntity) ois.readObject();
      if (codeRuleEntity == null) {
        continue;
      }
      CodeRuleEntity existCode = codeRuleService.findByRuleCode(codeRuleEntity.getRuleCode());
      if (existCode == null) {
        result.setCodeRuleCreate(result.getCodeRuleCreate() + 1);
      }
    }
  }

  /**
   * 验证字典
   *
   * @param zipFile
   * @param result
   * @throws IOException
   * @throws ClassNotFoundException
   */
  private void analysisDict(ZipFile zipFile, MigrateConfigAnalysisEntity result)
      throws IOException, ClassNotFoundException {
    ZipEntry dictEntry = zipFile.getEntry(DICT_FILENAME);
    if (dictEntry == null) {
      return;
    }
    InputStream is = zipFile.getInputStream(dictEntry);
    ObjectInputStream ois = new ObjectInputStream(is);
    // 读取字典分类数量，字典数量
    Integer dictCateNum = ois.readInt();
    Integer dictNum = ois.readInt();
    result.setDictCateNum(dictCateNum);
    result.setDictNum(dictNum);
    // 验证字典分组
    for (int i = 0; dictCateNum != null && i < dictCateNum; i++) {
      DictCategoryEntity dictCategory = (DictCategoryEntity) ois.readObject();
      if (dictCategory == null) {
        continue;
      }
      DictCategoryEntity existDictCategory =
          dictCategoryService.findByCode(dictCategory.getCateCode());
      if (existDictCategory == null) {
        result.setDictCateCreate(result.getDictCateCreate() + 1);
      }
    }
    // 验证字典
    for (int i = 0; dictNum != null && i < dictNum; i++) {
      DictEntity dictEntity = (DictEntity) ois.readObject();
      if (dictEntity == null) {
        continue;
      }
      DictEntity existDictEntity = dictService.findByDictCode(dictEntity.getDictCode());
      if (existDictEntity == null) {
        result.setDictCreate(result.getDictCreate() + 1);
      }
    }
  }

  /**
   * 验证环境变量
   *
   * @param zipFile
   * @param result
   * @throws IOException
   * @throws ClassNotFoundException
   */
  private void analysisEnv(ZipFile zipFile, MigrateConfigAnalysisEntity result)
      throws IOException, ClassNotFoundException {
    ZipEntry envEntry = zipFile.getEntry(ENV_FILENAME);
    if (envEntry == null) {
      return;
    }
    InputStream is = zipFile.getInputStream(envEntry);
    ObjectInputStream ois = new ObjectInputStream(is);
    // 数量
    Integer envNum = ois.readInt();
    result.setEnvNum(envNum);
    // 验证环境变量
    for (int i = 0; envNum != null && i < envNum; i++) {
      EnvironmentVariableEntity env = (EnvironmentVariableEntity) ois.readObject();
      if (env == null) {
        continue;
      }
      EnvironmentVariableEntity existCode =
          environmentVariableService.findByCode(env.getParamCode());
      EnvironmentVariableEntity existKey = environmentVariableService.findByKey(env.getParamKey());
      // 环境变量中，编码与参数KEY都必须唯一
      if (existCode == null && existKey == null) {
        result.setDictCreate(result.getDictCreate() + 1);
      }
    }
  }

  /**
   * 验证远端数据源
   *
   * @param zipFile
   * @param result
   * @throws IOException
   * @throws ClassNotFoundException
   */
  private void analysisRemote(ZipFile zipFile, MigrateConfigAnalysisEntity result)
      throws IOException, ClassNotFoundException {
    ZipEntry remoteEntry = zipFile.getEntry(REMOTE_FILENAME);
    if (remoteEntry == null) {
      return;
    }
    InputStream is = zipFile.getInputStream(remoteEntry);
    ObjectInputStream ois = new ObjectInputStream(is);
    // 获取远端服务地址数量,远端服务源数量
    Integer remoteAddressNum = ois.readInt();
    Integer remoteServiceNum = ois.readInt();
    result.setRemoteAddressNum(remoteAddressNum);
    result.setRemoteServiceNum(remoteServiceNum);
    // 验证远端服务地址
    for (int i = 0; remoteAddressNum != null && i < remoteAddressNum; i++) {
      RemoteServiceAddressEntity address = (RemoteServiceAddressEntity) ois.readObject();
      if (address == null) {
        continue;
      }
      RemoteServiceAddressEntity existAddress =
          remoteServiceAddressService.findByCode(address.getCode());
      if (existAddress == null) {
        result.setRemoteAddressCreate(result.getRemoteAddressCreate() + 1);
      }
    }
    // 验证端服务源
    for (int i = 0; remoteServiceNum != null && i < remoteServiceNum; i++) {
      RemoteServiceEntity service = (RemoteServiceEntity) ois.readObject();
      if (service == null) {
        continue;
      }
      RemoteServiceEntity existService = remoteServiceService.findByCode(service.getCode());
      if (existService == null) {
        result.setRemoteServiceCreate(result.getRemoteServiceCreate() + 1);
      }
    }
  }

  /**
   * 执行迁入数据视图
   *
   * @param zipFile
   * @param currentUser
   */
  private void executeDataView(ZipFile zipFile, Principal currentUser)
      throws IOException, ClassNotFoundException {
    ZipEntry dataViewEntry = zipFile.getEntry(DATA_VIEW_FILENAME);
    if (dataViewEntry == null) {
      return;
    }
    // 用来存储新旧ID的对应关系
    Map<String, String> idmap = new HashMap<>();
    InputStream is = zipFile.getInputStream(dataViewEntry);
    ObjectInputStream ois = new ObjectInputStream(is);
    // 获取数据源数量，数据分组数量，数据视图数量，数据权限-横向权限数量，数据权限-纵向权限数量
    Integer dataSourceNum = ois.readInt();
    Integer dataGroupNum = ois.readInt();
    Integer dataViewNum = ois.readInt();
    Integer dataAuthHorNum = ois.readInt();
    Integer dataAuthVerNum = ois.readInt();
    // 录入数据源
    for (int i = 0; dataSourceNum != null && i < dataSourceNum; i++) {
      DataSourceEntity sourceEntity = (DataSourceEntity) ois.readObject();
      if (sourceEntity == null) {
        continue;
      }
      // 验证数据源,主要验证编码是否有重复
      DataSourceEntity existSource = dataSourceService.findDetailsByCode(sourceEntity.getCode());
      if (existSource != null) {
        continue;
      }
      String oldSourceId = sourceEntity.getId();
      sourceEntity.setId(null);
      // 调用数据源管理类的新增方法，此步创建会进行连接验证
      sourceEntity = dynamicDataSourceManager.create(sourceEntity);
      migrateImportRepository.flush();
      idmap.put(oldSourceId, sourceEntity.getId());
    }
    // 录入数据分组
    for (int i = 0; dataGroupNum != null && i < dataGroupNum; i++) {
      DataViewGroupEntity groupEntity = (DataViewGroupEntity) ois.readObject();
      if (groupEntity == null) {
        continue;
      }
      DataViewGroupEntity existGroup = dataViewGroupService.findByCode(groupEntity.getCode());
      if (existGroup != null) {
        continue;
      }
      String oldGroupId = groupEntity.getId();
      groupEntity.setId(null);
      DataSourceEntity oldSource = groupEntity.getDataSource();
      // 数据源不存在，则说明是主数据源下的，不需要变更ID
      if (oldSource != null) {
        oldSource.setId(idmap.get(oldSource.getId()));
        groupEntity.setDataSource(oldSource);
      }
      groupEntity = dataViewGroupService.create(groupEntity, false);
      migrateImportRepository.flush();
      idmap.put(oldGroupId, groupEntity.getId());
    }
    // 录入数据视图
    for (int i = 0; dataViewNum != null && i < dataViewNum; i++) {
      DataViewEntity dataView = (DataViewEntity) ois.readObject();
      if (dataView == null) {
        continue;
      }
      DataViewEntity existDataView = dataViewService.findByCode(dataView.getCode());
      if (existDataView != null) {
        continue;
      }
      String oldDataViewId = dataView.getId();
      dataView.setId(null);
      DataSourceEntity dataSource = dataView.getDataSource();
      if (dataSource != null) {
        dataSource.setId(idmap.get(dataSource.getId()));
        dataView.setDataSource(dataSource);
      }
      DataViewGroupEntity dataViewGroup = dataView.getDataViewGroup();
      if (dataViewGroup != null) {
        dataViewGroup.setId(idmap.get(dataViewGroup.getId()));
        dataView.setDataViewGroup(dataViewGroup);
      }
      // 保存数据视图方法内，会同时保存其下的输出字段信息，用户过滤条件信息，系统查询字段信息，所以需要去除其ID
      this.removeDataViewItemId(dataView);
      dataView = dataViewService.create(dataView, true);
      migrateImportRepository.flush();
      idmap.put(oldDataViewId, dataView.getId());
    }
    // 录入数据视图横向权限
    for (int i = 0; dataAuthHorNum != null && i < dataAuthHorNum; i++) {
      DataViewAuthHorizontalEntity horizontalEntity =
          (DataViewAuthHorizontalEntity) ois.readObject();
      if (horizontalEntity == null) {
        continue;
      }
      // 更新数据视图 关联
      DataViewEntity oldDataView = horizontalEntity.getDataView();
      Validate.notNull(oldDataView,ERROR_MESS);
      DataViewEntity newDataView = new DataViewEntity();
      newDataView.setId(idmap.get(oldDataView.getId()));
      horizontalEntity.setDataView(newDataView);
      // 更新用户权限筛选字段 关联
      DataViewFieldEntity oldFieldEntity = horizontalEntity.getField();
      Validate.notNull(oldFieldEntity,ERROR_MESS);
      DataViewFieldEntity newFieldEntity = this.dataViewFieldService.findByCode(oldFieldEntity.getCode());
      horizontalEntity.setField(newFieldEntity);
      // 抹去 明细 Set<DataViewAuthHorizontalRelationEntity> 中的ID
      Set<DataViewAuthHorizontalRelationEntity> relations = horizontalEntity.getAuthRelations();
      if(!CollectionUtils.isEmpty(relations)){
        for(DataViewAuthHorizontalRelationEntity relation : relations){
          relation.setId(null);
        }
        horizontalEntity.setAuthRelations(relations);
      }
      Set<DataViewAuthHorizontalEntity> sets = new HashSet<>();
      horizontalEntity.setId(null);
      sets.add(horizontalEntity);
      this.dataViewAuthHorizontalService.create(sets);
      this.migrateImportRepository.flush();

    }
    // 录入数据视图纵向权限
    for (int i = 0; dataAuthVerNum != null && i < dataAuthVerNum; i++) {
      DataViewAuthVerticalEntity verticalEntity = (DataViewAuthVerticalEntity) ois.readObject();
      if (verticalEntity == null) {
        continue;
      }
      // 更新数据视图 关联
      DataViewEntity oldDataView = verticalEntity.getDataView();
      Validate.notNull(oldDataView,"数据视图纵向权限关联的数据视图不能为空！");
      DataViewEntity newDataView = new DataViewEntity();
      newDataView.setId(idmap.get(oldDataView.getId()));
      verticalEntity.setDataView(newDataView);
      // 抹去 明细 Set<DataViewAuthHorizontalRelationEntity> 中的ID
      Set<DataViewAuthVerticalRelationEntity> relations = verticalEntity.getAuthRelations();
      if(!CollectionUtils.isEmpty(relations)){
        for(DataViewAuthVerticalRelationEntity relation : relations){
          relation.setId(null);
        }
        verticalEntity.setAuthRelations(relations);
      }

      // 更新 displayFields ManyToMany 中的关联关系
      Set<DataViewFieldEntity> displayFields = verticalEntity.getDisplayFields();
      Set<DataViewFieldEntity> newDisplayFields = new HashSet<>();
      if(!CollectionUtils.isEmpty(displayFields)){
        for(DataViewFieldEntity displayField : displayFields){
          DataViewFieldEntity newDisplayField = this.dataViewFieldService.findByCode(displayField.getCode());
          Validate.notNull(newDisplayField,"未找到纵向数据视图关联的displayFields在新环境中的实体!");
          newDisplayFields.add(newDisplayField);
        }
        verticalEntity.setDisplayFields(newDisplayFields);
      }
      Set<DataViewAuthVerticalEntity> sets = new HashSet<>();
      verticalEntity.setId(null);
      sets.add(verticalEntity);
      this.dataViewAuthVerticalService.create(sets);
      this.migrateImportRepository.flush();
    }
  }

  /**
   * 移除数据视图中输出字段信息，用户过滤条件信息，系统查询字段信息的ID
   *
   * @param dataViewEntity
   */
  private void removeDataViewItemId(DataViewEntity dataViewEntity) {
    Validate.notNull(dataViewEntity, "插入的数据视图不能为空！");
    if (!CollectionUtils.isEmpty(dataViewEntity.getFields())) {
      for (DataViewFieldEntity field : dataViewEntity.getFields()) {
        field.setId(null);
      }
    }
    if (!CollectionUtils.isEmpty(dataViewEntity.getFilters())) {
      for (DataViewFilterEntity filter : dataViewEntity.getFilters()) {
        filter.setId(null);
      }
    }
    if (!CollectionUtils.isEmpty(dataViewEntity.getSystemFilters())) {
      for (DataViewSystemEntity system : dataViewEntity.getSystemFilters()) {
        system.setId(null);
      }
    }
  }

  /**
   * 执行迁入编码规则
   *
   * @param zipFile
   * @param currentUser
   */
  private void executeCodeRule(ZipFile zipFile)
      throws IOException, ClassNotFoundException {
    ZipEntry codeRuleEntry = zipFile.getEntry(CODE_RULE_FILENAME);
    if (codeRuleEntry == null) {
      return;
    }
    // 用来存储新旧ID的对应关系
    Map<String, String> idmap = new HashMap<>();
    InputStream is = zipFile.getInputStream(codeRuleEntry);
    ObjectInputStream ois = new ObjectInputStream(is);
    // 获取脚本数量，业务编码规则数量
    Integer scriptNum = ois.readInt();
    Integer codeRuleNum = ois.readInt();
    // 验证脚本，主要根据名称验证
    for (int i = 0; scriptNum != null && i < scriptNum; i++) {
      ScriptEntity scriptEntity = (ScriptEntity) ois.readObject();
      if (scriptEntity == null) {
        continue;
      }
      ScriptEntity existName = scriptService.findByName(scriptEntity.getName());
      if (existName != null) {
        continue;
      }
      // 读取出脚本文件内容
      Boolean startWith =
          StringUtils.startsWith(scriptEntity.getFileCode(), "/")
              || StringUtils.startsWith(scriptEntity.getFileCode(), "\\");
      String filePath =
          Boolean.TRUE.equals(startWith) ? scriptEntity.getFileCode().substring(1) : scriptEntity.getFileCode();
      // 将脚本文件从之前保存的ZIP文件特定路径下获取出
      String fullScriptFilePath = StringUtils.join(filePath, "/", scriptEntity.getFileName());
      ZipEntry scriptFile = zipFile.getEntry(fullScriptFilePath);
      InputStream inputStream = zipFile.getInputStream(scriptFile);
      byte[] bytes = IOUtils.toByteArray(inputStream);
      // 保存文件到新的本地目录下
      String base64 = Base64.getEncoder().encodeToString(bytes);
      String oldScriptId = scriptEntity.getId();
      scriptEntity.setId(null);
      scriptEntity = scriptService.create(scriptEntity, base64);
      migrateImportRepository.flush();
      idmap.put(oldScriptId, scriptEntity.getId());
    }
    // 验证业务编码，主要根据名称验证
    for (int i = 0; codeRuleNum != null && i < codeRuleNum; i++) {
      CodeRuleEntity codeRuleEntity = (CodeRuleEntity) ois.readObject();
      if (codeRuleEntity == null) {
        continue;
      }
      CodeRuleEntity existCode = codeRuleService.findByRuleCode(codeRuleEntity.getRuleCode());
      CodeRuleEntity existName = codeRuleService.findByRuleName(codeRuleEntity.getRuleName());
      ScriptEntity script = codeRuleEntity.getScript();
      if (existCode != null || existName != null || script == null) {
        continue;
      }
      script = scriptService.findByName(script.getName());
      codeRuleEntity.setScript(script);
      codeRuleEntity.setId(null);
      codeRuleService.create(codeRuleEntity);
      migrateImportRepository.flush();
    }
  }

  /**
   * 执行迁入字典
   *
   * @param zipFile
   * @param currentUser
   */
  private void executeDict(ZipFile zipFile) throws IOException, ClassNotFoundException {
    ZipEntry dictEntry = zipFile.getEntry(DICT_FILENAME);
    if (dictEntry == null) {
      return;
    }
    Map<String,String> idMaps = new HashMap<>();
    InputStream is = zipFile.getInputStream(dictEntry);
    ObjectInputStream ois = new ObjectInputStream(is);
    // 读取字典分类数量，字典数量
    Integer dictCateNum = ois.readInt();
    Integer dictNum = ois.readInt();
    // 验证字典分组
    for (int i = 0; dictCateNum != null && i < dictCateNum; i++) {
      DictCategoryEntity dictCategory = (DictCategoryEntity) ois.readObject();
      if (dictCategory == null) {
        continue;
      }
      DictCategoryEntity existDictCategory =
          dictCategoryService.findByCode(dictCategory.getCateCode());
      if (existDictCategory == null) {
        String oldId = dictCategory.getId();
        //获取改字典分类的父级分类
        DictCategoryEntity parentCate = dictCategory.getParentCategory();
        String parentId = null;
        if(parentCate!=null){
          // 如果有父级分类，则替换父级分类的ID为新ID
          parentId = idMaps.get(parentCate.getId());
          Validate.notBlank(parentId,"保存字典分组中，未找到字典分组的父级分组在迁移后环境中的ID");
        }
        dictCategory.setId(null);
        dictCategory = dictCategoryService.create(dictCategory,parentId);
        migrateImportRepository.flush();
        idMaps.put(oldId,dictCategory.getId());
      }else {
        String existDictCategoryId = existDictCategory.getId();
        // 如果导入的字典分组在当前环境中已经存在，则使用当前存在的分组
        if(StringUtils.isNotBlank(existDictCategoryId)){
          idMaps.put(existDictCategoryId,existDictCategoryId);
        }
      }
    }
    // 验证字典
    for (int i = 0; dictNum != null && i < dictNum; i++) {
      DictEntity dictEntity = (DictEntity) ois.readObject();
      if (dictEntity == null) {
        continue;
      }
      DictEntity existDictEntity = dictService.findByDictCode(dictEntity.getDictCode());
      if (existDictEntity == null) {
        DictCategoryEntity dictCategory = dictEntity.getCategory();
        Validate.notNull(dictCategory,"迁移字典中，未发现字典的相关分组信息！");
        String newParentId = idMaps.get(dictCategory.getId());
        Validate.notBlank(newParentId,"迁移字典中，未发现字典的相关分组ID信息！");
        dictCategory.setId(newParentId);
        dictEntity.setCategory(dictCategory);
        // 为了保存字典明细，需要将该字典设置为未上架状态
        dictEntity.setDictStatus(false);
        dictEntity.setId(null);
        // 保存字典
        dictService.create(dictEntity, newParentId);
        migrateImportRepository.flush();
      }
    }
  }

  /**
   * 执行迁入环境变量
   *
   * @param zipFile
   * @param currentUser
   */
  private void executeEnv(ZipFile zipFile, Principal currentUser) throws IOException, ClassNotFoundException {
    ZipEntry envEntry = zipFile.getEntry(ENV_FILENAME);
    if (envEntry == null) {
      return;
    }
    InputStream is = zipFile.getInputStream(envEntry);
    ObjectInputStream ois = new ObjectInputStream(is);
    // 数量
    Integer envNum = ois.readInt();
    // 验证环境变量
    for (int i = 0; envNum != null && i < envNum; i++) {
      EnvironmentVariableEntity env = (EnvironmentVariableEntity) ois.readObject();
      if (env == null) {
        continue;
      }
      EnvironmentVariableEntity existCode = environmentVariableService.findByCode(env.getParamCode());
      EnvironmentVariableEntity existKey = environmentVariableService.findByKey(env.getParamKey());
      // 环境变量中，编码与参数KEY都必须唯一
      if (existCode == null && existKey == null) {
        env.setId(null);
        env.setCreateUser(this.getCurrentUser(currentUser.getName()));
        env.setCreateDate(new Date());
        env.setModifyUser(this.getCurrentUser(currentUser.getName()));
        env.setModifyDate(new Date());
        this.environmentVariableService.save(env);
      }
    }
  }

  /**
   * 执行迁入远端数据源
   *
   * @param zipFile
   * @param currentUser
   */
  private void executeRemoteService(ZipFile zipFile) throws IOException, ClassNotFoundException {
    ZipEntry remoteEntry = zipFile.getEntry(REMOTE_FILENAME);
    if (remoteEntry == null) {
      return;
    }
    Map<String,String> idMaps = new HashMap<>();
    InputStream is = zipFile.getInputStream(remoteEntry);
    ObjectInputStream ois = new ObjectInputStream(is);
    // 获取远端服务地址数量,远端服务源数量
    Integer remoteAddressNum = ois.readInt();
    Integer remoteServiceNum = ois.readInt();
    // 验证远端服务地址
    for (int i = 0; remoteAddressNum != null && i < remoteAddressNum; i++) {
      RemoteServiceAddressEntity address = (RemoteServiceAddressEntity) ois.readObject();
      if (address == null) {
        continue;
      }
      RemoteServiceAddressEntity existAddress = remoteServiceAddressService.findByCode(address.getCode());
      if (existAddress == null) {
        String oldId = address.getId();
        address.setId(null);
        address = remoteServiceAddressService.create(address);
        migrateImportRepository.flush();
        idMaps.put(oldId,address.getId());
      }
    }
    // 验证端服务源
    for (int i = 0; remoteServiceNum != null && i < remoteServiceNum; i++) {
      RemoteServiceEntity service = (RemoteServiceEntity) ois.readObject();
      if (service == null) {
        continue;
      }
      RemoteServiceEntity existService = remoteServiceService.findByCode(service.getCode());
      if (existService == null) {
        service.setId(null);
        // 连接新的服务源地址
        RemoteServiceAddressEntity address = service.getRemoteServiceAddress();
        Validate.notNull(address,"保存服务源时，相关服务地址不能为空！");
        String newAddressId = idMaps.get(address.getId());
        RemoteServiceAddressEntity newAddress = new RemoteServiceAddressEntity();
        newAddress.setId(newAddressId);
        service.setRemoteServiceAddress(newAddress);
        //迁移JSON文件
        Boolean startWith = StringUtils.startsWith(service.getJsonRelativePath(), "/") || StringUtils.startsWith(service.getJsonRelativePath(), "\\");
        String filePath = Boolean.TRUE.equals(startWith) ? service.getJsonRelativePath().substring(1) : service.getJsonRelativePath();
        // 将脚本文件从之前保存的ZIP文件特定路径下获取出
        String fullScriptFilePath = StringUtils.join(filePath, "/", service.getJsonName());
        ZipEntry scriptFile = zipFile.getEntry(fullScriptFilePath);
        InputStream inputStream = zipFile.getInputStream(scriptFile);
        byte[] bytes = IOUtils.toByteArray(inputStream);
        // 保存文件到当前系统下
        nebulaFileService.saveFile(service.getJsonRelativePath(),service.getJsonName(),service.getJsonName(),bytes);
        // 保存实体
        remoteServiceService.create(service);
      }
    }
  }

  /**
   * 保存BYTE[]到本地临时文件中
   *
   * @param bytes
   * @return
   */
  private File saveByteToLocalFile(byte[] bytes) {
    // 首先保存 BYTE[] 到本地文件
    ByteArrayInputStream byteArrayInputStream = new ByteArrayInputStream(bytes);
    String uuid = UUID.randomUUID().toString();
    File localFile = new File(StringUtils.join(fileRoot, "/migrateImportTemp/", uuid, ".zip"));
    File parentFile = localFile.getParentFile();
    try {
      if(!parentFile.exists()){
        Validate.isTrue(parentFile.mkdirs() , "系统创建父文件夹错误，请检查文件系统设定!!");
      }
      if (!localFile.exists()) {
        Validate.isTrue(localFile.createNewFile() , "系统创建文件错误，请检查文件系统设定!!");
      }
    } catch (IOException e) {
      throw new IllegalArgumentException(e.getMessage(), e);
    }
    // 开始写入压缩文件
    try (FileOutputStream os = new FileOutputStream(localFile); ) {
      int maxLen = 4096;
      int realLen;
      byte[] contents = new byte[maxLen];
      while ((realLen = byteArrayInputStream.read(contents, 0, maxLen)) != -1) {
        os.write(contents, 0, realLen);
      }
    } catch (IOException e) {
      throw new IllegalArgumentException("解析上传文件数据失败，原因：" + e.getMessage());
    }
    return localFile;
  }
}
