package com.bizunited.platform.core.service.security;

import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.core.common.constant.Constants;
import com.bizunited.platform.core.common.enums.RbacRelationEnum;
import com.bizunited.platform.core.entity.OrganizationEntity;
import com.bizunited.platform.core.entity.PositionEntity;
import com.bizunited.platform.core.entity.RbacSettingEntity;
import com.bizunited.platform.core.entity.RoleEntity;
import com.bizunited.platform.core.entity.UserEntity;
import com.bizunited.platform.core.repository.OrganizationRepository;
import com.bizunited.platform.core.repository.PositionRepository;
import com.bizunited.platform.core.repository.RoleRepository;
import com.bizunited.platform.core.repository.UserRepository;
import com.bizunited.platform.core.service.NebulaToolkitService;
import com.bizunited.platform.rbac.server.service.PositionService;
import com.bizunited.platform.rbac.server.vo.PositionVo;
import com.bizunited.platform.rbac.server.vo.UserVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.redisson.api.RedissonClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

/**
 * @description:
 * @author: yanwe
 * @date: 17/Jan/2019 15:01
 */
public class PositionServiceImpl implements PositionService {

  @Autowired 
  private PositionRepository positionRepository;
  @Autowired 
  private OrganizationRepository organizationRepository;
  @Autowired 
  private UserRepository userRepository;
  @Autowired 
  private RoleRepository roleRepository;
  @Autowired
  private RbacSettingService rbacSettingService;
  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private RedissonClient redissonClient;

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.PositionService#create(com.bizunited.platform.rbac.server.vo.PositionVo)
   */
  @Override
  @Transactional
  public PositionVo create(PositionVo positionVo) {
    Validate.notNull(positionVo, "岗位信息不能为空，请检查");
    Validate.isTrue(StringUtils.isBlank(positionVo.getId()), "添加岗位时，不能传入岗位id,请重新添加!!");
    Validate.notBlank(positionVo.getName(), "岗位名称不能为空，请检查!!");
    Validate.notNull(positionVo.getCode(), "岗位编号不能为空");
    positionVo.setCreateTime(new Date());
    PositionEntity currentPosition = positionRepository.findByCode(positionVo.getCode());
    Validate.isTrue(null == currentPosition,"存在重复的code,请重新输入岗位编码!!");
    if(positionVo.getParent() != null){
      Validate.notBlank(positionVo.getParent().getId(), "未找到该岗位的上级岗位的id，请检查！");
      PositionEntity parent = positionRepository.findById(positionVo.getParent().getId()).orElse(null);
      Validate.notNull(parent, "未找到该岗位的上级岗位，请检查！");
    }
    // 转换后进行保存
    PositionEntity positionEntity = this.nebulaToolkitService.copyObjectByWhiteList(positionVo, PositionEntity.class, HashSet.class, ArrayList.class, "parent");
    positionRepository.save(positionEntity);
    positionVo.setId(positionEntity.getId());
    return positionVo;
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.PositionService#update(com.bizunited.platform.rbac.server.vo.PositionVo)
   */
  @Override
  @Transactional
  public PositionVo update(PositionVo positionVo) {
    // 验证
    Validate.notNull(positionVo, "岗位信息不能为空，请检查");
    Validate.isTrue(!StringUtils.isBlank(positionVo.getId()), "修改岗位时，必须传入岗位id,请重新修改!!");
    Validate.notBlank(positionVo.getName(), "岗位名称不能为空，请检查!!");
    Optional<PositionEntity> op = positionRepository.findById(positionVo.getId());
    PositionEntity position = op.orElse(null);
    Validate.notNull(position, "未在数据层找到对应的岗位信息");
    if(positionVo.getParent() != null){
      String parentId = positionVo.getParent().getId();
      Validate.notBlank(parentId, "未找到该岗位的上级岗位的id，请检查！");
      Validate.isTrue(!positionVo.getId().equals(parentId), "禁止将该岗位本身设置为上级岗位");
      PositionEntity parent = positionRepository.findById(parentId).orElse(null);
      Validate.notNull(parent, "未找到该岗位的上级岗位，请检查！");
      position.setParent(parent);
    }else {
      position.setParent(null);
    }
    // 赋值并保存
    position.setName(positionVo.getName());
    Set<String> positionStack = new HashSet<>();
    positionStack.add(position.getId());
    this.validateCircular(position, positionStack);
    positionRepository.save(position);
    // 转换后返回
    return this.nebulaToolkitService.copyObjectByWhiteList(position, PositionVo.class, HashSet.class, ArrayList.class);
  }

  /**
   *判断是否形成循环依赖
   * @param parent
   * @param positionStack
   */
  private void validateCircular(PositionEntity parent, Set<String> positionStack) {
    if(CollectionUtils.isEmpty(parent.getChildren())) {
      return;
    }
    for (PositionEntity positionEntity : parent.getChildren()) {
      Validate.isTrue(!positionStack.contains(positionEntity.getId()), "形成循环依赖，更新失败，请检查！");
      positionStack.add(positionEntity.getId());
      this.validateCircular(positionEntity, positionStack);
    }
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.PositionService#updateStatus(java.lang.String)
   */
  @Override
  @Transactional
  public PositionVo updateStatus(String positionId) {
    Validate.notBlank(positionId, "岗位id不能为空，请检查");
    Optional<PositionEntity> op = positionRepository.findById(positionId);
    PositionEntity position = op.orElse(null);
    Validate.notNull(position, "没有该岗位，请检查!!");
    Integer status = position.getTstatus();
    
    // 反转状态，0:禁用,1:启用
    status = (status == 1 ? 0 : 1);
    position.setTstatus(status);
    positionRepository.save(position);
    
    // 转换后返回
    return this.nebulaToolkitService.copyObjectByWhiteList(position, PositionVo.class, HashSet.class, ArrayList.class);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.PositionService#findById(java.lang.String)
   */
  @Override
  public PositionVo findDetailsById(String positionId) {
    if (StringUtils.isBlank(positionId)) {
      return null;
    }
    PositionEntity positionEntity = positionRepository.findDetailsById(positionId);
    Validate.notNull(positionEntity , "未找到指定的岗位信息，请检查传参!!");
    
    // 转换后返回
    return this.nebulaToolkitService.copyObjectByWhiteList(positionEntity, PositionVo.class, LinkedHashSet.class, ArrayList.class, "organization" , "roles" , "users");
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.PositionService#bindOrgAndPosition(java.lang.String, java.lang.String)
   */
  @Override
  @Transactional
  public void bindOrgAndPosition(String orgId, String positionId) {
    Validate.notBlank(orgId, "组织机构ID不能为空!");
    Validate.notBlank(positionId, "岗位ID不能为空!");
    Optional<OrganizationEntity> optional = organizationRepository.findById(orgId);
    Validate.isTrue(optional.isPresent(), "未找到指定的组织机构!");
    OrganizationEntity organization = optional.get();
    Optional<PositionEntity> op = positionRepository.findById(positionId);
    PositionEntity position = op.orElse(null);
    Validate.notNull(position, "未找到指定的岗位!");
    Validate.isTrue(null == position.getOrganization(), "该岗位已绑定有组织机构，请检查！");
    position.setOrganization(organization);
    positionRepository.saveAndFlush(position);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.PositionService#unbindOrgAndPosition(java.lang.String, java.lang.String)
   */
  @Override
  @Transactional
  public void unbindOrgAndPosition(String orgId, String positionId) {
    Validate.notBlank(orgId, "组织机构ID不能为空!");
    Validate.notBlank(positionId, "岗位ID不能为空!");
    Optional<OrganizationEntity> optional = organizationRepository.findById(orgId);
    Validate.isTrue(optional.isPresent(), "未找到指定的组织机构!");
    OrganizationEntity organization = optional.get();
    Optional<PositionEntity> op = positionRepository.findById(positionId);
    PositionEntity position = op.orElse(null);
    Validate.notNull(position, "未找到指定的岗位!");
    Validate.notNull(position.getOrganization(), "该岗位没有绑定的组织，请检查！");
    Validate.isTrue(organization.getId().equals(position.getOrganization().getId()), "待解绑的岗位与组织机构没有绑定关系，请检查！");
    position.setOrganization(null);
    positionRepository.saveAndFlush(position);
  }
  /**
   * 验证绑定角色和岗位
   * @param roleId
   * @param positionId
   */
  private void validateBindRole(String positionId, String roleId) {
    RbacSettingEntity rbacSettingEntity = rbacSettingService.findRbacSetting();
    if (rbacSettingEntity == null) {
      long count = positionRepository.countByRoleAndPosition(roleId,positionId);
      Validate.isTrue(count == 0, "在批量绑定时,指定角色[%s]已经绑定了，不能重复绑定，请检查!!", roleId);
    } else {
      int bindRelation = rbacSettingEntity.getRolePositionRelation();
      //如果角色和岗位绑定关系为多对多,多对一，一对多，一对一分别控制判断
      if (bindRelation == RbacRelationEnum.MANY_TO_MANY.getRelation()) {
        long count = positionRepository.countByRoleAndPosition(roleId,positionId);
        Validate.isTrue(count == 0, "在批量绑定时,角色和岗位多对多，指定角色[%s]已经绑定了，不能重复绑定，请检查!!", roleId);
      }
      if (bindRelation == RbacRelationEnum.MANY_TO_ONE.getRelation()) {
        long count = positionRepository.countPositionByRoleId(roleId);
        Validate.isTrue(count == 0, "在批量绑定时,角色和岗位多对一，指定角色[%s]已经绑定了岗位，不能继续绑定，请检查!!", roleId);
      }
      if (bindRelation == RbacRelationEnum.ONE_TO_MANY.getRelation()) {
        long count = positionRepository.countRoleByPositionId(positionId);
        Validate.isTrue(count == 0, "在批量绑定时,角色和岗位一对多，指定岗位[%s]已经绑定了角色，不能继续绑定，请检查!!", positionId);
      }
      if (bindRelation == RbacRelationEnum.ONE_TO_ONE.getRelation()) {
        long countRole = positionRepository.countRoleByPositionId(positionId);
        Validate.isTrue(countRole == 0, "在批量绑定时,角色和岗位一对一，指定岗位[%s]已经绑定了角色，不能继续绑定，请检查!!", positionId);
        long countPosition = positionRepository.countPositionByRoleId(roleId);
        Validate.isTrue(countPosition == 0, "在批量绑定时,角色和岗位一对一，指定角色[%s]已经绑定了岗位，不能继续绑定，请检查!!", roleId);
      }

    }
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.PositionService#bindRoleAndPosition(java.lang.String, java.lang.String)
   */
  @Override
  @Transactional
  public void bindRoleAndPosition(String roleId, String positionId) {
    Validate.notBlank(roleId, "角色ID不能为空!");
    Validate.notBlank(positionId, "岗位ID不能为空!");
    Optional<RoleEntity> optional = roleRepository.findById(roleId);
    Validate.isTrue(optional.isPresent(), "未找到指定的角色!");
    Optional<PositionEntity> op = positionRepository.findById(positionId);
    Validate.isTrue(op.isPresent(), "未找到指定的岗位!");
    this.validateBindRole(positionId, roleId);
    positionRepository.bindRole(roleId, positionId);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.PositionService#bindRolesAndPosition(java.util.List, java.lang.String)
   */
  @Override
  @Transactional
  public void bindRolesAndPosition(List<String> roleIds, String positionId) {
    Validate.isTrue(!CollectionUtils.isEmpty(roleIds), "角色ID集合不能为空，请检查！");
    roleIds.stream().forEach(o -> bindRoleAndPosition(o, positionId));
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.PositionService#unbindRoleAndPosition(java.lang.String, java.lang.String)
   */
  @Override
  @Transactional
  public void unbindRoleAndPosition(String roleId, String positionId) {
    Validate.notBlank(roleId, "角色ID不能为空!");
    Validate.notBlank(positionId, "岗位ID不能为空!");
    Optional<RoleEntity> optional = roleRepository.findById(roleId);
    Validate.isTrue(optional.isPresent(), "未找到指定的角色!");
    Optional<PositionEntity> op = positionRepository.findById(positionId);
    Validate.isTrue(op.isPresent(), "未找到指定的岗位!");
    positionRepository.unbindRole(roleId, positionId);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.PositionService#unbindRolesAndPosition(java.util.List, java.lang.String)
   */
  @Override
  @Transactional
  public void unbindRolesAndPosition(List<String> roleIds, String positionId) {
    Validate.isTrue(!CollectionUtils.isEmpty(roleIds), "角色ID集合不能为空，请检查！");
    roleIds.stream().forEach(o -> unbindRoleAndPosition(o, positionId));
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.PositionService#bindUsersAndPosition(java.util.List, java.lang.String)
   */
  @Override
  @Transactional
  public void bindUsersAndPosition(List<String> userIds, String positionId) {
    Validate.isTrue(!CollectionUtils.isEmpty(userIds), "用户ID集合不能为空，请检查！");
    RbacSettingEntity rbacSettingEntity = rbacSettingService.findRbacSetting();
    userIds.stream().forEach(o -> bindUserAndPosition(rbacSettingEntity, o, positionId));
  }
  /**
   * 验证绑定用户和岗位
   * @param userId
   * @param positionId
   * @param rbacSettingEntity
   */
  private void validateBindUser(RbacSettingEntity rbacSettingEntity,String positionId, String userId) {
    if (rbacSettingEntity == null) {
      long count = positionRepository.countByUserAndPosition(userId, positionId);
      Validate.isTrue(count == 0, "在批量绑定时,指定用户[%s]已经绑定了，不能重复绑定，请检查!!", userId);
    } else {
      int bindRelation = rbacSettingEntity.getUserPositionRelation();
      //如果用户和岗位绑定关系为多对多,多对一，一对多，一对一分别控制判断
      if (bindRelation == RbacRelationEnum.MANY_TO_MANY.getRelation()) {
        long count = positionRepository.countByUserAndPosition(userId, positionId);
        Validate.isTrue(count == 0, "在批量绑定时,用户和岗位多对多，指定用户[%s]已经绑定了，不能重复绑定，请检查!!", userId);
      }
      if (bindRelation == RbacRelationEnum.MANY_TO_ONE.getRelation()) {
        long count = positionRepository.countPositionByUserId(userId);

        Validate.isTrue(count == 0, "在批量绑定时,用户和岗位多对一，指定用户[%s]已经绑定了岗位，不能继续绑定，请检查!!", userId);
      }
      if (bindRelation == RbacRelationEnum.ONE_TO_MANY.getRelation()) {
        long count = positionRepository.countUserByPositionId(positionId);
        Validate.isTrue(count == 0, "在批量绑定时,用户和岗位一对多，指定岗位[%s]已经绑定了用户，不能继续绑定，请检查!!", positionId);
      }
      if (bindRelation == RbacRelationEnum.ONE_TO_ONE.getRelation()) {
        long countUser = positionRepository.countUserByPositionId(positionId);
        Validate.isTrue(countUser == 0, "在批量绑定时,用户和岗位一对一，指定岗位[%s]已经绑定了用户，不能继续绑定，请检查!!", positionId);
        long countPosition = positionRepository.countPositionByUserId(userId);
        Validate.isTrue(countPosition == 0, "在批量绑定时,用户和岗位一对一，指定用户[%s]已经绑定了岗位，不能继续绑定，请检查!!", userId);
      }

    }
  }
  /**
   * 绑定用户和岗位
   * @param userId
   * @param positionId
   */
  private void bindUserAndPosition(RbacSettingEntity rbacSettingEntity,String userId, String positionId) {
    Validate.notBlank(userId, "用户ID不能为空!");
    Validate.notBlank(positionId, "岗位ID不能为空!");
    Optional<UserEntity> op = userRepository.findById(userId);
    UserEntity userEntity = op.orElse(null);
    Validate.notNull(userEntity, "未找到指定的用户!");

    Optional<PositionEntity> optional = positionRepository.findById(positionId);
    Validate.isTrue(optional.isPresent(), "未找到指定的岗位!");
    this.validateBindUser(rbacSettingEntity, positionId, userId);
    positionRepository.binduser(userId, positionId);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.PositionService#unbindUsersAndPosition(java.util.List, java.lang.String)
   */
  @Override
  @Transactional
  public void unbindUsersAndPosition(List<String> userIds, String positionId) {
    Validate.isTrue(!CollectionUtils.isEmpty(userIds), "用户ID集合不能为空，请检查！");
    userIds.stream().forEach(o -> unbindUserAndPosition(o, positionId));
  }

  private void unbindUserAndPosition(String userId, String positionId) {
    Validate.notBlank(userId, "用户ID不能为空!");
    Validate.notBlank(positionId, "岗位ID不能为空!");
    Optional<UserEntity> op = userRepository.findById(userId);
    Validate.isTrue(op.isPresent(), "未找到指定的用户!");
    Optional<PositionEntity> optional = positionRepository.findById(positionId);
    Validate.isTrue(optional.isPresent(), "未找到指定的岗位!");
    Validate.isTrue(0L != positionRepository.countByUserAndPosition(userId, positionId), "该用户与该岗位未有绑定关系，请检查！");
    //主岗位不能解绑
    PositionVo positionVo = null;
    String key = Constants.REDIS_KEY_RBAC_USER + userId;
    Map<String, String> mainPosition = redissonClient.getMap(key);
    if(mainPosition.containsKey(Constants.MAIN_POSITION)){
      positionVo = JSONObject.parseObject(mainPosition.get(Constants.MAIN_POSITION), PositionVo.class);
    }
    if(positionVo != null) {
      Validate.isTrue(!positionId.equals(positionVo.getId()), "[%s]是用户的主岗位不能解绑", optional.get().getName());
    }
    positionRepository.unbinduser(userId, positionId);
  }
  
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.PositionService#findByUserId(java.lang.String)
   */
  @Override
  public List<PositionVo> findByUserId(String userId) {
    Validate.notBlank(userId,"传入的用户ID不能为空！");
    UserEntity user = this.userRepository.findDetailsById(userId);
    if(user==null){
      return null;
    }
    
    if(!CollectionUtils.isEmpty(user.getPositions())){
      Set<PositionEntity> positions = user.getPositions();
      // 转换后返回
      Collection<PositionVo> positionVos = this.nebulaToolkitService.copyCollectionByWhiteList(positions, PositionEntity.class, PositionVo.class, HashSet.class, ArrayList.class);
      return Lists.newArrayList(positionVos);
    }
    return Lists.newArrayList();
  }

  @Override
  public Set<PositionVo> findByIds(List<String> ids) {
    if(CollectionUtils.isEmpty(ids)){
      return Sets.newHashSet();
    }
    Set<PositionEntity> pos = positionRepository.findByIds(ids);
    if(CollectionUtils.isEmpty(pos)){
      return Sets.newHashSet();
    }
    Collection<PositionVo> pvo = this.nebulaToolkitService.copyCollectionByWhiteList(pos,PositionEntity.class,PositionVo.class,HashSet.class,ArrayList.class);
    return Sets.newHashSet(pvo);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.PositionService#findByConditions(java.lang.String, java.lang.String, java.lang.Integer, org.springframework.data.domain.Pageable)
   */
  @Override
  public Page<PositionVo> findByConditions(String code, String name, Integer tstatus, Pageable pageable) {
    Map<String, Object> conditions = new HashMap<>();
    if (StringUtils.isNotBlank(code)) {
      conditions.put("code", code);
    }
    if (StringUtils.isNotBlank(name)) {
      conditions.put("name", name);
    }
    if (tstatus != null) {
      conditions.put("tstatus", tstatus);
    }
    // 如果当前没有设定分页信息，则默认第一页，每页50条数据
    if (pageable == null) {
      pageable = PageRequest.of(0, 50);
    }
  
    Page<PositionEntity> positionEntityPage = positionRepository.queryPage(pageable, conditions);
    // 分页信息需要自行转换
    List<PositionEntity> positionEntitys = positionEntityPage.getContent();
    Page<PositionVo> positionVoPage = null;
    if(!positionEntitys.isEmpty()) {
      Collection<PositionVo> positionVos = this.nebulaToolkitService.copyCollectionByWhiteList(positionEntitys, PositionEntity.class, PositionVo.class, LinkedHashSet.class, ArrayList.class, "users", "parent", "children");
      positionVoPage = new PageImpl<>(new ArrayList<>(positionVos), pageable, positionEntityPage.getTotalElements());
    } else {
      positionVoPage = new PageImpl<>(Lists.newArrayList(), pageable, 0l);
    }
    return positionVoPage;
  }

  @Override
  public PositionVo findByCode(String code) {
    if(StringUtils.isBlank(code)) {
      return null;
    }
    PositionEntity position = positionRepository.findByCode(code);
    if(position == null) {
      return null;
    }
    return nebulaToolkitService.copyObjectByWhiteList(position, PositionVo.class, HashSet.class, ArrayList.class);
  }

  /**
   * 切换岗位
   * @param positionId
   */
  @Override
  public UserVo changePosition(String positionId, String userId){
    Validate.notBlank(positionId, "切换岗位id不能为空");
    Validate.notBlank(userId, "用户id不能为空");

    UserEntity user = userRepository.findById(userId).orElse(null);
    Validate.notNull(user, "未获取到用户信息，请检查");
    UserVo userVo = nebulaToolkitService.copyObjectByWhiteList(user, UserVo.class, HashSet.class, ArrayList.class, "roles");

    PositionEntity positionEntity = positionRepository.findById(positionId).orElse(null);
    Validate.notNull(positionEntity, "未获取到岗位信息，请检查");
    PositionVo positionVo = nebulaToolkitService.copyObjectByWhiteList(positionEntity, PositionVo.class, HashSet.class, ArrayList.class);

    List<PositionVo> positionVos = this.findByUserId(userId);
    userVo.setPositions(Sets.newLinkedHashSet(positionVos));
    userVo.setMainPosition(positionId);
    Map<String, String> mainPosition = redissonClient.getMap(Constants.REDIS_KEY_RBAC_USER + userId);
    mainPosition.put(Constants.MAIN_POSITION, JSONObject.toJSON(positionVo).toString());
    return userVo;
  }

  /**
   * 查询Redis中主岗位，如果redis中不存在，则默认主岗位为数据库查询的第一条数据，并且将该数据存入redis中
   * @param userId
   * @return
   */
  @Override
  public PositionVo findMainPositionByUserId(String userId){
    Validate.notBlank(userId, "传入的用户id不能为空");
    PositionVo positionVo = null;
    String key = Constants.REDIS_KEY_RBAC_USER + userId;
    Map<String, String> mainPosition = redissonClient.getMap(key);
    if(mainPosition.containsKey(Constants.MAIN_POSITION)){
      positionVo = JSONObject.parseObject(mainPosition.get(Constants.MAIN_POSITION), PositionVo.class);
    }
    if(positionVo == null){
      List<PositionVo> positionVoList = this.findByUserId(userId);
      if(CollectionUtils.isEmpty(positionVoList)){
        return null;
      }else {
        positionVo = positionVoList.get(0);
      }
      String json = JSONObject.toJSON(positionVo).toString();
      mainPosition.put(Constants.MAIN_POSITION, json);
    }
    return positionVo;
  }
}
