package com.bizunited.platform.core.service.security;

import com.bizunited.platform.core.entity.OrganizationEntity;
import com.bizunited.platform.core.entity.RoleEntity;
import com.bizunited.platform.core.entity.UserEntity;
import com.bizunited.platform.core.entity.UserGroupEntity;
import com.bizunited.platform.core.repository.OrganizationRepository;
import com.bizunited.platform.core.repository.RoleRepository;
import com.bizunited.platform.core.repository.UserGroupRepository;
import com.bizunited.platform.core.repository.UserRepository;
import com.bizunited.platform.core.service.NebulaToolkitService;
import com.bizunited.platform.rbac.server.service.PositionService;
import com.bizunited.platform.rbac.server.service.RoleService;
import com.bizunited.platform.rbac.server.vo.PositionVo;
import com.bizunited.platform.rbac.server.vo.RoleVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.domain.Sort;
import org.springframework.security.access.AccessDeniedException;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * @author yinwenjie
 */
public class RoleServiceImpl implements RoleService {

  /**
   * 角色repository自动注入.
   */
  @Autowired
  private RoleRepository roleRepository;
  @Autowired
  private UserGroupRepository userGroupRepository;
  @Autowired
  private OrganizationRepository organizationRepository;
  @Autowired
  private UserRepository userService;
  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private PositionService positionService;
  /**
   * 这个参数记录了那些可以忽略“功能”级别权限验证的角色
   */
  @Value("${rbac.ignoreMethodCheckRoles}")
  private String[] ignoreMethodCheckRoles; 
  /**
   * 配置的那些不允许被删除被作废的角色
   */
  @Value("${rbac.roles.deleteDeny}")
  private String[] deleteDenys;

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.RoleService#create(com.bizunited.platform.rbac.server.vo.RoleVo)
   */
  @Override
  @Transactional
  public RoleVo create(RoleVo role) {
    // 进入传入信息的检查
    this.validateRoleBeforeAdd(role);
    // 开始插入
    RoleEntity roleEntity = this.nebulaToolkitService.copyObjectByWhiteList(role, RoleEntity.class, HashSet.class, ArrayList.class, "parent");
    this.roleRepository.saveAndFlush(roleEntity);
    role.setId(roleEntity.getId());
    return role;
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.RoleService#update(com.bizunited.platform.rbac.server.vo.RoleVo)
   */
  @Override
  @Transactional
  public RoleVo update(RoleVo role) {
    /*
     * 修改角色的过程如下： 
     * 1、首先确定当前role是可以进行修改的role 
     * 2、只进行comment的修改，其它即使传递了也不进行修改
     */
    Validate.notNull(role, "role mssage not null");
    String roleId = role.getId();
    String updateComment = role.getComment();
    Validate.notEmpty(roleId, "role id not empty!");

    Optional<RoleEntity> op = this.roleRepository.findById(roleId);
    Validate.isTrue(op.isPresent(), "role not found");
    RoleEntity currentRole = op.get();
    // 1、========
    String currentName = currentRole.getRoleName();
    // 如果条件成立，说明这个角色信息不能被修改
    for (String deleteDeny : deleteDenys) {
      if (StringUtils.equals(currentName, deleteDeny)) {
        throw new AccessDeniedException("这个角色为系统特定默认角色，不允许修改！");
      }
    }

    // 2、========
    if(!StringUtils.isBlank(updateComment)){
      currentRole.setComment(updateComment);
    }

    if(role.getParent() != null){
      Validate.notBlank(role.getParent().getId(), "未找到该角色的父级角色的id，请检查！");
      RoleEntity parentRole = roleRepository.findById(role.getParent().getId()).orElse(null);
      Validate.notNull(parentRole, "未找到角色的父级角色");
      currentRole.setParent(parentRole);
    }else {
      currentRole.setParent(null);
    }
    this.roleRepository.saveAndFlush(currentRole);

    Set<String> roleStack = new HashSet<>();
    roleStack.add(currentRole.getId());
    this.handleCircular(currentRole, roleStack);
    return role;
  }

  /**
   *判断是否形成循环依赖
   * @param parent
   * @param roleStack
   */
  private void handleCircular(RoleEntity parent, Set<String> roleStack) {
    if(CollectionUtils.isEmpty(parent.getChildren())) {
      return;
    }
    for (RoleEntity roleEntity : parent.getChildren()) {
      Validate.isTrue(!roleStack.contains(roleEntity.getId()), "形成循环依赖，更新失败，请检查！");
      roleStack.add(roleEntity.getId());
      this.handleCircular(roleEntity, roleStack);
    }
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.RoleService#disable(java.lang.String)
   */
  @Override
  @Transactional
  public RoleVo disable(String roleId) {
    /*
     * 注意：在配置文件中已经设定的那些不能操作的角色信息，是不允许禁用的
     */
    Validate.notEmpty(roleId, "role id not be found!");
    Optional<RoleEntity> op = this.roleRepository.findById(roleId);
    Validate.isTrue(op.isPresent(), "未找到指定的角色信息，请检查!!");
    RoleEntity currentRole = op.get();

    // 如果条件成立，说明这个角色信息不能被删除（或者说作废）
    for (String deleteDeny : deleteDenys) {
      if (StringUtils.equals(currentRole.getRoleName(), deleteDeny)) {
        throw new AccessDeniedException("the role not allow be disable！");
      }
    }

    // 禁用角色
    currentRole.setTstatus(0);
    this.roleRepository.save(currentRole);
    return this.nebulaToolkitService.copyObjectByWhiteList(currentRole, RoleVo.class, HashSet.class, ArrayList.class);
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.RoleService#enable(java.lang.String)
   */
  @Override
  @Transactional
  public RoleVo enable(String roleId) {
    Validate.notEmpty(roleId, "role id not be found!");
    Optional<RoleEntity> op = this.roleRepository.findById(roleId);
    Validate.isTrue(op.isPresent(), "未找到指定的角色信息，请检查!!");
    RoleEntity currentRole = op.get();

    // 启用角色
    currentRole.setTstatus(1);
    this.roleRepository.save(currentRole);
    return this.nebulaToolkitService.copyObjectByWhiteList(currentRole, RoleVo.class, HashSet.class, ArrayList.class);
  }
  /**
   * 该私有方法在新增一个role信息前，检查传入信息的证确定   * @param role
   */
  private void validateRoleBeforeAdd(RoleVo role) {
    Validate.notNull(role, "role input object not be null!");
    // 开始验证
    Validate.notBlank(role.getRoleName(), "role name not be null!");
    // 必须是大写
    role.setRoleName(role.getRoleName().toUpperCase());
    RoleEntity oldRole = this.roleRepository.findByRoleName(role.getRoleName());
    Validate.isTrue(oldRole == null, "当前设定的角色名称（role name）已经被使用，请更换!");
    // code必须验证
    oldRole = this.roleRepository.findByRoleCode(role.getRoleCode());
    Validate.isTrue(oldRole == null, "当前设定的角色编号（role code）已经被使用，请更换!");
    // 当前的创建时间和修改时间要一起写入
    Date currentTime = new Date();
    role.setCreateDate(currentTime);
    // 说明性信息
    if(StringUtils.isBlank(role.getComment())) {
      role.setComment("[未填写]");
    }
    // 当前角色必须是状态正常的
    role.setTstatus(1);
    if(role.getParent() != null){
      Validate.notBlank(role.getParent().getId(), "未找到该角色的父级角色的id，请检查！");
      RoleEntity parentRole = roleRepository.findById(role.getParent().getId()).orElse(null);
      Validate.notNull(parentRole, "未找到角色的父级角色");
    }
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.RoleService#bindRoles(java.lang.String, java.lang.String[])
   */
  @Override
  @Transactional
  public void bindRoles(String userId, String[] roleNames) {
    Validate.notBlank(userId, "绑定权限时，用户id不能为空");
    UserEntity user = userService.findDetailsById(userId);
    Validate.notNull(user, "用户不存在");
    Validate.isTrue(roleNames != null && roleNames.length > 0 , "在进行角色绑定时，请至少传入一个角色名信息");
    
    // 去重
    Set<String> roleNameSet = new HashSet<>();
    roleNameSet.addAll(Arrays.asList(roleNames));
    // 开始添加（不用做存在性判断，没有自然会报错）
    for (String roleName : roleNameSet) {
      RoleEntity currentRole = this.roleRepository.findByRoleName(roleName);
      Validate.notNull(currentRole , "未找到指定的角色!!");
      // 一般来说，这里抛出异常就是重复绑定
      try {
        this.roleRepository.bindUser(userId, currentRole.getId());
        this.roleRepository.flush();
      } catch(Exception e) {
        throw new IllegalArgumentException("错误的绑定信息，请检测是否重复绑定，或者角色信息未传入!");
      } 
    }
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.RoleService#reBindRoles(java.lang.String, java.lang.String[])
   */
  @Override
  @Transactional
  public void reBindRoles(String userId, String[] roleNames) {
    Validate.notBlank(userId, "绑定权限时，用户id不能为空");
    UserEntity user = userService.findDetailsById(userId);
    Validate.notNull(user, "用户不存在");
    Validate.notEmpty(roleNames, "绑定或者解绑必须保证用户拥有至少一个直接绑定的角色!!");
    
    // 去重
    Set<String> roleNameSet = new HashSet<>();
    roleNameSet.addAll(Arrays.asList(roleNames));
    // 首先删除以前的绑定信息
    this.roleRepository.deleteRoleByUserId(userId);
    // 然后重新进行绑定
    for (String roleName : roleNameSet) {
      RoleEntity currentRole = this.roleRepository.findByRoleName(roleName);
      Validate.notNull(currentRole , "未找到指定的角色!!");
      // 一般来说，这里抛出异常就是重复绑定
      try {
        this.roleRepository.bindUser(userId, currentRole.getId());
      } catch(Exception e) {
        throw new IllegalArgumentException("错误的绑定信息，请检测是否重复绑定，或者角色信息未传入!");
      } 
    }
  }

  @Override
  public Set<RoleVo> findByUser(String userId){
    if(StringUtils.isBlank(userId)) {
      return Sets.newHashSet();
    }
    List<RoleEntity> roles = this.roleRepository.findByUserId(userId);
    Collection<RoleVo> roleVos = this.nebulaToolkitService.copyCollectionByWhiteList(roles, RoleEntity.class, RoleVo.class, LinkedHashSet.class, ArrayList.class);
    return Sets.newHashSet(roleVos);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.RoleService#findAllByUserId(java.lang.String)
   */
  @Override
  public List<RoleVo> findAllByUserId(String userId, Integer type) {
    /*
     * 在表单引擎V1.2+，按照一个用户数据库编号，查询这个用户对应的角色信息，就没有那么简单了，
     * 因为一个用户的角色可以由三方面决定：
     * a、这个用户直接绑定的角色信息
     * b、这个用户所属的一个或者多个用户组绑定的角色信息
     * c、这个用户所属的一个或者多个组织机构（不包括父级）绑定的角色信息
     * d、这个用户所属的一个或多个岗位（不包括父级）绑定的角色信息
     * 
     * 注意：数据层并没有使用status进行过滤，这里需要自行进行过滤
     * */
    Map<String, String> sourceMap = new HashMap<>();
    if(StringUtils.isBlank(userId)) {
      return Collections.emptyList();
    }
    
    // a、====
    Set<RoleEntity> resultRoles = new HashSet<>();
    List<RoleEntity> roles = this.roleRepository.findByUserId(userId);
    if(roles != null && !roles.isEmpty()) {
      this.setSource(Sets.newHashSet(roles), sourceMap, "user");
      resultRoles.addAll(roles);
    }
    
    // b、====
    Set<UserGroupEntity> groups = this.userGroupRepository.findByUserId(userId);
    if(groups != null && !groups.isEmpty()) {
      for(UserGroupEntity item : groups){
        this.setSource(item.getRoles(), sourceMap, "group");
      }
      groups.stream().filter(item -> item.getRoles() != null && item.getTstatus() == 1).forEach(item ->  resultRoles.addAll(item.getRoles()));
    }
    
    // c、====
    Set<OrganizationEntity> orgs = this.organizationRepository.findByUserId(userId);
    if(orgs != null && !orgs.isEmpty()) {
      for(OrganizationEntity item : orgs){
        this.setSource(item.getRoles(), sourceMap, "org");
      }
      orgs.stream().filter(item -> item.getRoles() != null && item.getTstatus() == 1).forEach(item ->  resultRoles.addAll(item.getRoles()));
    }

    // d、==== 获取当前岗位的绑的角色
    Set<RoleEntity> positionRoles = new HashSet<>();
    if(type != null && type.equals(1)) {
      List<PositionVo> positions = positionService.findByUserId(userId);
      if(!CollectionUtils.isEmpty(positions)) {
        List<String> positionIds = positions.stream().map(PositionVo::getId).collect(Collectors.toList());
        positionRoles = roleRepository.findByPositions(positionIds);
      }
    } else {
      PositionVo positionVo = positionService.findMainPositionByUserId(userId);
      if(positionVo != null){
        positionRoles = roleRepository.findByPosition(positionVo.getId());
      }
    }
    resultRoles.addAll(positionRoles);
    this.setSource(positionRoles, sourceMap, "position");

    // 过滤出唯一信息后返回
    if(resultRoles.isEmpty()) {
      return Collections.emptyList();
    }
    List<RoleEntity> roleEntitys = resultRoles.stream().filter(item -> item.getTstatus() == 1).collect(Collectors.toList());
    // 转换vo后进行返回
    List<RoleVo> roleVos = new ArrayList<>();
    for(RoleEntity entity : roleEntitys){
      RoleVo roleVo = this.nebulaToolkitService.copyObjectByWhiteList(entity, RoleVo.class, HashSet.class, ArrayList.class);
      String sourceName = sourceMap.get(roleVo.getId());
      roleVo.setSource(sourceName);
      roleVo.setParent(this.findParentRole(entity));
      roleVos.add(roleVo);
    }
    return roleVos.stream().sorted(Comparator.comparing(RoleVo::getCreateDate)).collect(Collectors.toList());
  }

  /**
   * 将获取的来源放进map中存储，key为role的id
   * 方便后期放入到roleVo中
   * @param roles
   * @param sourceMap
   * @param source
   */
  private void setSource(Set<RoleEntity> roles, Map<String, String> sourceMap, String source) {
    if(CollectionUtils.isEmpty(roles)){
      return;
    }
    for(RoleEntity roleEntity : roles){
      if(sourceMap.containsKey(roleEntity.getId())){
        String value = sourceMap.get(roleEntity.getId()) + "," + source;
        sourceMap.put(roleEntity.getId(), value);
      }else {
        sourceMap.put(roleEntity.getId(), source);
      }
    }
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.RoleService#findByCompetenceId(java.lang.String)
   */
  @Override
  public List<RoleVo> findByCompetenceId(String competenceId) {
    if(StringUtils.isBlank(competenceId)) {
      return Collections.emptyList();
    }
    List<RoleEntity> roles = this.roleRepository.findByCompetenceId(competenceId);
    if (roles == null || roles.isEmpty()) {
      return Collections.emptyList();
    }
    Collection<RoleVo> roleVos = this.nebulaToolkitService.copyCollectionByWhiteList(roles, RoleEntity.class, RoleVo.class, LinkedHashSet.class, ArrayList.class);
    return Lists.newArrayList(roleVos);
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.RoleService#findByStatus(java.lang.Integer)
   */
  @Override
  public List<RoleVo> findByStatus(Integer useStatus) {
    if(useStatus == null) {
      return Collections.emptyList();
    }
    List<RoleEntity> rolesList = this.roleRepository.findByTstatus(useStatus);
    if (rolesList == null || rolesList.isEmpty()) {
      return Collections.emptyList();
    }
    Collection<RoleVo> roleVos = this.nebulaToolkitService.copyCollectionByWhiteList(rolesList, RoleEntity.class, RoleVo.class, LinkedHashSet.class, ArrayList.class);
    return Lists.newArrayList(roleVos);
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.RoleService#findById(java.lang.String)
   */
  @Override
  public RoleVo findById(String roleId) {
    Validate.notNull(roleId, "role Id must not null!");
    Optional<RoleEntity> op = this.roleRepository.findById(roleId);
    if (!op.isPresent()) {
      return null;
    }
    return this.nebulaToolkitService.copyObjectByWhiteList(op.get(), RoleVo.class, HashSet.class, ArrayList.class);
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.RoleService#findAll()
   */
  @Override
  public List<RoleVo> findAll() {
    List<RoleEntity> rolesList = this.roleRepository.findAll(Sort.by(Sort.Order.desc("createDate")));
    // 如果条件成立说明系统该数据异常，这是直接抛出错误信息
    if (rolesList == null || rolesList.isEmpty()) {
      return Collections.emptyList();
    }
    Collection<RoleVo> roleVos = this.nebulaToolkitService.copyCollectionByWhiteList(rolesList, RoleEntity.class, RoleVo.class, LinkedHashSet.class, ArrayList.class);
    return Lists.newArrayList(roleVos);
  }

  @Override
  public Set<RoleVo> findByIds(List<String> ids) {
    if(CollectionUtils.isEmpty(ids)){
      return Sets.newHashSet();
    }
    Set<RoleEntity> roles = this.roleRepository.findByIds(ids);
    if(CollectionUtils.isEmpty(roles)){
      return Sets.newHashSet();
    }
    Collection<RoleVo> rvos = this.nebulaToolkitService.copyCollectionByWhiteList(roles, RoleEntity.class, RoleVo.class, HashSet.class, ArrayList.class);
    return Sets.newHashSet(rvos);
  }

  /**
   * 查询角色树状关系
   * @return
   */
  @Override
  public Set<RoleVo> findRoleTree() {
    LinkedHashSet<RoleVo> roleSet = new LinkedHashSet<>();
    Set<RoleEntity> rootRoles = roleRepository.findByParentIsNull();
    if(CollectionUtils.isEmpty(rootRoles)){
      return Sets.newHashSet();
    }

    for(RoleEntity roleEntity : rootRoles){
      RoleVo roleVo = this.nebulaToolkitService.copyObjectByWhiteList(roleEntity, RoleVo.class, HashSet.class, ArrayList.class);
      roleVo.setChildren(this.findChildrenRole(roleEntity));
      roleSet.add(roleVo);
    }
    return roleSet;
  }

  /**
   * 查询子角色
   * @param roleEntity
   * @return
   */
  public Set<RoleVo> findChildrenRole(RoleEntity roleEntity) {
    LinkedHashSet<RoleVo> roleVoSet = new LinkedHashSet<>();
    //  查询子级结构
    Set<RoleEntity> children = roleEntity.getChildren();
    if(CollectionUtils.isEmpty(children)){
      return Sets.newHashSet();
    }
    for(RoleEntity child : children){
      RoleVo roleVo = this.nebulaToolkitService.copyObjectByWhiteList(child, RoleVo.class, HashSet.class, ArrayList.class);
      roleVo.setChildren(this.findChildrenRole(child));
      roleVoSet.add(roleVo);
    }
    return roleVoSet;
  }

  /**
   * 查询父角色
   * @param child
   * @return
   */
  private RoleVo findParentRole(RoleEntity child) {
    RoleEntity parent = child.getParent();
    if(parent == null){
      return null;
    }
    parent = roleRepository.findById(parent.getId()).orElse(null);
    if(parent == null){
      return null;
    }
    RoleVo roleVo = this.nebulaToolkitService.copyObjectByWhiteList(parent, RoleVo.class, HashSet.class, ArrayList.class);
    roleVo.setParent(this.findParentRole(parent));
    return roleVo;
  }

  /**
   * 根据角色名称模糊查询
   * @return
   */
  @Override
  public Set<RoleVo> findByRoleNameLike(String roleName) {
    Set<RoleEntity> roleSet = this.roleRepository.findByRoleNameLike(roleName);
    if(CollectionUtils.isEmpty(roleSet)){
      return Sets.newHashSet();
    }
    Collection<RoleVo> roleVos = this.nebulaToolkitService.copyCollectionByWhiteList(roleSet, RoleEntity.class, RoleVo.class, HashSet.class, ArrayList.class);
    return Sets.newHashSet(roleVos);
  }

  @Override
  public RoleVo findByCode(String code) {
    if(StringUtils.isBlank(code)) {
      return null;
    }
    RoleEntity role = roleRepository.findByRoleCode(code);
    if(role == null) {
      return null;
    }
    return nebulaToolkitService.copyObjectByWhiteList(role, RoleVo.class, HashSet.class, ArrayList.class);
  }
}
