package com.bizunited.platform.core.service.security;

import com.bizunited.platform.core.common.enums.RbacRelationEnum;
import com.bizunited.platform.core.entity.RbacSettingEntity;
import com.bizunited.platform.core.entity.RoleEntity;
import com.bizunited.platform.core.entity.UserEntity;
import com.bizunited.platform.core.entity.UserGroupEntity;
import com.bizunited.platform.core.repository.RoleRepository;
import com.bizunited.platform.core.repository.UserGroupRepository;
import com.bizunited.platform.core.repository.UserRepository;
import com.bizunited.platform.core.service.NebulaToolkitService;
import com.bizunited.platform.rbac.server.service.UserGroupService;
import com.bizunited.platform.rbac.server.vo.UserGroupVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class UserGroupServiceImpl implements UserGroupService {
  @Autowired
  private UserGroupRepository userGroupRepository;
  @Autowired
  private UserRepository userRepository;
  @Autowired
  private RoleRepository roleRepository;
  @Autowired
  private RbacSettingService rbacSettingService;
  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;
  private static final String ERROR_GROUP_ID = "分组id不能为空，请检查";
  private static final String ERROR_NOT_FOUND = "没有该工作组，请检查!!";
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.UserGroupService#create(com.bizunited.platform.rbac.server.vo.UserGroupVo)
   */
  @Transactional
  @Override
  public UserGroupVo create(UserGroupVo userGroup) {
    // 验证传入分组信息的正确性
    Validate.notNull(userGroup, "分组信息不能为空，请检查");
    Validate.isTrue(StringUtils.isBlank(userGroup.getId()), "添加分组时，不能传入分组id,请重新添加!!");
    String groupName = userGroup.getGroupName();
    Validate.notBlank(groupName, "分组名称不能为空，请检查!!");
    Validate.isTrue(!isChinese(groupName), "分组名称不能包含中文，请重新输入!!");
    Validate.notBlank(userGroup.getGroupDescription(), "分组描述不能为空，请检查!!");
    Validate.notNull(userGroup.getTstatus(), "分组状态不能为空，请检查!!");
    UserGroupEntity oldUserGroup = userGroupRepository.findByGroupName(groupName);
    Validate.isTrue(null == oldUserGroup, "该分组已经存在，请重新输入分组名");
    
    // 转换后保存
    UserGroupEntity userGroupEntity = this.nebulaToolkitService.copyObjectByWhiteList(userGroup, UserGroupEntity.class, HashSet.class, ArrayList.class);
    userGroupRepository.save(userGroupEntity);
    userGroup.setId(userGroupEntity.getId());
    return userGroup;
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.UserGroupService#update(com.bizunited.platform.rbac.server.vo.UserGroupVo)
   */
  @Transactional
  @Override
  public UserGroupVo update(UserGroupVo userGroup) {
    // 验证传入的用户信息
    Validate.notNull(userGroup, "分组信息不能为空，请检查");
    String groupId = userGroup.getId();
    Validate.notNull(groupId, ERROR_GROUP_ID);
    Optional<UserGroupEntity> op = userGroupRepository.findById(groupId);
    Validate.isTrue(op.isPresent(), "该分组不存在，请重新输入分组名");
    UserGroupEntity currentGroup = op.get();

    // 只有分组描述信息能够修改
    if(!StringUtils.isBlank(userGroup.getGroupDescription())) {
      currentGroup.setGroupDescription(userGroup.getGroupDescription());
      userGroupRepository.save(currentGroup);
    }
    
    // 转换后输出
    return this.nebulaToolkitService.copyObjectByWhiteList(currentGroup, UserGroupVo.class, HashSet.class, ArrayList.class);
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.UserGroupService#updateStatus(java.lang.String)
   */
  @Transactional
  @Override
  public UserGroupVo updateStatus(String groupId) {
    Validate.notNull(groupId, ERROR_GROUP_ID);
    Optional<UserGroupEntity> op = userGroupRepository.findById(groupId);
    Validate.isTrue(op.isPresent(), ERROR_NOT_FOUND);
    UserGroupEntity userGroup = op.get();
    Integer status = userGroup.getTstatus();
    status = (status == 1 ? 0 : 1); //0:禁用,1:启用
    userGroup.setTstatus(status);
    userGroupRepository.save(userGroup);
    
    // 转换后输出
    return this.nebulaToolkitService.copyObjectByWhiteList(userGroup, UserGroupVo.class, HashSet.class, ArrayList.class);
  }
  /**
   * 验证绑定用户和分组
   * @param userId
   * @param groupId
   * @param rbacSettingEntity
   */
  private void validateBindUser(RbacSettingEntity rbacSettingEntity,String groupId, String userId){
    Optional<UserEntity> optional = userRepository.findById(userId);
    Validate.isTrue(optional.isPresent(), "没有该用户[%s],请检查!!", userId);
    if (rbacSettingEntity == null) {
      long count = userGroupRepository.countUserGroup(groupId, userId);
      Validate.isTrue(count == 0, "在批量绑定时,指定用户[%s]已经绑定了，不能重复绑定，请检查!!", userId);
    } else {
      int bindRelation = rbacSettingEntity.getUserGroupRelation();
      //如果用户和用户分组绑定关系为多对多,多对一，一对多，一对一分别控制判断
      if (bindRelation == RbacRelationEnum.MANY_TO_MANY.getRelation()) {
        long count = userGroupRepository.countUserGroup(groupId, userId);
        Validate.isTrue(count == 0, "在批量绑定时,用户和分组多对多，指定用户[%s]已经绑定了，不能重复绑定，请检查!!", userId);
      }
      if (bindRelation == RbacRelationEnum.MANY_TO_ONE.getRelation()) {
        long count = userGroupRepository.countGroupByUserId(userId);
        Validate.isTrue(count == 0, "在批量绑定时,用户和分组多对一，指定用户[%s]已经绑定了分组，不能继续绑定，请检查!!", userId);
      }
      if (bindRelation == RbacRelationEnum.ONE_TO_MANY.getRelation()) {
        long count = userGroupRepository.countUserByGroupId(groupId);
        Validate.isTrue(count == 0, "在批量绑定时,用户和分组一对多，指定分组[%s]已经绑定了用户，不能继续绑定，请检查!!", groupId);
      }
      if (bindRelation == RbacRelationEnum.ONE_TO_ONE.getRelation()) {
        long countGroup = userGroupRepository.countGroupByUserId(userId);
        Validate.isTrue(countGroup == 0, "在批量绑定时,用户和分组一对一，指定用户[%s]已经绑定了分组，不能继续绑定，请检查!!", userId);
        long countUser = userGroupRepository.countUserByGroupId(groupId);
        Validate.isTrue(countUser == 0, "在批量绑定时,用户和分组一对一，指定分组[%s]已经绑定了用户，不能继续绑定，请检查!!", groupId);
      }

    }
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.UserGroupService#bindUser(java.lang.String, java.lang.String[])
   */
  @Override
  @Transactional
  public void bindUser(String groupId, String[] userIds) {
    Validate.notNull(groupId, ERROR_GROUP_ID);
    Validate.isTrue(userIds != null && userIds.length > 0, "用户id不能为空（至少需要传入一个），请检查");
    Optional<UserGroupEntity> op = userGroupRepository.findById(groupId);
    Validate.isTrue(op.isPresent(), ERROR_NOT_FOUND);
    RbacSettingEntity rbacSettingEntity = rbacSettingService.findRbacSetting();
    for (String userId : userIds) {
      this.validateBindUser(rbacSettingEntity, groupId, userId);
      userGroupRepository.bindUser(groupId, userId);
    }
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.UserGroupService#unBindUser(java.lang.String, java.lang.String[])
   */
  @Override
  @Transactional
  public void unBindUser(String groupId, String[] userIds) {
    Validate.notNull(groupId, ERROR_GROUP_ID);
    Validate.isTrue(userIds != null && userIds.length > 0, "用户id不能为空（至少需要传入一个），请检查");
    Optional<UserGroupEntity> op = userGroupRepository.findById(groupId);
    Validate.isTrue(op.isPresent(), ERROR_NOT_FOUND);
    
    for (String userId : userIds) {
      Optional<UserEntity> optional = userRepository.findById(userId);
      Validate.isTrue(optional.isPresent(), "没有该用户,请检查!!");
      userGroupRepository.unBindUser(groupId, userId);
    }
  }
  /**
   * 验证绑定角色和分组
   * @param roleId
   * @param groupIdId
   * @param rbacSettingEntity
   */
  private void validateBindRole(RbacSettingEntity rbacSettingEntity,String groupId, String roleId){
    Optional<RoleEntity> optional =roleRepository.findById(roleId);
    Validate.isTrue(optional.isPresent(), "没有该角色[%s],请检查!!", roleId);
    if (rbacSettingEntity == null) {
      long count = userGroupRepository.countRoleGroup(groupId,roleId);
      Validate.isTrue(count == 0, "在批量绑定时,指定角色[%s]已经绑定了，不能重复绑定，请检查!!", roleId);
    } else {
      int bindRelation = rbacSettingEntity.getRoleGroupRelation();
      //如果角色和用户分组绑定关系为多对多,多对一，一对多，一对一分别控制判断
      if (bindRelation == RbacRelationEnum.MANY_TO_MANY.getRelation()) {
        long count = userGroupRepository.countRoleGroup(groupId,roleId);
        Validate.isTrue(count == 0, "在批量绑定时,角色和分组多对多，指定角色[%s]已经绑定了，不能重复绑定，请检查!!", roleId);
      }
      if (bindRelation == RbacRelationEnum.MANY_TO_ONE.getRelation()) {
        long count = userGroupRepository.countGroupByRoleId(roleId);
        Validate.isTrue(count == 0, "在批量绑定时,角色和分组多对一，指定角色[%s]已经绑定了分组，不能继续绑定，请检查!!", roleId);
      }
      if (bindRelation == RbacRelationEnum.ONE_TO_MANY.getRelation()) {
        long count = userGroupRepository.countRoleByGroupId(groupId);
        Validate.isTrue(count == 0, "在批量绑定时,角色和分组一对多，指定分组[%s]已经绑定了角色，不能继续绑定，请检查!!", groupId);
      }
      if (bindRelation == RbacRelationEnum.ONE_TO_ONE.getRelation()) {
        long countGroup = userGroupRepository.countGroupByRoleId(roleId);
        Validate.isTrue(countGroup == 0, "在批量绑定时,角色和分组一对一，指定角色[%s]已经绑定了分组，不能继续绑定，请检查!!", roleId);
        long countRole = userGroupRepository.countRoleByGroupId(groupId);
        Validate.isTrue(countRole == 0, "在批量绑定时,角色和分组一对一，指定分组[%s]已经绑定了角色，不能继续绑定，请检查!!", groupId);
      }

    }
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.UserGroupService#bindRole(java.lang.String, java.lang.String[])
   */
  @Override
  @Transactional
  public void bindRole(String groupId, String[] roleIds) {
    Validate.notNull(groupId, ERROR_GROUP_ID);
    Validate.isTrue(roleIds != null && roleIds.length > 0, "角色id不能为空（至少需要传入一个），请检查");
    Optional<UserGroupEntity> op = userGroupRepository.findById(groupId);
    Validate.isTrue(op.isPresent(), ERROR_NOT_FOUND);
    RbacSettingEntity rbacSettingEntity = rbacSettingService.findRbacSetting();
    for (String roleId : roleIds) {
      this.validateBindRole(rbacSettingEntity, groupId, roleId);
      userGroupRepository.bindRole(groupId, roleId);
    }
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.UserGroupService#unBindRole(java.lang.String, java.lang.String[])
   */
  @Override
  @Transactional
  public void unBindRole(String groupId, String[] roleIds) {
    Validate.notNull(groupId, ERROR_GROUP_ID);
    Validate.isTrue(roleIds != null && roleIds.length > 0, "角色id不能为空（至少需要传入一个），请检查");
    Optional<UserGroupEntity> op = userGroupRepository.findById(groupId);
    Validate.isTrue(op.isPresent(), ERROR_NOT_FOUND);
    
    for (String roleId : roleIds) {
      Optional<RoleEntity> optional = roleRepository.findById(roleId);
      Validate.isTrue(optional.isPresent(), "没有该角色[%s],请检查!!" , roleId);
      userGroupRepository.unBindRole(groupId, roleId);
    }
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.UserGroupService#findById(java.lang.String)
   */
  @Override
  public Set<UserGroupVo> findDetailsById(String id) {
    if (StringUtils.isBlank(id)) {
      return Sets.newHashSet();
    }
    Set<UserGroupEntity> userGroupEntitys = userGroupRepository.findDetailsById(id);
    if(userGroupEntitys == null || userGroupEntitys.isEmpty()) {
      return Sets.newHashSet();
    }
    
    // 转换后输出
    Collection<UserGroupVo> userGroupVos = this.nebulaToolkitService.copyCollectionByWhiteList(userGroupEntitys, UserGroupEntity.class, UserGroupVo.class, LinkedHashSet.class, ArrayList.class, "roles", "users");
    return Sets.newHashSet(userGroupVos);
  }

  @Override
  public Set<UserGroupVo> findByIds(List<String> ids) {
    if(CollectionUtils.isEmpty(ids)){
      return Sets.newHashSet();
    }
    Set<UserGroupEntity> groups = userGroupRepository.findByIds(ids);
    if(CollectionUtils.isEmpty(groups)){
      return Sets.newHashSet();
    }
    Collection<UserGroupVo> ugroups = this.nebulaToolkitService.copyCollectionByWhiteList(groups,UserGroupEntity.class,UserGroupVo.class,HashSet.class,ArrayList.class);
    return Sets.newHashSet(ugroups);
  }

  /**
   * 判断是否包含中文
   * @param str
   * @return
   */
  private static boolean isChinese(String str) {
    String regEx = "[\\u4e00-\\u9fa5]+";
    Pattern p = Pattern.compile(regEx);
    Matcher m = p.matcher(str);
    return m.find();
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.UserGroupService#findByUserId(java.lang.String)
   */
  @Override
  public Set<UserGroupVo> findByUserId(String userId) {
    if (StringUtils.isBlank(userId)) {
      return Sets.newHashSet();
    }
    Set<UserGroupEntity> userGroups = userGroupRepository.findByUserId(userId);
    if(CollectionUtils.isEmpty(userGroups)) {
      return Sets.newHashSet();
    }
    Collection<UserGroupVo> groups = this.nebulaToolkitService.copyCollectionByWhiteList(userGroups, UserGroupEntity.class, UserGroupVo.class, HashSet.class,ArrayList.class, "users", "roles");
    return Sets.newLinkedHashSet(groups);
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.UserGroupService#findByCondition(java.lang.String, java.lang.Integer, org.springframework.data.domain.Pageable)
   */
  @Override
  public Page<UserGroupVo> findByCondition(String groupName, Integer status, Pageable pageable) {
    Map<String, Object> conditions = new HashMap<>();
    if (StringUtils.isNotBlank(groupName)) {
      conditions.put("groupName", groupName);
    }
    if (status != null) {
      conditions.put("status", status);
    }
    // 如果当前没有设定分页信息，则默认第一页，每页50条数据
    if (pageable == null) {
      pageable = PageRequest.of(0, 50);
    }
    Page<UserGroupEntity> userGroupPage = userGroupRepository.queryPage(pageable, conditions);
    
    // 分页信息需要自行转换
    List<UserGroupEntity> userGroupEntitys = userGroupPage.getContent();
    Page<UserGroupVo> userGroupVoPage = null;
    if(!userGroupEntitys.isEmpty()) {
      Collection<UserGroupVo> userGroupVos = this.nebulaToolkitService.copyCollectionByWhiteList(userGroupEntitys, UserGroupEntity.class, UserGroupVo.class, LinkedHashSet.class, ArrayList.class);
      userGroupVoPage = new PageImpl<>(new ArrayList<>(userGroupVos), pageable, userGroupPage.getTotalElements());
    } else {
      userGroupVoPage = new PageImpl<>(Lists.newArrayList(), pageable, 0l);
    }
    return userGroupVoPage;
  }

  @Override
  public UserGroupVo findByGroupName(String groupName) {
    if(StringUtils.isBlank(groupName)) {
      return null;
    }
    UserGroupEntity userGroup = userGroupRepository.findByGroupName(groupName);
    if(userGroup == null) {
      return null;
    }
    return nebulaToolkitService.copyObjectByWhiteList(userGroup, UserGroupVo.class, HashSet.class, ArrayList.class);
  }

  /**
   * 查询所有用户信息
   * @return
   */
  @Override
  public List<UserGroupVo> findAll(){
    Sort sort = Sort.by(Sort.Order.desc("createTime"));
    List<UserGroupEntity> entitySet = userGroupRepository.findAll(sort);
    if(CollectionUtils.isEmpty(entitySet)){
      return Lists.newArrayList();
    }
    Collection<UserGroupVo> groups = nebulaToolkitService.copyCollectionByWhiteList(entitySet, UserGroupEntity.class, UserGroupVo.class, HashSet.class, ArrayList.class, "users", "roles");
    return Lists.newArrayList(groups);
  }
}
