package com.bizunited.platform.core.service.serviceable.handle;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.core.common.constant.ParamClassTypeConst;
import com.bizunited.platform.core.common.utils.ParamsAnalysisUtil;
import com.bizunited.platform.core.entity.ServicableMethodEntity;
import com.bizunited.platform.core.entity.ServicableMethodPropertyEntity;
import com.bizunited.platform.core.service.invoke.HandleChain;
import com.bizunited.platform.core.service.invoke.HandleChain.ChainLogic;
import com.bizunited.platform.core.service.invoke.InvokeProxyContext;
import com.bizunited.platform.core.service.invoke.InvokeProxyException;
import com.bizunited.platform.core.service.invoke.InvokeRequestHandle;
import com.bizunited.platform.core.service.invoke.model.InvokeOperations;
import com.bizunited.platform.core.service.invoke.model.InvokeParams;
import com.bizunited.platform.core.service.serviceable.ServicableMethodService;
import com.bizunited.platform.core.service.serviceable.model.InputParamsModel;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Component;

import java.lang.reflect.Array;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;

/**
 * 该处理过程对基于HTTP协议传递来的最原始的调用请求（包括服务名、服务参数等信息）进行分析，并形成处理链后续结构能够看懂的结构封装。
 * 解析成功后的结构将通过ServicableContext传递给后续处理节点。
 * @author yinwenjie
 */
@Component("SpringMVCParamsHandle")
public class ServicableInvokeHandle implements InvokeRequestHandle {
  
  private static final Logger LOGGER = LoggerFactory.getLogger(ServicableInvokeHandle.class);

  @Autowired
  private ServicableMethodService servicableMethodService;
  @Autowired
  private ApplicationContext applicationContext;

  /* (non-Javadoc)
   * @see com.bizunited.platform.core.service.invoke.invokeProxyRequestHandle#doHandle(com.bizunited.platform.core.service.invoke.ServicableContext, com.bizunited.platform.core.service.invoke.HandleChain)
   */
  @Override
  public void doHandle(InvokeProxyContext context, HandleChain chain) throws InvokeProxyException {
    // 检测上下文中需要的各种入参信息
    InputParamsModel model = (InputParamsModel)context.getChainParam(InputParamsModel.class.getName());
    Validate.notNull(model , "使用SpringMVCParamsHandle分析入参情况是，初始化的入参对象InputParamsModel，必须通过InputParamsModel.class.getName()作为key的方式在调用时传入");
    Object principalValue = context.getChainParam("principal");
    Validate.notNull(principalValue , "在处理SpringMVCParamsHandle时，没有发现登录者信息，请检查!!");
    String userName = principalValue.toString();
    String servicableMethodName = model.getServiceName();
    Validate.notBlank(servicableMethodName , "错误的静态服务源名，请检查!!");
    
    // 1、==== 从上下文取出被调用服务的详情
    ServicableMethodEntity currentServicableMethod = servicableMethodService.findByName(servicableMethodName);
    Validate.notNull(currentServicableMethod , "为找到服务源参数[ServicableMethodEntity]信息，请检查!!");
    // 方法没有入参时，则可直接进行代理调用
    if (currentServicableMethod.getProperties() == null || currentServicableMethod.getProperties().isEmpty()) {
      chain.doHandle(context, ChainLogic.BREAK);
    }
    
    // 2、==== （参数验证）根据从数据库或者缓存里获取到的服务信息，与前端传入的参数进行验证、筛选等
    // 用于组合model中的两个map(params , customData)
    Map<String, Object> assembleMap = new HashMap<>();
    // 对参数进行索引排序
    Set<ServicableMethodPropertyEntity> properties = new TreeSet<>((e1, e2) -> e1.getPropertyIndex().intValue() - e2.getPropertyIndex().intValue());
    properties.addAll(currentServicableMethod.getProperties());
    Object[] args = new Object[properties.size()];// 动态调用时的最终参数数组
    Class<?>[] argsClass = new Class[properties.size()];// 动态调用时的最终参数数组的类类型
    LOGGER.debug("****{}服务动态调用****开始准备处理参数*****需要处理的参数个数：{}",model.getServiceName(), (properties == null || properties.isEmpty() ? 0 : properties.size()));
    // 2.1.1) 判断合并前端传入的两个map
    if (model.getParams() != null && !model.getParams().isEmpty()) {
      assembleMap.putAll(model.getParams());
    }
    if (model.getCustomData() != null && !model.getCustomData().isEmpty()) {
      Set<String> customSet = model.getCustomData().keySet();
      Validate.isTrue(Sets.intersection(customSet, assembleMap.keySet()).isEmpty(), "传入的params和customData有键值冲突，请检查!!");
      assembleMap.putAll(model.getCustomData());
    }

    InvokeParams conditionParams = new InvokeParams();
    boolean existMap = properties.stream().anyMatch(e -> e.getParamType() == 5);//判断是否存在参数Map结构
    //如果定义的服务源中，参数存在Map结构，需要把前端传入的params和customData结构的参数，都封装到KuiperInvokeParams中
    if(existMap) {
      if(model.isFilterParamsExist()) {
        List<InvokeOperations> filterParams = model.getFilterParams();
        for(InvokeOperations k : filterParams) {
          conditionParams.add(k.getParamName(), k, true);
        }
      }
      for(Map.Entry<String, Object> entry : assembleMap.entrySet()) {
        InvokeOperations m = new InvokeOperations();
        m.setParamName(entry.getKey());
        m.setCompareValue(entry.getValue());
        conditionParams.add(entry.getKey(), m, false);
      }
    }

    // 2.1.2) 判断组装数据
    Class<?> iocBeanClass = null;
    try {
      this.analysis(context, model, args, argsClass, userName, properties, assembleMap, conditionParams);
      iocBeanClass = context.getClassLoader().loadClass(currentServicableMethod.getInterfaceName());
    } catch(ClassNotFoundException e) {
      throw new IllegalArgumentException(e.getMessage(), e);
    }
    
    // 进行调用
    LOGGER.debug("*******************参数处理结束**********************");
    Validate.notNull(iocBeanClass , "未在spring ioc容器中发现指定的class定义!!");
    Object iocBean = applicationContext.getBean(iocBeanClass);
    Validate.notNull(iocBean , "未在spring ioc容器中发现指定的bean定义!!");
    try {
      this.invoke(context, iocBean, currentServicableMethod.getSimpleMethodName(), argsClass, args);
    } catch (NoSuchMethodException | InvocationTargetException | IllegalAccessException e) {
      //由于jdk反射机制在调用时，会把异常重新封装，所以这里要判断一下，才能拿到真实的异常的记录
      if(e instanceof InvocationTargetException) {
        InvocationTargetException invocation = (InvocationTargetException) e;
        Throwable throwableException = invocation.getTargetException();
        throw new IllegalArgumentException(throwableException.getMessage(),e);
      }
      throw new IllegalArgumentException(e.getMessage(), e);
    }
    // 继续下一个处理节点
    chain.doHandle(context, ChainLogic.CONTINUE);
  }
  
  /**
   * JDK原生反射的方法调用
   */
  private void invoke(InvokeProxyContext context , Object objBean , String simpleMethodName , Class<?>[] argsClass , Object[] argsValues) throws NoSuchMethodException , InvocationTargetException , IllegalAccessException {
    Method method = objBean.getClass().getMethod(simpleMethodName, argsClass);
    Object result = null;
    result = method.invoke(objBean, argsValues);
    if(result != null) {
      context.setResult(result);
    }
  }
  
  /**
   * 判断组装数据
   */
  private void analysis(InvokeProxyContext context , InputParamsModel model , Object[] args , Class<?>[] argsClass , String userName , Set<ServicableMethodPropertyEntity> properties , Map<String, Object> assembleMap, InvokeParams conditionParams) throws ClassNotFoundException {
    for (ServicableMethodPropertyEntity pro : properties) {
      LOGGER.debug("****开始处理参数*****参数类型：" + pro.getParamType() + " 参数索引：" + pro.getPropertyIndex() + "  参数限定名：" + (StringUtils.isBlank(pro.getAnnonQualifiedName()) ? "无" : pro.getAnnonQualifiedName()));
      switch (pro.getParamType()) {
        case 2:
          args[pro.getPropertyIndex()] = userName;
          argsClass[pro.getPropertyIndex()] = Class.forName(pro.getParamClass());
          break;
        case 3:// 分页数据(分页只能有一个参数)
          int page = assembleMap.get("page") == null ? 0 : (int) assembleMap.remove("page");
          int size = assembleMap.get("size") == null ? 50 : (int) assembleMap.remove("size");
          
          Object pageData = PageRequest.of(page, size);
          args[pro.getPropertyIndex()] = pageData;
          argsClass[pro.getPropertyIndex()] = Class.forName(pro.getParamClass());
          break;
        case 4:// form表单数据(表单数据只能有一个)
          JSONObject form = model.getFormData();
          if (form != null) {
            String modelType = pro.getModelType();
            try {
              Class<?> clz = context.getClassLoader().loadClass(modelType);
              LOGGER.debug("<<<转换后的form参数>>>：{}",  form);
              args[pro.getPropertyIndex()] = JSON.toJavaObject(form, clz);
            } catch (Exception e2) {
              throw new IllegalArgumentException("json数据类型转换异常，请检查传入的json数据类型!!" + e2.getMessage());
            }
          } else {
            args[pro.getPropertyIndex()] = null;
          }
          argsClass[pro.getPropertyIndex()] = context.getClassLoader().loadClass(pro.getModelType());
          break;
        case 5:// KuiperInvokeParams数据(KuiperInvokeParams类型数据只能有一个，其实默认就是k-v结构的数据集)并且需要合并params和customData的k-v
          //this.checkKeys(model, conditionParams);//这里肯定是原map参数或者需要构造这样的kuiperInvokeParams
          args[pro.getPropertyIndex()] = conditionParams;
          argsClass[pro.getPropertyIndex()] = InvokeParams.class;
          break;
        // 以下的情况很可能有多个
        case 1:// params数据和customData数据的组合
          Object data = fetchObject(pro, assembleMap, properties);
          if (data == null) {
            args[pro.getPropertyIndex()] = null;
          } else {
            LOGGER.debug("<<<传入的参数被封装后的类型为>>>：{}", data.getClass().getName());
            String jdkClassName = pro.getParamClass();
            Class<?> jdkClass = ParamClassTypeConst.baseClassMapContainsKey(jdkClassName) ? ParamClassTypeConst.getBaseClass(jdkClassName) : Class.forName(jdkClassName);
            String frontClassName = data.getClass().getName();
            Class<?> frontClass = Class.forName(frontClassName);
            if (jdkClass.isAssignableFrom(frontClass)) {
              args[pro.getPropertyIndex()] = data;
            } else if (!jdkClassName.contains(".") && jdkClassName.startsWith("[")) {
              int n = 0;// 计数器
              int index = 0;// 指定字符的长度
              index = jdkClassName.indexOf('[');
              while (index != -1) {
                n++;
                index = jdkClassName.indexOf('[', index + 1);
              }

              if (n == 1) {
                try {
                  args = ParamsAnalysisUtil.unwapperArray(data, args, pro.getPropertyIndex(), jdkClassName);// 一维的java基本类型数组
                } catch (Exception e) {
                  throw new IllegalArgumentException("基本类型数组转换异常，请检查!!");
                }
              } else {
                try {
                  args = ParamsAnalysisUtil.unwapperArray2(data, args, pro.getPropertyIndex(), jdkClassName);// 二维的java基本类型数组
                } catch (Exception e) {
                  throw new IllegalArgumentException("二维基本类型数组转换异常，请检查!!");
                }
              }
            } else if (!jdkClassName.contains(".") && !jdkClassName.startsWith("[")) {
              try {
                args = ParamsAnalysisUtil.unwapper(data, args, pro.getPropertyIndex(), jdkClassName);// java基本类型
              } catch (Exception e) {
                throw new IllegalArgumentException("java基本类型数据转换异常，请检查!!");
              }
            } else {// 这里是多维(一维或二维)不同对象类型的数组(去除集合ArrayList的封装)只支持到二维
              List<?> source = null;
              try {
                source = (ArrayList<?>) data;
              } catch (Exception e) {
                throw new IllegalArgumentException("传入的数组数据类型转换异常，请检查!!");
              }
              // 判断一维还是二维
              Object oneDimensioArr = null;
              Object twoDimensioArr = null;
              boolean flag = false;

              for (int x = 0; x < source.size(); x++) {
                if (source.get(x) == null) {
                  continue;
                } else {
                  flag = source.get(x).getClass().getName().equals(ArrayList.class.getName());
                  break;
                }
              }
              
              if (!flag) {// 一维数组
                oneDimensioArr = Array.newInstance(Class.forName(jdkClassName.replaceAll("\\[L", "").replace(";", "")), source.size());
                for (int index = 0; index < source.size() && source.get(index) != null; index++) {
                  Array.set(oneDimensioArr, index, source.get(index));
                }
                args[pro.getPropertyIndex()] = oneDimensioArr;
              } else {// 二维数组
                String twoDiClassName = jdkClassName.replaceFirst("\\[", "");
                String oneDiClassName = jdkClassName.replaceAll("\\[\\[L", "").replace(";", "");
                twoDimensioArr = Array.newInstance(Class.forName(twoDiClassName), source.size());
                for (int index = 0; index < source.size() && source.get(index) != null; index++) {
                  Object oneDimensio = Array.newInstance(Class.forName(oneDiClassName), ((ArrayList<?>) source.get(index)).size());
                  ArrayList<?> temp = ((ArrayList<?>) source.get(index));
                  if (temp.isEmpty()) {
                    Array.set(twoDimensioArr, index, Array.newInstance(Class.forName(oneDiClassName), 0));
                    continue;
                  }
                  for (int x = 0; x < temp.size() && temp.get(x) != null; x++) {
                    Array.set(oneDimensio, x, temp.get(x));
                    Array.set(twoDimensioArr, index, oneDimensio);
                  }
                }
                args[pro.getPropertyIndex()] = twoDimensioArr;
              }
            }
          }

          if (Boolean.TRUE.equals(ParamClassTypeConst.baseClassMapContainsKey(pro.getParamClass()))) {
            argsClass[pro.getPropertyIndex()] = ParamClassTypeConst.getBaseClass(pro.getParamClass());
          } else {
            argsClass[pro.getPropertyIndex()] = Class.forName(pro.getParamClass());
          }
          break;
      }
    }
  }
  
  /**
   * 根据@ServiceMethodParam注解的限定名，从Map里获取指定数据
   * @param pro
   * @param source
   * @return
   */
  private Object fetchObject(ServicableMethodPropertyEntity pro , Map<String , Object> source , Set<ServicableMethodPropertyEntity> properties) {
    if(StringUtils.equals("ignoreValidate", pro.getAnnonQualifiedName())) {
      return false;
    }
    
    if(StringUtils.isNotBlank(pro.getAnnonQualifiedName())) {
      if(properties.size() == 1 && pro.getParamType() == 1 && source.values().size() == 1) {
        return source.values().iterator().next();
      }
      return source.remove(pro.getAnnonQualifiedName());
    }
    
    return null;
  }

}