package com.bizunited.platform.core.controller;

import com.bizunited.platform.core.controller.model.ResponseModel;
import com.bizunited.platform.rbac.server.service.ButtonService;
import com.bizunited.platform.rbac.server.vo.ButtonVo;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.security.Principal;
import java.util.Set;

/**
 * 按钮的管理类，负责提供相关接口
 * @author: zengxingwang
 * @date: 2020/1/13 14:37
 */
@RestController
@RequestMapping(value="/v1/nebula/buttons")
public class ButtonController extends BaseController{

  /** 日志 */
  private static final Logger LOGGER = LoggerFactory.getLogger(ButtonController.class);

  @Autowired
  private ButtonService buttonService;


  /**
   * 创建按钮
   * @param button
   * @return
   */
  @PostMapping("")
  @ApiOperation(value = "创建按钮", notes = "创建按钮，必须要传递菜单对象，并且菜单对象ID不能为空")
  public ResponseModel create(@ApiParam("按钮实体信息") @RequestBody ButtonVo button){
    try {
      ButtonVo result = this.buttonService.create(button);
      return buildHttpResultW(result);
    }catch (RuntimeException e){
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 批量创建按钮
   * @param buttons
   * @return
   */
  @PostMapping("batchCreate")
  @ApiOperation(value = "批量创建按钮", notes = "创建按钮，必须要传递菜单对象，并且菜单对象ID不能为空")
  public ResponseModel batchCreate(@ApiParam("按钮实体信息") @RequestBody Set<ButtonVo> buttons){
    try {
      this.buttonService.batchCreate(buttons);
      return buildHttpResult();
    }catch (RuntimeException e){
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 修改按钮
   * @param button
   * @return
   */
  @PatchMapping("")
  @ApiOperation(value = "修改按钮", notes = "修改按钮")
  public ResponseModel update(@ApiParam("按钮实体信息") @RequestBody ButtonVo button){
    try {
      ButtonVo result = this.buttonService.update(button);
      return buildHttpResultW(result);
    }catch (RuntimeException e){
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }


  /**
   * 根据id删除按钮(批量)
   * @param ids
   * @return
   */
  @DeleteMapping("batchDelete")
  @ApiOperation(value = "根据id批量删除按钮")
  public ResponseModel batchDelete(@RequestParam(name = "ids") @ApiParam(name = "ids", value = "指定的按钮id信息(可一个是一个或者多个)") String[] ids){
    try {
      this.buttonService.batchDelete(ids);
      return buildHttpResult();
    }catch (RuntimeException e){
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 根据id删除按钮
   * @param id
   * @return
   */
  @DeleteMapping("/{id}")
  @ApiOperation(value = "根据id删除按钮", notes = "根据id删除按钮")
  public ResponseModel deleteById(@ApiParam(name = "id", value = "按钮ID", required = true) @PathVariable("id") String id){
    try {
      this.buttonService.deleteById(id);
      return buildHttpResult();
    }catch (RuntimeException e){
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 根据菜单ID查询按钮
   * @param competenceId
   * @return
   */
  @GetMapping("findByCompetenceId")
  @ApiOperation(value = "根据菜单ID查询按钮", notes = "根据菜单ID查询按钮")
  public ResponseModel findByCompetenceId(@ApiParam(name = "competenceId", value = "按钮ID", required = true) @RequestParam("competenceId") String competenceId){
    try {
      Set<ButtonVo> result = this.buttonService.findByCompetenceId(competenceId);
      return buildHttpResultW(result);
    }catch (RuntimeException e){
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 根据角色code查询关联的按钮
   * @param roleCode
   * @return
   */
  @GetMapping("findByRoleCode")
  @ApiOperation(value = "根据角色code查询关联的按钮", notes = "根据角色code查询关联的按钮")
  public ResponseModel findByRoleCode(@ApiParam(name = "roleCode", value = "角色编码", required = true) @RequestParam("roleCode") String roleCode){
    try {
      Set<ButtonVo> result = this.buttonService.findByRoleCode(roleCode);
      return buildHttpResultW(result);
    }catch (RuntimeException e){
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 建立指定按钮和指定接口的绑定关系
   * @param buttonId
   * @param competenceIds
   * @return
   */
  @ApiOperation(value = "建立指定按钮和指定接口的绑定关系")
  @PatchMapping(value = "/bindCompetences")
  public ResponseModel bindCompetences(@RequestParam(name = "buttonId") @ApiParam(name = "buttonId", value = "指定的按钮编号信息") String buttonId,
                                   @RequestParam(name = "competenceIds") @ApiParam(name = "competenceIds", value = "指定的接口编号信息(可一个是一个或者多个)") String[] competenceIds) {
    try {
      this.buttonService.bindCompetences(buttonId, competenceIds);
      return this.buildHttpResult();
    } catch(Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 取消指定按钮和指定接口的绑定关系
   * @param buttonId
   * @param competenceIds
   * @return
   */
  @ApiOperation(value = "取消指定按钮和指定接口的绑定关系")
  @PatchMapping(value = "/unbindCompetences")
  public ResponseModel unbindCompetences(@RequestParam(name="buttonId") @ApiParam(name = "buttonId", value = "指定的按钮编号信息") String buttonId ,
                                           @RequestParam(name="competenceIds") @ApiParam(name = "competenceIds", value = "指定的接口编号信息(可一个是一个或者多个)") String[] competenceIds) {
    try {
      this.buttonService.unbindCompetences(buttonId, competenceIds);
      return this.buildHttpResult();
    } catch(Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据用户ID查询用户的按钮权限
   * @param userId
   * @return
   */
  @ApiOperation(value = "根据用户ID查询用户的按钮权限")
  @GetMapping("findByUserId")
  public ResponseModel findByUserId(@RequestParam(name = "userId") @ApiParam(name = "userId", value = "指定的用户的编号信息") String userId){
    try {
      Set<ButtonVo> buttonVos = this.buttonService.findByUserId(userId);
      return this.buildHttpResultW(buttonVos);
    } catch(Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 查询当前登录用户的按钮权限，如果当前登录用户有忽略权限的按钮，那就返回所有的权限按钮
   * @param userId
   * @return
   */
  @GetMapping("findByCurrentUser")
  @ApiOperation(value = "查询当前登录用户的按钮权限", notes = "查询当前登录用户的按钮权限,如果当前登录用户有忽略权限的按钮，那就返回所有的权限按钮")
  public ResponseModel findByCurrentUser(){
    try {
      Principal principal = super.getPrincipal();
      Set<ButtonVo> buttonVos = this.buttonService.findByCurrentUser(principal);
      return this.buildHttpResultW(buttonVos);
    } catch(Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据按钮id查询按钮信息
   * @param buttonId
   * @return
   */
  @ApiOperation(value = "根据按钮id查询按钮信息")
  @GetMapping("findDateilsById")
  public ResponseModel findDateilsById(@RequestParam(name = "buttonId") @ApiParam(name = "buttonId", value = "指定的按钮的编号信息") String buttonId ){
    try {
      ButtonVo buttonVo = this.buttonService.findDateilsById(buttonId);
      return this.buildHttpResultW(buttonVo, "competences", "competence", "roles");
    } catch(Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
}
