package com.bizunited.platform.core.controller;

import java.security.Principal;
import java.util.Set;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.core.controller.model.ResponseModel;
import com.bizunited.platform.core.entity.DynamicTaskSchedulerEntity;
import com.bizunited.platform.core.service.DynamicTaskSchedulerService;

import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;

/**
 * 这个controller用于对动态任务的基本信息、动态任务的脚本内容、启动、停止功能进行操作。但是本controller所提供的HTTP接口，并不包括直接对动态任务进行执行
 * @author yinwenjie
 */
@RestController
@RequestMapping("/v1/nebula/dynamicTasks")
public class DynamicTaskSchedulerController extends BaseController {
  @Autowired
  private DynamicTaskSchedulerService dynamicTaskSchedulerService;
  
  /**
   * 创建一个新的定时任务
   */
  @ApiOperation(value = "创建一个新的定时任务")
  @RequestMapping(value = "", method = RequestMethod.POST)
  public ResponseModel create(@RequestParam("taskstr") @ApiParam(name = "taskstr" , value = "以json字符串形式表单的动态任务描述") String taskstr , 
                              @RequestParam("scriptContent")  @ApiParam(name = "scriptContent" , value = "目前仅支持groovy形式的脚本内容") String scriptContent) {
    try {
      Principal principal = this.getPrincipal();
      DynamicTaskSchedulerEntity task = JSONObject.parseObject(taskstr, DynamicTaskSchedulerEntity.class);
      task = dynamicTaskSchedulerService.create(task, scriptContent, principal);
      return buildHttpResultW(task , "script");
    } catch (Exception e) {
      return buildHttpResultForException(e);
    }
  }
  /**
   * 更新一个指定的task状态，只有状态同时处于“有效”的“未工作”任务，才能进行更新操作
   */
  @ApiOperation(value = "更新一个指定的task状态，只有状态同时处于“有效”的“未工作”任务，才能进行更新操作" , notes = "该方法仅提供对一些基本属性进行更新")
  @RequestMapping(value = "", method = RequestMethod.PATCH)
  public ResponseModel update(@RequestParam("taskstr") @ApiParam(name = "taskstr" , value = "以json字符串形式表单的动态任务描述") String taskstr,
                              @RequestParam("scriptContent")  @ApiParam(name = "scriptContent" , value = "目前仅支持groovy形式的脚本内容") String scriptContent) {
    try {
      Principal principal = this.getPrincipal();
      DynamicTaskSchedulerEntity task = JSONObject.parseObject(taskstr, DynamicTaskSchedulerEntity.class);
      task = dynamicTaskSchedulerService.update(task, scriptContent, principal);
      return buildHttpResultW(task);
    } catch (Exception e) {
      return buildHttpResultForException(e);
    }
  }
  /**
   * 将指定的一个或者多个动态任务的状态变更为“失效”
   * @param taskCodes 可以一次传入多个需要强制停止的动态任务编号
   */
  @ApiOperation(value = "停止指定的任务，只有处于“运行中”的任务能够被停止" , notes = "由于当前的“轻质请求”可能不来自于master节点，所以这里只是更改任务状态，以便master节点下一次任务状态扫描时停止该节点" )
  @RequestMapping(value = "/updateInvalidByTaskCode", method = RequestMethod.PATCH)
  public ResponseModel updateInvalidByTaskCode(@RequestParam("taskCodes") @ApiParam(name = "taskCodes" , value = "可以一次传入多个需要强制停止的动态任务编号code(可以是多个)") String[] taskCodes) {
    try {
      dynamicTaskSchedulerService.updateInvalidByTaskCode(taskCodes);
      return buildHttpResult();
    } catch (Exception e) {
      return buildHttpResultForException(e);
    }
  }
  /**
   * 停止指定的任务，只有处于“运行中”的任务能够被停止，
   *由于当前的“轻质请求”可能不来自于master节点，所以这里只是更改任务状态，以便master节点下一次任务状态扫描时停止该节点
   * @param taskCode
   */
  @ApiOperation(value = "停止指定的任务，只有处于“运行中”的任务能够被停止" , notes = "由于当前的“轻质请求”可能不来自于master节点，所以这里只是更改任务状态，以便master节点下一次任务状态扫描时停止该节点" )
  @RequestMapping(value = "/interrupt", method = RequestMethod.POST)
  public ResponseModel interrupt(@RequestParam("taskCode") @ApiParam(name = "taskCode" , value = "指定的动态任务编号信息") String taskCode) {
    try {
      this.dynamicTaskSchedulerService.interrupt(taskCode);
      return this.buildHttpResult();
    } catch(Exception e) {
      return buildHttpResultForException(e);
    }
  }
  /**
   * 启动指定的任务，由于当前的“启动请求”可能不来自于master节点，所以这里只是更改任务状态，以便master节点下一次任务状态扫描时启动该节点
   * @param taskCode
   */
  @ApiOperation(value = "启动指定的任务" , notes = "由于当前的“启动请求”可能不来自于master节点，所以这里只是更改任务状态，以便master节点下一次任务状态扫描时启动该节点,无效任务也可以使用该方法，一旦使用无效任务就会变成有效任务" )
  @RequestMapping(value = "/start", method = RequestMethod.POST)
  public ResponseModel start(@RequestParam("taskCode") @ApiParam(name = "taskCode" , value = "指定的动态任务编号信息") String taskCode) {
    try {
      this.dynamicTaskSchedulerService.start(taskCode);
      return this.buildHttpResult();
    } catch(Exception e) {
      return buildHttpResultForException(e);
    }
  }
  /**
   * 查询当前已设定所有动态任务，无论其状态如何；并且返回信息按照创建时间倒序排列。
   * 查询信息还包括了其所有的直接关联信息
   */
  @ApiOperation(value = "查询当前已设定所有动态任务，无论其状态如何；并且返回信息按照创建时间倒序排列。")
  @RequestMapping(value = "/findAll", method = RequestMethod.GET)
  public ResponseModel findAll() {
    try {
      Set<DynamicTaskSchedulerEntity> resutls = this.dynamicTaskSchedulerService.findAll();
      return this.buildHttpResultW(resutls);
    } catch(Exception e) {
      return buildHttpResultForException(e);
    }
  }
  /**
   * 按照指定的任务业务编号，查询动态任务详细情况。只包括动态任务的基本信息和可能的创建者、修改者信息
   * @param taskCode 指定的任务详细情况编号
   * @return 
   */
  @ApiOperation(value = "按照指定的任务业务编号，查询动态任务详细情况。只包括动态任务的基本信息和可能的创建者、修改者信息")
  @RequestMapping(value = "/findByTaskCode", method = RequestMethod.GET)
  public ResponseModel findByTaskCode(@RequestParam(name="taskCode" , required=false) @ApiParam(name = "taskCode", value = "任务唯一编号（只能由英文、数字、下杠构成）" , required=false) String taskCode) {
    try {
      DynamicTaskSchedulerEntity result = this.dynamicTaskSchedulerService.findByTaskCode(taskCode);
      return this.buildHttpResultW(result , "modifyUser","createUser");
    } catch(Exception e) {
      return buildHttpResultForException(e);
    }
  }
  /**
   * -该方法用于按照分页标准，对动态任务列表进行查询
   * @param pageable 分页信息
   * @param taskType 任务类型：1、一次性执行；2：周期性执行
   * @param taskCode 任务唯一编号（只能由英文、数字、下杠构成）
   * @param tstatus 任务状态：0：无效任务；1、有效任务
   * @param workingStatus 工作状态：0：要求运行， 1：人工停止，2：已运行
   * @param taskDesc 任务中文说明（任务名）
   * @return 
   */
  @ApiOperation(value = "查询当前已设定所有动态任务，无论其状态如何；并且返回信息按照创建时间倒序排列。")
  @RequestMapping(value = "/findByConditions", method = RequestMethod.GET)
  public ResponseModel findByConditions(@PageableDefault(value = 50) @ApiParam(name = "pageable", value = "分页参数，当指定page时为查询当前页码（页码从0开始）；当指定size时，为指定每页大小，默认为50")  Pageable pageable, 
                                        @RequestParam(name="taskType" , required=false) @ApiParam(name = "taskType", value = "任务类型：1、一次性执行；2：周期性执行" , required=false) Integer taskType , 
                                        @RequestParam(name="taskCode" , required=false) @ApiParam(name = "taskCode", value = "任务唯一编号（只能由英文、数字、下杠构成）" , required=false) String taskCode ,
                                        @RequestParam(name="tstatus" , required=false) @ApiParam(name = "tstatus", value = "任务状态：0：无效任务；1、有效任务" , required=false) Integer tstatus , 
                                        @RequestParam(name="workingStatus" , required=false) @ApiParam(name = "workingStatus", value = "工作状态：0：要求运行， 1：人工停止，2：已运行" , required=false) Integer workingStatus , 
                                        @RequestParam(name="taskDesc" , required=false) @ApiParam(name = "taskDesc", value = "任务中文说明（任务名）" , required=false) String taskDesc , 
                                        @RequestParam(name="invokeType" , required=false) @ApiParam(name = "invokeType", value = "定时器任务内容的执行方式：1：groovy脚本执行器；2：基于注解的Java method" , required=false) Integer invokeType) {
    try {
      Page<DynamicTaskSchedulerEntity> results = this.dynamicTaskSchedulerService.findByConditions(pageable, taskType, taskCode, tstatus, workingStatus, taskDesc , invokeType);
      return this.buildHttpResultW(results, "modifyUser","createUser");
    } catch(Exception e) {
      return buildHttpResultForException(e);
    }
  }
}
