package com.bizunited.platform.core.controller;

import java.io.IOException;
import java.security.Principal;
import java.util.List;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;

import com.bizunited.platform.core.controller.model.ResponseModel;
import com.bizunited.platform.core.entity.OrdinaryFileEntity;
import com.bizunited.platform.core.service.image.FileUpdateService;
import com.bizunited.platform.core.service.image.FileViewService;

import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;

/**
 * 这个controller用于文件附件的上传，支持由白名单放行的一般附件，以及位图图片文件（jpeg、jpg、gif、png、bmp）
 * @author yinwenjie
 */
@RestController
@RequestMapping("/v1/nebula/files")
public class FileController extends BaseController {
  @Autowired
  private FileUpdateService fileUpdateService;
  @Autowired
  private FileViewService fileViewService;
  private static final String UPLOAD_ERROR = "you must upload least one file !";

  /**
   * 这个方法用于上传若干张附件文件，且有子系统区别时使用
   * @param subsystem
   * @param effective
   * @param files
   * @return
   */
  @ApiOperation(value = "这个方法可用于一次上传多个文件附件（支持文件附件上传）", notes = ""
      + "1、目前支持普通文件，普通文件通过配置文件进行后缀名的白名单控制"
      + "2、只能上传大小默认不超过2048KB的文件（可设置），也就是说如果上传的文件较大，那么客户端需要自行压缩一下<br>"
      + "3、上传成功后，这个请求将返回文件在服务器端保存的唯一id还包括了文件对象的其它信息，如果是图片文件，那么在文件访问时还可以加相应特效哦，具体请参见/v1/kuiper/images下的controller api说明信息）")
  @RequestMapping(path = "/{subsystem}/fileUpload", method = RequestMethod.POST)
  public ResponseModel fileUpload(@PathVariable("subsystem") String subsystem, 
                                  @RequestParam(name="effective" , required=false) @ApiParam(required=false , name="effective" , value="设定的文件有效期时长，时长单位为“天”") Integer effective , 
                                  @RequestParam(name="file" , required=true) @ApiParam(required=true , name="file" , value="上传的文件对象，提交的表单信息中，请命名为files（支持多个文件，建议不超过5个文件）") MultipartFile[] files) {
    if (files == null || files.length == 0) {
      throw new IllegalArgumentException(UPLOAD_ERROR);
    }
    // 依次处理每个文件
    try {
      Principal opUser = this.getPrincipal();
       List<OrdinaryFileEntity> imageUploadPojos = this.fileUpdateService.fileUpload(subsystem, opUser.getName(), effective, files);
      return this.buildHttpResult(imageUploadPojos);
    } catch(Exception e) {
      return this.buildHttpResultForException(e);
    }
  }
  
  /**
   * 文件上传服务，这个文件不一定是图片，只要满足配置信息中规定后缀的文件都可以上传，注意这些文件的有效期为永久。
   * 如果要更改文件有效时间，则需要指定effective参数，该参数是指定文件有效时长，单位为“天”</p>
   * 并且上传的文件内容为base64编码
   * @param subsystem 指代进行文件上传的子系统信息，子系统将单独生成一个文件夹。
   * @param effective 这些文件的有效时长，单位为“天”。如果没有传入则默认这些文件有效期为永久
   * @param fileNanmes 文件上传时的文件名
   * @param base64Contents 文件内容的base64编码描述
   * @return 
   * @throws IllegalArgumentException
   */
  @ApiOperation(value = "这个方法可用于一次上传多个文件附件（支持文件附件上传）", notes = ""
      + "1、目前支持普通文件，普通文件通过配置文件进行后缀名的白名单控制"
      + "2、只能上传大小默认不超过2048KB的文件（可设置），也就是说如果上传的文件较大，那么客户端需要自行压缩一下<br>"
      + "3、上传成功后，这个请求将返回文件在服务器端保存的唯一id还包括了文件对象的其它信息，如果是图片文件，那么在文件访问时还可以加相应特效哦，具体请参见/v1/kuiper/images下的controller api说明信息）")
  @RequestMapping(path = "/{subsystem}/fileUploadBase64", method = RequestMethod.POST)
  public ResponseModel fileUpload(@PathVariable("subsystem") String subsystem , 
                                  @RequestParam(name="effective" , required=false) @ApiParam(required=false , name="effective" , value="设定的文件有效期时长，时长单位为“天”") Integer effective , 
                                  @RequestParam(name="fileNanmes" , required=true) @ApiParam(required=true , name="fileNanmes" , value="带有文件后缀，但是不带有文件路径的的文件名(一个或者多个)") String[] fileNanmes , 
                                  @RequestParam(name="base64Contents" , required=true) @ApiParam(required=true , name="base64Contents" , value="用base64编码的文件内容（一个或者多个）") String[] base64Contents) {
    try {
      Principal opUser = this.getPrincipal();
       List<OrdinaryFileEntity> imageUploadPojos = this.fileUpdateService.fileUpload(subsystem, opUser.getName(), effective, fileNanmes, base64Contents);
      return this.buildHttpResult(imageUploadPojos);
    } catch(Exception e) {
      return this.buildHttpResultForException(e);
    }
  }

  /**
   *
   * 这个方法用于上传若干张图片文件，且有子系统区别时使用
   * @param subsystem
   * @param effective
   * @param files
   * @return
   */
  @ApiOperation(value = "这个方法可用于一次上传多个文件附件（支持文件附件上传）", notes = ""
      + "1、只支持图片文件的上传"
      + "2、只能上传大小默认不超过2048KB的图片文件（可设置）<br>"
      + "3、上传成功后，这个请求将返回文件在服务器端保存的唯一id还包括了文件对象的其它信息，如果是图片文件，那么在文件访问时还可以加相应特效哦，具体请参见/v1/kuiper/images下的controller api说明信息）")
  @RequestMapping(path = "/{subsystem}/fileImageUpload", method = RequestMethod.POST)
  public ResponseModel fileImageUpload(@PathVariable("subsystem") String subsystem, 
                                       @RequestParam(name="effective" , required=false) @ApiParam(required=false , name="effective" , value="设定的文件有效期时长，时长单位为“天”") Integer effective ,
                                       @RequestParam(name="file" , required=true) @ApiParam(required=true , name="file" , value="上传的文件对象，提交的表单信息中，请命名为files（支持多个文件，建议不超过5个文件）") MultipartFile[] files) {
    if (files == null || files.length == 0) {
      throw new IllegalArgumentException(UPLOAD_ERROR);
    }
    Principal opUser = this.getPrincipal();
    // 依次处理每个文件
     List<OrdinaryFileEntity> imageUploadPojos = this.fileUpdateService.fileImageUpload(subsystem, opUser.getName(), effective, files);
    return this.buildHttpResult(imageUploadPojos);
  }
  
  @ApiOperation(value = "这个方法可用于一次上传多个文件附件（支持文件附件上传）", notes = ""
      + "1、只支持图片文件的上传"
      + "2、只能上传大小默认不超过2048KB的图片文件（可设置）<br>"
      + "3、上传成功后，这个请求将返回文件在服务器端保存的唯一id还包括了文件对象的其它信息，如果是图片文件，那么在文件访问时还可以加相应特效哦，具体请参见/v1/kuiper/images下的controller api说明信息）")
  @RequestMapping(path = "/{subsystem}/fileImageUploadBase64", method = RequestMethod.POST)
  public ResponseModel fileImageUpload(@PathVariable("subsystem") String subsystem, 
                                       @RequestParam(name="effective" , required=false) @ApiParam(required=false , name="effective" , value="设定的文件有效期时长，时长单位为“天”") Integer effective ,
                                       @RequestParam(name="fileNames" , required=true) @ApiParam(required=true , name="fileNames" , value="上传的文件名，包括扩展名并且不包括相对路径") String[] fileNames,
                                       @RequestParam(name="base64Contents" , required=true) @ApiParam(required=true , name="base64Contents" , value="用base64编码的文件内容（一个或者多个）") String[] base64Contents) {
    if (fileNames == null || fileNames.length == 0) {
      throw new IllegalArgumentException(UPLOAD_ERROR);
    }
    // 依次处理每个文件
    try {
      Principal opUser = this.getPrincipal();
       List<OrdinaryFileEntity> imageUploadPojos = this.fileUpdateService.fileImageUpload(subsystem, opUser.getName(), effective, fileNames, base64Contents);
      return this.buildHttpResult(imageUploadPojos);
    } catch(Exception e) {
      return this.buildHttpResultForException(e);
    }
  }
  
  @ApiOperation(value = "将指定文件的有效期设置为“永久”，实际上有效期截止时间将设置为3999-01-01 00:00:00", notes = "文件信息是文件存储的相对路径 + 重命名后的文件名。例如/file/20190402/3/e18ab4f4-9fad-46c7-9754-fdfc3f08c488.jpg")
  @RequestMapping(path = "/updatePermanentEffective", method = RequestMethod.PATCH)
  public ResponseModel updatePermanentEffective(@RequestParam(name="fileNanmes" , required=true) @ApiParam(required=true , name="fileNanmes" , value="带有文件后缀，且带有文件相对路径的的文件名(一个或者多个)") String[] fileNanmes) {
    try {
      this.fileUpdateService.updateEffective(fileNanmes);
      return this.buildHttpResult();
    } catch(Exception e) {
      return this.buildHttpResultForException(e);
    }
  }
  
  /**
   * 按照给定的文件路径和文件名，下载一般性文件。
   * @param response
   * @param relativeLocal
   * @param fileName
   * @throws IOException
   */
  @ApiOperation(value = "按照给定的文件路径和文件名，下载一般性文件。")
  @RequestMapping(value = "/download", method = RequestMethod.GET)
  public void download(HttpServletRequest request, HttpServletResponse response ,
                       @ApiParam(name = "relativeLocal", value = "相对路径", required = true) @RequestParam("relativeLocal") String relativeLocal,
                       @ApiParam(name = "fileName", value = "文件名", required = true) @RequestParam("fileName") String fileName) throws IOException {
    byte[] bytes = fileViewService.fileQuery(relativeLocal, fileName);
    if (bytes == null) {
      return;
    }
    OrdinaryFileEntity ordinaryFile = this.fileViewService.findByFileNameAndRelativeLocal(fileName, relativeLocal);
    if(ordinaryFile == null) {
      return;
    }
    // 附件文件下载
    this.writeResponseFile(request, response, bytes, ordinaryFile.getOriginalFileName());
  }

}
