package com.bizunited.platform.core.controller;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.bizunited.platform.core.controller.model.ResponseModel;
import com.bizunited.platform.core.entity.RemoteServiceEntity;
import com.bizunited.platform.core.service.RemoteServiceService;

import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.web.bind.annotation.*;

/**
 * RemoteSourceController 远端调用源服务
 *
 * @description:
 * @author: yanwe
 * @date: 05/May/2019 11:14
 */
@RestController
@RequestMapping("/v1/nebula/remoteServices")
public class RemoteServiceController extends BaseController {
  /** 日志 */
  private static final Logger LOGGER = LoggerFactory.getLogger(RemoteServiceController.class);

  @Autowired private RemoteServiceService remoteServiceService;

  /**
   * 测试一个新的远端调用服务
   *
   * @param entity 指定远端调用服务基本信息
   * @param paramJson 测试远端调用服务参数结构
   */
  @ApiOperation(value = "测试一个新的远端调用源地址")
  @RequestMapping(value = "/valid", method = RequestMethod.POST)
  public ResponseModel valid(
      @ApiParam(name = "entity", value = "指定远端调用服务基本信息") @RequestParam(name = "entity") String entity,
      @ApiParam(name = "paramJson", value = "测试远端调用服务参数结构") @RequestParam(name = "paramJson") String paramJson) {
    try {
      JSON resultJSON = this.remoteServiceService.valid(entity, paramJson);
      return this.buildHttpResultW(resultJSON);
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 执行一个远端调用服务
   * @param remoteServiceId 指定远端调用服务id
   * @param paramJson 测试远端调用服务参数结构
   */
  @ApiOperation(value = "执行一个远端调用服务")
  @RequestMapping(value = "/execute", method = RequestMethod.POST)
  public ResponseModel execute(
      @ApiParam(name = "code", value = "指定远端调用服务业务编号") @RequestParam(name = "code") String code,
      @ApiParam(name = "paramJson", value = "执行远端调用服务参数结构") @RequestParam(name = "paramJson") String paramJson) {
    try {
      JSON resultJSON = this.remoteServiceService.execute(code, paramJson);
      return this.buildHttpResultW(resultJSON);
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 创建一个新的远端调用服务
   *
   * @param entity 指定远端调用服务基本信息
   * @param paramJson 新增远端调用服务参数结构
   */
  @ApiOperation(value = "创建一个新的远端调用源地址")
  @RequestMapping(value = "", method = RequestMethod.POST)
  public ResponseModel create(
      @ApiParam(name = "entity", value = "指定远端调用服务基本信息") @RequestParam(name = "entity")
          String entity,
      @ApiParam(name = "paramJson", value = "新增远端调用服务参数结构") @RequestParam(name = "paramJson")
          String paramJson) {
    try {
      RemoteServiceEntity remoteServiceAddressEntity =
          this.remoteServiceService.create(entity, paramJson);
      return this.buildHttpResultW(remoteServiceAddressEntity);
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 更新一个新的远端调用服务
   *
   * @param entity 指定远端调用服务基本信息
   * @param paramJson 更新远端调用服务参数结构
   */
  @ApiOperation(value = "更新一个新的远端调用源地址")
  @RequestMapping(value = "/update", method = RequestMethod.POST)
  public ResponseModel update(
      @ApiParam(name = "entity", value = "指定远端调用服务基本信息") @RequestParam(name = "entity")
          String entity,
      @ApiParam(name = "paramJson", value = "更新远端调用服务参数结构") @RequestParam(name = "paramJson")
          String paramJson) {
    try {
      RemoteServiceEntity remoteServiceAddressEntity =
          this.remoteServiceService.update(entity, paramJson);
      return this.buildHttpResultW(remoteServiceAddressEntity);
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 禁用用远端调用服务
   *
   * @param id
   * @return
   */
  @ApiOperation(value = "禁用用远端调用服务")
  @RequestMapping(
      value = "/disable",
      method = {RequestMethod.POST})
  public ResponseModel disable(@RequestParam(name = "id") @ApiParam("id") String id) {
    try {
      RemoteServiceEntity entity = this.remoteServiceService.disable(id);
      return this.buildHttpResultW(entity);
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 启用用远端调用服务
   * @return
   */
  @PostMapping("/enable")
  @ApiOperation("启用用远端调用服务")
  public ResponseModel enable(@RequestParam(name = "id") @ApiParam("id") String id) {
    try {
      RemoteServiceEntity entity = this.remoteServiceService.enable(id);
      return this.buildHttpResultW(entity);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 导入json格式
   * @param jsonStr
   * @return
   */
  @ApiOperation(value = "导入模板：由用户手动输入一段json，并让其作为http接口的json映射结构，进行导入")
  @RequestMapping(
      value = "/importJson",
      method = {RequestMethod.POST})
  public ResponseModel importJson(@RequestParam(name = "jsonStr") @ApiParam("jsonStr") String jsonStr) {
    try {
      JSONArray jsonArray = this.remoteServiceService.importJson(jsonStr);
      return this.buildHttpResultW(jsonArray);
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
  
  /** 分页查询，根据编号，别名 */
  @ApiOperation(value = "分页查询，根据编号，别名")
  @RequestMapping(value = "/findByConditions", method = {RequestMethod.GET})
  public ResponseModel findByConditions(
      @RequestParam(value = "remoteServiceType", required = false) @ApiParam("remoteServiceType") Integer remoteServiceType,
      @RequestParam(value = "addressCode", required = false) @ApiParam("addressCode") String addressCode,
      @RequestParam(value = "addressAlias", required = false) @ApiParam("addressAlias") String addressAlias,
      @RequestParam(value = "requestMethod", required = false) @ApiParam("requestMethod") String requestMethod,
      @RequestParam(value = "remoteServiceStatus", required = false) @ApiParam("remoteServiceStatus") Integer remoteServiceStatus,
      Pageable pageable) {
    try {
      Page<RemoteServiceEntity> page = this.remoteServiceService.findByConditions(remoteServiceType, addressCode, addressAlias, requestMethod,remoteServiceStatus, pageable);
      return this.buildHttpResultW(page, new String[] {"remoteServiceAddress"});
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "获取草稿，按照业务编号。注意这个json中的信息不回包括head、请求参数设定等信息")
  @RequestMapping(value = "/findJsonByCode", method = {RequestMethod.GET})
  public ResponseModel findJsonByCode(@RequestParam(name = "code") @ApiParam(name = "code" , value="系统唯一的业务编号信息") String code) {
    try {
      String json = this.remoteServiceService.findJsonByCode(code);
      return this.buildHttpResultW(json);
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "根据业务编号查询保存在文件中的格式JSON，注意这个json信息是包括head信息、请求参数设定等的完整信息")
  @RequestMapping(value = "/findCompleteByCode", method = {RequestMethod.GET})
  public ResponseModel findCompleteByCode(@RequestParam(name = "code") @ApiParam(name="code" , value="系统唯一的业务编号信息") String code) {
    try {
      JSON json = this.remoteServiceService.findCompleteByCode(code);
      return this.buildHttpResultW(json);
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
}