package com.bizunited.platform.core.repository;

import com.bizunited.platform.core.entity.RoleEntity;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.JpaSpecificationExecutor;
import org.springframework.data.jpa.repository.Modifying;
import org.springframework.data.jpa.repository.Query;
import org.springframework.data.repository.query.Param;
import org.springframework.stereotype.Repository;

import java.util.List;
import java.util.Set;

/**
 * @author yinwenjie
 */
@Repository("RoleRepository")
public interface RoleRepository
    extends
      JpaRepository<RoleEntity, String>,
      JpaSpecificationExecutor<RoleEntity> {
  
  /**
   * 按照指定的用户id，删除目前这个用户已经有的绑定信息
   * @param userId
   */
  @Modifying
  @Query(value="delete from engine_role_user_mapping where user_id = :userId" , nativeQuery=true)
  void deleteRoleByUserId(@Param("userId") String userId);
  /**
   * 形成用户和角色的绑定关系，如果重复绑定，则会报错
   * @param userId
   * @param roleId
   */
  @Modifying
  @Query(value="insert into engine_role_user_mapping(role_id , user_id) values (:roleId , :userId)" , nativeQuery=true)
  void bindUser(@Param("userId") String userId ,@Param("roleId") String roleId);
  /**
   * 形成功能和角色的绑定关系，如果重复绑定，则会报错
   * @param competenceId
   * @param roleId
   */
  @Modifying
  @Query(value="insert into engine_role_competence_mapping(role_id , competence_id) values (:roleId , :competenceId)" , nativeQuery=true)
  void bindCompetence(@Param("competenceId") String competenceId ,@Param("roleId") String roleId);
  /**
   * 查询符合角色状态的信息
   * @param useStatus
   * @return
   */
  List<RoleEntity> findByTstatus(@Param("useStatus") Integer useStatus);
  /**
   * 按照填写的角色业务编号进行判断
   * @param roleCode
   * @return
   */
  RoleEntity findByRoleCode(String roleCode);
  /**
   * 查询指定的后台用户所绑定的角色信息(只包括角色基本信息)
   * @return
   */
  @Query("select distinct r from RoleEntity r left join fetch r.users a where a.id = :userId")
  List<RoleEntity> findByUserId(@Param("userId") String userId);
  /**
   * 查询指定的功能描述所绑定的角色信息
   * @param competenceId 功能描述信息
   * @return 
   */
  @Query("select distinct r from RoleEntity r inner join r.buttons b inner join b.competences c where c.id = :competenceId")
  List<RoleEntity> findByCompetenceId(@Param("competenceId") String competenceId);
  /**
   * 按照角色名，查询指定的角色信息
   * @param roleName
   * @return
   */
  RoleEntity findByRoleName(String roleName);

  /**
   * 查询多个角色信息
   * @param ids
   * @return
   */
  @Query("from RoleEntity m where m.id in :ids")
  Set<RoleEntity> findByIds(@Param("ids") List<String> ids);

  /**
   * 查询根角色
   * @return
   */
  Set<RoleEntity> findByParentIsNull();

  /**
   * 根据角色名称模糊查询
   * @return
   */
  @Query("select u from RoleEntity u where u.roleName like CONCAT(CONCAT('%',:roleName),'%')")
  Set<RoleEntity> findByRoleNameLike(@Param("roleName") String roleName);

  /**
   * 根据岗位id查询关联的角色信息
   * @param positionId
   * @return
   */
  @Query("select distinct r from RoleEntity r left join fetch r.positions a where a.id = :positionId")
  Set<RoleEntity> findByPosition(@Param("positionId") String positionId);

  /**
   * 查询岗位绑定的角色
   * @param positionIds
   * @return
   */
  @Query("select distinct r from RoleEntity r left join fetch r.positions a where a.id in :positionIds")
  Set<RoleEntity> findByPositions(@Param("positionIds") List<String> positionIds);

  /**
   * 查询指定的角色编号和按钮编号的是否有且只有一个绑定关系
   * @param roleId 指定的角色编号
   * @param buttonId 指定的按钮编号
   * @return
   */
  @Query(value="select count(*) from engine_role_button_mapping where role_id = :roleId and button_id = :buttonId", nativeQuery=true)
  long countByRoleIdAndButtonId(@Param("roleId") String roleId, @Param("buttonId") String buttonId);

  /**
   * 建立指定角色和指定按钮的绑定关系
   * @param roleId 指定的角色编号信息
   * @param buttonId 指定的按钮编号
   */
  @Modifying
  @Query(value="insert into engine_role_button_mapping(role_id,button_id) values (:roleId,:buttonId)", nativeQuery=true)
  void bindButton(@Param("roleId") String roleId, @Param("buttonId") String buttonId);

  /**
   * 取消指定角色和指定按钮的绑定关系
   * @param roleId 指定的角色编号信息
   * @param buttonId 指定的按钮编号
   */
  @Modifying
  @Query(value="delete from engine_role_button_mapping where role_id = :roleId and button_id = :buttonId", nativeQuery=true)
  void unbindButton(@Param("roleId") String roleId, @Param("buttonId") String buttonId);
}
