package com.bizunited.platform.core.repository.dataview.analysis;

import com.bizunited.platform.core.common.PlatformContext;
import com.bizunited.platform.core.common.enums.ClassTypeEnum;
import com.bizunited.platform.core.common.enums.SQLCorrelationEnum;
import com.bizunited.platform.core.common.utils.ParamsAnalysisUtil;
import com.bizunited.platform.core.entity.DataViewAuthHorizontalEntity;
import com.bizunited.platform.core.entity.DataViewAuthInterceptorEntity;
import com.bizunited.platform.core.entity.DataViewAuthVerticalEntity;
import com.bizunited.platform.core.entity.DataViewEntity;
import com.bizunited.platform.core.entity.DataViewFieldEntity;
import com.bizunited.platform.core.entity.DataViewFilterEntity;
import com.bizunited.platform.core.entity.DataViewSystemEntity;
import com.bizunited.platform.core.interceptor.DataViewAuthInterceptor;
import com.bizunited.platform.core.service.dataview.model.SQLParamModel;
import com.bizunited.platform.core.service.invoke.model.InvokeParams;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.gradle.internal.impldep.com.google.common.collect.Sets;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.data.domain.Pageable;
import org.springframework.util.CollectionUtils;

import javax.persistence.Query;
import java.math.BigDecimal;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;

/**
 * sql处理的抽象类，主要用于sql的通用处理，如sql的条件等
 * @Author: Paul Chan
 * @Date: 2019-10-09 17:08
 */
public abstract class AbstractSqlAnalysis implements SqlAnalysis {

  private static final Logger LOGGER = LoggerFactory.getLogger(AbstractSqlAnalysis.class);

  protected static final String ERROR_SQL = "SQL执行传入的的参数";

  protected static final String ERROR_INTERCEPTOR_IMPLEMENTS = "权限拦截器类必须实现接口：com.bizunited.platform.core.interceptor.DataViewAuthInterceptor";

  /**所有的字段信息*/
  protected Set<DataViewFieldEntity> allFileds;

  /**分页信息*/
  protected Pageable pageable;

  /**视图信息*/
  protected DataViewEntity dataView;

  /**视图用户筛选条件*/
  protected Set<DataViewFilterEntity> filters;

  /**视图系统参数筛选条件*/
  protected Set<DataViewSystemEntity> systemParams;

  /**视图横向权限*/
  protected Set<DataViewAuthHorizontalEntity> authHorizontals;

  /**视图竖向权限*/
  protected Set<DataViewAuthVerticalEntity> authVerticals;

  /**
   * 权限拦截器
   */
  protected DataViewAuthInterceptorEntity dataViewAuthInterceptor;

  /**视图调用传参*/
  protected InvokeParams params;

  /**SQL分析结果*/
  protected Map<SQLCorrelationEnum,Object> result;

  @Override
  public void analysis() {
    SQLAuthVerticalAnalysis.buildSQLAuthVertical(authVerticals, dataView.getSourceSql(), result, allFileds);
    String sql = result.get(SQLCorrelationEnum.RESULT_SQL).toString();
    SQLSystemParamAnalysis.buildSystemParamAnalysis(systemParams, params, result, sql);
    SQLConditionFilterAnalysis.buildSQLConditions(filters, result, params, this);
    SQLAuthHorizontalAnalysis.buildSQLAuthHorizoncal(authHorizontals, result);
    SQLOrderbyAnalysis.buildSQLConditionsOrderBy(filters, params, result);
    this.interceptor();
  }

  /**
   * 拼接sql
   * @param strs
   * @return
   */
  abstract String concatSql(String... strs);

  public AbstractSqlAnalysis(DataViewEntity dataView,Set<DataViewFilterEntity> filters,Set<DataViewSystemEntity> systemParams,
                             Set<DataViewAuthHorizontalEntity> authHorizontals,Set<DataViewAuthVerticalEntity> authVerticals,InvokeParams params,Pageable pageable,
                             Set<DataViewFieldEntity> allFileds, DataViewAuthInterceptorEntity dataViewAuthInterceptor){
    this.dataView = dataView;
    this.filters = filters;
    this.systemParams = systemParams;
    this.authHorizontals = authHorizontals;
    this.authVerticals = authVerticals;
    this.allFileds = allFileds;
    this.params = params;
    this.pageable = pageable;
    this.dataViewAuthInterceptor = dataViewAuthInterceptor;
    this.initResultMap();
  }

  /**
   * 初始化值信息
   * @return
   */
  private void initResultMap() {
    result = new HashMap<>();
    result.put(SQLCorrelationEnum.RESULT_SQL, Constants.EMPTY_CHAR);
    result.put(SQLCorrelationEnum.CONDITION_VALUES, new LinkedHashSet<SQLParamModel>(4));
    result.put(SQLCorrelationEnum.SYSTEM_PARAM_VALUES, new LinkedHashSet<SQLParamModel>(4));
    result.put(SQLCorrelationEnum.AUTH_HORIZONTAL_VALUES, new LinkedHashSet<SQLParamModel>(4));
    result.put(SQLCorrelationEnum.AUTH_VERTICAL_VALUES, new LinkedHashSet<SQLParamModel>(4));
    result.put(SQLCorrelationEnum.COUNTER, 1);
  }

  @Override
  public Map<SQLCorrelationEnum, Object> getResult() {
    return result;
  }

  @Override
  public Pageable getPageable() {
    return pageable;
  }

  /**
   * 设置预置信息
   * @param systems
   * @param authHorzontals
   * @param verticalPresets
   */
  @Override
  public void setPresets(Map<String,Object> systems , Map<String,Object> authHorzontals, Map<Integer,Object> verticalPresets){
    Map<String,Object> values = new HashMap<>();
    if(!CollectionUtils.isEmpty(systems)){
      values.putAll(systems);
    }
    if(!CollectionUtils.isEmpty(authHorzontals)){
      values.putAll(authHorzontals);
    }
    if(!CollectionUtils.isEmpty(verticalPresets)){
      result.put(SQLCorrelationEnum.VERTICAL_PRESETS,verticalPresets);
    }
    result.put(SQLCorrelationEnum.PRESETS,values);
  }

  @SuppressWarnings("unchecked")
  public void processSQLParams(Object object){
    Validate.notNull(object,"传入的Query或PreparedStatement对象不能为空！");
    Set<SQLParamModel> systemPart = (LinkedHashSet<SQLParamModel>)result.get(SQLCorrelationEnum.SYSTEM_PARAM_VALUES);
    Set<SQLParamModel> conditionPart = (LinkedHashSet<SQLParamModel>)result.get(SQLCorrelationEnum.CONDITION_VALUES);
    Set<SQLParamModel> authHorizontalPart = (LinkedHashSet<SQLParamModel>)result.get(SQLCorrelationEnum.AUTH_HORIZONTAL_VALUES);
    if(!CollectionUtils.isEmpty(systemPart)){
      this.setSQLParams(object,systemPart);
    }

    if(!CollectionUtils.isEmpty(conditionPart)){
      this.setSQLParams(object , conditionPart);
    }

    if(!CollectionUtils.isEmpty(authHorizontalPart)){
      this.setSQLParams(object , authHorizontalPart);
    }
  }

  protected void setSQLParams(Object obj , Set<SQLParamModel> params){
    Validate.notEmpty(params,"当前传递的SQL参数不能为空，请检查!!");
    Validate.notNull(obj,"JPA Query或PrepareStatement对象不能为空，请检查!!");
    for(SQLParamModel model : params){
      Validate.notNull(model,"当前传递的SQL参数不能为空，请检查!!");
      Object value = this.transData(model);
      Validate.notNull(value,"转换结果不能为空！");
      if(obj instanceof Query){
        matchJPAParamType((Query)obj,value,model);
      }else if(obj instanceof PreparedStatement){
        matchPreparedStatementParamType((PreparedStatement)obj,value,model);
      }else{
        throw new IllegalArgumentException("未知的对象信息，请检查!!");
      }
    }
  }

  protected Object transData(SQLParamModel model){
    Validate.notNull(model,"SQL参数不能为空，请检查!!");
    Validate.notNull(model.getTransferType(),"参数传递类型值不能为空，请检查!!");
    switch (model.getTransferType()){
      case 1 : //外部传入
      case 2 : //固定值
        return ParamsAnalysisUtil.doTrans(model.getValue(),model.getMappingType());
      case 3 : //预制值
        return ParamsAnalysisUtil.doTrans(model.getValue(),model.getMappingType());
      default:
        throw new IllegalArgumentException("未知的参数传递方式，请检查!!");
    }
  }

  /**
   * JPA方式的传参处理分析
   * @param query
   * @param value
   * @param model
   */
  public static void matchJPAParamType(Query query, Object value, SQLParamModel model){
    Validate.notNull(query,"传入query对象不能为空！");
    Validate.notNull(value,"传入数据对象不能为空！");
    Validate.notNull(model,"匹配JPA方式解析时，传入SQL参数数据对象不能为空！");
    Class<?> clz = value.getClass();
    String className = clz.getName();
    Validate.isTrue(Constants.ALL_SUPPORT_CLASS.contains(className) , ERROR_SQL + model.getParamName() + "有误!!未知的类型!!");
    try {
      matchJPAParamTypeAnalysis(query, value, model, className);
    } catch(Exception e) {
      throw new IllegalArgumentException(ERROR_SQL + model.getParamName()+"有误!!未知的类型!!");
    }
  }

  protected static void matchJPAParamTypeAnalysis(Query query , Object value , SQLParamModel model , String className) {
    switch (className) {
      case "java.lang.String" :
        query.setParameter(model.getIndex(), value.toString());
        break;
      case "int" :
      case "java.lang.Integer" :
        query.setParameter(model.getIndex(), ((Number)value).intValue());
        break;
      case "double" :
      case "java.lang.Double" :
        query.setParameter(model.getIndex(), ((Number)value).doubleValue());
        break;
      case "float":
      case "java.lang.Float":
        query.setParameter(model.getIndex(), ((Number)value).floatValue());
        break;
      case "long":
      case "java.lang.Long":
        query.setParameter(model.getIndex(), ((Number)value).longValue());
        break;
      case "short":
      case "java.lang.Short":
        query.setParameter(model.getIndex(), ((Number)value).shortValue());
        break;
      case "byte":
      case "java.lang.Byte":
        query.setParameter(model.getIndex(), ((Number)value).byteValue());
        break;
      case "boolean":
      case "java.lang.Boolean":
        query.setParameter(model.getIndex(), value);
        break;
      case "char":
      case "java.lang.Character":
        query.setParameter(model.getIndex(), value.toString());
        break;
      case "java.util.Date" :
        query.setParameter(model.getIndex(), value);
        break;
      case "java.math.BigDecimal" :
        query.setParameter(model.getIndex(), value);
        break;
      default:
        query.setParameter(model.getIndex(), value);
        break;
    }
  }

  /**
   * PreparedStatement传参方式的分析处理
   * @param st
   * @param value
   * @param model
   */
  public static void matchPreparedStatementParamType(PreparedStatement st,Object value ,SQLParamModel model){
    Validate.notNull(st,"传入PreparedStatement对象不能为空！");
    Validate.notNull(value,"传入数据对象不能为空！");
    Validate.notNull(model,"匹配PreparedStatement方式解析时，传入SQL参数数据对象不能为空！");
    Class<?> clz = value.getClass();
    String className = clz.getName();
    Validate.isTrue(Sets.newHashSet(Constants.ALL_SUPPORT_CLASS).contains(className) , ERROR_SQL + model.getParamName()+"有误!!未知的类型!!");
    try {
      matchPreparedStatementParamTypeAnalysis(st, value, model, className);
    } catch (SQLException e) {
      throw new IllegalArgumentException(ERROR_SQL + model.getParamName()+"有误!!未知的类型!!");
    }
  }

  /**
   * 对参数值进行数据类型判断，并设置值
   * @param st
   * @param value
   * @param model
   * @param className
   * @throws SQLException
   */
  protected static void matchPreparedStatementParamTypeAnalysis(PreparedStatement st, Object value , SQLParamModel model , String className) throws SQLException {
    switch (className) {
      case "java.lang.String":
        st.setString(model.getIndex(), value.toString());
        break;
      case "int":
      case "java.lang.Integer":
        st.setInt(model.getIndex(), ((Number) value).intValue());
        break;
      case "double":
      case "java.lang.Double":
        st.setDouble(model.getIndex(), ((Number) value).doubleValue());
        break;
      case "float":
      case "java.lang.Float":
        st.setFloat(model.getIndex(), ((Number) value).floatValue());
        break;
      case "long":
      case "java.lang.Long":
        st.setLong(model.getIndex(), ((Number) value).longValue());
        break;
      case "short":
      case "java.lang.Short":
        st.setShort(model.getIndex(), ((Number) value).shortValue());
        break;
      case "byte":
      case "java.lang.Byte":
        st.setByte(model.getIndex(), ((Number) value).byteValue());
        break;
      case "boolean":
      case "java.lang.Boolean":
        st.setBoolean(model.getIndex(), (Boolean) value);
        break;
      case "char":
      case "java.lang.Character":
        st.setString(model.getIndex(), value.toString());
        break;
      case "java.util.Date":
        st.setDate(model.getIndex(), (java.sql.Date) value);
        break;
      case "java.math.BigDecimal":
        st.setBigDecimal(model.getIndex(), (BigDecimal) value);
        break;
      default:
        break;
    }
  }

  /**
   * 执行拦截器
   */
  protected void interceptor() {
    DataViewAuthInterceptor interceptor = this.getInterceptor();
    if(interceptor == null) {
      return;
    }
    interceptor.intercept(this);
  }

  /**
   * 获取拦截器
   * @return
   */
  protected DataViewAuthInterceptor getInterceptor() {
    if(dataViewAuthInterceptor == null) {
      return null;
    }
    String interceptorClass = dataViewAuthInterceptor.getInterceptor();
    if(StringUtils.isBlank(interceptorClass)) {
      return null;
    }
    Integer classType = dataViewAuthInterceptor.getClassType();
    if(ClassTypeEnum.CLASS_PATH.getType().equals(classType)) {
      // 根据类路径获取拦截器
      try {
        Class<?> clazz = Class.forName(interceptorClass);
        Validate.isTrue(DataViewAuthInterceptor.class.isAssignableFrom(clazz), ERROR_INTERCEPTOR_IMPLEMENTS);
        return (DataViewAuthInterceptor) clazz.newInstance();
      } catch (ClassNotFoundException e) {
        LOGGER.error(e.getMessage(), e);
        throw new IllegalArgumentException(String.format("未找到java类：%s", interceptorClass));
      } catch (IllegalAccessException|InstantiationException e) {
        LOGGER.error(e.getMessage(), e);
        throw new IllegalArgumentException(String.format("实例化类错误：%s", interceptorClass));
      }
    } else if (ClassTypeEnum.BEAN_NAME.getType().equals(classType)) {
      // 从ioc容器获取拦截器
      ApplicationContext applicationContext = PlatformContext.getApplicationContext();
      Validate.isTrue(applicationContext.containsBean(interceptorClass), "ioc容器中未发现bean：%s", interceptorClass);
      Object bean = applicationContext.getBean(interceptorClass);
      Validate.isTrue(DataViewAuthInterceptor.class.isAssignableFrom(bean.getClass()), ERROR_INTERCEPTOR_IMPLEMENTS);
      return (DataViewAuthInterceptor) bean;
    } else {
      throw new IllegalArgumentException(String.format("不支持的拦截器类型:%s", classType));
    }
  }


}
