package com.bizunited.platform.core.repository.dataview.analysis;

import com.bizunited.platform.core.common.enums.SQLCorrelationEnum;
import com.bizunited.platform.core.entity.DataViewAuthVerticalEntity;
import com.bizunited.platform.core.entity.DataViewAuthVerticalRelationEntity;
import com.bizunited.platform.core.entity.DataViewFieldEntity;
import com.bizunited.platform.rbac.server.vo.OrganizationVo;
import com.bizunited.platform.rbac.server.vo.PositionVo;
import com.bizunited.platform.rbac.server.vo.RoleVo;
import com.bizunited.platform.rbac.server.vo.UserGroupVo;
import com.bizunited.platform.rbac.server.vo.UserVo;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.Validate;
import org.springframework.util.CollectionUtils;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 用于数据视图之纵向数据权限条件的分析以及构造SQL工具类
 */
public class SQLAuthVerticalAnalysis {
  /**
   * 通过私有构造函数，让该类暂时不对外开发
   * 因为后期可能会对该类进行重新设计
   */
  private SQLAuthVerticalAnalysis() {
  }

  /**
   * 纵向数据权限 参数筛选条件的分析及构造
   * 默认Prestatement的构建方式(?传参)
   * 注意：使用该类前，需要把显示字段的多对多关系数据 和 经过处理后的系统参数SQL(如果没有，就是视图的原始SQL) 组装好
   *
   * @param auths
   * @param map
   * @param sql   可能是外部已经处理过的SQL，比如经过了系统参数筛选条件处理的SQL
   * @return
   */
  @SuppressWarnings("unchecked")
  public static void buildSQLAuthVertical(Set<DataViewAuthVerticalEntity> auths, String sql, Map<SQLCorrelationEnum, Object> map, Set<DataViewFieldEntity> allFileds) {
    if (map == null) {
      map = new HashMap<>();
    }

    if (CollectionUtils.isEmpty(auths)) {
      map.put(SQLCorrelationEnum.RESULT_SQL, String.format("select %s from (%s) %s where 1=1 ", "*", sql, Constants.ALIAS_STRING));
      return;
    }

    //开始处理数据视图的纵向权限逻辑
    Set<String> finalFieldNames = new HashSet<>();
    Map<Integer, Object> presets = (Map<Integer, Object>) map.get(SQLCorrelationEnum.VERTICAL_PRESETS);
    Map<Integer, Set<String>> verticalDispalyFields = new HashMap<>();
    for (int i = 1; i <= Constants.AUTH_TYPE_COUNT; i++) {
      final Integer authType = i;
      //获取相同权限类型的数据
      Set<DataViewAuthVerticalEntity> sameAuths = auths.stream().filter(e -> authType.compareTo(e.getAuthType()) == 0).collect(Collectors.toSet());
      if (CollectionUtils.isEmpty(sameAuths)) {
        continue;
      }
      Set<String> fieldNames = buildFields(sameAuths, presets, i);
      verticalDispalyFields.put(i, finalFieldNames);
      finalFieldNames.addAll(fieldNames);
    }
    map.put(SQLCorrelationEnum.DISPLAY_FIELDS, finalFieldNames);
    Validate.notEmpty(finalFieldNames, "查询字段为空，请检查sql或纵向权限配置");
    String verticalFields = String.join(",", finalFieldNames);
    map.put(SQLCorrelationEnum.RESULT_SQL, String.format("select %s from (%s) %s where 1=1 ", verticalFields, sql, Constants.ALIAS_STRING));
  }

  /**
   * 构建查询字段
   * @return
   */
  public static Set<String> buildFields(Set<DataViewAuthVerticalEntity> auths, Map<Integer, Object> presets) {
    Set<String> fieldNames = new HashSet<>();
    if(CollectionUtils.isEmpty(auths)) {
      return fieldNames;
    }
    for (int i = 1; i <= Constants.AUTH_TYPE_COUNT; i++) {
      final Integer authType = i;
      Set<DataViewAuthVerticalEntity> sameAuths = auths.stream().filter(e -> authType.compareTo(e.getAuthType()) == 0).collect(Collectors.toSet());
      if (CollectionUtils.isEmpty(sameAuths)) {
        continue;
      }
      Set<String> fields = buildFields(sameAuths, presets, i);
      fieldNames.addAll(fields);
    }
    return fieldNames;
  }

  /**
   * 构建查询字段
   * @param sameAuths
   * @param presets
   * @param index
   * @return
   */
  private static Set<String> buildFields(Set<DataViewAuthVerticalEntity> sameAuths, Map<Integer, Object> presets, Integer index) {
    Set<String> fieldNames = new HashSet<>();
    for (DataViewAuthVerticalEntity e : sameAuths) {
      Set<DataViewFieldEntity> displayFields = e.getDisplayFields();
      Validate.notEmpty(displayFields, "纵向权限中没有指定显示字段信息，请检查!!");
      Set<DataViewAuthVerticalRelationEntity> relations = e.getAuthRelations();
      if(CollectionUtils.isEmpty(relations)) {
        continue;
      }
      Object o = presets.get(index);
      boolean result = validateAuth(relations, e.getOprtType(), index, o);
      if (result) {
        fieldNames.addAll(displayFields.stream().map(DataViewFieldEntity::getFieldName).collect(Collectors.toSet()));
      }
    }
    return fieldNames;
  }

  /**
   * 验证用户关联信息是否满足权限规则
   * @param relations
   * @param oprtType
   * @param relaType
   * @param o
   * @return
   */
  @SuppressWarnings("unchecked")
  public static boolean validateAuth(Set<DataViewAuthVerticalRelationEntity> relations, String oprtType, Integer relaType, Object o) {
    if (o == null) {
      if ("EQ".equals(oprtType) || "IN".equals(oprtType)) {
        return false;
      }
      if ("NEQ".equals(oprtType) || "NIN".equals(oprtType)) {
        return true;
      }
    }
    //用户，角色，岗位，分组，组织
    Set<String> result = new HashSet<>();
    String relationType = String.valueOf(relaType);
    if (SQLCorrelationEnum.USER_RELATION.getCode().equals(relationType)) {
      UserVo user = (UserVo) o;
      Set<String> userAccount = new HashSet<>();
      userAccount.add(user.getAccount());
      result = userAccount;
    }
    if (SQLCorrelationEnum.ROLE_RELATION.getCode().equals(relationType)) {
      List<RoleVo> roles = (List<RoleVo>) o;
      result = roles.stream().map(RoleVo::getRoleCode).collect(Collectors.toSet());
    }
    if (SQLCorrelationEnum.POSITION_RELATION.getCode().equals(relationType)) {
      List<PositionVo> positions = (List<PositionVo>) o;
      result = positions.stream().map(PositionVo::getCode).collect(Collectors.toSet());
    }
    if (SQLCorrelationEnum.USER_GROUP_RELATION.getCode().equals(relationType)) {
      Set<UserGroupVo> userGroups = (HashSet<UserGroupVo>) o;
      result = userGroups.stream().map(UserGroupVo::getGroupName).collect(Collectors.toSet());
    }
    if (SQLCorrelationEnum.ORGS_RELATION.getCode().equals(relationType)) {
      Set<OrganizationVo> orgs = (HashSet<OrganizationVo>) o;
      result = orgs.stream().map(OrganizationVo::getCode).collect(Collectors.toSet());
    }
    Set<String> relationStrs = relations.stream().map(DataViewAuthVerticalRelationEntity::getAuthRelationIds).collect(Collectors.toSet());
    Sets.SetView<String> intersectionSets = Sets.intersection(result, relationStrs);
    Boolean flag = CollectionUtils.isEmpty(intersectionSets);
    if ("EQ".equals(oprtType) || "IN".equals(oprtType)) {
      //存在交集，满足包含等于规则
      if (!flag) {
        return true;
      }
    }
    if ("NEQ".equals(oprtType) || "NIN".equals(oprtType)) {
      //不存在交集，满足不包含不等于规则
      if (flag) {
        return true;
      }
    }
    return false;
  }

}
