package com.bizunited.platform.core.repository.dataview.analysis;

import com.bizunited.platform.core.common.constant.DataViewSystemParamKey;
import com.bizunited.platform.core.entity.DataViewAuthHorizontalEntity;
import com.bizunited.platform.core.entity.DataViewAuthVerticalEntity;
import com.bizunited.platform.core.entity.DataViewSystemEntity;
import com.bizunited.platform.rbac.server.service.OrganizationService;
import com.bizunited.platform.rbac.server.service.PositionService;
import com.bizunited.platform.rbac.server.service.RoleService;
import com.bizunited.platform.rbac.server.service.UserGroupService;
import com.bizunited.platform.rbac.server.service.UserService;
import com.bizunited.platform.rbac.server.vo.OrganizationVo;
import com.bizunited.platform.rbac.server.vo.PositionVo;
import com.bizunited.platform.rbac.server.vo.RoleVo;
import com.bizunited.platform.rbac.server.vo.UserGroupVo;
import com.bizunited.platform.rbac.server.vo.UserVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.security.Principal;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import static com.bizunited.platform.core.common.constant.Constants.DEFAULT_QUERY_COVER_CODE;

@Component("SQLPresetValueAnalysis")
public class SQLPresetValueAnalysis {

  @Autowired
  private UserService userService;
  @Autowired
  private PositionService positionService;
  @Autowired
  private UserGroupService userGroupService;
  @Autowired
  private OrganizationService organizationService;
  @Autowired
  private RoleService roleService;
  private static final String ANALYSIS_ERROR = "系统参数分析时，Principal信息不能为空，请检查!!";
  private static final String ANALYSIS_H_ERROR = "横向权限分析时，Principal信息不能为空，请检查!!";
  private static final String ANALYSIS_V_ERROR = "纵向权限分析时，Principal信息不能为空，请检查!!";
  private static final String ANALYSIS_P_ERROR = "预置信息分析时，Principal信息不能为空，请检查!!";

  /**
   * 系统参数筛选时，获取指定的预制标识值的预制信息
   *
   * @param systems
   * @param principal
   * @return
   */
  public Map<String, Object> systemPresetValues(Set<DataViewSystemEntity> systems, Principal principal) {
    if (CollectionUtils.isEmpty(systems)) {
      return null;
    }
    Validate.notNull(principal, ANALYSIS_ERROR);
    Validate.notBlank(principal.getName(), ANALYSIS_ERROR);
    Map<String, Object> presets = new HashMap<>();
    Set<DataViewSystemEntity> presetSystems = systems.stream().filter(e -> e.getParamSourceType() == 3).collect(Collectors.toSet());
    for (DataViewSystemEntity element : presetSystems) {
      //当系统参数设置为预制信息的方式时，需要遍历预制标识信息ParamValue
      Object o = this.getPresetValueByParamKey(element.getParamValue(), principal);
      String key = element.getParamName() + "|DataViewSystemEntity";
      presets.put(key, o);
    }
    return presets;
  }


  /**
   * 横向权限筛选时，获取指定的预制标识值的预制信息
   *
   * @param authHorizontals
   * @param principal
   * @return
   */
  public Map<String, Object> authHorizontalPresetValues(Set<DataViewAuthHorizontalEntity> authHorizontals, Principal principal) {
    if (CollectionUtils.isEmpty(authHorizontals)) {
      return null;
    }
    Validate.notNull(principal, ANALYSIS_H_ERROR);
    Validate.notBlank(principal.getName(), ANALYSIS_H_ERROR);
    Map<String, Object> presets = new HashMap<>();
    Set<DataViewAuthHorizontalEntity> presetAuths = authHorizontals.stream().filter(e -> e.getParamSourceType() == 3).collect(Collectors.toSet());
    for (DataViewAuthHorizontalEntity auth : presetAuths) {
      //当系统参数设置为预制信息的方式时，需要遍历预制标识信息ParamValue
      if (auth.getParamSourceType() == 3) {
        Object o = this.getPresetValueByParamKey(auth.getParamKey(), principal);
        String key = auth.getField().getFieldName() + "|DataViewAuthHorizontalEntity";
        presets.put(key, o);
      }
    }
    return presets;
  }

  /**
   * 纵向权限时，获取当前人信息
   *
   * @param authVerticals
   * @param principal
   * @return
   */
  public Map<Integer, Object> authVerticalPresetValues(Set<DataViewAuthVerticalEntity> authVerticals, Principal principal) {
    if (CollectionUtils.isEmpty(authVerticals)) {
      return null;
    }
    Validate.notNull(principal, ANALYSIS_V_ERROR);
    Validate.notBlank(principal.getName(), ANALYSIS_V_ERROR);
    Map<Integer, Object> presets = new HashMap<>();
    for (int i = 1; i <= Constants.AUTH_TYPE_COUNT; i++) {
      final Integer authType = i;
      //获取相同权限类型的数据
      Set<DataViewAuthVerticalEntity> sameAuths = authVerticals.stream().filter(e -> authType.compareTo(e.getAuthType()) == 0).collect(Collectors.toSet());
      if (CollectionUtils.isEmpty(sameAuths)) {
        continue;
      }
      Integer key = i;
      Object value = this.getPresetValueByAuthType(principal, i);
      presets.put(key, value);
    }
    return presets;
  }


  /**
   * 用于查询固定值情况，纵向才会使用
   * 因为纵向的本身就是当前用户人
   */
  private Object getPresetValueByAuthType(Principal principal, int authType) {
    Validate.notNull(principal, ANALYSIS_P_ERROR);
    Validate.notBlank(principal.getName(), ANALYSIS_P_ERROR);
    String account = principal.getName();
    Validate.notBlank(account, "未能获取到用户的账户信息，请检查!!");
    UserVo currentUser = this.userService.findByAccount(account);
    Validate.notNull(currentUser, "非法的用户信息，请检查!!");
    switch (authType) {
      case 1://用户
        return currentUser;
      case 2://角色
        List<RoleVo> roles = this.roleService.findAllByUserId(currentUser.getId(), 0);
        return roles;
      case 3://岗位
        List<PositionVo> positions = this.positionService.findByUserId(currentUser.getId());
        return positions;
      case 4://用户组
        Set<UserGroupVo> groups = userGroupService.findByUserId(currentUser.getId());
        return groups;
      case 5://组织机构
        Set<OrganizationVo> orgs = organizationService.findOrgByUserId(currentUser.getId());
        return orgs;
      default:
        return null;
    }
  }

  /**
   * 根据预置值的标识信息，获取对应的预置值
   * 当控件设置了日期的预置值时，暂时忽略前端设置参数类型，统一返回日期Date类型，以备后续的逻辑处理
   */
  private Object getPresetValueByParamKey(String paramKey, Principal principal) {
    Validate.notNull(principal, ANALYSIS_P_ERROR);
    Validate.notBlank(principal.getName(), ANALYSIS_P_ERROR);
    Validate.notBlank(paramKey, "预置控件中，未能发现预置值标识信息，请检查!!");
    String account = principal.getName();
    Validate.notBlank(account, "未能获取到用户的账户信息，请检查!!");
    UserVo currentUser = this.userService.findByAccount(account);
    Validate.notNull(currentUser, "非法的用户信息，请检查!!");
    Object result = null;
    switch (paramKey) {
      //用户相关
      case DataViewSystemParamKey.OPT_USER_ID:
        result = currentUser.getId();
        break;
      case DataViewSystemParamKey.OPT_USER_NAME:
        result = currentUser.getUserName();
        break;
      case DataViewSystemParamKey.OPT_USER_ACCOUNT:
        result = currentUser.getAccount();
        break;
      case DataViewSystemParamKey.OPT_USER_CODE:
        result = currentUser.getId();
        break;
      case DataViewSystemParamKey.OPT_USER_PHONE:
        result = currentUser.getPhone();
        break;
      case DataViewSystemParamKey.OPT_USER_STATUS:
        result = currentUser.getUseStatus();
        break;
      case DataViewSystemParamKey.OPT_USER_IDCARD:
        result = currentUser.getIdcard();
        break;

      //岗位相关
      case DataViewSystemParamKey.OPT_USER_POSITION_ID:
      case DataViewSystemParamKey.OPT_USER_POSITION:
      case DataViewSystemParamKey.OPT_USER_POSITION_CODE:
      case DataViewSystemParamKey.OPT_USER_POSITION_STATUS:
        List<String> res = this.getPositionByUserId(currentUser.getId());
        if(CollectionUtils.isEmpty(res)) {
          res = Lists.newArrayList(DEFAULT_QUERY_COVER_CODE);
        }
        result = res;
        break;

      //组织机构相关
      case DataViewSystemParamKey.OPT_USER_ORG_STATUS:
      case DataViewSystemParamKey.OPT_USER_ORG_DESC:
      case DataViewSystemParamKey.OPT_USER_ORG_TYPE:
        List<Object> orgInfos =  this.findOrgInfosByUserId(currentUser.getId(), paramKey);
        result = orgInfos;
        break;
      case DataViewSystemParamKey.OPT_USER_ORG:
      case DataViewSystemParamKey.OPT_USER_ORG_ID:
      case DataViewSystemParamKey.OPT_CUR_ORGS:
      case DataViewSystemParamKey.OPT_PARENT_ORGS:
      case DataViewSystemParamKey.OPT_PARENT_ALL_ORGS:
      case DataViewSystemParamKey.OPT_CHILDREN_ORGS:
      case DataViewSystemParamKey.OPT_CHILDREN_ALL_ORGS:
        List<String> orgIds = this.findOrgsByUserId(currentUser.getId(), paramKey);
        if(CollectionUtils.isEmpty(orgIds)) {
          orgIds = Lists.newArrayList(DEFAULT_QUERY_COVER_CODE);
        }
        result = orgIds;
        break;
      //用户组相关
      case DataViewSystemParamKey.OPT_USER_GROUP:
        Set<UserGroupVo> groups = userGroupService.findByUserId(currentUser.getId());
        List<String> userGroups = groups.stream().map(UserGroupVo::getId).collect(Collectors.toList());
        if(CollectionUtils.isEmpty(userGroups)) {
          userGroups = Lists.newArrayList(DEFAULT_QUERY_COVER_CODE);
        }
        result = userGroups;
        break;

      //用户角色相关
      case DataViewSystemParamKey.OPT_USER_ROLE:
        List<RoleVo> roles = this.roleService.findAllByUserId(currentUser.getId(), 0);
        List<String> roleIds = roles.stream().map(RoleVo::getId).collect(Collectors.toList());
        if(CollectionUtils.isEmpty(roleIds)) {
          roleIds = Lists.newArrayList(DEFAULT_QUERY_COVER_CODE);
        }
        result = roleIds;
        break;


      //时间日期相关
      case DataViewSystemParamKey.CURRENT_USER_LOG_OUT_TIME:
        result = currentUser.getLastloginTime();
        break;
      case DataViewSystemParamKey.CURRENT_USER_LOGIN_TIME:
        result = currentUser.getEntryTime();
        break;
      case DataViewSystemParamKey.NOW_DATE_MONTH:
      case DataViewSystemParamKey.NOW_DATE_DAY:
      case DataViewSystemParamKey.NOW_DATE_MINUTE:
      case DataViewSystemParamKey.NOW_DATE_SECOND:
        result = new Date();
        break;
      default:
        break;
    }
    return result;
  }

  /**
   * 获取当前用户的组织机构信息
   * @param userId
   * @param paramKey
   * @return
   */
  private List<Object> findOrgInfosByUserId(String userId, String paramKey) {
    Set<OrganizationVo> orgs = organizationService.findOrgByUserId(userId);
    List<Object> result = new ArrayList<>();
    if(CollectionUtils.isEmpty(orgs)) {
      return result;
    }
    switch (paramKey) {
      case DataViewSystemParamKey.OPT_USER_ORG_STATUS:
        result = orgs.stream().map(OrganizationVo::getTstatus).collect(Collectors.toList());
        break;
      case DataViewSystemParamKey.OPT_USER_ORG_DESC:
        result = orgs.stream().map(OrganizationVo::getDescription).collect(Collectors.toList());
        break;
      case DataViewSystemParamKey.OPT_USER_ORG_TYPE:
        result = orgs.stream().map(OrganizationVo::getType).collect(Collectors.toList());
        break;
      default:
        break;
    }
    return result;
  }


  private List<String> getPositionByUserId(String id) {
    Validate.notBlank("获取岗位信息时，传入的用户id不能为空，请检查!!");
    PositionVo position = this.positionService.findMainPositionByUserId(id);
    Validate.notNull(position, "预制信息处理时，没有获取到岗位信息，请检查!!");
    return Lists.newArrayList(position.getId());
  }

  /**
   * 获取用户的组织机构信息
   * @param userId
   * @param paramKey
   * @return
   */
  private List<String> findOrgsByUserId(String userId, String paramKey) {
    Validate.notBlank("获取组织信息时，传入的用户id不能为空，请检查!!");
    Set<OrganizationVo> orgs;
    switch (paramKey) {
      case DataViewSystemParamKey.OPT_CUR_ORGS:
      case DataViewSystemParamKey.OPT_USER_ORG:
      case DataViewSystemParamKey.OPT_USER_ORG_ID:
        orgs = organizationService.findOrgByUserId(userId);
        break;
      case DataViewSystemParamKey.OPT_PARENT_ORGS:
        orgs = this.findParentOrgsByUserId(userId);
        break;
      case DataViewSystemParamKey.OPT_PARENT_ALL_ORGS:
        orgs = this.findAllParentOrgsByUserId(userId);
        break;
      case DataViewSystemParamKey.OPT_CHILDREN_ORGS:
        orgs = this.findChildrenOrgsByUserId(userId);
        break;
      case DataViewSystemParamKey.OPT_CHILDREN_ALL_ORGS:
        orgs = this.findAllChildrenOrgsByUserId(userId);
        break;
      default:
        orgs = Sets.newHashSet();
        break;
    }
    if(!CollectionUtils.isEmpty(orgs)) {
      return orgs.stream().map(OrganizationVo::getId).collect(Collectors.toList());
    }
    return Lists.newArrayList();
  }

  /**
   * 获取用户的上级组织
   * @param userId
   * @return
   */
  private Set<OrganizationVo> findParentOrgsByUserId(String userId) {
    Set<OrganizationVo> orgs = organizationService.findOrgByUserId(userId);
    if (CollectionUtils.isEmpty(orgs)) {
      return orgs;
    }
    Set<OrganizationVo> parents = new HashSet<>(16);
    for (OrganizationVo org : orgs) {
      OrganizationVo parent = organizationService.findByChild(org.getId());
      if (parent != null) {
        parents.add(parent);
      }
    }
    return parents;
  }
  /**
   * 获取用户的下级组织
   * @param userId
   * @return
   */
  private Set<OrganizationVo> findChildrenOrgsByUserId(String userId) {
    Set<OrganizationVo> orgs = organizationService.findOrgByUserId(userId);
    if (CollectionUtils.isEmpty(orgs)) {
      return orgs;
    }
    Set<OrganizationVo> Sons = new HashSet<>(16);
    for (OrganizationVo org : orgs) {
      List<OrganizationVo> son = organizationService.findByParent(org.getId());
      if (!CollectionUtils.isEmpty(son)) {
        Sons.addAll(son);
      }
    }
    return Sons;
  }

  /**
   * 获取用户的所有上级组织
   * @param userId
   * @return
   */
  private Set<OrganizationVo> findAllParentOrgsByUserId(String userId) {
    Set<OrganizationVo> orgs = organizationService.findOrgByUserId(userId);
    if (CollectionUtils.isEmpty(orgs)) {
      return orgs;
    }
    Set<OrganizationVo> parents = new HashSet<>(16);
    for (OrganizationVo org : orgs) {
      this.findParentOrg(parents, org.getId());
    }
    return parents;
  }
  /**
   * 获取用户的所有下级组织
   * @param userId
   * @return
   */
  private Set<OrganizationVo> findAllChildrenOrgsByUserId(String userId) {
    Set<OrganizationVo> orgs = organizationService.findOrgByUserId(userId);
    if (CollectionUtils.isEmpty(orgs)) {
      return orgs;
    }
    Set<OrganizationVo> Sons = new HashSet<>(16);
    for (OrganizationVo org : orgs) {
      this.findChildrenOrgs(Sons, org.getId());
    }
    return Sons;
  }

  /**
   * 寻找一个组织机构的所有父级组织机构
   * @param orgs
   * @param childOrgId
   */
  private void findParentOrg(Set<OrganizationVo> orgs, String childOrgId) {
    OrganizationVo parent = organizationService.findByChild(childOrgId);
    if (parent == null) {
      return;
    }
    orgs.add(parent);
    findParentOrg(orgs, parent.getId());
  }

  /**
   * 寻找一个组织机构的所有子级组织机构
   * @param orgs
   * @param parentOrgId
   */
  private void findChildrenOrgs(Set<OrganizationVo> orgs, String parentOrgId) {
    List<OrganizationVo> children = organizationService.findByParent(parentOrgId);
    if (CollectionUtils.isEmpty(children)) {
      return;
    }
    orgs.addAll(children);
    for (OrganizationVo org : children) {
      findChildrenOrgs(orgs, org.getId());
    }

  }


}
