package com.bizunited.platform.core.service;

import java.security.Principal;
import java.util.List;
import java.util.Set;

import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;

import com.bizunited.platform.core.entity.DynamicTaskSchedulerEntity;

/**
 * 动态任务调度服务
 * @author yinwenjie
 */
public interface DynamicTaskSchedulerService {
  /**
   * 当这个进程抢占到了动态调度任务的执行权限后，通过该方法进行“可执行”任务的加载
   */
  public void loading();
  /**
   * 当这个进程没有抢占到了动态调度任务的执行权限后，通过该方法进行“可执行”任务的卸载
   */
  public void unloading();
  /**
   * 创建一个新的定时任务
   * @param task
   * @return
   */
  public DynamicTaskSchedulerEntity create(DynamicTaskSchedulerEntity task, String scriptContent, Principal principal);
  /**
   * 更新一个指定的task状态，只有状态同时处于“有效”的“未工作”任务，才能进行更新操作
   * @param task
   * @return
   */
  public DynamicTaskSchedulerEntity update(DynamicTaskSchedulerEntity task, String scriptContent, Principal principal);
  /**
   * 将指定的一个或者多个动态任务的状态变更为“失效”
   * @param taskCodes 可以一次传入多个需要强制停止的动态任务编号
   */
  public void updateInvalidByTaskCode(String[] taskCodes);
  /**
   * 该方法对当前已存在于系统中的基于DynamicTaskService注解的java method动态执行方法（集合形态），参照currentScanTasks进行更新处理</br>
   * 通过两个集合交集、差集对基于DynamicTaskService注解的java method动态执行方法进行添加、修改、删除操作</p>
   * 注意，该方法只对基于DynamicTaskService注解的java method动态执行方法起作用，对基于groovy脚本的执行内容无效；另外该服务也不会对WEB接口层开放</br>
   * 另外注意，该方法只会在定时器模块启动前执行，以及该方法只会对当前服务名一致的任务产生影响
   * @param currentScanTasks
   */
  public void updateExsitReflexTasks(List<DynamicTaskSchedulerEntity> currentScanTasks);
  /**
   * 停止指定的任务，只有处于“运行中”的任务能够被停止，
   *由于当前的“轻质请求”可能不来自于master节点，所以这里只是更改任务状态，以便master节点下一次任务状态扫描时会停止该节点
   * @param taskCode
   */
  public void interrupt(String taskCode);
  /**
   * 启动指定的任务，由于当前的“启动请求”可能不来自于master节点，所以这里只是更改任务状态，以便master节点下一次任务状态扫描时会启动该节点</br>
   * 无效任务也可以使用该方法，一旦使用无效任务就会变成有效任务
   * @param taskCode
   */
  public void start(String taskCode);
  /**
   * 查询当前已设定所有动态任务，无论其状态如何；并且返回信息按照创建时间倒序排列。
   * 查询信息还包括了其所有的直接关联信息
   * @return
   */
  public Set<DynamicTaskSchedulerEntity> findAll();
  /**
   * 按照指定的任务业务编号，查询动态任务详细情况。只包括动态任务的基本信息
   * @param taskCode 指定的任务详细情况编号
   * @return
   */
  public DynamicTaskSchedulerEntity findByTaskCode(String taskCode);
  /**
   * -该方法用于按照分页标准，对动态任务列表进行查询
   * @param pageable 分页信息
   * @param taskType 任务类型：1、一次性执行；2：周期性执行
   * @param taskCode 任务唯一编号（只能由英文、数字、下杠构成）
   * @param tstatus 任务状态：0：无效任务；1、有效任务
   * @param workingStatus 工作状态：0：要求运行， 1：人工停止，2：已运行
   * @param taskDesc 任务中文说明（任务名）
   * @param invokeType 定时器任务内容的执行方式：1：groovy脚本执行器；2：基于注解的Java method
   * @return 
   */
  Page<DynamicTaskSchedulerEntity> findByConditions(Pageable pageable, Integer taskType , String taskCode , Integer tstatus , Integer workingStatus , String taskDesc , Integer invokeType);
}