package com.bizunited.platform.core.service;

import com.bizunited.platform.saturn.utils.TStringUtils;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.RegExUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.cglib.beans.BeanCopier;
import org.springframework.cglib.core.Converter;

import java.lang.reflect.Field;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.time.temporal.Temporal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 这是一个工具包性质的服务类，主要为了减少生成代码的工作量以及基于表单引擎完成的后端业务开发的工作量。
 * @author yinwenjie
 */
@SuppressWarnings("rawtypes")
public class NebulaToolkitService {
  
  private static final Logger LOGGER = LoggerFactory.getLogger(NebulaToolkitService.class);
  
  /**
   * 该工具方法可用于从两个相同泛型的集合中，依据指定的属性（由function确认）从两个集合中分离出“差集”
   * @param targetCollections 需要进行“差集”分析的集合
   * @param resouceCollections 依据参考的集合。
   * @param function 泛型集合中获取属性的那个方法。
   * @return targetCollections集合中没有存在于resouceCollections集合中的元素将被返回
   */
  public <T , R> Set<R> collectionDiffent(Collection<T> targetCollections , Collection<T> resouceCollections , Function<T , R> function) {
    Set<R> currentTargetCollections = new HashSet<>();
    Set<R> currentResouceCollections = new HashSet<>();
    if(targetCollections != null && !targetCollections.isEmpty()) {
      currentTargetCollections = targetCollections.stream().map(function).collect(Collectors.toSet());
    }
    if(resouceCollections != null && !resouceCollections.isEmpty()) {
      currentResouceCollections = resouceCollections.stream().map(function).collect(Collectors.toSet());
    } 
    
    // 进行差集比较
    return Sets.difference(currentTargetCollections, currentResouceCollections);
  }
  
  /**
   * 该工具方法可用于从两个相同泛型的集合中，依据指定的属性（由function确认）从两个集合中识别出“交集”
   * @param targetCollections 需要进行“交集”分析的集合
   * @param resouceCollections 依据参考的集合。
   * @param function 泛型集合中获取属性的那个方法，这个方法的值将作为元素在交集比较过程中所呈现的值。
   * @return targetCollections集合中存在，且存在于resouceCollections集合中的元素将被返回。
   */
  public <T , R> Set<R> collectionIntersection(Collection<T> targetCollections , Collection<T> resouceCollections , Function<T , R> function) {
    Set<R> currentTargetCollections = new HashSet<>();
    Set<R> currentResouceCollections = new HashSet<>();
    if(targetCollections != null && !targetCollections.isEmpty()) {
      currentTargetCollections = targetCollections.stream().map(function).collect(Collectors.toSet());
    }
    if(resouceCollections != null && !resouceCollections.isEmpty()) {
      currentResouceCollections = resouceCollections.stream().map(function).collect(Collectors.toSet());
    }
    
    // 进行交集
    return Sets.intersection(currentTargetCollections, currentResouceCollections);
  }
  
  /**
   * 在业务代码的处理逻辑中，我们经常会遇上判定当前调用者提交的明细集合中的数据，哪些明细数据项是需要进行修改操作的、哪些明细数据项需要进行删除操作，哪些明细数据项需要进行新增操作。</p>
   * 该工具方法专门提供给业务开发人员，通过最简易的代码段就可以在任何需要的地方实现以上需求逻辑。</p>
   * 
   * 请看以下举例：例如要找到业务表A中，若干明细项B，哪些需要进行添加操作、哪些需要进行修改操作、哪些需要进行删除操作，那么可以使用以下代码直接实现：</p>
   * 
   * <code>
   * Set<B> deleteCollections = new HashSet<>();</br>
   * Set<B> updateCollections = new HashSet<>();</br>
   * Set<B> createCollections = new HashSet<>();</br>
   * nebulaToolkitService.collectionDiscrepancy(requestBCollections , dbBCollections , B::getId , deleteCollections , updateCollections , createCollections);</br>
   * // 调用成功后，deleteCollections将存储那些需要被删除的明细项</br>
   * // 调用成功后，updateCollections将存储那些需要被修改的明细项</br>
   * // 调用成功后，createCollections将存储那些需要被添加的明细项</p>
   * </code>
   * 
   * @param <T> 当前作为参照函数的返回值类型
   * @param <E> 使用该方法的业务模型类型必须在技术中台进行了注册
   * @param currentCollections 当前要进行CUD操作类型拆分的原始集合，从业务开发的角度讲，该集合一般来自于前端页面的提交的各种明细项集合数据
   * @param referenceCollections 当前进行CUD操作类型拆分的参照集合，从业务开发的角度讲，该集合一般来自于从数据持久层查询出来的之前已经存在的明细项集合信息
   * @param function 参照函数，一般以getId为典型函数代表
   * @param deleteCollections 当前currentCollections集合中不存在但是referenceCollections集合中存在的信息，将从referenceCollections集合中复制引用到deleteCollections集合
   * @param updateCollections 当前currentCollections集合中存在，referenceCollections集合中也存在的信息，将从currentCollections集合中复制引用到updateCollections集合
   * @param createCollections 当前存在于currentCollections集合中，但是其function参照函数获取的值为null，或者为“空字符串”（只对String起作用）；或者currentCollections集合中存在但是referenceCollections集合中不存在的信息，这些数据从currentCollections集合中复制引用到createCollections集合
   */
  public <T,E> void  collectionDiscrepancy(Collection<E> currentCollections , Collection<E> referenceCollections , Function<E , T> function , Collection<E> deleteCollections , Collection<E> updateCollections, Collection<E> createCollections) {
    Validate.notNull(currentCollections, "当前进行CUD操作类型拆分的原始集合不能为空!!");
    Validate.notNull(function , "指定的参照函数不能为空!!");
    Validate.notNull(deleteCollections , "拆分分析完成后，用于承装“删除数据”的集合不能为空!!");
    Validate.notNull(updateCollections , "拆分分析完成后，用于承装“修改数据”的集合不能为空!!");
    Validate.notNull(createCollections , "拆分分析完成后，用于承装“新增数据”的集合不能为空!!");
    Validate.isTrue(referenceCollections.stream().filter(item -> {
      Object result = function.apply(item);
      if(result == null) {
        return true;
      }
      return (result instanceof String && StringUtils.isBlank((String)result));
    }).count() == 0l , "在进行拆分分析时，参照集合referenceCollections中各元素针对函数式[function]的返回值不能为空，请检查!!");
    
    // 1、首先判定删除集合
    // currentCollections集合中不存在但是referenceCollections集合中存在的信息
    Set<T> deleteIds = this.collectionDiffent(referenceCollections , currentCollections, function);
    if(deleteIds != null) {
      List<E> deleteResult = referenceCollections.stream().filter(currentItem -> deleteIds.contains(function.apply(currentItem))).collect(Collectors.toList());
      deleteCollections.addAll(deleteResult);
    }
    
    // 2、然后判定修改集合
    // 当前currentCollections集合中存在，referenceCollections集合中也存在的信息
    Collection<T> updateIds = this.collectionIntersection(currentCollections, referenceCollections, function);
    if(updateIds != null) {
      List<E> updateResult =  referenceCollections.stream().filter(currentItem -> updateIds.contains(function.apply(currentItem))).collect(Collectors.toList());
      updateCollections.addAll(updateResult);
    }
    
    // 3、最后找到哪些要新增的集合
    // 第一种情况：当前存在于currentCollections集合中，但是其function参照函数获取的值为null，或者为“空字符串”（只对String起作用）
    List<E> createResult1 = currentCollections.stream().filter(item -> {
      Object result = function.apply(item);
      if(result == null) {
        return true;
      }
      return (result instanceof String && StringUtils.isBlank((String)result));
    }).collect(Collectors.toList());
    // 第二种情况：currentCollections集合中存在但是referenceCollections集合中不存在的信息
    Set<T> createIds = this.collectionDiffent(currentCollections, referenceCollections , function);
    List<E> createResult2 = null;
    if(createIds != null) {
      createResult2 = currentCollections.stream().filter(currentItem -> createIds.contains(function.apply(currentItem))).collect(Collectors.toList());
    }
    if(createResult1 != null) {
      createCollections.addAll(createResult1);
    }
    if(createResult2 != null) {
      createCollections.addAll(createResult2);
    }
  }
  
  /**
   * 该工具方法可完成对象到对象的中度复制过程。这里解释如下：
   * 
   * 对象与对象存在依赖关系，例如A对象中依赖了B对象，且还依赖了一个C对象的集合。
   * 那么目前的转换工具（PropertyUtils,BeanUtils,BeanCopier等）默认都只提供单机转换，也就是说可以把基于X1类的A对象转换为基于X2的A1对象，
   * 但是A对象的中对于B对象、C对象集合的引用关系并不会发生变化(称为浅层复制)。<p>
   * 
   * 举个例子，基于X1的类定义的A对象有以下属性定义：</P>
   * class X1 {</br>
   * &nbsp;&nbsp;private String p1;</br>
   * &nbsp;&nbsp;private Date p2;</br>
   * &nbsp;&nbsp;private Integer p3;</br>
   * &nbsp;&nbsp;private float p4;</br>
   * &nbsp;&nbsp;private B p5;</br>
   * &nbsp;&nbsp;private Set&lt;C&gt; p6;</br>
   * }</P>
   * 
   * 基于X2的类定义的A1对象有以下属性定义：</P>
   * class X2 {</br>
   * &nbsp;&nbsp;private String p1;</br>
   * &nbsp;&nbsp;private Date p2;</br>
   * &nbsp;&nbsp;private Long p3;</br>
   * &nbsp;&nbsp;private Float p4;</br>
   * &nbsp;&nbsp;private B p5;</br>
   * &nbsp;&nbsp;private Set&lt;C&gt; p6;</br>
   * }</p>
   * 
   * 那么一般的基于X1类的A对象转换为基于X2的A1对象的浅层复制的效果如下</p>
   * 
   * X1:A1(p1[对象地址1],p2[对象地址2],p3[对象地址3],p4[对象地址4],p5[对象地址5],p6[对象地址6])<br>
   * -&gt; <br>
   * X2:A2(p1[对象地址1],p2[对象地址2],p3[null],p4[null],p5[对象地址5],p6[对象地址6]) </p>
   * 
   * 我们来分析一下形成以上浅层复制效果的现象：<br>
   * a)、p1、p2属性在浅层复制后保持应用地址不变的原因是可以知晓的，这相当于我们在业务代码手动书写如下代码片段：<br>
   * a2.setP2(a1.getP2());<br>
   * b)、p3、p4属性在浅层复制后为null的原因也是可以知晓的，这主要是因为复制前后p3、p4属性类型是无法对应的，甚至一些浅层复制工具本身并不支持自动装拆箱<br>
   * c)、p5、p6属性在浅层复制后，也将被引用同样的对象地址。虽然这在理论上是正确的，但并不是开发人员想要的效果。原因如下:<p>
   * 
   * ==========================================================<p>
   * 
   * <b>在普遍的业务系统开发过程中存在大量的因为VO&lt;-&gt;PO、VO&lt;-&gt;DTO、PO&lt;-&gt;DTO转换编写的机械化代码。</b>
   * 这些对象模型中的属性和对象间的关联结构都基本类型，但类的定义又完全不同。例如如下两个类结构定义：<p>
   * 
   * 以下是一个VO，这个VO类记为AVO，其中关联了另外两个视图蹭的类定义：BVO和CVO；<br>
   * class AVO {</br>
   * &nbsp;&nbsp;private String p1;</br>
   * &nbsp;&nbsp;private Date p2;</br>
   * &nbsp;&nbsp;private Integer p3;</br>
   * &nbsp;&nbsp;private float p4;</br>
   * &nbsp;&nbsp;private BVO p5;</br>
   * &nbsp;&nbsp;private Set&lt;CVO&gt; p6;</br>
   * }</P>
   * 
   * 同时在数据持久层存在AVO、BVO、CVO对应的类定义，分别为APO、BPO、CPO，并且这些类定义中拥有基本相似的属性定义和依赖结构，如下所示：<br>
   * class APO {</br>
   * &nbsp;&nbsp;private String p1;</br>
   * &nbsp;&nbsp;private Date p2;</br>
   * &nbsp;&nbsp;private Integer p3;</br>
   * &nbsp;&nbsp;private float p4;</br>
   * &nbsp;&nbsp;private BPO p5;</br>
   * &nbsp;&nbsp;private Set&lt;CPO&gt; p6;</br>
   * }</P>
   * 
   * <b>很显然在业务系统中要求进行AVO -&gt; APO的属性拷贝过程中，我们并不是要BPO和CPO最终的结果为null，而是要BVO的基本属性值拷贝到BPO中，CVO的基本属性值拷贝到CPO中。
   * 。而这样的对象自动复制过程，前文提到的对象拷贝工具都不支持，需要在成熟的工具基础上自行实现。</b></p>
   * 
   * 本工具方法支持以下特性的对象复制操作：<br>
   * 1、在对象复制过程中，建立基本类型（Integer、Float等）、常用类型（String、StringBuffer）、常用工具类型（Date、BigDecimal等）的直接引用传递<br>
   * 2、在对象复制过程中，支持基本类型的自动装拆箱过程<br>
   * 3、在对象复制过程中，支持已在Kuiper注册的模型类型的自动映射（既是上文提到的BVO映射为BPO——只要它们在上层模型中属性名定义相同）。<br>
   * 4、支持特性3中，以Set、List性质存在的模型集合。<br>
   * 注意：至于模型类如何在系统启动时，向Kuiper表单引擎注册，请参见相关技术文档（《Kuiper表单引擎用户手册》），使用示例如下：<p>
   * 
   * <code>
   * // currentStudent对象是一个StudentEntity类的实例<br>
   * StudentVO vo = this.copyObjectByWhiteList(currentStudent, StudentVO.class, LinkedHashSet.class, ArrayList.class, new String[]{"master","master.creator","achievements"});<br>
   * // 以上工具方法的调用，可以将currentStudent中的属性复制到一个新的StudentVO类的实例中，并作为方法调用的返回值进行返回。<br>
   * // 其中的currentStudent参数是复制数据的来源；StudentVO.class表示复制目标的类实例；LinkedHashSet表示当复制过程中遇到Set性质的关联模型属性，则使用哪一个Set接口的具体实现来创建新的集合；<br>
   * // ArrayList表示当复制过程中遇到List性质的关联模型属性，则使用哪一个List接口的具体实现来创建新的集合；<br>
   * // 最后一个参数为白名单列表，指示位于currentStudent对象中的那些关联模型属性需要进行中度赋值。在以上示例白名单指定了三个关联属性，<br>
   * // 既是currentStudent（StudentEntity类）对象中的master属性（MasterEntity类）、master属性继续关联的creator属性（UserEntity类）、currentStudent对象中的achievements属性（AchievementEntity类）
   * </code>
   * 
   * @param sourceObject 需要进行赋值的数据来源
   * @param targetClass 最终要复制成的目标类，如果中度复制成功则输出的对象就是这个类的具体实例
   * @param setCollectionClass 如果在对象中度复制过程中，遇到了set集合性质的关联模型对象集合，则使用该参数给定的Set.class创建一个新的Set集合，例如LinkedHashSet.class、HashSet.class
   * @param listCollectionClass 如果在对象中度复制过程中，遇到了list集合性质的关联模型对象集合，则使用该参数给定的List.class创建一个新的List集合，例如ArrayList.class、LinkedList.class
   * @param propertiesFilter 由于对象与对象的依赖关系可能存在多层，甚至可能存在循环依赖的情况，所以为了避免不必要的深层赋值以及循环复制，所以需要这个白名单指定那些需要进行中度复制的关联模型。请参见上文中的具体使用说明
   */
  public <T> T copyObjectByWhiteList(Object sourceObject , Class<T> targetClass , Class<? extends Set> setCollectionClass , Class<? extends List> listCollectionClass , String... propertiesFilter) {
    Validate.notNull(sourceObject , "进行中考时，必须传入源对象!!");
    Validate.notNull(targetClass , "进行中考时，必须传入目标类型!!");
    Class<?> sourceClass = sourceObject.getClass();
    
    T targetObject = null;
    try {
      BeanCopier copier = BeanCopier.create(sourceClass, targetClass, true);
      PropertyConverter propertyConverter = new PropertyConverter(targetClass , setCollectionClass == null?LinkedHashSet.class:setCollectionClass, listCollectionClass == null?ArrayList.class:listCollectionClass, propertiesFilter);
      targetObject = targetClass.newInstance();
      copier.copy(sourceObject, targetObject , propertyConverter);
    } catch(IllegalAccessException | InstantiationException e) {
      throw new IllegalArgumentException(e);
    }
    return targetObject;
  }
  
  /**
   * @see #copyObjectByWhiteList(Object, Class, Class, Class, String...)
   */
  @SuppressWarnings("unchecked")
  public <S , T> Collection<T> copyCollectionByWhiteList(Iterable<S> sourceCollections , Class<S> sourceClass, Class<T> targetClass, Class<? extends Set> setCollectionClass , Class<? extends List> listCollectionClass , String... propertiesFilter) {
    Validate.notNull(sourceCollections , "进行中考时，必须传入源集合对象!!");
    Validate.notNull(targetClass , "进行中考时，必须传入目标类型!!");
    Collection targetCollection = null;
    try {
      if(Set.class.isAssignableFrom(sourceCollections.getClass())) {
        targetCollection = setCollectionClass.newInstance();
      } else {
        targetCollection = listCollectionClass.newInstance();
      } 
      
      BeanCopier copier = BeanCopier.create(sourceClass, targetClass, true);
      PropertyConverter propertyConverter = new PropertyConverter(targetClass , setCollectionClass == null?LinkedHashSet.class:setCollectionClass, listCollectionClass == null?ArrayList.class:listCollectionClass, propertiesFilter);
      for (S item : sourceCollections) {
        T targetItem = targetClass.newInstance();
        copier.copy(item, targetItem, propertyConverter);
        targetCollection.add(targetItem);
      }
    } catch(IllegalAccessException | InstantiationException e) {
      throw new IllegalArgumentException(e);
    }
    
    return targetCollection;
  }
  
  private class PropertyConverter implements Converter {
    
    private Class<?> targetClass;
    
    private Class<? extends Set> setCollectionClass;
    
    private Class<? extends List> listCollectionClass;
    /**
     * 以当前递归级别中，指定属性为根节点的子树
     */
    private PropertiesFilterNode currentRoot = new PropertiesFilterNode("");
    /**
     * PropertyConverter转换器构造函数
     * @param targetClass 当前要转换成的目标类；在递归过程中，targetClass根据当前关联属性类来确认。
     * @param setCollectionClass 如果在对象中度复制过程中，遇到了set集合性质的关联模型对象集合，则使用该参数给定的Set.class创建一个新的Set集合，例如LinkedHashSet.class、HashSet.class
     * @param listCollectionClass 如果在对象中度复制过程中，遇到了list集合性质的关联模型对象集合，则使用该参数给定的List.class创建一个新的List集合，例如ArrayList.class、LinkedList.class
     * @param propertiesFilter 由于对象与对象的依赖关系可能存在多层，甚至可能存在循环依赖的情况，所以为了避免不必要的深层赋值以及循环复制，所以需要这个白名单指定那些需要进行中度复制的关联模型
     */
    public PropertyConverter(Class<?> targetClass , Class<? extends Set> setCollectionClass , Class<? extends List> listCollectionClass , String[] propertiesFilter) {
      this.targetClass = targetClass;
      this.setCollectionClass = setCollectionClass;
      this.listCollectionClass = listCollectionClass;
      this.buildPropertyTree(propertiesFilter);
    }
    
    /**
     * 这个私有构造器主要在递归Converter过程中进行使用
     */
    private PropertyConverter(Class<?> targetClass , Class<? extends Set> setCollectionClass , Class<? extends List> listCollectionClass , PropertiesFilterNode currentRoot) {
      this.targetClass = targetClass;
      this.setCollectionClass = setCollectionClass;
      this.listCollectionClass = listCollectionClass;
      this.currentRoot = currentRoot;
    }
    
    /**
     * 支持对象初始化时对指定的多级属性构造PropertiesFilterNode树形结构
     * @param propertiesFilter
     */
    private void buildPropertyTree(String[] propertiesFilter) {
      NULLNODE:for(int index = 0 ; propertiesFilter != null && index < propertiesFilter.length ; index++) {
        String propertiyFilterItem = propertiesFilter[index];
        String[] propertiyItemArrays = StringUtils.split(propertiyFilterItem, ".");
        PropertiesFilterNode currentPropertyNode = this.currentRoot; 
        
        // 采用深度模型进行树结构的生成
        for(int sum = 0 ; sum < propertiyItemArrays.length ; sum++) {
          String propertiyItem = propertiyItemArrays[sum];
          currentPropertyNode = currentPropertyNode.addAndGetChildProperty(propertiyItem);
          if(currentPropertyNode == null) {
            continue NULLNODE;
          }
        }
      }
    }
    
    /* (non-Javadoc)
     * @see org.springframework.cglib.core.Converter#convert(java.lang.Object, java.lang.Class, java.lang.Object)
     */
    @Override
    public Object convert(Object fieldValue, Class targetFieldClass, Object fieldMethodObject) {
      if(fieldValue == null) {
        return null;
      }
      Validate.notNull(targetFieldClass , "转换时目标class不能为空，请检查");
      Validate.notNull(fieldMethodObject , "元对象的set方法必须编写，请检查!!");
      /*
       * 转换方式为：
       * 1、如果当前属性的值为属于java基础类型（包括数组形式），或者常用java工具类，或者是日期类型，或者是枚举类型则直接赋值返回对象本身
       * 2、根据白名单过滤规则，确定是否还要进行下一级关联对象的中考，否则就直接返回null了
       * 3、如果当前类型是一个集合java.util.Collection，且集合中的对象类型已在Kuiper表单引擎中注册，则取出其中的每一个对象，进行递归中考
       *  (注意：规则是基于当前是黑名单或者白名单的设定，保证当前关联属性 存在/不存在于当前 过滤树的直接子级节点中)
       * 4、如果当前类属于静态模型（既是该静态模型已在Kuiper表单引擎中注册），那么就进行递归中考
       * (注意：规则是基于当前是黑名单或者白名单的设定，保证当前关联属性 存在/不存在于当前 过滤树的直接子级节点中)
       * 注意：其它目前不支持递归中考的类型：例如数组、多维度数组（[I、[[I、[[[D、[[Lclassname），则直接将当前值（fieldValue）进行返回
       * 也就是说，不再进行中考
       * */
      // 1、=======
      if(targetFieldClass.isPrimitive()
          || targetFieldClass == Byte.class || targetFieldClass == Short.class || targetFieldClass == Integer.class
          || targetFieldClass == Long.class || targetFieldClass == Float.class || targetFieldClass == Double.class 
          || targetFieldClass == Character.class || targetFieldClass == Boolean.class) {
        return fieldValue;
      }
      if(Date.class.isAssignableFrom(targetFieldClass)) {
        return fieldValue;
      }
      if(Temporal.class.isAssignableFrom(targetFieldClass)) {
        return fieldValue;
      }
      if(targetFieldClass.isEnum()) {
        return fieldValue;
      }
      
      String fieldTypeName = targetFieldClass.getName();
      if(StringUtils.startsWith(fieldTypeName, "java.lang.")
        || StringUtils.startsWith(fieldTypeName, "java.math.")
        || StringUtils.startsWith(fieldTypeName, "java.io.")
        || StringUtils.startsWith(fieldTypeName, "java.text.")) {
        return fieldValue;
      }

      // 2、=======
      // 判断黑名单/白名单的设定是否当前集合属性能够被中拷
      String fieldSetMethodName = fieldMethodObject.toString();
      String fieldGetMethodName = StringUtils.replace(fieldSetMethodName, "s", "g", 1);
      String fieldName = RegExUtils.replaceFirst(fieldGetMethodName, "get", "");
      fieldName = TStringUtils.letterLowercase(fieldName);
      // 如果满足以下过滤条件说明无需进行下一层过滤了，直接返回null
      PropertiesFilterNode nextFilterNode = null;
      if(currentRoot == null) {
        return null;
      } 
      nextFilterNode = this.currentRoot.getChildProperty(fieldName);
      if(nextFilterNode == null) {
        return null;
      }
      
      try {
        // 3、========
        if(Collection.class.isAssignableFrom(targetFieldClass)) {
          Collection results = this.convertPersistentCollectionClass(this.targetClass , fieldValue, targetFieldClass , fieldName , nextFilterNode);
          return results == null || results.isEmpty()?null:results;
        } 
        // 4、==========
        else {
          return convertPersistentClass(fieldValue, targetFieldClass , nextFilterNode);
        }
      } catch(InstantiationException | IllegalAccessException | NoSuchFieldException e) {
        throw new IllegalArgumentException(e);
      }
    }
    
    /**
     * 该方法完成和当前对象关联的另一单个对象的中考过程。为了有效反馈异常，这里不进行异常的捕获。
     */
    @SuppressWarnings("unchecked")
    private Collection convertPersistentCollectionClass(Class<?> targetClass , Object fieldValue, Class targetFieldClass , String fieldName , PropertiesFilterNode nextFilterNode) throws NoSuchFieldException, IllegalAccessException , InstantiationException {
      /*
       * 第3步的处理过程为：
       * 3.1、寻找中考的四个要素：源对象属性中的值（既是fieldValues）；源对象类型（fieldValue中的泛型）
       * 目标对象类型（基于targetFieldClass，通过targetFieldClass进行寻找的元素泛型）;一个目标类型的初始化对象(通过获取的泛型进行的初始化)
       * 3.2、新创建一个新的PropertyConverter，并循环中考
       * */
      Collection<?> fieldValues = (Collection<?>)fieldValue;
      if(fieldValues.isEmpty()) {
        return Sets.newHashSet();
      } 
      // 判定使用什么样的集合类型完成中考后的元素集中
      Collection targetFieldValues = null;
      if(Set.class.isAssignableFrom(targetFieldClass)) {
        targetFieldValues = this.setCollectionClass.newInstance();
      } else if(List.class.isAssignableFrom(targetFieldClass)) {
        targetFieldValues = this.listCollectionClass.newInstance();
      } else  {
        return Sets.newHashSet();
      }
      
      // 3.1、========
      Class<?> targetItemClass = null;
      // 取得目标类中的指定字段，注意，目标类中可能没有指定的字段
      Field field = null;
      try {
        field = targetClass.getDeclaredField(fieldName);
      } catch(NoSuchFieldException e) {
        LOGGER.warn(e.getMessage(),"在中拷时未找到目标类的属性" );
        return Sets.newHashSet();
      }
      
      Type genericType = field.getGenericType();
      if(genericType instanceof ParameterizedType) {
        ParameterizedType pt = (ParameterizedType) genericType;  
        targetItemClass = (Class<?>)pt.getActualTypeArguments()[0];
        if(targetItemClass == null) {
          return Sets.newHashSet();
        }
        String fieldTypeName = targetItemClass.getName();
        if(StringUtils.isBlank(fieldTypeName)) {
          return Sets.newHashSet();
        }
      } else {
        return Sets.newHashSet();
      }
      
      // 3.2、========循环中拷其中的每一个元素
      for (Object fieldValueItem : fieldValues) {
        Class<?> fieldValueItemClass = fieldValueItem.getClass();
        BeanCopier fieldCopier = BeanCopier.create(fieldValueItemClass, targetItemClass, true);
        Object currentTargetFieldValue = targetItemClass.newInstance();
        // 下一层对象copy
        fieldCopier.copy(fieldValueItem, currentTargetFieldValue, new PropertyConverter(targetItemClass , this.setCollectionClass, this.listCollectionClass , nextFilterNode));
        targetFieldValues.add(currentTargetFieldValue);
      }
      return targetFieldValues;
    }
    
    /**
     * 该方法完成和当前对象关联的另一多个（集合性质）对象的中考过程。为了有效反馈异常，这里不进行异常的捕获。
     */
    private Object convertPersistentClass(Object fieldValue, Class targetFieldClass, PropertiesFilterNode nextFilterNode) throws IllegalAccessException , InstantiationException {
      /*
       * 第2步的处理过程为：
       * 2.1、寻求到以下四个要素
       * 源对象属性中的值（既是fieldValue）；源对象类型（通过fieldValue中的class来获取）
       * 目标类型（既是targetFieldClass）;一个目标类型的初始化对象（通过targetFieldClass进行newinstance来获取）
       * 2.2、新创建一个新的PropertyConverter，并进行转换
       * */
      // 2.1、=========
      Object currentFieldValue = fieldValue;
      Class<?> currentFieldItemClass = fieldValue.getClass();
      Class<?> currentTargetFieldClass = targetFieldClass;
      Object currentTargetFieldValue = null;
      // 创建一个目标对象的实例
      currentTargetFieldValue = currentTargetFieldClass.newInstance();
      // 2.2、=========
      BeanCopier fieldCopier = BeanCopier.create(currentFieldItemClass, currentTargetFieldClass, true);
      // 下一层对象copy
      fieldCopier.copy(currentFieldValue, currentTargetFieldValue, new PropertyConverter(currentTargetFieldClass , LinkedHashSet.class, ArrayList.class, nextFilterNode));
      return currentTargetFieldValue;
    }
  }
  
  /**
   * 该私有类为指定的黑名单或者白名单属性构造一棵树结构，以便在递归过程中使用
   * @author yinwenjie
   */
  private class PropertiesFilterNode {
    /**
     * 当前递归层级正在使用的属性名
     */
    private String propertyName;
    /**
     * 当前级别已设置的下级关联属性
     */
    private Set<PropertiesFilterNode> childproperties = new HashSet<>();
    
    public PropertiesFilterNode(String propertyName) {
      this.propertyName = propertyName;
    }
    
    /**
     * 如果当前递归级别存在指定的子级propertyName属性，则进行返回；否则返回null
     * @param propertyName 指定的子级属性名
     */
    public PropertiesFilterNode getChildProperty(String propertyName) {
      List<PropertiesFilterNode> resultChilds = this.childproperties.stream().filter(item -> StringUtils.equals(item.getPropertyName(), propertyName)).collect(Collectors.toList());
      if(resultChilds == null || resultChilds.isEmpty()) {
        return null;
      }
      
      return resultChilds.get(0);
    }
    
    /**
     * 检查当前递归级别的子级propertyName是否存在，如果不存在则创建一个，并进行返回。
     * 如果存在则直接进行返回；
     * @param propertyName 指定的子级属性名
     * @return 除非传参出现问题，否则都不会返回null
     */
    private PropertiesFilterNode addAndGetChildProperty(String propertyName) {
      if(StringUtils.isBlank(propertyName)) {
        return null;
      }
      PropertiesFilterNode currentNode = this.getChildProperty(propertyName);
      if(currentNode != null) {
        return currentNode;
      }
      
      currentNode = new PropertiesFilterNode(propertyName);
      this.childproperties.add(currentNode);
      return currentNode;
    }
    
    public String getPropertyName() {
      return propertyName;
    }
  }
}